/*
 * The Kuali Financial System, a comprehensive financial management system for higher education.
 *
 * Copyright 2005-2023 Kuali, Inc.
 *
 * This program is free software: you can redistribute it and/or modify
 * it under the terms of the GNU Affero General Public License as published by
 * the Free Software Foundation, either version 3 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU Affero General Public License for more details.
 *
 * You should have received a copy of the GNU Affero General Public License
 * along with this program.  If not, see <http://www.gnu.org/licenses/>.
 */
package org.kuali.kfs.module.ar.businessobject.lookup;

import org.apache.commons.lang3.StringUtils;
import org.kuali.kfs.core.web.format.Formatter;
import org.kuali.kfs.kim.impl.identity.Person;
import org.kuali.kfs.kns.document.authorization.BusinessObjectRestrictions;
import org.kuali.kfs.kns.lookup.CollectionIncomplete;
import org.kuali.kfs.kns.lookup.HtmlData;
import org.kuali.kfs.kns.lookup.HtmlData.AnchorHtmlData;
import org.kuali.kfs.kns.web.comparator.CellComparatorHelper;
import org.kuali.kfs.kns.web.struts.form.LookupForm;
import org.kuali.kfs.kns.web.ui.Column;
import org.kuali.kfs.kns.web.ui.ResultRow;
import org.kuali.kfs.krad.bo.BusinessObject;
import org.kuali.kfs.krad.bo.PersistableBusinessObject;
import org.kuali.kfs.krad.util.GlobalVariables;
import org.kuali.kfs.krad.util.ObjectUtils;
import org.kuali.kfs.module.ar.ArPropertyConstants;
import org.kuali.kfs.module.ar.businessobject.CollectionActivityReport;
import org.kuali.kfs.module.ar.report.service.CollectionActivityReportService;
import org.kuali.kfs.sys.KFSConstants;

import java.util.Collection;
import java.util.HashMap;
import java.util.List;
import java.util.Map;

/**
 * LookupableHelperService class for Collection Activity Report.
 */
public class CollectionActivityReportLookupableHelperServiceImpl extends
        CollectionsReportLookupableHelperServiceImplBase {

    protected CollectionActivityReportService collectionActivityReportService;

    /**
     * Get the search results that meet the input search criteria.
     *
     * @param fieldValues Map containing prop name keys and search values
     * @return a List of found business objects
     */
    @Override
    public List<? extends BusinessObject> getSearchResultsUnbounded(final Map<String, String> fieldValues) {
        setBackLocation(fieldValues.get(KFSConstants.BACK_LOCATION));
        setDocFormKey(fieldValues.get(KFSConstants.DOC_FORM_KEY));
        final List<CollectionActivityReport> results =
                collectionActivityReportService.filterEventsForCollectionActivity(fieldValues);
        return new CollectionIncomplete<>(results, (long) results.size());
    }

    @Override
    public Collection<? extends BusinessObject> performLookup(
            final LookupForm lookupForm, final Collection<ResultRow> resultTable,
            final boolean bounded) {
        final List<? extends BusinessObject> displayList = getSearchResultsUnbounded(lookupForm.getFieldsForLookup());

        boolean hasReturnableRow = false;

        final Person user = GlobalVariables.getUserSession().getPerson();
        final List<String> pkNames = getBusinessObjectMetaDataService().listPrimaryKeyFieldNames(getBusinessObjectClass());

        // iterate through result list and wrap rows with return url and action urls
        for (final BusinessObject element : displayList) {
            final BusinessObjectRestrictions businessObjectRestrictions = getBusinessObjectAuthorizationService()
                    .getLookupResultRestrictions(element, user);

            if (ObjectUtils.isNotNull(getColumns())) {
                final List<Column> columns = getColumns();
                for (final Object column : columns) {

                    final Column col = (Column) column;
                    final Formatter formatter = col.getFormatter();

                    // pick off result column from result list, do formatting
                    final Object prop = ObjectUtils.getPropertyValue(element, col.getPropertyName());

                    String propValue = ObjectUtils.getFormattedPropertyValue(element, col.getPropertyName(),
                            col.getFormatter());
                    final Class propClass = getPropertyClass(element, col.getPropertyName());

                    // formatters
                    if (ObjectUtils.isNotNull(prop)) {
                        propValue = getContractsGrantsReportHelperService().formatByType(prop, formatter);
                    }

                    // comparator
                    col.setComparator(CellComparatorHelper.getAppropriateComparatorForPropertyClass(propClass));
                    col.setValueComparator(CellComparatorHelper.getAppropriateValueComparatorForPropertyClass(propClass));

                    propValue = super.maskValueIfNecessary(element, col.getPropertyName(), propValue,
                            businessObjectRestrictions);
                    col.setPropertyValue(propValue);

                    // Add url when property is invoiceNumber
                    if (col.getPropertyName().equals(ArPropertyConstants.INVOICE_NUMBER)) {
                        final String url = contractsGrantsReportHelperService.getDocSearchUrl(propValue);

                        final Map<String, String> fieldList = new HashMap<>();
                        fieldList.put(ArPropertyConstants.INVOICE_NUMBER, propValue);
                        final AnchorHtmlData a = new AnchorHtmlData(url, KFSConstants.EMPTY_STRING);
                        a.setTitle(HtmlData.getTitleText(getContractsGrantsReportHelperService()
                                .createTitleText(getBusinessObjectClass()), getBusinessObjectClass(), fieldList));

                        col.setColumnAnchor(a);
                    } else if (StringUtils.isNotBlank(propValue)) {
                        col.setColumnAnchor(getInquiryUrl(element, col.getPropertyName()));
                    }
                }

                final ResultRow row = new ResultRow(columns, KFSConstants.EMPTY_STRING, getActionUrls(element, pkNames,
                        businessObjectRestrictions));
                if (element instanceof PersistableBusinessObject) {
                    row.setObjectId(((PersistableBusinessObject) element).getObjectId());
                }

                final boolean rowReturnable = isResultReturnable(element);
                row.setRowReturnable(rowReturnable);
                if (rowReturnable) {
                    hasReturnableRow = true;
                }

                resultTable.add(row);
            }

            lookupForm.setHasReturnableRow(hasReturnableRow);
        }

        return displayList;
    }

    public CollectionActivityReportService getCollectionActivityReportService() {
        return collectionActivityReportService;
    }

    public void setCollectionActivityReportService(final CollectionActivityReportService collectionActivityReportService) {
        this.collectionActivityReportService = collectionActivityReportService;
    }
}
