/*
 * The Kuali Financial System, a comprehensive financial management system for higher education.
 *
 * Copyright 2005-2023 Kuali, Inc.
 *
 * This program is free software: you can redistribute it and/or modify
 * it under the terms of the GNU Affero General Public License as published by
 * the Free Software Foundation, either version 3 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU Affero General Public License for more details.
 *
 * You should have received a copy of the GNU Affero General Public License
 * along with this program.  If not, see <http://www.gnu.org/licenses/>.
 */
package org.kuali.kfs.module.ar.document;

import org.apache.commons.lang3.StringUtils;
import org.apache.logging.log4j.LogManager;
import org.apache.logging.log4j.Logger;
import org.kuali.kfs.datadictionary.legacy.DataDictionaryService;
import org.kuali.kfs.integration.ar.AccountsReceivableCustomerCreditMemo;
import org.kuali.kfs.krad.exception.ValidationException;
import org.kuali.kfs.krad.rules.rule.event.KualiDocumentEvent;
import org.kuali.kfs.krad.util.ObjectUtils;
import org.kuali.kfs.module.ar.ArConstants;
import org.kuali.kfs.module.ar.businessobject.AccountsReceivableDocumentHeader;
import org.kuali.kfs.module.ar.businessobject.CustomerCreditMemoDetail;
import org.kuali.kfs.module.ar.businessobject.CustomerInvoiceDetail;
import org.kuali.kfs.module.ar.businessobject.ReceivableCustomerCreditMemoDetail;
import org.kuali.kfs.module.ar.businessobject.ReceivableCustomerInvoiceDetail;
import org.kuali.kfs.module.ar.businessobject.SalesTaxCustomerCreditMemoDetail;
import org.kuali.kfs.module.ar.document.service.AccountsReceivableDocumentHeaderService;
import org.kuali.kfs.module.ar.document.service.AccountsReceivablePendingEntryService;
import org.kuali.kfs.module.ar.document.service.AccountsReceivableTaxService;
import org.kuali.kfs.module.ar.document.service.CustomerCreditMemoDocumentService;
import org.kuali.kfs.sys.businessobject.GeneralLedgerPendingEntry;
import org.kuali.kfs.sys.businessobject.GeneralLedgerPendingEntrySequenceHelper;
import org.kuali.kfs.sys.businessobject.GeneralLedgerPendingEntrySourceDetail;
import org.kuali.kfs.sys.businessobject.TaxDetail;
import org.kuali.kfs.sys.context.SpringContext;
import org.kuali.kfs.sys.document.AmountTotaling;
import org.kuali.kfs.sys.document.GeneralLedgerPendingEntrySource;
import org.kuali.kfs.sys.document.GeneralLedgerPostingDocumentBase;
import org.kuali.kfs.sys.service.GeneralLedgerPendingEntryService;
import org.kuali.kfs.sys.service.TaxService;
import org.kuali.kfs.sys.service.UniversityDateService;
import org.kuali.kfs.sys.util.KfsDateUtils;
import org.kuali.kfs.core.api.datetime.DateTimeService;
import org.kuali.kfs.core.api.util.type.KualiDecimal;
import org.kuali.kfs.core.web.format.CurrencyFormatter;
import org.kuali.kfs.kew.framework.postprocessor.DocumentRouteStatusChange;

import java.math.BigDecimal;
import java.math.RoundingMode;
import java.sql.Date;
import java.sql.Timestamp;
import java.util.ArrayList;
import java.util.Iterator;
import java.util.List;
import java.util.Locale;

public class CustomerCreditMemoDocument extends GeneralLedgerPostingDocumentBase implements GeneralLedgerPendingEntrySource, AmountTotaling, AccountsReceivableCustomerCreditMemo {
    private static final Logger LOG = LogManager.getLogger();

    protected String statusCode;
    protected String financialDocumentReferenceInvoiceNumber;

    protected KualiDecimal crmTotalItemAmount = KualiDecimal.ZERO;
    protected KualiDecimal crmTotalTaxAmount = KualiDecimal.ZERO;
    protected KualiDecimal crmTotalAmount = KualiDecimal.ZERO;

    protected Integer invOutstandingDays;

    protected CustomerInvoiceDocument invoice;
    protected AccountsReceivableDocumentHeader accountsReceivableDocumentHeader;

    protected List<CustomerCreditMemoDetail> creditMemoDetails;

    protected transient TaxService taxService;
    protected transient AccountsReceivableTaxService arTaxService;

    public CustomerCreditMemoDocument() {
        super();
        setPostingYear(SpringContext.getBean(UniversityDateService.class).getCurrentFiscalYear());
        creditMemoDetails = new ArrayList<>();
        setGeneralLedgerPendingEntries(new ArrayList<>());
    }

    public List<CustomerCreditMemoDetail> getCreditMemoDetails() {
        return creditMemoDetails;
    }

    public void setCreditMemoDetails(final List<CustomerCreditMemoDetail> creditMemoDetails) {
        this.creditMemoDetails = creditMemoDetails;
    }

    public String getFinancialDocumentReferenceInvoiceNumber() {
        return financialDocumentReferenceInvoiceNumber;
    }

    public void setFinancialDocumentReferenceInvoiceNumber(String financialDocumentReferenceInvoiceNumber) {
        if (financialDocumentReferenceInvoiceNumber != null) {
            financialDocumentReferenceInvoiceNumber = financialDocumentReferenceInvoiceNumber.toUpperCase(Locale.US);
        }

        this.financialDocumentReferenceInvoiceNumber = financialDocumentReferenceInvoiceNumber;
    }

    public CustomerInvoiceDocument getInvoice() {
        if (ObjectUtils.isNull(invoice) && StringUtils.isNotEmpty(financialDocumentReferenceInvoiceNumber)) {
            refreshReferenceObject("invoice");
        }

        return invoice;
    }

    public void setInvoice(final CustomerInvoiceDocument invoice) {
        this.invoice = invoice;
    }

    public String getStatusCode() {
        return statusCode;
    }

    public void setStatusCode(final String statusCode) {
        this.statusCode = statusCode;
    }

    /**
     * Initializes the values for a new document.
     */
    public void initiateDocument() {
        LOG.debug("initiateDocument() started");
        setStatusCode(ArConstants.CustomerCreditMemoStatuses.INITIATE);
    }

    /**
     * Clear out the initially populated fields.
     */
    public void clearInitFields() {
        LOG.debug("clearDocument() started");

        // Clearing document Init fields
        setFinancialDocumentReferenceInvoiceNumber(null);
    }

    public KualiDecimal getCrmTotalAmount() {
        return crmTotalAmount;
    }

    /**
     * This method returns the crmTotalAmount as a currency formatted string.
     *
     * @return String
     */
    public String getCurrencyFormattedCrmTotalAmount() {
        return (String) new CurrencyFormatter().format(crmTotalAmount);
    }

    public void setCrmTotalAmount(final KualiDecimal crmTotalAmount) {
        this.crmTotalAmount = crmTotalAmount;
    }

    public KualiDecimal getCrmTotalItemAmount() {
        return crmTotalItemAmount;
    }

    /**
     * This method returns the crmTotalItemAmount as a currency formatted string.
     *
     * @return String
     */
    public String getCurrencyFormattedCrmTotalItemAmount() {
        return (String) new CurrencyFormatter().format(crmTotalItemAmount);
    }

    public void setCrmTotalItemAmount(final KualiDecimal crmTotalItemAmount) {
        this.crmTotalItemAmount = crmTotalItemAmount;
    }

    public KualiDecimal getCrmTotalTaxAmount() {
        return crmTotalTaxAmount;
    }

    /**
     * This method returns the crmTotalTaxAmount as a currency formatted string.
     *
     * @return String
     */
    public String getCurrencyFormattedCrmTotalTaxAmount() {
        return (String) new CurrencyFormatter().format(crmTotalTaxAmount);
    }

    public void setCrmTotalTaxAmount(final KualiDecimal crmTotalTaxAmount) {
        this.crmTotalTaxAmount = crmTotalTaxAmount;
    }

    public Integer getInvOutstandingDays() {
        final Timestamp invBillingDateTimestamp = new Timestamp(invoice.getBillingDate().getTime());
        final Timestamp todayDateTimestamp = new Timestamp(SpringContext.getBean(DateTimeService.class).getCurrentSqlDate().getTime());
        final double diffInDays = KfsDateUtils.getDifferenceInDays(invBillingDateTimestamp, todayDateTimestamp);
        invOutstandingDays = new KualiDecimal(diffInDays).intValue();

        return invOutstandingDays;
    }

    public void setInvOutstandingDays(final Integer invOutstandingDays) {
        this.invOutstandingDays = invOutstandingDays;
    }

    public void recalculateTotalsBasedOnChangedItemAmount(final CustomerCreditMemoDetail customerCreditMemoDetail) {
        final KualiDecimal duplicateCreditMemoItemTotalAmount = customerCreditMemoDetail.getDuplicateCreditMemoItemTotalAmount();
        final KualiDecimal creditMemoItemTotalAmount = customerCreditMemoDetail.getCreditMemoItemTotalAmount();

        // subtract the 'old' item amount, tax amount, and total amount accordingly from totals
        if (ObjectUtils.isNotNull(duplicateCreditMemoItemTotalAmount)) {
            prepareTotalsForUpdate(duplicateCreditMemoItemTotalAmount, getArTaxService().isCustomerInvoiceDetailTaxable(getInvoice(), customerCreditMemoDetail.getCustomerInvoiceDetail()));
        }

        recalculateTotals(creditMemoItemTotalAmount, getArTaxService().isCustomerInvoiceDetailTaxable(getInvoice(), customerCreditMemoDetail.getCustomerInvoiceDetail()));

        // update duplicate credit memo item amount with 'new' value
        customerCreditMemoDetail.setDuplicateCreditMemoItemTotalAmount(creditMemoItemTotalAmount);
    }

    public void recalculateTotals(final CustomerCreditMemoDetail customerCreditMemoDetail) {
        final KualiDecimal duplicateCreditMemoItemTotalAmount = customerCreditMemoDetail.getDuplicateCreditMemoItemTotalAmount();

        // subtract the 'old' item amount, tax amount, and total amount accordingly from totals
        if (ObjectUtils.isNotNull(duplicateCreditMemoItemTotalAmount)) {
            prepareTotalsForUpdate(duplicateCreditMemoItemTotalAmount,
                    getArTaxService().isCustomerInvoiceDetailTaxable(getInvoice(),
                            customerCreditMemoDetail.getCustomerInvoiceDetail()));
            customerCreditMemoDetail.setDuplicateCreditMemoItemTotalAmount(null);
        }
    }

    public void recalculateTotals(final KualiDecimal itemAmount, final boolean isTaxableItemFlag) {
        crmTotalItemAmount = crmTotalItemAmount.add(itemAmount);
        if (isTaxableItemFlag) {
            crmTotalTaxAmount = crmTotalTaxAmount.add(getTaxService().getTotalSalesTaxAmount(invoice.getBillingDate(), getPostalCode(), itemAmount));
        }
        crmTotalAmount = crmTotalItemAmount.add(crmTotalTaxAmount);
        getDocumentHeader().setFinancialDocumentTotalAmount(crmTotalAmount);
    }

    protected void prepareTotalsForUpdate(final KualiDecimal oldItemAmount, final boolean isTaxableItemFlag) {
        KualiDecimal oldItemTaxAmount = KualiDecimal.ZERO;
        if (isTaxableItemFlag) {
            oldItemTaxAmount = getTaxService().getTotalSalesTaxAmount(invoice.getBillingDate(), getPostalCode(), oldItemAmount);
        }

        crmTotalItemAmount = crmTotalItemAmount.subtract(oldItemAmount);
        crmTotalTaxAmount = crmTotalTaxAmount.subtract(oldItemTaxAmount);
        crmTotalAmount = crmTotalAmount.subtract(oldItemAmount.add(oldItemTaxAmount));
    }

    public void resetTotals() {
        crmTotalItemAmount = KualiDecimal.ZERO;
        crmTotalTaxAmount = KualiDecimal.ZERO;
        crmTotalAmount = KualiDecimal.ZERO;
    }

    /*
     * populate customer credit memo details based on the invoice info
     */
    public void populateCustomerCreditMemoDetails() {
        CustomerCreditMemoDetail customerCreditMemoDetail;
        KualiDecimal openInvoiceAmount;

        setStatusCode(ArConstants.CustomerCreditMemoStatuses.IN_PROCESS);

        //set accounts receivable document header if not already set
        if (getAccountsReceivableDocumentHeader() == null) {
            final AccountsReceivableDocumentHeader accountsReceivableDocumentHeader = SpringContext.getBean(AccountsReceivableDocumentHeaderService.class).getNewAccountsReceivableDocumentHeaderForCurrentUser();
            accountsReceivableDocumentHeader.setDocumentNumber(getDocumentNumber());
            accountsReceivableDocumentHeader.setCustomerNumber(invoice.getAccountsReceivableDocumentHeader().getCustomerNumber());
            setAccountsReceivableDocumentHeader(accountsReceivableDocumentHeader);
        }

        final List<CustomerInvoiceDetail> customerInvoiceDetails = invoice.getCustomerInvoiceDetailsWithoutDiscounts();
        for (final CustomerInvoiceDetail customerInvoiceDetail : customerInvoiceDetails) {
            customerCreditMemoDetail = new CustomerCreditMemoDetail();

            if (ObjectUtils.isNull(customerInvoiceDetail.getInvoiceItemTaxAmount())) {
                customerInvoiceDetail.setInvoiceItemTaxAmount(KualiDecimal.ZERO);
            }
            customerCreditMemoDetail.setInvoiceLineTotalAmount(customerInvoiceDetail.getInvoiceItemTaxAmount(), customerInvoiceDetail.getInvoiceItemPreTaxAmount());
            customerCreditMemoDetail.setReferenceInvoiceItemNumber(customerInvoiceDetail.getSequenceNumber());
            openInvoiceAmount = customerInvoiceDetail.getAmountOpen();

            customerCreditMemoDetail.setInvoiceOpenItemAmount(openInvoiceAmount);
            customerCreditMemoDetail.setInvoiceOpenItemQuantity(getInvoiceOpenItemQuantity(customerCreditMemoDetail, customerInvoiceDetail));
            customerCreditMemoDetail.setDocumentNumber(documentNumber);
            customerCreditMemoDetail.setFinancialDocumentReferenceInvoiceNumber(financialDocumentReferenceInvoiceNumber);

            // this is a hookup for institution custom to update financial object code for prior year(s) invoice
            //customerInvoiceDetailService.updateFinancialObjectCode(customerCreditMemoDetail);

            creditMemoDetails.add(customerCreditMemoDetail);
        }

    }

    /**
     * This method populates credit memo details that aren't saved in database
     */
    public void populateCustomerCreditMemoDetailsAfterLoad() {
        KualiDecimal openInvoiceAmount;
        KualiDecimal creditMemoItemAmount;
        KualiDecimal creditMemoTaxAmount;

        for (final CustomerCreditMemoDetail creditMemoDetail : creditMemoDetails) {

            creditMemoDetail.setFinancialDocumentReferenceInvoiceNumber(financialDocumentReferenceInvoiceNumber);
            final CustomerInvoiceDetail customerInvoiceDetail = creditMemoDetail.getCustomerInvoiceDetail();
            openInvoiceAmount = customerInvoiceDetail.getAmountOpen();
            creditMemoDetail.setInvoiceOpenItemAmount(openInvoiceAmount);

            creditMemoDetail.setInvoiceOpenItemQuantity(getInvoiceOpenItemQuantity(creditMemoDetail, customerInvoiceDetail));

            if (ObjectUtils.isNull(customerInvoiceDetail.getInvoiceItemTaxAmount())) {
                customerInvoiceDetail.setInvoiceItemTaxAmount(KualiDecimal.ZERO);
            }
            creditMemoDetail.setInvoiceLineTotalAmount(customerInvoiceDetail.getInvoiceItemTaxAmount(), customerInvoiceDetail.getInvoiceItemPreTaxAmount());

            creditMemoItemAmount = creditMemoDetail.getCreditMemoItemTotalAmount();
            creditMemoDetail.setDuplicateCreditMemoItemTotalAmount(creditMemoItemAmount);
            if (ObjectUtils.isNotNull(creditMemoItemAmount)) {
                if (getArTaxService().isCustomerInvoiceDetailTaxable(invoice, customerInvoiceDetail)) {
                    creditMemoTaxAmount = getTaxService().getTotalSalesTaxAmount(invoice.getBillingDate(), getPostalCode(), creditMemoItemAmount);
                } else {
                    creditMemoTaxAmount = KualiDecimal.ZERO;
                }
                creditMemoDetail.setCreditMemoItemTaxAmount(creditMemoTaxAmount);
                creditMemoDetail.setCreditMemoLineTotalAmount(creditMemoItemAmount.add(creditMemoTaxAmount));

                crmTotalItemAmount = crmTotalItemAmount.add(creditMemoItemAmount);
                crmTotalTaxAmount = crmTotalTaxAmount.add(creditMemoTaxAmount);
                crmTotalAmount = crmTotalAmount.add(creditMemoItemAmount.add(creditMemoTaxAmount));
            }

            // this is a hookup for institution custom to update financial object code for prior year(s) invoice
            //customerInvoiceDetailService.updateFinancialObjectCode(creditMemoDetail);
        }
    }

    public BigDecimal getInvoiceOpenItemQuantity(final CustomerCreditMemoDetail customerCreditMemoDetail, final CustomerInvoiceDetail customerInvoiceDetail) {
        final BigDecimal invoiceOpenItemQuantity;
        final BigDecimal invoiceItemUnitPrice = customerInvoiceDetail.getInvoiceItemUnitPrice();
        if (ObjectUtils.isNull(invoiceItemUnitPrice) || invoiceItemUnitPrice.equals(BigDecimal.ZERO)) {
            invoiceOpenItemQuantity = BigDecimal.ZERO;
        } else {
            final KualiDecimal invoiceOpenItemAmount = customerCreditMemoDetail.getInvoiceOpenItemAmount();
            KualiDecimal invoiceOpenItemPretaxAmount = invoiceOpenItemAmount;
            if (getArTaxService().isCustomerInvoiceDetailTaxable(getInvoice(), customerInvoiceDetail)) {
                invoiceOpenItemPretaxAmount = getCustomerInvoiceDetailOpenPretaxAmount(invoiceOpenItemAmount);
            }

            invoiceOpenItemQuantity = invoiceOpenItemPretaxAmount.bigDecimalValue().divide(invoiceItemUnitPrice, ArConstants.ITEM_QUANTITY_SCALE,
                    RoundingMode.HALF_UP);
        }
        return invoiceOpenItemQuantity;
    }

    protected KualiDecimal getCustomerInvoiceDetailOpenPretaxAmount(final KualiDecimal openAmount) {
        final Date dateOfTransaction = getInvoice().getBillingDate();
        return SpringContext.getBean(TaxService.class).getPretaxAmount(dateOfTransaction, getPostalCode(), openAmount);
    }

    /**
     * do all the calculations before the document gets saved gets called for 'Submit', 'Save', and 'Blanket Approved'
     */
    @Override
    public void prepareForSave(final KualiDocumentEvent event) {
        final CustomerCreditMemoDocument customerCreditMemoDocument = (CustomerCreditMemoDocument) event.getDocument();
        final CustomerCreditMemoDocumentService customerCreditMemoDocumentService = SpringContext.getBean(CustomerCreditMemoDocumentService.class);
        customerCreditMemoDocumentService.recalculateCustomerCreditMemoDocument(customerCreditMemoDocument, false);

        // generate GLPEs
        if (!SpringContext.getBean(GeneralLedgerPendingEntryService.class).generateGeneralLedgerPendingEntries(this)) {
            logErrors();
            throw new ValidationException("general ledger GLPE generation failed");
        }
        super.prepareForSave(event);
    }

    @Override
    public boolean generateDocumentGeneralLedgerPendingEntries(final GeneralLedgerPendingEntrySequenceHelper sequenceHelper) {
        return true;
    }

    @Override
    public boolean isDebit(final GeneralLedgerPendingEntrySourceDetail postable) {
        return false;
    }

    @Override
    public void clearAnyGeneralLedgerPendingEntries() {
        generalLedgerPendingEntries = new ArrayList<>();
    }

    @Override
    public List<GeneralLedgerPendingEntrySourceDetail> getGeneralLedgerPendingEntrySourceDetails() {
        final List<GeneralLedgerPendingEntrySourceDetail> generalLedgerPendingEntrySourceDetails = new ArrayList<>();
        if (creditMemoDetails != null) {
            final Iterator iter = creditMemoDetails.iterator();
            CustomerCreditMemoDetail customerCreditMemoDetail;
            KualiDecimal amount;
            while (iter.hasNext()) {
                customerCreditMemoDetail = (CustomerCreditMemoDetail) iter.next();
                amount = customerCreditMemoDetail.getCreditMemoItemTotalAmount();

                // get only non empty credit memo details to generate GLPEs
                if (ObjectUtils.isNotNull(amount) && amount.isGreaterThan(KualiDecimal.ZERO)) {
                    generalLedgerPendingEntrySourceDetails.add(customerCreditMemoDetail);
                }
            }
        }
        return generalLedgerPendingEntrySourceDetails;
    }

    @Override
    public void addPendingEntry(final GeneralLedgerPendingEntry entry) {
        generalLedgerPendingEntries.add(entry);
    }

    @Override
    public KualiDecimal getGeneralLedgerPendingEntryAmountForDetail(final GeneralLedgerPendingEntrySourceDetail postable) {
        return postable.getAmount();
    }

    /**
     * Returns the general ledger input type code for the given document, using the DataDictionaryService
     *
     * @return the general ledger input type code for the given document
     */
    @Override
    public String getFinancialDocumentTypeCode() {
        return SpringContext.getBean(DataDictionaryService.class).getDocumentTypeNameByClass(getClass());
    }

    @Override
    public List<String> getWorkflowEngineDocumentIdsToLock() {
        // a credit memo wont always update the source invoice, but sometimes it will so we include it here
        if (StringUtils.isNotBlank(getFinancialDocumentReferenceInvoiceNumber())) {
            final List<String> documentIds = new ArrayList<>();
            documentIds.add(getFinancialDocumentReferenceInvoiceNumber());
            return documentIds;
        }
        return null;
    }

    /**
     * When document is processed do the following:
     * <p>
     * 1) Apply amounts to writeoff invoice
     * 2) Mark off invoice indicator
     */
    @Override
    public void doRouteStatusChange(final DocumentRouteStatusChange statusChangeEvent) {
        super.doRouteStatusChange(statusChangeEvent);
        if (getDocumentHeader().getWorkflowDocument().isProcessed()) {

            //have to populate because not all the customer credit memo details are populated while doc is in workflow
            populateCustomerCreditMemoDetailsAfterLoad();

            // apply writeoff amounts by only retrieving only the invoice details that ARE NOT discounts
            final CustomerCreditMemoDocumentService service = SpringContext.getBean(CustomerCreditMemoDocumentService.class);
            service.completeCustomerCreditMemo(this);
        }
    }

    /**
     * This method creates the following GLPE's for the customer credit memo
     * <p>
     * 1. Credit to receivable object for total line amount (excluding sales tax)
     * 2. Debit to income object for total line amount (excluding sales tax)
     * 3. Credit to receivable object in sales tax account(if sales tax exists)
     * 4. Debit to liability object in sales tax account(if sales tax exists)
     * 5. Credit to receivable object in district tax account(if district tax exists)
     * 6. Debit to liability object code in district tax account(if district tax exists)
     */
    @Override
    public boolean generateGeneralLedgerPendingEntries(
            final GeneralLedgerPendingEntrySourceDetail glpeSourceDetail,
            final GeneralLedgerPendingEntrySequenceHelper sequenceHelper) {
        addReceivableGLPEs(sequenceHelper, glpeSourceDetail);
        sequenceHelper.increment();
        addIncomeGLPEs(sequenceHelper, glpeSourceDetail);

        //if sales tax is enabled generate GLPEs
        final CustomerInvoiceDetail invoiceDetail = ((CustomerCreditMemoDetail) glpeSourceDetail).getCustomerInvoiceDetail();
        if (getArTaxService().isCustomerInvoiceDetailTaxable(getInvoice(), invoiceDetail)) {
            addSalesTaxGLPEs(sequenceHelper, glpeSourceDetail);
        }

        return true;
    }

    /**
     * This method creates the receivable GLPEs for credit memo detail lines.
     *
     * @param sequenceHelper
     * @param glpeSourceDetail
     */
    protected void addReceivableGLPEs(
            final GeneralLedgerPendingEntrySequenceHelper sequenceHelper,
            final GeneralLedgerPendingEntrySourceDetail glpeSourceDetail) {
        final CustomerCreditMemoDetail customerCreditMemoDetail = (CustomerCreditMemoDetail) glpeSourceDetail;
        final CustomerInvoiceDetail customerInvoiceDetail = customerCreditMemoDetail.getCustomerInvoiceDetail();
        final ReceivableCustomerInvoiceDetail receivableCustomerInvoiceDetail = new ReceivableCustomerInvoiceDetail(customerInvoiceDetail, getInvoice());
        final boolean isDebit = false;

        final AccountsReceivablePendingEntryService service = SpringContext.getBean(AccountsReceivablePendingEntryService.class);
        service.createAndAddGenericInvoiceRelatedGLPEs(this, receivableCustomerInvoiceDetail, sequenceHelper, isDebit, false, customerCreditMemoDetail.getCreditMemoItemTotalAmount());
    }

    /**
     * This method adds pending entry with transaction ledger entry amount set to item price * quantity
     *
     * @param sequenceHelper
     * @param glpeSourceDetail
     */
    protected void addIncomeGLPEs(
            final GeneralLedgerPendingEntrySequenceHelper sequenceHelper,
            final GeneralLedgerPendingEntrySourceDetail glpeSourceDetail) {

        final CustomerCreditMemoDetail customerCreditMemoDetail = (CustomerCreditMemoDetail) glpeSourceDetail;
        final boolean isDebit = true;

        final AccountsReceivablePendingEntryService service = SpringContext.getBean(AccountsReceivablePendingEntryService.class);
        service.createAndAddGenericInvoiceRelatedGLPEs(this, customerCreditMemoDetail, sequenceHelper, isDebit, false, customerCreditMemoDetail.getCreditMemoItemTotalAmount());
    }

    /**
     * This method add pending entries for every tax detail that exists for a particular postal code
     *
     * @param sequenceHelper
     * @param glpeSourceDetail
     */
    protected void addSalesTaxGLPEs(
            final GeneralLedgerPendingEntrySequenceHelper sequenceHelper,
            final GeneralLedgerPendingEntrySourceDetail glpeSourceDetail) {

        final CustomerCreditMemoDetail customerCreditMemoDetail = (CustomerCreditMemoDetail) glpeSourceDetail;
        final boolean isDebit = false;

        final String postalCode = getPostalCode();
        final Date dateOfTransaction = getInvoice().getBillingDate();

        final List<TaxDetail> salesTaxDetails = getTaxService().getSalesTaxDetails(dateOfTransaction, postalCode, customerCreditMemoDetail.getCreditMemoItemTotalAmount());

        final AccountsReceivablePendingEntryService service = SpringContext.getBean(AccountsReceivablePendingEntryService.class);
        SalesTaxCustomerCreditMemoDetail salesTaxCustomerCreditMemoDetail;
        ReceivableCustomerCreditMemoDetail receivableCustomerCreditMemoDetail;
        for (final TaxDetail salesTaxDetail : salesTaxDetails) {

            salesTaxCustomerCreditMemoDetail = new SalesTaxCustomerCreditMemoDetail(salesTaxDetail, customerCreditMemoDetail);
            salesTaxCustomerCreditMemoDetail.setCustomerInvoiceDetail(customerCreditMemoDetail.getCustomerInvoiceDetail());
            receivableCustomerCreditMemoDetail = new ReceivableCustomerCreditMemoDetail(salesTaxCustomerCreditMemoDetail, this);
            receivableCustomerCreditMemoDetail.setCustomerInvoiceDetail(customerCreditMemoDetail.getCustomerInvoiceDetail());

            sequenceHelper.increment();
            service.createAndAddGenericInvoiceRelatedGLPEs(this, receivableCustomerCreditMemoDetail, sequenceHelper, isDebit, false, salesTaxDetail.getTaxAmount());

            sequenceHelper.increment();
            service.createAndAddGenericInvoiceRelatedGLPEs(this, salesTaxCustomerCreditMemoDetail, sequenceHelper, !isDebit, false, salesTaxDetail.getTaxAmount());
        }
    }

    @Override
    public KualiDecimal getTotalDollarAmount() {
        return getDocumentHeader().getFinancialDocumentTotalAmount();
    }

    public AccountsReceivableDocumentHeader getAccountsReceivableDocumentHeader() {
        return accountsReceivableDocumentHeader;
    }

    public void setAccountsReceivableDocumentHeader(final AccountsReceivableDocumentHeader accountsReceivableDocumentHeader) {
        this.accountsReceivableDocumentHeader = accountsReceivableDocumentHeader;
    }

    @Override
    public void setAccountsReceivableDocumentHeader(final org.kuali.kfs.integration.ar.AccountsReceivableDocumentHeader arDocHeader) {
        accountsReceivableDocumentHeader = (org.kuali.kfs.module.ar.businessobject.AccountsReceivableDocumentHeader) arDocHeader;
    }

    public String getPostalCode() {
        return getArTaxService().getPostalCodeForTaxation(getInvoice());
    }

    public TaxService getTaxService() {
        //  lazy init the service if its been nullified by session-izing the document
        if (taxService == null) {
            taxService = SpringContext.getBean(TaxService.class);
        }
        return taxService;
    }

    public AccountsReceivableTaxService getArTaxService() {
        //  lazy init the service if its been nullified by session-izing the document
        if (arTaxService == null) {
            arTaxService = SpringContext.getBean(AccountsReceivableTaxService.class);
        }
        return arTaxService;
    }

}
