/*
 * The Kuali Financial System, a comprehensive financial management system for higher education.
 *
 * Copyright 2005-2023 Kuali, Inc.
 *
 * This program is free software: you can redistribute it and/or modify
 * it under the terms of the GNU Affero General Public License as published by
 * the Free Software Foundation, either version 3 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU Affero General Public License for more details.
 *
 * You should have received a copy of the GNU Affero General Public License
 * along with this program.  If not, see <http://www.gnu.org/licenses/>.
 */
package org.kuali.kfs.module.ar.document.dataaccess.impl;

import org.apache.commons.lang3.StringUtils;
import org.apache.logging.log4j.LogManager;
import org.apache.logging.log4j.Logger;
import org.apache.ojb.broker.query.Criteria;
import org.apache.ojb.broker.query.Query;
import org.apache.ojb.broker.query.QueryByCriteria;
import org.apache.ojb.broker.query.QueryFactory;
import org.apache.ojb.broker.query.ReportQueryByCriteria;
import org.kuali.kfs.krad.util.ObjectUtils;
import org.kuali.kfs.module.ar.ArConstants;
import org.kuali.kfs.module.ar.ArPropertyConstants;
import org.kuali.kfs.module.ar.businessobject.CustomerInvoiceDetail;
import org.kuali.kfs.module.ar.document.CustomerInvoiceDocument;
import org.kuali.kfs.module.ar.document.dataaccess.CustomerInvoiceDocumentDao;
import org.kuali.kfs.sys.KFSConstants;
import org.kuali.kfs.sys.KFSPropertyConstants;
import org.kuali.kfs.core.framework.persistence.ojb.dao.PlatformAwareDaoBaseOjb;

import java.sql.Date;
import java.util.ArrayList;
import java.util.Collection;
import java.util.HashSet;
import java.util.Iterator;
import java.util.List;
import java.util.Locale;
import java.util.Set;

public class CustomerInvoiceDocumentDaoOjb extends PlatformAwareDaoBaseOjb implements CustomerInvoiceDocumentDao {

    private static final Logger LOG = LogManager.getLogger();

    @Override
    public List<String> getPrintableCustomerInvoiceDocumentNumbersFromUserQueue() {

        final Criteria criteria = new Criteria();
        criteria.addEqualTo("printInvoiceIndicator", ArConstants.PrintInvoiceOptions.PRINT_BY_USER);
        criteria.addIsNull("printDate");
        criteria.addEqualTo("documentHeader.financialDocumentStatusCode", KFSConstants.DocumentStatusCodes.APPROVED);

        //  Why use the OJB reports approach here, rather than a list of CustomerInvoiceDocuments?
        //
        //  This was done because even if we had the invoice documents, we then need to do a proper document load
        // via the documentService, which loads up the workflow information as well and properly prepares the document.
        //
        //  Therefore, at this stage, there's no reason to load entire documents, all we need are document numbers.  And with
        // OJB, this is how you get just a collection of a single column's value out.  Given the performance issues associated
        // with live reporting like this, the attempt was made to minimize the resource usage.

        final ReportQueryByCriteria rqbc = QueryFactory.newReportQuery(CustomerInvoiceDocument.class,
                new String[]{"documentNumber"}, criteria, false);

        final Iterator<Object[]> iter = getPersistenceBrokerTemplate().getReportQueryIteratorByQuery(rqbc);
        final List<String> invoiceNumbers = new ArrayList<>();
        while (iter.hasNext()) {
            invoiceNumbers.add((String) iter.next()[0]);
        }
        return invoiceNumbers;
    }

    @Override
    public List<String> getPrintableCustomerInvoiceDocumentNumbersByProcessingChartAndOrg(
            final String chartOfAccountsCode,
            final String organizationCode) {
        if (StringUtils.isBlank(chartOfAccountsCode)) {
            throw new IllegalArgumentException("The method was called with a Null or Blank chartOfAccountsCode parameter.");
        }
        if (StringUtils.isBlank(organizationCode)) {
            throw new IllegalArgumentException("The method was called with a Null or Blank organizationCode parameter.");
        }

        //  Why use the OJB reports approach here, rather than a list of CustomerInvoiceDocuments?
        //
        //  This was done because even if we had the invoice documents, we then need to do a proper document load
        // via the documentService, which loads up the workflow information as well and properly prepares the document.
        //
        //  Therefore, at this stage, there's no reason to load entire documents, all we need are document numbers.  And with
        // OJB, this is how you get just a collection of a single column's value out.  Given the performance issues associated
        // with live reporting like this, the attempt was made to minimize the resource usage.

        // select i.fdoc_nbr
        // from ar_doc_hdr_t h inner join ar_inv_doc_t i
        //   on h.fdoc_nbr = i.fdoc_nbr
        // where h.prcs_fin_coa_cd = ? and h.prcs_org_cd = ?

        final Criteria criteria = new Criteria();
        criteria.addEqualTo("accountsReceivableDocumentHeader.processingChartOfAccountCode", chartOfAccountsCode);
        criteria.addEqualTo("accountsReceivableDocumentHeader.processingOrganizationCode", organizationCode);
        criteria.addEqualTo("printInvoiceIndicator", ArConstants.PrintInvoiceOptions.PRINT_BY_PROCESSING_ORG);
        criteria.addIsNull("printDate");
        criteria.addEqualTo("documentHeader.financialDocumentStatusCode", KFSConstants.DocumentStatusCodes.APPROVED);

        final ReportQueryByCriteria rqbc = QueryFactory.newReportQuery(CustomerInvoiceDocument.class, new String[]{"documentNumber"}, criteria, false);

        final Iterator<Object[]> iter = getPersistenceBrokerTemplate().getReportQueryIteratorByQuery(rqbc);
        final List<String> invoiceNumbers = new ArrayList<>();
        while (iter.hasNext()) {
            invoiceNumbers.add((String) iter.next()[0]);
        }
        return new ArrayList<>(invoiceNumbers);
    }

    @Override
    public List<String> getPrintableCustomerInvoiceDocumentNumbersByBillingChartAndOrg(
            final String chartOfAccountsCode,
            final String organizationCode) {
        if (StringUtils.isBlank(chartOfAccountsCode)) {
            throw new IllegalArgumentException("The method was called with a Null or Blank chartOfAccountsCode parameter.");
        }
        if (StringUtils.isBlank(organizationCode)) {
            throw new IllegalArgumentException("The method was called with a Null or Blank organizationCode parameter.");
        }

        //  Why use the OJB reports approach here, rather than a list of CustomerInvoiceDocuments?
        //
        //  This was done because even if we had the invoice documents, we then need to do a proper document load
        // via the documentService, which loads up the workflow information as well and properly prepares the document.
        //
        //  Therefore, at this stage, there's no reason to load entire documents, all we need are document numbers.  And with
        // OJB, this is how you get just a collection of a single column's value out.  Given the performance issues associated
        // with live reporting like this, the attempt was made to minimize the resource usage.

        final Criteria criteria = new Criteria();
        criteria.addEqualTo("billByChartOfAccountCode", chartOfAccountsCode);
        criteria.addEqualTo("billedByOrganizationCode", organizationCode);
        criteria.addEqualTo("printInvoiceIndicator", ArConstants.PrintInvoiceOptions.PRINT_BY_BILLING_ORG);
        criteria.addIsNull("printDate");
        criteria.addEqualTo("documentHeader.financialDocumentStatusCode", KFSConstants.DocumentStatusCodes.APPROVED);

        final ReportQueryByCriteria rqbc = QueryFactory.newReportQuery(CustomerInvoiceDocument.class, new String[]{"documentNumber"}, criteria, false);

        final Iterator<Object[]> iter = getPersistenceBrokerTemplate().getReportQueryIteratorByQuery(rqbc);
        final List<String> invoiceNumbers = new ArrayList<>();
        while (iter.hasNext()) {
            invoiceNumbers.add((String) iter.next()[0]);
        }
        return new ArrayList<>(invoiceNumbers);
    }

    /**
     * Very similar to above except lacks check for print invoice indicator and print date.
     */
    @Override
    public List<String> getPrintableCustomerInvoiceDocumentNumbersForBillingStatementByBillingChartAndOrg(
            final String chartOfAccountsCode, final String organizationCode) {
        if (StringUtils.isBlank(chartOfAccountsCode)) {
            throw new IllegalArgumentException("The method was called with a Null or Blank chartOfAccountsCode parameter.");
        }
        if (StringUtils.isBlank(organizationCode)) {
            throw new IllegalArgumentException("The method was called with a Null or Blank organizationCode parameter.");
        }

        //  Why use the OJB reports approach here, rather than a list of CustomerInvoiceDocuments?
        //
        //  This was done because even if we had the invoice documents, we then need to do a proper document load
        // via the documentService, which loads up the workflow information as well and properly prepares the document.
        //
        //  Therefore, at this stage, there's no reason to load entire documents, all we need are document numbers.  And with
        // OJB, this is how you get just a collection of a single column's value out.  Given the performance issues associated
        // with live reporting like this, the attempt was made to minimize the resource usage.

        final Criteria criteria = new Criteria();
        criteria.addEqualTo("billByChartOfAccountCode", chartOfAccountsCode);
        criteria.addEqualTo("billedByOrganizationCode", organizationCode);
        criteria.addEqualTo("documentHeader.financialDocumentStatusCode", KFSConstants.DocumentStatusCodes.APPROVED);

        final ReportQueryByCriteria rqbc = QueryFactory.newReportQuery(CustomerInvoiceDocument.class, new String[]{"documentNumber"}, criteria, false);

        final Iterator<Object[]> iter = getPersistenceBrokerTemplate().getReportQueryIteratorByQuery(rqbc);
        final List<String> invoiceNumbers = new ArrayList<>();
        while (iter.hasNext()) {
            invoiceNumbers.add((String) iter.next()[0]);
        }
        return new ArrayList<>(invoiceNumbers);
    }

    @Override
    public List<String> getCustomerInvoiceDocumentNumbersByProcessingChartAndOrg(final String chartOfAccountsCode, final String organizationCode) {
        if (StringUtils.isBlank(chartOfAccountsCode)) {
            throw new IllegalArgumentException("The method was called with a Null or Blank chartOfAccountsCode parameter.");
        }
        if (StringUtils.isBlank(organizationCode)) {
            throw new IllegalArgumentException("The method was called with a Null or Blank organizationCode parameter.");
        }

        //  Why use the OJB reports approach here, rather than a list of CustomerInvoiceDocuments?
        //
        //  This was done because even if we had the invoice documents, we then need to do a proper document load
        // via the documentService, which loads up the workflow information as well and properly prepares the document.
        //
        //  Therefore, at this stage, there's no reason to load entire documents, all we need are document numbers.  And with
        // OJB, this is how you get just a collection of a single column's value out.  Given the performance issues associated
        // with live reporting like this, the attempt was made to minimize the resource usage.

        // select i.fdoc_nbr
        // from ar_doc_hdr_t h inner join ar_inv_doc_t i
        //   on h.fdoc_nbr = i.fdoc_nbr
        // where h.prcs_fin_coa_cd = ? and h.prcs_org_cd = ?

        final Criteria criteria = new Criteria();
        criteria.addEqualTo("accountsReceivableDocumentHeader.processingChartOfAccountCode", chartOfAccountsCode);
        criteria.addEqualTo("accountsReceivableDocumentHeader.processingOrganizationCode", organizationCode);

        final ReportQueryByCriteria rqbc = QueryFactory.newReportQuery(CustomerInvoiceDocument.class, new String[]{"documentNumber"}, criteria, false);

        final Iterator<Object[]> iter = getPersistenceBrokerTemplate().getReportQueryIteratorByQuery(rqbc);
        final List<String> invoiceNumbers = new ArrayList<>();
        while (iter.hasNext()) {
            invoiceNumbers.add((String) iter.next()[0]);
        }
        return new ArrayList<>(invoiceNumbers);
    }

    @Override
    public List<String> getCustomerInvoiceDocumentNumbersByBillingChartAndOrg(final String chartOfAccountsCode, final String organizationCode) {
        if (StringUtils.isBlank(chartOfAccountsCode)) {
            throw new IllegalArgumentException("The method was called with a Null or Blank chartOfAccountsCode parameter.");
        }
        if (StringUtils.isBlank(organizationCode)) {
            throw new IllegalArgumentException("The method was called with a Null or Blank organizationCode parameter.");
        }

        //  Why use the OJB reports approach here, rather than a list of CustomerInvoiceDocuments?
        //
        //  This was done because even if we had the invoice documents, we then need to do a proper document load
        // via the documentService, which loads up the workflow information as well and properly prepares the document.
        //
        //  Therefore, at this stage, there's no reason to load entire documents, all we need are document numbers.  And with
        // OJB, this is how you get just a collection of a single column's value out.  Given the performance issues associated
        // with live reporting like this, the attempt was made to minimize the resource usage.

        final Criteria criteria = new Criteria();
        criteria.addEqualTo("billByChartOfAccountCode", chartOfAccountsCode);
        criteria.addEqualTo("billedByOrganizationCode", organizationCode);

        final ReportQueryByCriteria rqbc = QueryFactory.newReportQuery(CustomerInvoiceDocument.class, new String[]{"documentNumber"}, criteria, false);

        final Iterator<Object[]> iter = getPersistenceBrokerTemplate().getReportQueryIteratorByQuery(rqbc);
        final List<String> invoiceNumbers = new ArrayList<>();
        while (iter.hasNext()) {
            invoiceNumbers.add((String) iter.next()[0]);
        }
        return new ArrayList<>(invoiceNumbers);
    }

    @Override
    public Collection getAllOpen() {
        final Criteria criteria = new Criteria();
        criteria.addEqualTo("openInvoiceIndicator", true);
        criteria.addEqualTo("documentHeader.financialDocumentStatusCode", KFSConstants.DocumentStatusCodes.APPROVED);

        final QueryByCriteria qbc = QueryFactory.newQuery(CustomerInvoiceDocument.class, criteria);

        final Collection customerInvoiceDocuments = getPersistenceBrokerTemplate().getCollectionByQuery(qbc);
        return new ArrayList(customerInvoiceDocuments);
    }

    @Override
    public Collection getOpenByCustomerNumber(final String customerNumber) {
        // select i.*
        // from ar_doc_hdr_t h inner join ar_inv_doc_t i
        //   on h.fdoc_nbr = i.fdoc_nbr
        // where h.cust_nbr = ?

        //  OJB deals with the inner join automatically, because we have it setup with
        // accountsReceivableDocumentHeader as a ReferenceDescriptor to Invoice.
        final Criteria criteria = new Criteria();
        criteria.addEqualTo("accountsReceivableDocumentHeader.customerNumber", customerNumber == null ?
                customerNumber : customerNumber.toUpperCase(Locale.US));
        criteria.addEqualTo("openInvoiceIndicator", "true");
        criteria.addEqualTo("documentHeader.financialDocumentStatusCode", KFSConstants.DocumentStatusCodes.APPROVED);

        final QueryByCriteria qbc = QueryFactory.newQuery(CustomerInvoiceDocument.class, criteria);

        final Collection customerInvoiceDocuments = getPersistenceBrokerTemplate().getCollectionByQuery(qbc);
        return new ArrayList(customerInvoiceDocuments);
    }

    @Override
    public Collection getOpenByCustomerNameByCustomerType(final String customerName, final String customerTypeCode) {
        // select i.*
        // from ar_doc_hdr_t h inner join ar_inv_doc_t i
        //   on h.fdoc_nbr = i.fdoc_nbr
        //   inner join ar_cust_t c
        //   on h.cust_nbr = c.cust_nbr
        // where c.cust_nm like ? and c.cust_typ_cd = ?

        final Criteria criteria = new Criteria();
        criteria.addLike("accountsReceivableDocumentHeader.customer.customerName", customerName);
        criteria.addEqualTo("accountsReceivableDocumentHeader.customer.customerTypeCode", customerTypeCode);
        criteria.addEqualTo("openInvoiceIndicator", "true");
        criteria.addEqualTo("documentHeader.financialDocumentStatusCode", KFSConstants.DocumentStatusCodes.APPROVED);

        final QueryByCriteria qbc = QueryFactory.newQuery(CustomerInvoiceDocument.class, criteria);

        final Collection customerInvoiceDocuments = getPersistenceBrokerTemplate().getCollectionByQuery(qbc);
        return new ArrayList(customerInvoiceDocuments);
    }

    @Override
    public Collection getOpenByCustomerName(final String customerName) {
        // select i.*
        // from ar_doc_hdr_t h inner join ar_inv_doc_t i
        //   on h.fdoc_nbr = i.fdoc_nbr
        //   inner join ar_cust_t c
        //   on h.cust_nbr = c.cust_nbr
        // where c.cust_nm like ?

        final Criteria criteria = new Criteria();
        criteria.addLike("accountsReceivableDocumentHeader.customer.customerName", customerName);
        criteria.addEqualTo("openInvoiceIndicator", "true");
        criteria.addEqualTo("documentHeader.financialDocumentStatusCode", KFSConstants.DocumentStatusCodes.APPROVED);

        final QueryByCriteria qbc = QueryFactory.newQuery(CustomerInvoiceDocument.class, criteria);

        final Collection customerInvoiceDocuments = getPersistenceBrokerTemplate().getCollectionByQuery(qbc);
        return new ArrayList(customerInvoiceDocuments);
    }

    @Override
    public Collection getOpenByCustomerType(final String customerTypeCode) {
        // select i.*
        // from ar_doc_hdr_t h inner join ar_inv_doc_t i
        //   on h.fdoc_nbr = i.fdoc_nbr
        //   inner join ar_cust_t c
        //   on h.cust_nbr = c.cust_nbr
        // where c.cust_typ_cd = ?

        //  OJB deals with the inner join automatically, because we have it setup with
        // accountsReceivableDocumentHeader as a ReferenceDescriptor to Invoice, and Customer
        // as a reference descriptor to accountsReceivableDocumentHeader.
        final Criteria criteria = new Criteria();
        criteria.addEqualTo("accountsReceivableDocumentHeader.customer.customerTypeCode", customerTypeCode);
        criteria.addEqualTo("openInvoiceIndicator", "true");
        criteria.addEqualTo("documentHeader.financialDocumentStatusCode", KFSConstants.DocumentStatusCodes.APPROVED);

        final QueryByCriteria qbc = QueryFactory.newQuery(CustomerInvoiceDocument.class, criteria);

        final Collection customerInvoiceDocuments = getPersistenceBrokerTemplate().getCollectionByQuery(qbc);
        return new ArrayList(customerInvoiceDocuments);
    }

    @Override
    public CustomerInvoiceDocument getInvoiceByOrganizationInvoiceNumber(final String organizationInvoiceNumber) {
        final Criteria criteria = new Criteria();
        criteria.addEqualTo("organizationInvoiceNumber", organizationInvoiceNumber);

        return (CustomerInvoiceDocument) getPersistenceBrokerTemplate().getObjectByQuery(
                QueryFactory.newQuery(CustomerInvoiceDocument.class, criteria));
    }

    @Override
    public CustomerInvoiceDocument getInvoiceByInvoiceDocumentNumber(final String documentNumber) {
        final Criteria criteria = new Criteria();
        criteria.addEqualTo("documentNumber", documentNumber);
        return (CustomerInvoiceDocument) getPersistenceBrokerTemplate().getObjectByQuery(
                QueryFactory.newQuery(CustomerInvoiceDocument.class, criteria));
    }

    @Override
    public Collection<CustomerInvoiceDocument> getAllAgingInvoiceDocumentsByBilling(
            final List<String> charts,
            final List<String> organizations, final Date invoiceBillingDateFrom, final Date invoiceBillingDateTo) {
        final Criteria criteria = getAllAgingInvoiceDocumentsCriteria(StringUtils.EMPTY, invoiceBillingDateFrom,
                invoiceBillingDateTo);

        if (ObjectUtils.isNotNull(charts)) {
            criteria.addIn(ArPropertyConstants.CustomerInvoiceDocumentFields.BILL_BY_CHART_OF_ACCOUNT_CODE, charts);
        }

        if (ObjectUtils.isNotNull(organizations)) {
            criteria.addIn(ArPropertyConstants.CustomerInvoiceDocumentFields.BILLED_BY_ORGANIZATION_CODE, organizations);
        }

        criteria.addIsNull(ArPropertyConstants.AGING_REPORT_SENT_TIME);

        final Query query = QueryFactory.newQuery(CustomerInvoiceDocument.class, criteria);

        return getPersistenceBrokerTemplate().getCollectionByQuery(query);
    }

    @Override
    public Collection<CustomerInvoiceDocument> getAllAgingInvoiceDocumentsByProcessing(
            final List<String> charts,
            final List<String> organizations, final Date invoiceBillingDateFrom, final Date invoiceBillingDateTo) {
        final Criteria criteria = getAllAgingInvoiceDocumentsCriteria(StringUtils.EMPTY, invoiceBillingDateFrom,
                invoiceBillingDateTo);

        if (ObjectUtils.isNotNull(charts) && !charts.isEmpty()) {
            criteria.addIn(ArPropertyConstants.CustomerInvoiceDocumentFields.PROCESSING_CHART_OF_ACCOUNT_CODE, charts);
        }

        if (ObjectUtils.isNotNull(organizations) && !organizations.isEmpty()) {
            criteria.addIn(ArPropertyConstants.CustomerInvoiceDocumentFields.PROCESSING_ORGANIZATION_CODE, organizations);
        }

        criteria.addIsNull(ArPropertyConstants.AGING_REPORT_SENT_TIME);

        final Query query = QueryFactory.newQuery(CustomerInvoiceDocument.class, criteria);

        return getPersistenceBrokerTemplate().getCollectionByQuery(query);
    }

    @Override
    public Collection<CustomerInvoiceDocument> getAllAgingInvoiceDocumentsByAccounts(
            final List<String> charts,
            final List<String> accounts, final Date invoiceBillingDateFrom, final Date invoiceBillingDateTo) {
        final Collection<CustomerInvoiceDocument> customerInvoiceDocuments = new ArrayList<>();

        final Criteria criteria = getAllAgingInvoiceDocumentsCriteria(ArPropertyConstants.CUSTOMER_INVOICE_DOCUMENT +
                                                                      ".", invoiceBillingDateFrom, invoiceBillingDateTo);

        if (ObjectUtils.isNotNull(charts)) {
            criteria.addIn(KFSPropertyConstants.CHART_OF_ACCOUNTS_CODE, charts);
        }

        if (ObjectUtils.isNotNull(accounts)) {
            criteria.addIn(KFSPropertyConstants.ACCOUNT_NUMBER, accounts);
        }

        criteria.addIsNull(ArPropertyConstants.AGING_REPORT_SENT_TIME);

        final Query query = QueryFactory.newQuery(CustomerInvoiceDetail.class, criteria);
        final Collection<CustomerInvoiceDetail> customerInvoiceDetails = getPersistenceBrokerTemplate().getCollectionByQuery(query);

        final Set<String> invoiceDocumentNumbers = new HashSet<>();
        for (final CustomerInvoiceDetail detail : customerInvoiceDetails) {
            final CustomerInvoiceDocument customerInvoiceDocument = detail.getCustomerInvoiceDocument();
            final String documentNumber = customerInvoiceDocument.getDocumentNumber();

            if (!invoiceDocumentNumbers.contains(documentNumber)) {
                customerInvoiceDocuments.add(customerInvoiceDocument);

                invoiceDocumentNumbers.add(documentNumber);
            }
        }

        return customerInvoiceDocuments;
    }

    /**
     * get selection criteria for aging invoice document
     */
    protected Criteria getAllAgingInvoiceDocumentsCriteria(
            final String prefix, final Date invoiceDueDateFrom,
            final Date invoiceDueDateTo) {
        final Criteria criteria = new Criteria();
        if (ObjectUtils.isNotNull(invoiceDueDateFrom)) {
            criteria.addGreaterOrEqualThan(ArPropertyConstants.CustomerInvoiceDocumentFields.INVOICE_DUE_DATE,
                    invoiceDueDateFrom);
        }

        if (ObjectUtils.isNotNull(invoiceDueDateTo)) {
            criteria.addLessThan(ArPropertyConstants.CustomerInvoiceDocumentFields.INVOICE_DUE_DATE, invoiceDueDateTo);
        }

        criteria.addEqualTo(prefix + ArPropertyConstants.CustomerInvoiceDocumentFields.OPEN_INVOICE_INDICATOR, true);
        criteria.addEqualTo(prefix + "documentHeader.financialDocumentStatusCode",
                KFSConstants.DocumentStatusCodes.APPROVED);

        return criteria;
    }

    @Override
    public Collection<CustomerInvoiceDocument> getAllAgingInvoiceDocumentsByCustomerTypes(
            final List<String> customerTypes,
            final Date invoiceDueDateFrom, final Date invoiceDueDateTo) {
        LOG.info("invoiceDueDateFrom :::::{}", invoiceDueDateFrom);
        LOG.info("invoiceDueDateTo ::::::::{}", invoiceDueDateTo);
        final Criteria criteria = getAllAgingInvoiceDocumentsCriteria(StringUtils.EMPTY, invoiceDueDateFrom,
                invoiceDueDateTo);

        if (ObjectUtils.isNotNull(customerTypes)) {
            criteria.addIn(ArPropertyConstants.CustomerInvoiceDocumentFields.CUSTOMER_TYPE_CODE, customerTypes);
        }

        final Query query = QueryFactory.newQuery(CustomerInvoiceDocument.class, criteria);

        return getPersistenceBrokerTemplate().getCollectionByQuery(query);
    }
}
