/*
 * The Kuali Financial System, a comprehensive financial management system for higher education.
 *
 * Copyright 2005-2023 Kuali, Inc.
 *
 * This program is free software: you can redistribute it and/or modify
 * it under the terms of the GNU Affero General Public License as published by
 * the Free Software Foundation, either version 3 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU Affero General Public License for more details.
 *
 * You should have received a copy of the GNU Affero General Public License
 * along with this program.  If not, see <http://www.gnu.org/licenses/>.
 */
package org.kuali.kfs.module.ar.document.service;

import org.kuali.kfs.module.ar.businessobject.CashControlDetail;
import org.kuali.kfs.module.ar.businessobject.CustomerInvoiceDetail;
import org.kuali.kfs.module.ar.businessobject.InvoicePaidApplied;
import org.kuali.kfs.module.ar.document.CashControlDocument;
import org.kuali.kfs.module.ar.document.CustomerInvoiceDocument;
import org.kuali.kfs.module.ar.document.PaymentApplicationAdjustmentDocument;
import org.kuali.kfs.module.ar.document.PaymentApplicationDocument;

import java.util.List;

/**
 * Service methods for Payment Application Document.
 */
public interface PaymentApplicationDocumentService {

    /**
     * Creates a new {@link PaymentApplicationAdjustmentDocument} from the supplied {@code document} and persists it.
     *
     * @param document A valid {@link PaymentApplicationDocument}.
     * @return The newly created, and saved, {@code PaymentApplicationAdjustmentDocument}.
     */
    PaymentApplicationAdjustmentDocument createPaymentApplicationAdjustment(PaymentApplicationDocument document);

    /**
     * Retrieves the CashControlDetail line associated with the passed-in PaymentApplication Document.
     *
     * @param document A valid PaymentApplication Document
     * @return The associated CashControlDetail, if exists, or null if not.
     */
    CashControlDetail getCashControlDetailForPaymentApplicationDocument(PaymentApplicationDocument document);

    /**
     * Retrieves the CashControlDetail line associated with the passed-in PaymentApplication Document number.
     *
     * @param payAppDocNumber A valid PaymentApplication Document Number
     * @return The associated CashControlDetail, if exists, or null if not.
     */
    CashControlDetail getCashControlDetailForPayAppDocNumber(String payAppDocNumber);

    /**
     * Retrieves the CashControlDocument associated with the passed-in PaymentApplication Document.
     *
     * @param document A valid PaymentApplication Document
     * @return The associated CashControlDocument, if exists, or null if not.
     */
    CashControlDocument getCashControlDocumentForPaymentApplicationDocument(PaymentApplicationDocument document);

    /**
     * Retrieves the CashControlDocument associated with the passed-in PaymentApplication Document number.
     *
     * @param payAppDocNumber A valid PaymentApplication Document number
     * @return The associated CashControlDocument, if exists, or null if not.
     */
    CashControlDocument getCashControlDocumentForPayAppDocNumber(String payAppDocNumber);

    /**
     * Creates PaidApplieds for all the invoice lines on the passed in InvoiceDocument, on the passed in PaymentApplicationDocument.
     * This method will overwrite any existing PaidApplieds on the document, it assumes an empty PayApp doc with no paidapplieds.
     * This method does no checking to prevent over or under applying, it assumes that the documents have been setup such that it
     * will work correctly. So if this method is used to over or under apply, then the resulting PaymentApplicationDocument will
     * fail business rules validation.
     *
     * @param customerInvoiceDocument
     * @param paymentApplicationDocument
     * @return
     */
    PaymentApplicationDocument createInvoicePaidAppliedsForEntireInvoiceDocument(
            CustomerInvoiceDocument customerInvoiceDocument, PaymentApplicationDocument paymentApplicationDocument);

    /**
     * This method creates an invoice paid applied for the given customer invoice detail. This method assumes that no existing
     * paidApplieds are already on the document.
     *
     * @param customerInvoiceDetail      the customer invoice detail for which we want to create the invoice paid applied
     * @param paymentApplicationDocument the payment application document
     * @param paidAppliedItemNumber
     * @return the created invoice paid applied if it did not exist, null otherwise
     */
    InvoicePaidApplied createInvoicePaidAppliedForInvoiceDetail(CustomerInvoiceDetail customerInvoiceDetail,
            PaymentApplicationDocument paymentApplicationDocument, Integer paidAppliedItemNumber);

    /**
     * This method is used in the lockbox process to create a PA document which is then auto-approved when the amount
     * on the invoice matches the amount on the lockbox.
     *
     * @param customerInvoiceDocument
     * @return
     */
    PaymentApplicationDocument createPaymentApplicationToMatchInvoice(
            CustomerInvoiceDocument customerInvoiceDocument);

    /**
     * This method creates, saves and approves a PaymentApplication document for a given invoice with the
     * given approval annotation and adhoc recipients.
     *
     * @param customerInvoiceDocument        invoice to create PaymentApplication from
     * @param approvalAnnotation             annotation for approval action
     * @param workflowNotificationRecipients adhoc notification recipients
     * @return PaymentApplicationDocument that was created
     */
    PaymentApplicationDocument createSaveAndApprovePaymentApplicationToMatchInvoice(
            CustomerInvoiceDocument customerInvoiceDocument, String approvalAnnotation,
            List workflowNotificationRecipients);

    /**
     * This method creates, saves and approves a PaymentApplication document for a given invoice.
     *
     * @param customerInvoiceDocument invoice to create PaymentApplication from
     * @return PaymentApplicationDocument that was created
     */
    PaymentApplicationDocument createAndSavePaymentApplicationToMatchInvoice(
            CustomerInvoiceDocument customerInvoiceDocument);

    /**
     * This method returns true if invoicePaidApplied is the applied payment for
     * the customer invoice detail based on document number and item/sequence number.
     *
     * @param customerInvoiceDetail
     * @param invoicePaidApplied
     * @return
     */
    boolean customerInvoiceDetailPairsWithInvoicePaidApplied(CustomerInvoiceDetail customerInvoiceDetail,
            InvoicePaidApplied invoicePaidApplied);

}
