/*
 * The Kuali Financial System, a comprehensive financial management system for higher education.
 *
 * Copyright 2005-2023 Kuali, Inc.
 *
 * This program is free software: you can redistribute it and/or modify
 * it under the terms of the GNU Affero General Public License as published by
 * the Free Software Foundation, either version 3 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU Affero General Public License for more details.
 *
 * You should have received a copy of the GNU Affero General Public License
 * along with this program.  If not, see <http://www.gnu.org/licenses/>.
 */
package org.kuali.kfs.module.ar.document.service.impl;

import org.apache.commons.lang3.StringUtils;
import org.kuali.kfs.krad.bo.Note;
import org.kuali.kfs.krad.service.BusinessObjectService;
import org.kuali.kfs.krad.service.NoteService;
import org.kuali.kfs.krad.service.SequenceAccessorService;
import org.kuali.kfs.krad.util.GlobalVariables;
import org.kuali.kfs.krad.util.ObjectUtils;
import org.kuali.kfs.module.ar.businessobject.Customer;
import org.kuali.kfs.module.ar.document.CustomerInvoiceDocument;
import org.kuali.kfs.module.ar.document.service.CustomerInvoiceDocumentService;
import org.kuali.kfs.module.ar.document.service.CustomerService;
import org.kuali.kfs.sys.KFSConstants;

import java.util.ArrayList;
import java.util.Collection;
import java.util.HashMap;
import java.util.List;
import java.util.Map;

public class CustomerServiceImpl implements CustomerService {

    protected SequenceAccessorService sequenceAccessorService;
    protected BusinessObjectService businessObjectService;
    protected CustomerInvoiceDocumentService customerInvoiceDocumentService;
    protected NoteService noteService;
    protected static final String CUSTOMER_NUMBER_SEQUENCE = "CUST_NBR_SEQ";

    @Override
    public Customer getByPrimaryKey(final String customerNumber) {
        return businessObjectService.findBySinglePrimaryKey(Customer.class, customerNumber);
    }

    @Override
    public Customer getByTaxNumber(final String taxNumber) {
        final Map<String, Object> fieldValues = new HashMap<>();
        fieldValues.put("customerTaxNbr", taxNumber);

        final Collection<Customer> customers = getBusinessObjectService().findMatching(Customer.class, fieldValues);
        Customer customer = null;
        for (final Customer c : customers) {
            customer = c;
        }

        return customer;
    }

    @Override
    public String getNextCustomerNumber(final Customer newCustomer) {
        try {
            final Long customerNumberSuffix = sequenceAccessorService.getNextAvailableSequenceNumber(
                CUSTOMER_NUMBER_SEQUENCE, Customer.class);
            final String customerNumberPrefix = newCustomer.getCustomerName().substring(0, 3);
            return customerNumberPrefix + customerNumberSuffix;
        } catch (final StringIndexOutOfBoundsException sioobe) {
            // The customer number generation expects all customer names to be at least three characters long.
            throw new StringIndexOutOfBoundsException("Customer name is less than three characters in length.");
        }
    }

    public SequenceAccessorService getSequenceAccessorService() {
        return sequenceAccessorService;
    }

    public void setSequenceAccessorService(final SequenceAccessorService sequenceAccessorService) {
        this.sequenceAccessorService = sequenceAccessorService;
    }

    @Override
    public Customer getCustomerByName(final String customerName) {
        final Map<String, Object> fieldValues = new HashMap<>();
        fieldValues.put("customerName", customerName);

        final Collection<Customer> customers = getBusinessObjectService().findMatching(Customer.class, fieldValues);
        Customer customer = null;
        for (final Customer c : customers) {
            customer = c;
        }

        return customer;
    }

    public BusinessObjectService getBusinessObjectService() {
        return businessObjectService;
    }

    public void setBusinessObjectService(final BusinessObjectService businessObjectService) {
        this.businessObjectService = businessObjectService;
    }

    public CustomerInvoiceDocumentService getCustomerInvoiceDocumentService() {
        return customerInvoiceDocumentService;
    }

    public void setCustomerInvoiceDocumentService(final CustomerInvoiceDocumentService customerInvoiceDocumentService) {
        this.customerInvoiceDocumentService = customerInvoiceDocumentService;
    }

    @Override
    public Collection<CustomerInvoiceDocument> getInvoicesForCustomer(final Customer customer) {
        Collection<CustomerInvoiceDocument> invoices = null;
        if (null != customer) {
            invoices = getInvoicesForCustomer(customer.getCustomerNumber());
        }
        return invoices;
    }

    @Override
    public Collection<CustomerInvoiceDocument> getInvoicesForCustomer(final String customerNumber) {
        return customerInvoiceDocumentService.getCustomerInvoiceDocumentsByCustomerNumber(customerNumber);
    }

    @Override
    public void createCustomerNote(final String customerNumber, final String customerNote) {
        final Customer customer = getByPrimaryKey(customerNumber);
        if (StringUtils.isNotBlank(customerNote)) {
            final Note newBONote = new Note();
            newBONote.setNoteText(customerNote);
            newBONote.setNotePostedTimestampToCurrent();
            newBONote.setNoteTypeCode(KFSConstants.NoteTypeEnum.BUSINESS_OBJECT_NOTE_TYPE.getCode());
            final Note note = noteService.createNote(newBONote, customer, GlobalVariables.getUserSession().getPrincipalId());
            noteService.save(note);
        }
    }

    @Override
    public List<Note> getCustomerNotes(final String customerNumber) {
        final Customer customer = getByPrimaryKey(customerNumber);
        List<Note> notes = new ArrayList<>();
        if (ObjectUtils.isNotNull(customer)) {
            notes = noteService.getByRemoteObjectId(customer.getObjectId());
        }
        return notes;
    }

    public NoteService getNoteService() {
        return noteService;
    }

    public void setNoteService(final NoteService noteService) {
        this.noteService = noteService;
    }

}
