/*
 * The Kuali Financial System, a comprehensive financial management system for higher education.
 *
 * Copyright 2005-2023 Kuali, Inc.
 *
 * This program is free software: you can redistribute it and/or modify
 * it under the terms of the GNU Affero General Public License as published by
 * the Free Software Foundation, either version 3 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU Affero General Public License for more details.
 *
 * You should have received a copy of the GNU Affero General Public License
 * along with this program.  If not, see <http://www.gnu.org/licenses/>.
 */
package org.kuali.kfs.module.ar.document.service.impl;

import org.apache.commons.lang3.StringUtils;
import org.apache.commons.lang3.Validate;
import org.kuali.kfs.core.api.config.property.ConfigurationService;
import org.kuali.kfs.core.api.util.type.KualiDecimal;
import org.kuali.kfs.coreservice.framework.parameter.ParameterService;
import org.kuali.kfs.kim.api.identity.PersonService;
import org.kuali.kfs.krad.service.BusinessObjectService;
import org.kuali.kfs.krad.service.DocumentService;
import org.kuali.kfs.module.ar.businessobject.AccountsReceivableDocumentHeader;
import org.kuali.kfs.module.ar.businessobject.CashControlDetail;
import org.kuali.kfs.module.ar.businessobject.CustomerInvoiceDetail;
import org.kuali.kfs.module.ar.businessobject.InvoicePaidApplied;
import org.kuali.kfs.module.ar.businessobject.NonAppliedHolding;
import org.kuali.kfs.module.ar.document.CashControlDocument;
import org.kuali.kfs.module.ar.document.CustomerInvoiceDocument;
import org.kuali.kfs.module.ar.document.PaymentApplicationAdjustmentDocument;
import org.kuali.kfs.module.ar.document.PaymentApplicationDocument;
import org.kuali.kfs.module.ar.document.dataaccess.CashControlDetailDao;
import org.kuali.kfs.module.ar.document.service.CustomerAddressService;
import org.kuali.kfs.module.ar.document.service.NonAppliedHoldingService;
import org.kuali.kfs.module.ar.document.service.PaymentApplicationDocumentService;
import org.kuali.kfs.module.ar.document.service.SystemInformationService;
import org.kuali.kfs.sys.businessobject.DocumentHeader;
import org.kuali.kfs.sys.service.UniversityDateService;
import org.springframework.transaction.annotation.Transactional;

import java.util.Arrays;
import java.util.Collection;
import java.util.List;

/**
 * Service method implementations for Payment Application Document.
 */
@Transactional
public class PaymentApplicationDocumentServiceImpl implements PaymentApplicationDocumentService {

    private DocumentService documentService;
    private BusinessObjectService businessObjectService;
    private NonAppliedHoldingService nonAppliedHoldingService;
    private UniversityDateService universityDateService;
    private CashControlDetailDao cashControlDetailDao;
    private ConfigurationService kualiConfigurationService;
    private SystemInformationService systemInformationService;
    private CustomerAddressService customerAddressService;
    private ParameterService parameterService;
    private PersonService personService;

    /**
     * @param customerInvoiceDocument
     * @return
     */
    @Override
    public PaymentApplicationDocument createPaymentApplicationToMatchInvoice(
            final CustomerInvoiceDocument customerInvoiceDocument) {
        final PaymentApplicationDocument applicationDocument = (PaymentApplicationDocument) documentService
                .getNewDocument(PaymentApplicationDocument.class);

        // get the processing chart & org off the invoice, we'll create the payapp with the same processing org
        final String processingChartCode = customerInvoiceDocument.getAccountsReceivableDocumentHeader().getProcessingChartOfAccountCode();
        final String processingOrgCode = customerInvoiceDocument.getAccountsReceivableDocumentHeader().getProcessingOrganizationCode();

        final AccountsReceivableDocumentHeader arDocHeader = new AccountsReceivableDocumentHeader();
        arDocHeader.setProcessingChartOfAccountCode(processingChartCode);
        arDocHeader.setProcessingOrganizationCode(processingOrgCode);
        arDocHeader.setDocumentNumber(applicationDocument.getDocumentNumber());
        applicationDocument.setAccountsReceivableDocumentHeader(arDocHeader);

        // This code is basically copied from PaymentApplicationAction.quickApply
        int paidAppliedItemNumber = 1;
        for (final CustomerInvoiceDetail customerInvoiceDetail : customerInvoiceDocument.getCustomerInvoiceDetailsWithoutDiscounts()) {
            final InvoicePaidApplied invoicePaidApplied = createInvoicePaidAppliedForInvoiceDetail(customerInvoiceDetail, applicationDocument, paidAppliedItemNumber);
            // if there was not another invoice paid applied already created for the current detail then invoicePaidApplied will not
            // be null
            if (invoicePaidApplied != null) {
                // add it to the payment application document list of applied payments
                applicationDocument.getInvoicePaidApplieds().add(invoicePaidApplied);
                paidAppliedItemNumber++;
            }
        }

        return applicationDocument;
    }

    /**
     * @param customerInvoiceDocument
     * @return
     */
    @Override
    public PaymentApplicationDocument createAndSavePaymentApplicationToMatchInvoice(
            final CustomerInvoiceDocument customerInvoiceDocument) {
        final PaymentApplicationDocument applicationDocument = createPaymentApplicationToMatchInvoice(customerInvoiceDocument);
        documentService.saveDocument(applicationDocument);
        return applicationDocument;
    }

    /**
     * @param customerInvoiceDocument
     * @param approvalAnnotation
     * @param workflowNotificationRecipients
     * @return
     */
    @Override
    public PaymentApplicationDocument createSaveAndApprovePaymentApplicationToMatchInvoice(
            final CustomerInvoiceDocument customerInvoiceDocument, final String approvalAnnotation,
            final List workflowNotificationRecipients) {
        final PaymentApplicationDocument applicationDocument = createAndSavePaymentApplicationToMatchInvoice(
                customerInvoiceDocument);
        documentService.approveDocument(applicationDocument, approvalAnnotation, workflowNotificationRecipients);
        return applicationDocument;
    }

    /**
     * @param document
     * @return
     */
    public KualiDecimal getTotalAppliedAmountForPaymentApplicationDocument(final PaymentApplicationDocument document) {
        KualiDecimal total = KualiDecimal.ZERO;
        final Collection<InvoicePaidApplied> invoicePaidApplieds = document.getInvoicePaidApplieds();

        for (final InvoicePaidApplied invoicePaidApplied : invoicePaidApplieds) {
            total = total.add(invoicePaidApplied.getInvoiceItemAppliedAmount());
        }

        // Include non-ar funds as well
        total = total.add(document.getSumOfNonInvoiceds());

        return total;
    }

    /**
     * @param document
     * @return
     */
    public KualiDecimal getTotalUnappliedFundsForPaymentApplicationDocument(final PaymentApplicationDocument document) {
        KualiDecimal total = KualiDecimal.ZERO;

        final String customerNumber = document.getAccountsReceivableDocumentHeader().getCustomerNumber();
        final Collection<NonAppliedHolding> nonAppliedHoldings = nonAppliedHoldingService.getNonAppliedHoldingsForCustomer(
                customerNumber);

        for (final NonAppliedHolding nonAppliedHolding : nonAppliedHoldings) {
            total = total.add(nonAppliedHolding.getFinancialDocumentLineAmount());
        }

        // Add the amount for this document, if it's set
        final NonAppliedHolding nonAppliedHolding = document.getNonAppliedHolding();
        if (null != nonAppliedHolding) {
            final KualiDecimal amount = nonAppliedHolding.getFinancialDocumentLineAmount();
            if (null != amount) {
                total = total.add(amount);
            }
        }

        return total;
    }

    /**
     * @param document
     * @return
     */
    public KualiDecimal getTotalUnappliedFundsToBeAppliedForPaymentApplicationDocument(
            final PaymentApplicationDocument document) {
        final KualiDecimal totalUnapplied = getTotalUnappliedFundsForPaymentApplicationDocument(document);
        final KualiDecimal totalApplied = getTotalAppliedAmountForPaymentApplicationDocument(document);
        return totalUnapplied.subtract(totalApplied);
    }

    @Override
    public CashControlDocument getCashControlDocumentForPaymentApplicationDocument(
            final PaymentApplicationDocument paymentApplicationDocument) {
        Validate.isTrue(paymentApplicationDocument != null, "paymentApplicationDocument cannot be null");
        final String payAppDocNumber = paymentApplicationDocument.getDocumentHeader().getDocumentNumber();
        return getCashControlDocumentForPayAppDocNumber(payAppDocNumber);
    }

    @Override
    public CashControlDocument getCashControlDocumentForPayAppDocNumber(final String paymentApplicationDocumentNumber) {
        Validate.isTrue(StringUtils.isNotBlank(paymentApplicationDocumentNumber),
                "paymentApplicationDocumentNumber cannot be blank");
        final CashControlDetail cashControlDetail = getCashControlDetailForPayAppDocNumber(paymentApplicationDocumentNumber);
        if (cashControlDetail == null) {
            return null;
        }
        return (CashControlDocument) documentService.getByDocumentHeaderId(cashControlDetail.getDocumentNumber());
    }

    @Override
    public PaymentApplicationAdjustmentDocument createPaymentApplicationAdjustment(
            final PaymentApplicationDocument adjusteeDocument
    ) {

        final PaymentApplicationAdjustmentDocument adjustmentDocument =
                (PaymentApplicationAdjustmentDocument) documentService
                        .getNewDocument(PaymentApplicationAdjustmentDocument.class);

        final DocumentHeader documentHeader = adjustmentDocument.getDocumentHeader();
        documentHeader.setDocumentDescription("Created by Application Adjustment");

        final AccountsReceivableDocumentHeader arDocumentHeader = adjusteeDocument.getAccountsReceivableDocumentHeader();
        final String processingChartCode = arDocumentHeader.getProcessingChartOfAccountCode();
        final String processingOrgCode = arDocumentHeader.getProcessingOrganizationCode();

        final String documentNumber = adjustmentDocument.getDocumentNumber();

        final AccountsReceivableDocumentHeader arDocHeader = new AccountsReceivableDocumentHeader();
        arDocHeader.setProcessingChartOfAccountCode(processingChartCode);
        arDocHeader.setProcessingOrganizationCode(processingOrgCode);
        arDocHeader.setDocumentNumber(documentNumber);
        arDocHeader.setCustomerNumber(arDocumentHeader.getCustomerNumber());
        adjustmentDocument.setAccountsReceivableDocumentHeader(arDocHeader);

        adjustmentDocument.setAdjusteeDocumentNumber(adjusteeDocument.getDocumentNumber());
        adjustmentDocument.setInvoicePaidApplieds(adjusteeDocument.getInvoicePaidApplieds());

        final NonAppliedHolding nonAppliedHolding = adjusteeDocument.getNonAppliedHolding();
        if (nonAppliedHolding != null) {
            nonAppliedHolding.setReferenceFinancialDocumentNumber(documentNumber);
            adjustmentDocument.setNonAppliedHoldings(Arrays.asList(nonAppliedHolding));
        }

        documentService.saveDocument(adjustmentDocument);

        adjusteeDocument.refreshReferenceObject("nonAppliedHoldings");
        adjusteeDocument.setAdjusterDocumentNumber(adjustmentDocument.getDocumentNumber());
        documentService.updateDocument(adjusteeDocument);

        return adjustmentDocument;
    }

    @Override
    public CashControlDetail getCashControlDetailForPaymentApplicationDocument(final PaymentApplicationDocument document) {
        Validate.isTrue(document != null, "document cannot be null");
        final String payAppDocumentNumber = document.getDocumentNumber();
        return getCashControlDetailForPayAppDocNumber(payAppDocumentNumber);
    }

    @Override
    public CashControlDetail getCashControlDetailForPayAppDocNumber(final String payAppDocNumber) {
        Validate.isTrue(StringUtils.isNotBlank(payAppDocNumber), "payAppDocNumber cannot be blank");
        return cashControlDetailDao.getCashControlDetailByRefDocNumber(payAppDocNumber);
    }

    public void setBusinessObjectService(final BusinessObjectService businessObjectService) {
        this.businessObjectService = businessObjectService;
    }

    @Override
    public PaymentApplicationDocument createInvoicePaidAppliedsForEntireInvoiceDocument(
            final CustomerInvoiceDocument customerInvoiceDocument, final PaymentApplicationDocument paymentApplicationDocument) {
        // clear any existing paidapplieds
        paymentApplicationDocument.getInvoicePaidApplieds().clear();

        int paidAppliedItemNumber = 1;
        for (final CustomerInvoiceDetail detail : customerInvoiceDocument.getCustomerInvoiceDetailsWithoutDiscounts()) {
            // create the new paidapplied
            final InvoicePaidApplied invoicePaidApplied = createInvoicePaidAppliedForInvoiceDetail(detail,
                    paymentApplicationDocument, paidAppliedItemNumber);
            // add it to the payment application document list of applied payments
            paymentApplicationDocument.getInvoicePaidApplieds().add(invoicePaidApplied);
            paidAppliedItemNumber++;
        }

        return paymentApplicationDocument;
    }

    @Override
    public InvoicePaidApplied createInvoicePaidAppliedForInvoiceDetail(
            final CustomerInvoiceDetail customerInvoiceDetail,
            final PaymentApplicationDocument paymentApplicationDocument, final Integer paidAppliedItemNumber) {
        final Integer universityFiscalYear = universityDateService.getCurrentFiscalYear();
        final String universityFiscalPeriodCode = universityDateService.getCurrentUniversityDate().getAccountingPeriod()
                .getUniversityFiscalPeriodCode();

        final InvoicePaidApplied invoicePaidApplied = new InvoicePaidApplied();

        // set the document number for the invoice paid applied to the payment application document number.
        invoicePaidApplied.setDocumentNumber(paymentApplicationDocument.getDocumentNumber());

        // Set the invoice paid applied ref doc number to the document number for the customer invoice document
        invoicePaidApplied.setFinancialDocumentReferenceInvoiceNumber(customerInvoiceDetail.getDocumentNumber());

        invoicePaidApplied.setInvoiceItemNumber(customerInvoiceDetail.getSequenceNumber());
        invoicePaidApplied.setInvoiceItemAppliedAmount(customerInvoiceDetail.getAmountOpen());
        invoicePaidApplied.setUniversityFiscalYear(universityFiscalYear);
        invoicePaidApplied.setUniversityFiscalPeriodCode(universityFiscalPeriodCode);
        invoicePaidApplied.setPaidAppliedItemNumber(paidAppliedItemNumber);

        return invoicePaidApplied;
    }

    @Override
    public boolean customerInvoiceDetailPairsWithInvoicePaidApplied(
            final CustomerInvoiceDetail customerInvoiceDetail,
            final InvoicePaidApplied invoicePaidApplied) {
        boolean pairs = customerInvoiceDetail.getSequenceNumber().equals(invoicePaidApplied.getInvoiceItemNumber());
        pairs &= customerInvoiceDetail.getDocumentNumber().equals(
                invoicePaidApplied.getFinancialDocumentReferenceInvoiceNumber());
        return pairs;
    }

    public DocumentService getDocumentService() {
        return documentService;
    }

    public void setDocumentService(final DocumentService documentService) {
        this.documentService = documentService;
    }

    public BusinessObjectService getBusinessObjectService() {
        return businessObjectService;
    }

    public NonAppliedHoldingService getNonAppliedHoldingService() {
        return nonAppliedHoldingService;
    }

    public void setNonAppliedHoldingService(final NonAppliedHoldingService nonAppliedHoldingService) {
        this.nonAppliedHoldingService = nonAppliedHoldingService;
    }

    public void setUniversityDateService(final UniversityDateService universityDateService) {
        this.universityDateService = universityDateService;
    }

    public void setCashControlDetailDao(final CashControlDetailDao cashControlDetailDao) {
        this.cashControlDetailDao = cashControlDetailDao;
    }

    public ConfigurationService getKualiConfigurationService() {
        return kualiConfigurationService;
    }

    public void setKualiConfigurationService(final ConfigurationService kualiConfigurationService) {
        this.kualiConfigurationService = kualiConfigurationService;
    }

    protected SystemInformationService getSystemInformationService() {
        return systemInformationService;
    }

    public void setSystemInformationService(final SystemInformationService systemInformationService) {
        this.systemInformationService = systemInformationService;
    }

    protected CustomerAddressService getCustomerAddressService() {
        return customerAddressService;
    }

    public void setCustomerAddressService(final CustomerAddressService customerAddressService) {
        this.customerAddressService = customerAddressService;
    }

    protected ParameterService getParameterService() {
        return parameterService;
    }

    public void setParameterService(final ParameterService parameterService) {
        this.parameterService = parameterService;
    }

    public PersonService getPersonService() {
        return personService;
    }

    public void setPersonService(final PersonService personService) {
        this.personService = personService;
    }

}
