/*
 * The Kuali Financial System, a comprehensive financial management system for higher education.
 *
 * Copyright 2005-2023 Kuali, Inc.
 *
 * This program is free software: you can redistribute it and/or modify
 * it under the terms of the GNU Affero General Public License as published by
 * the Free Software Foundation, either version 3 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU Affero General Public License for more details.
 *
 * You should have received a copy of the GNU Affero General Public License
 * along with this program.  If not, see <http://www.gnu.org/licenses/>.
 */
package org.kuali.kfs.module.ar.document.validation.event;

import org.apache.commons.lang3.StringUtils;
import org.apache.logging.log4j.LogManager;
import org.apache.logging.log4j.Logger;
import org.kuali.kfs.krad.document.Document;
import org.kuali.kfs.krad.document.TransactionalDocument;
import org.kuali.kfs.krad.rules.rule.BusinessRule;
import org.kuali.kfs.krad.rules.rule.event.KualiDocumentEventBase;
import org.kuali.kfs.krad.util.ObjectUtils;
import org.kuali.kfs.module.ar.businessobject.CashControlDetail;
import org.kuali.kfs.module.ar.document.validation.AddCashControlDetailRule;

public final class AddCashControlDetailEvent extends KualiDocumentEventBase {

    private static final Logger LOG = LogManager.getLogger();
    private final CashControlDetail cashControlDetail;

    public AddCashControlDetailEvent(final String errorPathPrefix, final Document document, final CashControlDetail cashControlDetail) {
        super("Adding cash control detail to document " + getDocumentId(document), errorPathPrefix, document);

        // by doing a deep copy, we are ensuring that the business rule class can't update
        // the original object by reference
        this.cashControlDetail = (CashControlDetail) ObjectUtils.deepCopy(cashControlDetail);

        logEvent();
    }

    @Override
    @SuppressWarnings("unchecked")
    public Class getRuleInterfaceClass() {
        return AddCashControlDetailRule.class;
    }

    @Override
    @SuppressWarnings("unchecked")
    public boolean invokeRuleMethod(final BusinessRule rule) {
        return ((AddCashControlDetailRule) rule)
                .processAddCashControlDetailBusinessRules((TransactionalDocument) getDocument(), getCashControlDetail());
    }

    public CashControlDetail getCashControlDetail() {
        return cashControlDetail;
    }

    @Override
    public void validate() {
        super.validate();
        if (getCashControlDetail() == null) {
            throw new IllegalArgumentException("invalid (null) cash control detail");
        }
    }

    /**
     * Logs the event type and some information about the associated cashControlDetail
     */
    private void logEvent() {
        if (LOG.isDebugEnabled()) {
            final StringBuffer logMessage = new StringBuffer(StringUtils.substringAfterLast(getClass().getName(), "."));
            logMessage.append(" with ");

            // vary logging detail as needed
            if (cashControlDetail == null) {
                logMessage.append("null cashControlDetail");
            } else {
                logMessage.append(" cashControlDetail# ");
                logMessage.append(cashControlDetail.getDocumentNumber());
            }

            LOG.debug(logMessage);
        }
    }
}
