/*
 * The Kuali Financial System, a comprehensive financial management system for higher education.
 *
 * Copyright 2005-2023 Kuali, Inc.
 *
 * This program is free software: you can redistribute it and/or modify
 * it under the terms of the GNU Affero General Public License as published by
 * the Free Software Foundation, either version 3 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU Affero General Public License for more details.
 *
 * You should have received a copy of the GNU Affero General Public License
 * along with this program.  If not, see <http://www.gnu.org/licenses/>.
 */
package org.kuali.kfs.module.ar.document.web.struts;

import org.apache.commons.lang3.StringUtils;
import org.kuali.kfs.kns.document.authorization.TransactionalDocumentPresentationController;
import org.kuali.kfs.kns.service.DocumentHelperService;
import org.kuali.kfs.kns.web.ui.ExtraButton;
import org.kuali.kfs.krad.util.KRADConstants;
import org.kuali.kfs.module.ar.ArAuthorizationConstants;
import org.kuali.kfs.module.ar.ArConstants;
import org.kuali.kfs.module.ar.document.CustomerCreditMemoDocument;
import org.kuali.kfs.sys.KFSConstants;
import org.kuali.kfs.sys.context.SpringContext;
import org.kuali.kfs.sys.document.web.struts.FinancialSystemTransactionalDocumentFormBase;

import javax.servlet.http.HttpServletRequest;
import java.util.List;
import java.util.Set;

public class CustomerCreditMemoForm extends FinancialSystemTransactionalDocumentFormBase {

    public CustomerCreditMemoForm() {
        super();
    }

    @Override
    protected String getDefaultDocumentTypeName() {
        return "CRM";
    }

    /**
     * Setup workflow doc in the document.
     */
    @Override
    public void populate(final HttpServletRequest request) {

        //populate document using request
        super.populate(request);

        final CustomerCreditMemoDocument customerCreditMemoDocument = (CustomerCreditMemoDocument) getDocument();
        final String customerInvoiceNumber = customerCreditMemoDocument.getFinancialDocumentReferenceInvoiceNumber();

        //this will make sure that every action has fully populated invoice
        if (StringUtils.isNotEmpty(customerInvoiceNumber) /*&& customerCreditMemoDocument.getInvoice() == null*/) {
            customerCreditMemoDocument.refreshReferenceObject("invoice");
        }
    }

    /**
     * Build additional customer credit memo specific buttons and set extraButtons list.
     *
     * @return - list of extra buttons to be displayed to the user
     * <p>
     * KRAD Conversion: Performs the creation of extra buttons.
     * No data dictionary is involved here.
     */
    @Override
    public List<ExtraButton> getExtraButtons() {

        // clear out the extra buttons array
        extraButtons.clear();

        final CustomerCreditMemoDocument creditMemoDoc = (CustomerCreditMemoDocument) getDocument();
        final DocumentHelperService docHelperService = SpringContext.getBean(DocumentHelperService.class);
        final TransactionalDocumentPresentationController presoController =
            (TransactionalDocumentPresentationController) docHelperService.getDocumentPresentationController(creditMemoDoc);
        final Set<String> editModes = presoController.getEditModes(creditMemoDoc);

        //  special buttons for the first 'init' screen
        if (editModes.contains(ArAuthorizationConstants.CustomerCreditMemoEditMode.DISPLAY_INIT_TAB)) {
            final String externalImageURL = getConfigurationService().getPropertyValueAsString(KFSConstants.RICE_EXTERNALIZABLE_IMAGES_URL_KEY);
            addExtraButton("methodToCall.continueCreditMemo", externalImageURL + "buttonsmall_continue.gif", "Continue");
            addExtraButton("methodToCall.clearInitTab", externalImageURL + "buttonsmall_clear.gif", "Clear");
        }

        //  show the print button if appropriate
        if (editModes.contains(ArAuthorizationConstants.CustomerCreditMemoEditMode.DISPLAY_PRINT_BUTTON)) {
            final String printButtonURL = getConfigurationService().getPropertyValueAsString(KFSConstants.EXTERNALIZABLE_IMAGES_URL_KEY);
            addExtraButton("methodToCall.print", printButtonURL + "buttonsmall_genprintfile.gif", "Print", "excludeSubmitRestriction=true");
        }

        return extraButtons;
    }

    /**
     * Adds a new button to the extra buttons collection.
     *
     * @param property - property for button
     * @param source   - location of image
     * @param altText  - alternate text for button if images don't appear
     */
    protected void addExtraButton(final String property, final String source, final String altText) {
        addExtraButton(property, source, altText, "");
    }

    protected void addExtraButton(final String property, final String source, final String altText, final String onclick) {
        final ExtraButton newButton = new ExtraButton();

        newButton.setExtraButtonProperty(property);
        newButton.setExtraButtonSource(source);
        newButton.setExtraButtonAltText(altText);
        newButton.setExtraButtonOnclick(onclick);

        extraButtons.add(newButton);
    }

    /**
     * @see org.kuali.kfs.kns.web.struts.form.KualiDocumentFormBase#shouldMethodToCallParameterBeUsed(java.lang.String, java.lang.String, javax.servlet.http.HttpServletRequest)
     */
    @Override
    public boolean shouldMethodToCallParameterBeUsed(final String methodToCallParameterName, final String methodToCallParameterValue, final HttpServletRequest request) {
        if (KRADConstants.DISPATCH_REQUEST_PARAMETER.equals(methodToCallParameterName) && ArConstants.PRINT_CREDIT_MEMO_PDF_METHOD.equals(methodToCallParameterValue)) {
            return true;
        }
        return super.shouldMethodToCallParameterBeUsed(methodToCallParameterName, methodToCallParameterValue, request);
    }

}
