/*
 * The Kuali Financial System, a comprehensive financial management system for higher education.
 *
 * Copyright 2005-2023 Kuali, Inc.
 *
 * This program is free software: you can redistribute it and/or modify
 * it under the terms of the GNU Affero General Public License as published by
 * the Free Software Foundation, either version 3 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU Affero General Public License for more details.
 *
 * You should have received a copy of the GNU Affero General Public License
 * along with this program.  If not, see <http://www.gnu.org/licenses/>.
 */
package org.kuali.kfs.module.ar.report.service.impl;

import org.kuali.kfs.krad.util.ObjectUtils;
import org.kuali.kfs.module.ar.ArConstants;
import org.kuali.kfs.module.ar.businessobject.ContractsGrantsLOCReport;
import org.kuali.kfs.module.ar.report.ContractsGrantsLOCReportDetailDataHolder;
import org.kuali.kfs.module.ar.report.ContractsGrantsReportDataHolder;
import org.kuali.kfs.module.ar.report.service.ContractsGrantsReportDataBuilderService;
import org.kuali.kfs.module.ar.report.service.ContractsGrantsReportHelperService;
import org.kuali.kfs.sys.report.ReportInfo;
import org.kuali.kfs.core.api.util.type.KualiDecimal;
import org.kuali.kfs.krad.bo.BusinessObject;

import java.math.BigDecimal;
import java.util.HashMap;
import java.util.List;
import java.util.Map;

/**
 * Implementation of ContractsGrantsReportDataBuilderService to build the Contracts & Grants LOC Draw Details Report
 */
public class ContractsGrantsLOCReportBuilderServiceImpl implements ContractsGrantsReportDataBuilderService {

    protected ReportInfo reportInfo;
    protected ContractsGrantsReportHelperService contractsGrantsReportHelperService;

    /**
     * Builds the report
     */
    @Override
    public ContractsGrantsReportDataHolder buildReportDataHolder(
            final List<? extends BusinessObject> displayList,
            final String sortPropertyName) {
        // check field is valid for subtotal
        final boolean isFieldSubtotalRequired = ArConstants.ReportsConstants.cgLOCReportSubtotalFieldsList
                .contains(sortPropertyName);
        Map<String, KualiDecimal> subTotalMap = new HashMap<>();

        if (isFieldSubtotalRequired) {
            subTotalMap = buildSubTotalMap((List<ContractsGrantsLOCReport>) displayList, sortPropertyName);
        }

        final ContractsGrantsReportDataHolder cgLOCDrawDetailsReportDataHolder = new ContractsGrantsReportDataHolder();
        final List<ContractsGrantsLOCReportDetailDataHolder> details = cgLOCDrawDetailsReportDataHolder.getDetails();

        for (final ContractsGrantsLOCReport cgLOCDrawDetailsReportEntry : (List<ContractsGrantsLOCReport>) displayList) {
            final ContractsGrantsLOCReportDetailDataHolder reportDetail = new ContractsGrantsLOCReportDetailDataHolder();
            // set report data
            setReportDate(cgLOCDrawDetailsReportEntry, reportDetail);

            if (isFieldSubtotalRequired) {
                // set sortedFieldValue for grouping in the report
                reportDetail.setSortedFieldValue(getContractsGrantsReportHelperService()
                        .getPropertyValue(cgLOCDrawDetailsReportEntry, sortPropertyName));
                reportDetail.setDisplaySubtotal(true);
                // set subTotal from subTotalMap
                reportDetail.setSubTotal(subTotalMap.get(getContractsGrantsReportHelperService()
                        .getPropertyValue(cgLOCDrawDetailsReportEntry, sortPropertyName)).bigDecimalValue());
            } else {
                // set this to empty string for not displaying subtotal
                reportDetail.setDisplaySubtotal(false);
            }
            details.add(reportDetail);
        }
        cgLOCDrawDetailsReportDataHolder.setDetails(details);
        return cgLOCDrawDetailsReportDataHolder;
    }

    /**
     * Returns the class of ContractsGrantsLOCReport
     */
    @Override
    public Class<? extends BusinessObject> getDetailsClass() {
        return ContractsGrantsLOCReport.class;
    }

    /**
     * @param displayList
     * @param sortPropertyName
     * @return
     */
    protected Map<String, KualiDecimal> buildSubTotalMap(
            final List<ContractsGrantsLOCReport> displayList,
            final String sortPropertyName) {
        final Map<String, KualiDecimal> returnSubTotalMap = new HashMap<>();
        // get list of sort fields
        final List<String> valuesOfsortProperty = getContractsGrantsReportHelperService()
                .getListOfValuesSortedProperties(displayList, sortPropertyName);

        // calculate sub_total and build subTotalMap
        for (final String value : valuesOfsortProperty) {
            KualiDecimal subTotal = KualiDecimal.ZERO;
            for (final ContractsGrantsLOCReport cgLOCDrawDetailsReportEntry : displayList) {
                // set fieldValue as "" when it is null
                if (value.equals(getContractsGrantsReportHelperService().getPropertyValue(cgLOCDrawDetailsReportEntry,
                        sortPropertyName))) {
                    subTotal = subTotal.add(cgLOCDrawDetailsReportEntry.getAmountToDraw());
                }
            }
            returnSubTotalMap.put(value, subTotal);
        }
        return returnSubTotalMap;
    }

    /**
     * @param cgLOCDrawDetailsReportEntry
     * @param reportDetail
     */
    protected void setReportDate(
            final ContractsGrantsLOCReport cgLOCDrawDetailsReportEntry,
            final ContractsGrantsLOCReportDetailDataHolder reportDetail) {
        reportDetail.setDocumentNumber(cgLOCDrawDetailsReportEntry.getDocumentNumber());
        reportDetail.setLetterOfCreditFundCode(cgLOCDrawDetailsReportEntry.getLetterOfCreditFundCode());
        reportDetail.setLetterOfCreditFundGroupCode(cgLOCDrawDetailsReportEntry.getLetterOfCreditFundGroupCode());
        reportDetail.setLetterOfCreditReviewCreateDate(cgLOCDrawDetailsReportEntry.getLetterOfCreditReviewCreateDate());
        final BigDecimal amountAvailableToDraw = ObjectUtils.isNull(cgLOCDrawDetailsReportEntry.getAmountAvailableToDraw()) ?
                BigDecimal.ZERO : cgLOCDrawDetailsReportEntry.getAmountAvailableToDraw().bigDecimalValue();
        reportDetail.setAmountAvailableToDraw(amountAvailableToDraw);
        final BigDecimal claimOnCashBalance = ObjectUtils.isNull(cgLOCDrawDetailsReportEntry.getClaimOnCashBalance()) ?
                BigDecimal.ZERO : cgLOCDrawDetailsReportEntry.getClaimOnCashBalance().bigDecimalValue();
        reportDetail.setClaimOnCashBalance(claimOnCashBalance);
        final BigDecimal amountToDraw = ObjectUtils.isNull(cgLOCDrawDetailsReportEntry.getAmountToDraw()) ?
                BigDecimal.ZERO : cgLOCDrawDetailsReportEntry.getAmountToDraw().bigDecimalValue();
        reportDetail.setAmountToDraw(amountToDraw);
        final BigDecimal fundsNotDrawn = ObjectUtils.isNull(cgLOCDrawDetailsReportEntry.getFundsNotDrawn()) ?
                BigDecimal.ZERO : cgLOCDrawDetailsReportEntry.getFundsNotDrawn().bigDecimalValue();
        reportDetail.setFundsNotDrawn(fundsNotDrawn);
    }

    @Override
    public ReportInfo getReportInfo() {
        return reportInfo;
    }

    public void setReportInfo(final ReportInfo reportInfo) {
        this.reportInfo = reportInfo;
    }

    public ContractsGrantsReportHelperService getContractsGrantsReportHelperService() {
        return contractsGrantsReportHelperService;
    }

    public void setContractsGrantsReportHelperService(
            final ContractsGrantsReportHelperService contractsGrantsReportHelperService) {
        this.contractsGrantsReportHelperService = contractsGrantsReportHelperService;
    }
}
