/*
 * The Kuali Financial System, a comprehensive financial management system for higher education.
 *
 * Copyright 2005-2023 Kuali, Inc.
 *
 * This program is free software: you can redistribute it and/or modify
 * it under the terms of the GNU Affero General Public License as published by
 * the Free Software Foundation, either version 3 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU Affero General Public License for more details.
 *
 * You should have received a copy of the GNU Affero General Public License
 * along with this program.  If not, see <http://www.gnu.org/licenses/>.
 */
package org.kuali.kfs.module.ar.service.impl;

import org.apache.commons.io.IOUtils;
import org.apache.commons.lang3.StringUtils;
import org.apache.logging.log4j.LogManager;
import org.apache.logging.log4j.Logger;
import org.kuali.kfs.coa.businessobject.Organization;
import org.kuali.kfs.coreservice.framework.parameter.ParameterService;
import org.kuali.kfs.datadictionary.legacy.DataDictionaryService;
import org.kuali.kfs.integration.cg.ContractsAndGrantsBillingAward;
import org.kuali.kfs.kim.impl.identity.Person;
import org.kuali.kfs.krad.bo.Attachment;
import org.kuali.kfs.krad.bo.Note;
import org.kuali.kfs.krad.exception.InvalidAddressException;
import org.kuali.kfs.krad.service.BusinessObjectService;
import org.kuali.kfs.krad.service.DocumentService;
import org.kuali.kfs.krad.service.KualiModuleService;
import org.kuali.kfs.krad.service.NoteService;
import org.kuali.kfs.krad.util.ObjectUtils;
import org.kuali.kfs.module.ar.ArConstants;
import org.kuali.kfs.module.ar.ArKeyConstants;
import org.kuali.kfs.module.ar.ArParameterConstants;
import org.kuali.kfs.module.ar.ArPropertyConstants;
import org.kuali.kfs.module.ar.batch.UpcomingMilestoneNotificationStep;
import org.kuali.kfs.module.ar.businessobject.CustomerAddress;
import org.kuali.kfs.module.ar.businessobject.InvoiceAddressDetail;
import org.kuali.kfs.module.ar.businessobject.Milestone;
import org.kuali.kfs.module.ar.businessobject.TransmissionDetailStatus;
import org.kuali.kfs.module.ar.document.ContractsGrantsInvoiceDocument;
import org.kuali.kfs.module.ar.document.service.ContractsGrantsInvoiceDocumentService;
import org.kuali.kfs.module.ar.service.AREmailService;
import org.kuali.kfs.module.ar.service.ContractsGrantsBillingUtilityService;
import org.kuali.kfs.sys.KFSConstants;
import org.kuali.kfs.sys.KFSParameterKeyConstants;
import org.kuali.kfs.sys.KFSPropertyConstants;
import org.kuali.kfs.sys.mail.BodyMailMessage;
import org.kuali.kfs.sys.service.EmailService;
import org.kuali.kfs.core.api.config.property.ConfigContext;
import org.kuali.kfs.core.api.config.property.ConfigurationService;

import javax.mail.MessagingException;

import java.io.IOException;
import java.io.InputStream;
import java.text.MessageFormat;
import java.util.Collection;
import java.util.HashMap;
import java.util.List;
import java.util.Map;
import java.util.Map.Entry;
import java.util.Properties;
import java.util.stream.Collectors;

public class AREmailServiceImpl implements AREmailService {

    private static final Logger LOG = LogManager.getLogger();

    protected ParameterService parameterService;
    protected DataDictionaryService dataDictionaryService;
    protected ConfigurationService kualiConfigurationService;
    protected BusinessObjectService businessObjectService;
    protected DocumentService documentService;
    protected NoteService noteService;
    protected KualiModuleService kualiModuleService;
    protected ContractsGrantsBillingUtilityService contractsGrantsBillingUtilityService;
    protected ContractsGrantsInvoiceDocumentService contractsGrantsInvoiceDocumentService;
    protected EmailService emailService;

    /**
     * This method is used to send emails to the agency
     *
     * @param invoices
     */
    @Override
    public boolean sendInvoicesViaEmail(final Collection<ContractsGrantsInvoiceDocument> invoices)
            throws InvalidAddressException, MessagingException {
        LOG.debug("sendInvoicesViaEmail() starting.");

        boolean success = true;

        for (final ContractsGrantsInvoiceDocument invoice : invoices) {
            final List<InvoiceAddressDetail> invoiceAddressDetails = invoice.getInvoiceAddressDetails();
            for (final InvoiceAddressDetail invoiceAddressDetail : invoiceAddressDetails) {
                if (ArConstants.InvoiceTransmissionMethod.EMAIL.equals(
                        invoiceAddressDetail.getInvoiceTransmissionMethodCode())
                        && StringUtils.equals(TransmissionDetailStatus.Queued.getCode(),
                        invoiceAddressDetail.getTransmissionStatusCode())) {
                    final Note note = noteService.getNoteByNoteId(invoice.getInvoiceGeneralDetail().getInvoiceNoteId());

                    if (ObjectUtils.isNotNull(note)) {
                        final BodyMailMessage message = new BodyMailMessage();

                        final String sender = parameterService.getParameterValueAsString(
                                KFSConstants.OptionalModuleNamespaces.ACCOUNTS_RECEIVABLE,
                                ArConstants.CONTRACTS_GRANTS_INVOICE_COMPONENT, KFSParameterKeyConstants.FROM_EMAIL
                        );
                        message.setFromAddress(sender);

                        final CustomerAddress customerAddress = invoice.getInvoiceGeneralDetail().getCustomerAddress();
                        final String recipients = invoiceAddressDetail.getCustomerEmailAddress();
                        if (StringUtils.isNotEmpty(recipients)) {
                            message.getToAddresses().add(recipients);
                        } else {
                            LOG.warn("sendInvoicesViaEmail() No recipients indicated.");
                        }

                        final String subject = getSubject(invoice);
                        message.setSubject(subject);
                        if (StringUtils.isEmpty(subject)) {
                            LOG.warn("sendInvoicesViaEmail() Empty subject being sent.");
                        }

                        final String bodyText = getMessageBody(invoice, customerAddress);
                        message.setMessage(bodyText);
                        if (StringUtils.isEmpty(bodyText)) {
                            LOG.warn("sendInvoicesViaEmail() Empty bodyText being sent.");
                        }

                        final Attachment attachment = note.getAttachment();
                        if (ObjectUtils.isNotNull(attachment)) {
                            try (InputStream attachmentContents = attachment.getAttachmentContents()) {
                                message.setAttachmentContent(IOUtils.toByteArray(attachmentContents));
                            } catch (final IOException ex) {
                                LOG.error("Error setting attachment contents", ex);
                                throw new RuntimeException(ex);
                            }
                            message.setAttachmentFileName(attachment.getAttachmentFileName());
                            message.setAttachmentContentType(attachment.getAttachmentMimeTypeCode());
                        }

                        emailService.sendMessage(message, false);

                        invoiceAddressDetail.markSent();
                        contractsGrantsInvoiceDocumentService.addInvoiceTransmissionNote(invoice, invoiceAddressDetail.getInvoiceTransmissionMethodCode());
                        documentService.updateDocument(invoice);
                    } else {
                        success = false;
                    }
                }
            }
        }
        return success;
    }

    protected String getSubject(final ContractsGrantsInvoiceDocument invoice) {
        final String grantNumber = invoice.getInvoiceGeneralDetail().getAward().getProposal().getGrantNumber();
        final String subject;
        final String message;
        if (StringUtils.isBlank(grantNumber)) {
            subject = kualiConfigurationService.getPropertyValueAsString(
                    ArKeyConstants.CGINVOICE_EMAIL_SUBJECT_NO_GRANT_NUMBER);
            message = MessageFormat.format(subject, invoice.getDocumentNumber());
        } else {
            subject = kualiConfigurationService.getPropertyValueAsString(ArKeyConstants.CGINVOICE_EMAIL_SUBJECT);
            final ContractsAndGrantsBillingAward award = invoice.getInvoiceGeneralDetail().getAward();
            message = MessageFormat.format(subject, award.getProposal().getGrantNumber(), invoice.getDocumentNumber(),
                    award.getAwardPrimaryProjectDirector().getProjectDirector().getName());
        }
        return message;
    }

    protected String getMessageBody(final ContractsGrantsInvoiceDocument invoice, final CustomerAddress customerAddress) {
        final String message = kualiConfigurationService.getPropertyValueAsString(ArKeyConstants.CGINVOICE_EMAIL_BODY);

        String department = "";
        final String[] orgCode = invoice.getInvoiceGeneralDetail().getAward().getAwardPrimaryFundManager().getFundManager()
                .getPrimaryDepartmentCode().split("-");
        final Map<String, Object> key = new HashMap<>();
        key.put(KFSPropertyConstants.CHART_OF_ACCOUNTS_CODE, orgCode[0].trim());
        key.put(KFSPropertyConstants.ORGANIZATION_CODE, orgCode[1].trim());
        final Organization org = businessObjectService.findByPrimaryKey(Organization.class, key);
        if (ObjectUtils.isNotNull(org)) {
            department = org.getOrganizationName();
        }

        final Person fundManager = invoice.getInvoiceGeneralDetail().getAward().getAwardPrimaryFundManager().getFundManager();
        return MessageFormat.format(message, customerAddress.getCustomerAddressName(),
            fundManager.getFirstName() + " " + fundManager.getLastName(), department, fundManager.getPhoneNumber(),
            fundManager.getEmailAddress());
    }

    /**
     * This method sends out emails for upcoming milestones.
     */
    @Override
    public void sendEmailNotificationsForMilestones(final List<Milestone> milestones, final ContractsAndGrantsBillingAward award) {
        LOG.debug("sendEmailNotificationsForMilestones() started");

        final Map<String, List<Milestone>> fundManagerMilestoneMap = milestones.stream().
                collect(Collectors.groupingBy(m -> m.getAward().getAwardPrimaryFundManager().getFundManager().getEmailAddress()));

        for (final Entry<String, List<Milestone>> entry : fundManagerMilestoneMap.entrySet()) {

            final BodyMailMessage message = new BodyMailMessage();

            message.setFromAddress(emailService.getDefaultFromAddress());
            message.setSubject(getEmailSubject(ArParameterConstants.EMAIL_SUBJECT));
            message.getToAddresses().add(entry.getKey());
            final StringBuffer body = new StringBuffer();

            String messageKey = kualiConfigurationService.getPropertyValueAsString(
                    ArKeyConstants.MESSAGE_CG_UPCOMING_MILESTONES_EMAIL_LINE_1);
            body.append(messageKey).append("\n\n");

            for (final Milestone milestone : entry.getValue()) {
                final String proposalNumber = dataDictionaryService.getAttributeLabel(Milestone.class,
                        KFSPropertyConstants.PROPOSAL_NUMBER);
                final String chartOfAccountsCode = dataDictionaryService.getAttributeLabel(Milestone.class,
                        KFSPropertyConstants.CHART_OF_ACCOUNTS_CODE);
                final String accountNumber = dataDictionaryService.getAttributeLabel(Milestone.class,
                        KFSPropertyConstants.ACCOUNT_NUMBER);
                final String milestoneNumber = dataDictionaryService.getAttributeLabel(Milestone.class,
                        ArPropertyConstants.MilestoneFields.MILESTONE_NUMBER);
                final String milestoneDescription = dataDictionaryService.getAttributeLabel(Milestone.class,
                        ArPropertyConstants.MilestoneFields.MILESTONE_DESCRIPTION);
                final String milestoneAmount = dataDictionaryService.getAttributeLabel(Milestone.class,
                        ArPropertyConstants.MilestoneFields.MILESTONE_AMOUNT);
                final String milestoneExpectedCompletionDate = dataDictionaryService.getAttributeLabel(Milestone.class,
                        ArPropertyConstants.MilestoneFields.MILESTONE_EXPECTED_COMPLETION_DATE);

                body.append(proposalNumber).append(": ").append(milestone.getProposalNumber()).append(" \n");
                body.append(chartOfAccountsCode).append(": ").append(milestone.getChartOfAccountsCode()).append(" \n");
                body.append(accountNumber).append(": ").append(milestone.getAccountNumber()).append(" \n");
                body.append(milestoneNumber).append(": ").append(milestone.getMilestoneNumber()).append(" \n");
                body.append(milestoneDescription).append(": ").append(milestone.getMilestoneDescription()).append(" \n");
                body.append(milestoneAmount).append(": ").append(milestone.getMilestoneAmount()).append(" \n");
                body.append(milestoneExpectedCompletionDate).append(": ")
                        .append(milestone.getMilestoneExpectedCompletionDate()).append(" \n");

                body.append("\n\n");
            }
            body.append("\n\n");

            messageKey = kualiConfigurationService.getPropertyValueAsString(
                    ArKeyConstants.MESSAGE_CG_UPCOMING_MILESTONES_EMAIL_LINE_2);
            body.append(MessageFormat.format(messageKey, new Object[]{null})).append("\n\n");

            message.setMessage(body.toString());

            emailService.sendMessage(message, false);
        }
    }

    /**
     * Retrieves the email subject text from system parameter then checks environment code and prepends to message if
     * not production.
     *
     * @param subjectParameterName name of parameter giving the subject text
     * @return subject text
     */
    protected String getEmailSubject(final String subjectParameterName) {
        return parameterService.getParameterValueAsString(UpcomingMilestoneNotificationStep.class,
                subjectParameterName);
    }

    protected Properties getConfigProperties() {
        return ConfigContext.getCurrentContextConfig().getProperties();
    }

    public void setParameterService(final ParameterService parameterService) {
        this.parameterService = parameterService;
    }

    public void setDataDictionaryService(final DataDictionaryService dataDictionaryService) {
        this.dataDictionaryService = dataDictionaryService;
    }

    public void setNoteService(final NoteService noteService) {
        this.noteService = noteService;
    }

    public void setKualiConfigurationService(final ConfigurationService kualiConfigurationService) {
        this.kualiConfigurationService = kualiConfigurationService;
    }

    public void setContractsGrantsBillingUtilityService(
            final ContractsGrantsBillingUtilityService contractsGrantsBillingUtilityService) {
        this.contractsGrantsBillingUtilityService = contractsGrantsBillingUtilityService;
    }

    public void setKualiModuleService(final KualiModuleService kualiModuleService) {
        this.kualiModuleService = kualiModuleService;
    }

    public void setBusinessObjectService(final BusinessObjectService businessObjectService) {
        this.businessObjectService = businessObjectService;
    }

    public void setDocumentService(final DocumentService documentService) {
        this.documentService = documentService;
    }

    public void setContractsGrantsInvoiceDocumentService(
            final ContractsGrantsInvoiceDocumentService contractsGrantsInvoiceDocumentService) {
        this.contractsGrantsInvoiceDocumentService = contractsGrantsInvoiceDocumentService;
    }

    public void setEmailService(final EmailService emailService) {
        this.emailService = emailService;
    }
}
