/*
 * The Kuali Financial System, a comprehensive financial management system for higher education.
 *
 * Copyright 2005-2023 Kuali, Inc.
 *
 * This program is free software: you can redistribute it and/or modify
 * it under the terms of the GNU Affero General Public License as published by
 * the Free Software Foundation, either version 3 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU Affero General Public License for more details.
 *
 * You should have received a copy of the GNU Affero General Public License
 * along with this program.  If not, see <http://www.gnu.org/licenses/>.
 */
package org.kuali.kfs.module.ar.web.struts;

import org.apache.logging.log4j.LogManager;
import org.apache.logging.log4j.Logger;
import org.apache.struts.action.ActionForm;
import org.apache.struts.action.ActionForward;
import org.apache.struts.action.ActionMapping;
import org.kuali.kfs.kns.lookup.Lookupable;
import org.kuali.kfs.kns.web.struts.action.KualiAction;
import org.kuali.kfs.kns.web.struts.form.LookupForm;
import org.kuali.kfs.kns.web.ui.Field;
import org.kuali.kfs.kns.web.ui.ResultRow;
import org.kuali.kfs.kns.web.ui.Row;
import org.kuali.kfs.kns.lookup.CollectionIncomplete;
import org.kuali.kfs.krad.util.GlobalVariables;
import org.kuali.kfs.krad.util.ObjectUtils;
import org.kuali.kfs.module.ar.businessobject.lookup.CustomerAgingReportLookupableHelperServiceImpl;
import org.kuali.kfs.sys.KFSConstants;
import org.kuali.kfs.sys.KFSKeyConstants;
import org.kuali.kfs.sys.KFSPropertyConstants;

import javax.servlet.ServletException;
import javax.servlet.http.HttpServletRequest;
import javax.servlet.http.HttpServletResponse;
import java.io.IOException;
import java.util.ArrayList;
import java.util.Collection;
import java.util.HashMap;
import java.util.List;
import java.util.Map;

/**
 * This class handles Actions for lookup flow for CustomerAgingReport.
 */
public class CustomerAgingReportAction extends KualiAction {

    private static final Logger LOG = LogManager.getLogger();

    private static final String TOTALS_TABLE_KEY = "totalsTable";

    public ActionForward start(
            final ActionMapping mapping, final ActionForm form, final HttpServletRequest request,
            final HttpServletResponse response) throws Exception {
        return mapping.findForward(KFSConstants.MAPPING_BASIC);
    }

    /**
     * Search - sets the values of the data entered on the form on the jsp into a map and then searches for the results.
     *
     * @param mapping
     * @param form
     * @param request
     * @param response
     * @return
     * @throws Exception KRAD Conversion: Lookupable performs customization of the results.
     *                   <p>
     *                   Fields are in data dictionary.
     */
    public ActionForward search(
            final ActionMapping mapping, final ActionForm form, final HttpServletRequest request,
            final HttpServletResponse response) throws Exception {
        final CustomerAgingReportForm lookupForm = (CustomerAgingReportForm) form;

        final Lookupable lookupable = lookupForm.getLookupable();

        if (lookupable == null) {
            LOG.error("Lookupable is null.");
            throw new RuntimeException("Lookupable is null.");
        }

        final List<ResultRow> resultTable = new ArrayList<>();

        try {
            final Collection displayList = lookupable.performLookup(lookupForm, resultTable, true);

            final Long totalSize = ((CollectionIncomplete) displayList).getActualSizeIfTruncated();

            request.setAttribute(KFSConstants.REQUEST_SEARCH_RESULTS_SIZE, totalSize);
            request.setAttribute(KFSConstants.REQUEST_SEARCH_RESULTS, resultTable);

            String resultsKey = request.getParameter(KFSConstants.SEARCH_LIST_REQUEST_KEY);
            if (resultsKey != null) {
                GlobalVariables.getUserSession().removeObject(resultsKey);
                GlobalVariables.getUserSession().removeObject(resultsKey + ".form");
            }

            resultsKey = GlobalVariables.getUserSession().addObjectWithGeneratedKey(resultTable);
            GlobalVariables.getUserSession().addObject(resultsKey + ".form", form);
            request.setAttribute(KFSConstants.SEARCH_LIST_REQUEST_KEY, resultsKey);
        } catch (final NumberFormatException e) {
            GlobalVariables.getMessageMap().putError(KFSPropertyConstants.UNIVERSITY_FISCAL_YEAR,
                    KFSKeyConstants.ERROR_CUSTOM, "Fiscal Year must be a four-digit number");
        }
        return mapping.findForward(KFSConstants.MAPPING_BASIC);
    }

    /**
     * Refresh - is called when one quickFinder returns to the previous one. Sets all the values and performs the new search.
     * <p>
     * KRAD Conversion: Lookupable performs customization of the fields and check for additional fields.
     * <p>
     * Data dictionary is used to retrieve the field properties.
     */
    @Override
    public ActionForward refresh(
            final ActionMapping mapping, final ActionForm form, final HttpServletRequest request,
            final HttpServletResponse response) throws Exception {
        final LookupForm lookupForm = (LookupForm) form;
        final Lookupable lookupable = lookupForm.getLookupable();
        if (ObjectUtils.isNull(lookupable)) {
            LOG.error("Lookupable is null.");
            throw new RuntimeException("Lookupable is null.");
        }

        final Map<String, String> fieldValues = new HashMap<>();
        final Map values = lookupForm.getFields();

        for (final Row row : lookupable.getRows()) {
            for (final Field field : row.getFields()) {
                if (field.getPropertyName() != null && !field.getPropertyName().equals("")) {
                    if (request.getParameter(field.getPropertyName()) != null) {
                        field.setPropertyValue(request.getParameter(field.getPropertyName()));
                    } else if (values.get(field.getPropertyName()) != null) {
                        field.setPropertyValue(values.get(field.getPropertyName()));
                    }
                }
                fieldValues.put(field.getPropertyName(), field.getPropertyValue());
            }
        }
        fieldValues.put(KFSConstants.DOC_FORM_KEY, lookupForm.getFormKey());
        fieldValues.put(KFSConstants.BACK_LOCATION, lookupForm.getBackLocation());

        if (lookupable.checkForAdditionalFields(fieldValues)) {
            for (final Row row : lookupable.getRows()) {
                for (final Field field : row.getFields()) {
                    if (field.getPropertyName() != null && !field.getPropertyName().equals("")) {
                        if (request.getParameter(field.getPropertyName()) != null) {
                            field.setPropertyValue(request.getParameter(field.getPropertyName()));
                            fieldValues.put(field.getPropertyName(), request.getParameter(field.getPropertyName()));
                        } else if (values.get(field.getPropertyName()) != null) {
                            field.setPropertyValue(values.get(field.getPropertyName()));
                        }
                    }
                }
            }
        }

        return mapping.findForward(KFSConstants.MAPPING_BASIC);
    }

    /**
     * Cancels the action and returns to portal main page.
     *
     * @param mapping
     * @param form
     * @param request
     * @param response
     * @return
     * @throws Exception
     */
    public ActionForward cancel(
            final ActionMapping mapping, final ActionForm form, final HttpServletRequest request,
            final HttpServletResponse response) throws Exception {
        return mapping.findForward(KFSConstants.MAPPING_CANCEL);
    }

    /**
     * Clears the values of all the fields on the jsp.
     *
     * KRAD Conversion: Lookupable performs customization of the fields.
     * <p>
     * Data dictionary is used to retrieve the field properties.
     *
     * @param mapping
     * @param form
     * @param request
     * @param response
     * @return
     * @throws IOException
     * @throws ServletException
     */
    public ActionForward clearValues(
            final ActionMapping mapping, final ActionForm form, final HttpServletRequest request,
            final HttpServletResponse response) throws IOException, ServletException {
        final LookupForm lookupForm = (LookupForm) form;
        final Lookupable lookupable = lookupForm.getLookupable();
        if (ObjectUtils.isNull(lookupable)) {
            LOG.error("Lookupable is null.");
            throw new RuntimeException("Lookupable is null.");
        }

        for (final Row row : lookupable.getRows()) {
            for (final Field field : row.getFields()) {
                if (!field.getFieldType().equals(Field.RADIO)) {
                    field.setPropertyValue(field.getDefaultValue());
                }
            }
        }

        return mapping.findForward(KFSConstants.MAPPING_BASIC);
    }

    /**
     * View results from balance inquiry action
     *
     * @param mapping
     * @param form
     * @param request
     * @param response
     * @return
     * @throws Exception
     */
    public ActionForward viewResults(
            final ActionMapping mapping, final ActionForm form, final HttpServletRequest request,
            final HttpServletResponse response) throws Exception {
        final CustomerAgingReportForm customerAgeForm = (CustomerAgingReportForm) form;
        final String resultKey = request.getParameter(KFSConstants.SEARCH_LIST_REQUEST_KEY);
        request.setAttribute(KFSConstants.SEARCH_LIST_REQUEST_KEY, resultKey);
        request.setAttribute(KFSConstants.REQUEST_SEARCH_RESULTS, GlobalVariables.getUserSession().retrieveObject(resultKey));
        final CustomerAgingReportForm prevForm = (CustomerAgingReportForm) GlobalVariables.getUserSession().retrieveObject(resultKey + ".form");
        customerAgeForm.setTotal0to30(prevForm.getTotal0to30());
        customerAgeForm.setTotal31to60(prevForm.getTotal31to60());
        customerAgeForm.setTotal61to90(prevForm.getTotal61to90());
        customerAgeForm.setTotal91toSYSPR(prevForm.getTotal91toSYSPR());
        customerAgeForm.setTotalSYSPRplus1orMore(prevForm.getTotalSYSPRplus1orMore());
        request.setAttribute(KFSConstants.REQUEST_SEARCH_RESULTS_SIZE, request.getParameter(KFSConstants.REQUEST_SEARCH_RESULTS_SIZE));

        // TODO: use inheritance instead of this if statement
        if (customerAgeForm.getLookupable().getLookupableHelperService() instanceof CustomerAgingReportLookupableHelperServiceImpl) {
            final Object totalsTable = GlobalVariables.getUserSession().retrieveObject(TOTALS_TABLE_KEY);
            request.setAttribute(TOTALS_TABLE_KEY, totalsTable);
        }

        return mapping.findForward(KFSConstants.MAPPING_BASIC);
    }

}
