/*
 * The Kuali Financial System, a comprehensive financial management system for higher education.
 *
 * Copyright 2005-2022 Kuali, Inc.
 *
 * This program is free software: you can redistribute it and/or modify
 * it under the terms of the GNU Affero General Public License as published by
 * the Free Software Foundation, either version 3 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU Affero General Public License for more details.
 *
 * You should have received a copy of the GNU Affero General Public License
 * along with this program.  If not, see <http://www.gnu.org/licenses/>.
 */
package org.kuali.kfs.module.ar.batch.service.impl;

import com.lowagie.text.Chunk;
import com.lowagie.text.DocumentException;
import com.lowagie.text.Font;
import com.lowagie.text.FontFactory;
import com.lowagie.text.PageSize;
import com.lowagie.text.Paragraph;
import com.lowagie.text.pdf.PdfWriter;
import jakarta.xml.bind.JAXBContext;
import jakarta.xml.bind.JAXBException;
import jakarta.xml.bind.Marshaller;
import org.apache.commons.io.IOUtils;
import org.apache.commons.lang3.StringUtils;
import org.apache.logging.log4j.LogManager;
import org.apache.logging.log4j.Logger;
import org.kuali.kfs.core.api.datetime.DateTimeService;
import org.kuali.kfs.kim.impl.identity.Person;
import org.kuali.kfs.module.ar.ArConstants;
import org.kuali.kfs.module.ar.batch.service.CustomerInvoiceWriteoffBatchService;
import org.kuali.kfs.module.ar.batch.vo.CustomerInvoiceWriteoffBatchVO;
import org.kuali.kfs.module.ar.businessobject.Customer;
import org.kuali.kfs.module.ar.document.service.CustomerInvoiceDocumentService;
import org.kuali.kfs.module.ar.document.service.CustomerInvoiceWriteoffDocumentService;
import org.kuali.kfs.module.ar.document.service.CustomerService;
import org.kuali.kfs.sys.batch.BatchInputFileType;
import org.kuali.kfs.sys.batch.service.BatchInputFileService;
import org.kuali.kfs.sys.exception.ParseException;
import org.kuali.kfs.sys.util.CDataXMLStreamWriter;
import org.springframework.transaction.annotation.Transactional;

import javax.xml.stream.XMLOutputFactory;
import javax.xml.stream.XMLStreamException;
import javax.xml.stream.XMLStreamWriter;
import java.awt.Color;
import java.io.BufferedOutputStream;
import java.io.File;
import java.io.FileInputStream;
import java.io.FileNotFoundException;
import java.io.FileOutputStream;
import java.io.IOException;
import java.io.InputStream;
import java.text.SimpleDateFormat;
import java.util.ArrayList;
import java.util.List;
import java.util.Locale;

@Transactional
public class CustomerInvoiceWriteoffBatchServiceImpl implements CustomerInvoiceWriteoffBatchService {
    private static final Logger LOG = LogManager.getLogger();

    private CustomerService customerService;
    private CustomerInvoiceDocumentService invoiceDocumentService;
    private DateTimeService dateTimeService;
    private BatchInputFileService batchInputFileService;
    private BatchInputFileType batchInputFileType;
    private String reportsDirectory;
    private CustomerInvoiceWriteoffDocumentService customerInvoiceWriteoffDocumentService;

    public CustomerInvoiceWriteoffBatchServiceImpl() {
    }

    @Override
    public boolean loadFiles() {
        LOG.info("Beginning processing of all available files for AR Customer Invoice Writeoff Batch Documents.");

        boolean result = true;

        //  create a list of the files to process
        final List<String> fileNamesToLoad = getListOfFilesToProcess();
        LOG.info("Found {} file(s) to process.", fileNamesToLoad::size);
        final boolean anyFilesFound = fileNamesToLoad.size() > 0;

        //  process each file in turn
        final List<String> processedFiles;
        try (com.lowagie.text.Document pdfdoc = anyFilesFound ? getPdfDoc() : null) {
            processedFiles = new ArrayList<>();
            for (final String inputFileName : fileNamesToLoad) {
                LOG.info("Beginning processing of filename: {}.", inputFileName);

                //  setup the results reporting
                writeFileNameSectionTitle(pdfdoc, inputFileName);

                //  load the file
                boolean success = false;
                try {
                    success = loadFile(inputFileName, pdfdoc);
                } catch (final Exception e) {
                    LOG.error("An unhandled error occurred.  {}", e::getMessage);
                    writeInvoiceSectionMessage(pdfdoc, "ERROR - Unhandled exception caught.");
                    writeInvoiceSectionMessage(pdfdoc, e.getMessage());
                }
                result &= success;

                //  handle result
                if (success) {
                    writeInvoiceSectionMessage(pdfdoc, "File successfully completed processing.");
                    processedFiles.add(inputFileName);
                } else {
                    writeInvoiceSectionMessage(pdfdoc, "File failed to process successfully.");
                    result = false;
                }
            }

            //  remove done files
            removeDoneFiles(processedFiles);
        } catch (IOException | DocumentException ex) {
            throw new RuntimeException("Could not load customer invoice writeoff files", ex);
        }

        return result;
    }

    /**
     * Clears out associated .done files for the processed data files.
     */
    protected void removeDoneFiles(final List<String> dataFileNames) {
        for (final String dataFileName : dataFileNames) {
            final String doneFileName = doneFileName(dataFileName);
            final File doneFile = new File(doneFileName);
            if (doneFile.exists()) {
                doneFile.delete();
            }
        }
    }

    public boolean loadFile(final String fileName, final com.lowagie.text.Document pdfdoc) {
        final boolean result = true;

        //  load up the file into a byte array
        final byte[] fileByteContent = safelyLoadFileBytes(fileName);

        //  parse the file against the XSD schema and load it into an object
        LOG.info("Attempting to parse the file using JAXB.");
        final Object parsedObject;
        try {
            parsedObject = batchInputFileService.parse(batchInputFileType, fileByteContent);
        } catch (final ParseException e) {
            LOG.error("Error parsing batch file: {}", e::getMessage);
            writeInvoiceSectionMessage(pdfdoc, "Error parsing batch file: " + e.getMessage());
            throw new ParseException(e.getMessage());
        }

        //  make sure we got the type we expected, then cast it
        if (!(parsedObject instanceof CustomerInvoiceWriteoffBatchVO)) {
            LOG.error("Parsed file was not of the expected type.  Expected [{}] but got [{}].",
                    () -> CustomerInvoiceWriteoffBatchVO.class,
                    parsedObject::getClass
            );
            writeInvoiceSectionMessage(
                    pdfdoc,
                    "Parsed file was not of the expected type.  Expected [" + CustomerInvoiceWriteoffBatchVO.class
                    + "] but got [" + parsedObject.getClass() + "]."
            );
            throw new RuntimeException(
                    "Parsed file was not of the expected type.  Expected [" + CustomerInvoiceWriteoffBatchVO.class
                    + "] but got [" + parsedObject.getClass() + "].");
        }

        //  convert to the real object type
        final CustomerInvoiceWriteoffBatchVO batchVO = (CustomerInvoiceWriteoffBatchVO) parsedObject;

        LOG.info("Beginning validation and preparation of batch file.");
        createCustomerInvoiceWriteoffDocumentsFromBatchVO(batchVO, pdfdoc);

        return result;
    }

    protected void createCustomerInvoiceWriteoffDocumentsFromBatchVO(CustomerInvoiceWriteoffBatchVO batchVO,
            com.lowagie.text.Document pdfdoc) {
        //  retrieve the user note
        final String note = batchVO.getNote();

        //  add submittedOn and submittedBy to the pdf
        writeInvoiceSectionMessage(pdfdoc, "Batch Submitted By: " + batchVO.getSubmittedByPrincipalName());
        writeInvoiceSectionMessage(pdfdoc, "Batch Submitted On: " + batchVO.getSubmittedOn());
        if (StringUtils.isNotBlank(note)) {
            writeInvoiceSectionMessage(pdfdoc, "NOTE: " + note);
        }

        //  create a new Invoice Writeoff document for each invoice number in the batch file
        boolean customerNoteIsSet = false;
        for (final String invoiceNumber : batchVO.getInvoiceNumbers()) {
            //  set the customer note
            if (!customerNoteIsSet) {
                final Customer customer = invoiceDocumentService.getCustomerByInvoiceDocumentNumber(invoiceNumber);
                if (customer != null) {
                    customerService.createCustomerNote(customer.getCustomerNumber(), note);
                    customerNoteIsSet = true;
                }
            }

            //  write the doc # we're trying to write off
            writeInvoiceSectionTitle(pdfdoc, "INVOICE DOC#: " + invoiceNumber);

            //  attempt to create the writeoff document
            final String writeoffDocNumber =
                    customerInvoiceWriteoffDocumentService.createCustomerInvoiceWriteoffDocument(invoiceNumber, note);

            //  write the successful information if we got it
            if (StringUtils.isNotBlank(writeoffDocNumber)) {
                writeInvoiceSectionMessage(
                        pdfdoc,
                        "SUCCESS - Created new Invoice Writeoff Document #" + writeoffDocNumber
                );
            } else {
                writeInvoiceSectionMessage(
                        pdfdoc,
                        "FAILURE - No error occurred, but a new Invoice Writeoff Document number was not created.  "
                        + "Check the logs."
                );
            }
        }
    }

    /**
     * Accepts a file name and returns a byte-array of the file name contents, if possible.
     * <p>
     * Throws RuntimeExceptions if FileNotFound or IOExceptions occur.
     *
     * @param fileName String containing valid path & filename (relative or absolute) of file to load.
     * @return A Byte Array of the contents of the file.
     */
    protected byte[] safelyLoadFileBytes(final String fileName) {
        final InputStream fileContents;
        final byte[] fileByteContent;
        try {
            fileContents = new FileInputStream(fileName);
        } catch (final FileNotFoundException e1) {
            LOG.error("Batch file not found [{}]. {}", () -> fileName, e1::getMessage);
            throw new RuntimeException("Batch File not found [" + fileName + "]. " + e1.getMessage());
        }
        try {
            fileByteContent = IOUtils.toByteArray(fileContents);
        } catch (final IOException e1) {
            LOG.error("IO Exception loading: [{}]. {}", () -> fileName, e1::getMessage);
            throw new RuntimeException("IO Exception loading: [" + fileName + "]. " + e1.getMessage());
        }
        return fileByteContent;
    }

    protected List<String> getListOfFilesToProcess() {
        //  create a list of the files to process
        final List<String> fileNamesToLoad = batchInputFileService.listInputFileNamesWithDoneFile(batchInputFileType);

        if (fileNamesToLoad == null) {
            LOG.error(
                    "BatchInputFileService.listInputFileNamesWithDoneFile({}) returned NULL which should never happen.",
                    batchInputFileType::getFileTypeIdentifier
            );
            throw new RuntimeException(
                    "BatchInputFileService.listInputFileNamesWithDoneFile(" + batchInputFileType.getFileTypeIdentifier()
                    + ") returned NULL which should never happen.");
        }

        //  filenames returned should never be blank/empty/null
        for (final String inputFileName : fileNamesToLoad) {
            if (StringUtils.isBlank(inputFileName)) {
                LOG.error("One of the file names returned as ready to process [{}] was blank.  This should not "
                          + "happen, " + "so throwing an error to investigate.",
                        inputFileName
                );
                throw new RuntimeException("One of the file names returned as ready to process [" + inputFileName
                                           + "] was blank.  This should not happen, so throwing an error to "
                                           + "investigate.");
            }
        }

        return fileNamesToLoad;
    }

    protected com.lowagie.text.Document getPdfDoc() throws IOException, DocumentException {
        final String reportDropFolder =
                reportsDirectory + "/" + ArConstants.CustomerInvoiceWriteoff.CUSTOMER_INVOICE_WRITEOFF_REPORT_SUBFOLDER
                + "/";
        final String fileName = ArConstants.CustomerInvoiceWriteoff.BATCH_REPORT_BASENAME + "_" + new SimpleDateFormat(
                "yyyyMMdd_HHmmssSSS",
                Locale.US
        ).format(dateTimeService.getCurrentDate()) + ".pdf";

        //  setup the writer
        final File reportFile = new File(reportDropFolder + fileName);
        final FileOutputStream fileOutStream;
        fileOutStream = new FileOutputStream(reportFile);
        final BufferedOutputStream buffOutStream = new BufferedOutputStream(fileOutStream);

        final com.lowagie.text.Document pdfdoc = new com.lowagie.text.Document(PageSize.LETTER, 54, 54, 72, 72);
        PdfWriter.getInstance(pdfdoc, buffOutStream);

        pdfdoc.open();

        return pdfdoc;
    }

    protected void writeFileNameSectionTitle(final com.lowagie.text.Document pdfDoc, final String filenameLine) {
        final Font font = FontFactory.getFont(FontFactory.COURIER, 10, Font.BOLD);

        //  file name title, get title only, on windows & unix platforms
        String fileNameOnly = filenameLine.toUpperCase(Locale.US);
        int indexOfSlashes = fileNameOnly.lastIndexOf('\\');
        if (indexOfSlashes < fileNameOnly.length()) {
            fileNameOnly = fileNameOnly.substring(indexOfSlashes + 1);
        }
        indexOfSlashes = fileNameOnly.lastIndexOf('/');
        if (indexOfSlashes < fileNameOnly.length()) {
            fileNameOnly = fileNameOnly.substring(indexOfSlashes + 1);
        }

        final Paragraph paragraph = new Paragraph();
        paragraph.setAlignment(com.lowagie.text.Element.ALIGN_LEFT);
        final Chunk chunk = new Chunk(fileNameOnly, font);
        chunk.setBackground(Color.LIGHT_GRAY, 5, 5, 5, 5);
        paragraph.add(chunk);

        //  blank line
        paragraph.add(new Chunk("", font));

        try {
            pdfDoc.add(paragraph);
        } catch (final DocumentException e) {
            LOG.error("iText DocumentException thrown when trying to write content.", e);
            throw new RuntimeException("iText DocumentException thrown when trying to write content.", e);
        }
    }

    protected void writeInvoiceSectionTitle(final com.lowagie.text.Document pdfDoc, final String customerNameLine) {
        final Font font = FontFactory.getFont(FontFactory.COURIER, 8, Font.BOLD + Font.UNDERLINE);

        final Paragraph paragraph = new Paragraph();
        paragraph.setAlignment(com.lowagie.text.Element.ALIGN_LEFT);
        paragraph.add(new Chunk(customerNameLine, font));

        //  blank line
        paragraph.add(new Chunk("", font));

        try {
            pdfDoc.add(paragraph);
        } catch (final DocumentException e) {
            LOG.error("iText DocumentException thrown when trying to write content.", e);
            throw new RuntimeException("iText DocumentException thrown when trying to write content.", e);
        }
    }

    protected void writeInvoiceSectionMessage(final com.lowagie.text.Document pdfDoc, final String resultLine) {
        final Font font = FontFactory.getFont(FontFactory.COURIER, 8, Font.NORMAL);

        final Paragraph paragraph = new Paragraph();
        paragraph.setAlignment(com.lowagie.text.Element.ALIGN_LEFT);
        paragraph.add(new Chunk(resultLine, font));

        //  blank line
        paragraph.add(new Chunk("", font));

        try {
            pdfDoc.add(paragraph);
        } catch (final DocumentException e) {
            LOG.error("iText DocumentException thrown when trying to write content.", e);
            throw new RuntimeException("iText DocumentException thrown when trying to write content.", e);
        }
    }

    @Override
    public String createBatchDrop(final Person person, final CustomerInvoiceWriteoffBatchVO writeoffBatchVO) {
        final String batchXmlFileName = transformVOtoFile(person, writeoffBatchVO);

        createDoneFile(batchXmlFileName);

        return batchXmlFileName;
    }

    protected String doneFileName(final String filename) {
        final String fileNoExtension = filename.substring(0, filename.lastIndexOf('.'));
        return fileNoExtension + ".done";
    }

    protected void createDoneFile(final String filename) {
        final String fileNoExtension = doneFileName(filename);
        final File doneFile = new File(fileNoExtension);
        try {
            doneFile.createNewFile();
        } catch (final IOException e) {
            throw new RuntimeException("Exception while trying to create .done file.", e);
        }
    }

    protected String getBatchFilePathAndName(final Person person) {
        final String filename = batchInputFileType.getFileName(person.getPrincipalId(), "", "");

        String filepath = batchInputFileType.getDirectoryPath();
        if (!filepath.endsWith("/")) {
            filepath += "/";
        }

        final String extension = batchInputFileType.getFileExtension();

        return filepath + filename + "." + extension;
    }

    protected String transformVOtoFile(final Person person, final CustomerInvoiceWriteoffBatchVO writeoffBatchVO) {
        //  determine file paths and names
        final String filePathAndName = getBatchFilePathAndName(person);
        final File file = new File(filePathAndName);

        try (FileOutputStream fileOutputStream = new FileOutputStream(file)) {

            final JAXBContext jaxbContext = JAXBContext.newInstance(CustomerInvoiceWriteoffBatchVO.class);
            final Marshaller jaxbMarshaller = jaxbContext.createMarshaller();

            // XMLStreamWriter would remove JAXB formatting
            final XMLOutputFactory xof = XMLOutputFactory.newInstance();
            final XMLStreamWriter streamWriter = xof.createXMLStreamWriter(fileOutputStream, "UTF-8");
            final CDataXMLStreamWriter cdataStreamWriter = new CDataXMLStreamWriter(streamWriter);
            jaxbMarshaller.marshal(writeoffBatchVO, cdataStreamWriter);
        } catch (JAXBException | XMLStreamException | IOException e) {
            LOG.fatal("Failed to serialize xml: {}", filePathAndName, e);
            throw new RuntimeException(e);
        }

        return filePathAndName;
    }

    public void setCustomerInvoiceWriteoffDocumentService(
            final CustomerInvoiceWriteoffDocumentService customerInvoiceWriteoffDocumentService
    ) {
        this.customerInvoiceWriteoffDocumentService = customerInvoiceWriteoffDocumentService;
    }

    public void setDateTimeService(final DateTimeService dateTimeService) {
        this.dateTimeService = dateTimeService;
    }

    public void setBatchInputFileService(final BatchInputFileService batchInputFileService) {
        this.batchInputFileService = batchInputFileService;
    }

    public void setBatchInputFileType(final BatchInputFileType batchInputFileType) {
        this.batchInputFileType = batchInputFileType;
    }

    public void setReportsDirectory(final String reportsDirectory) {
        this.reportsDirectory = reportsDirectory;
    }

    public void setCustomerService(final CustomerService customerService) {
        this.customerService = customerService;
    }

    public void setInvoiceDocumentService(final CustomerInvoiceDocumentService invoiceDocumentService) {
        this.invoiceDocumentService = invoiceDocumentService;
    }

}
