/*
 * The Kuali Financial System, a comprehensive financial management system for higher education.
 *
 * Copyright 2005-2022 Kuali, Inc.
 *
 * This program is free software: you can redistribute it and/or modify
 * it under the terms of the GNU Affero General Public License as published by
 * the Free Software Foundation, either version 3 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU Affero General Public License for more details.
 *
 * You should have received a copy of the GNU Affero General Public License
 * along with this program.  If not, see <http://www.gnu.org/licenses/>.
 */
package org.kuali.kfs.module.ar.batch.service.impl;

import org.apache.commons.collections4.CollectionUtils;
import org.apache.logging.log4j.LogManager;
import org.apache.logging.log4j.Logger;
import org.kuali.kfs.core.api.config.property.ConfigurationService;
import org.kuali.kfs.core.api.datetime.DateTimeService;
import org.kuali.kfs.core.api.search.SearchOperator;
import org.kuali.kfs.core.api.util.type.KualiDecimal;
import org.kuali.kfs.coreservice.framework.parameter.ParameterService;
import org.kuali.kfs.kew.api.document.WorkflowDocumentService;
import org.kuali.kfs.krad.service.DocumentService;
import org.kuali.kfs.krad.service.KualiModuleService;
import org.kuali.kfs.module.ar.ArConstants;
import org.kuali.kfs.module.ar.ArKeyConstants;
import org.kuali.kfs.module.ar.ArParameterConstants;
import org.kuali.kfs.module.ar.ArPropertyConstants;
import org.kuali.kfs.module.ar.batch.service.LetterOfCreditCreateService;
import org.kuali.kfs.module.ar.businessobject.AccountsReceivableDocumentHeader;
import org.kuali.kfs.module.ar.businessobject.CashControlDetail;
import org.kuali.kfs.module.ar.document.CashControlDocument;
import org.kuali.kfs.module.ar.document.ContractsGrantsInvoiceDocument;
import org.kuali.kfs.module.ar.document.PaymentApplicationDocument;
import org.kuali.kfs.module.ar.document.service.CashControlDocumentService;
import org.kuali.kfs.module.ar.document.service.ContractsGrantsInvoiceDocumentService;
import org.kuali.kfs.module.ar.document.service.PaymentApplicationDocumentService;
import org.kuali.kfs.sys.KFSConstants;
import org.kuali.kfs.sys.document.service.FinancialSystemDocumentService;
import org.kuali.kfs.sys.service.GeneralLedgerPendingEntryService;
import org.springframework.transaction.annotation.Transactional;

import java.io.File;
import java.io.IOException;
import java.io.PrintWriter;
import java.nio.charset.StandardCharsets;
import java.sql.Timestamp;
import java.text.MessageFormat;
import java.util.Collection;
import java.util.HashMap;
import java.util.Map;

/**
 * Defines a service class for creating Cash Control documents from the LOC Review Document.
 */
public class LetterOfCreditCreateServiceImpl implements LetterOfCreditCreateService {

    private static final Logger LOG = LogManager.getLogger();
    protected CashControlDocumentService cashControlDocumentService;
    protected ConfigurationService configService;
    protected ContractsGrantsInvoiceDocumentService contractsGrantsInvoiceDocumentService;
    protected DateTimeService dateTimeService;
    protected DocumentService documentService;
    protected FinancialSystemDocumentService financialSystemDocumentService;
    protected GeneralLedgerPendingEntryService generalLedgerPendingEntryService;
    protected KualiModuleService kualiModuleService;
    protected ParameterService parameterService;
    protected WorkflowDocumentService workflowDocumentService;
    protected PaymentApplicationDocumentService paymentApplicationDocumentService;

    @Override
    @Transactional
    public void routeCashControlDocument(CashControlDocument cashControlDocument, PrintWriter errorFile) {
        LOG.info("Routing Cash control document # {}.", cashControlDocument::getDocumentNumber);
        documentService.prepareWorkflowDocument(cashControlDocument);

        // calling workflow service to bypass business rule checks
        workflowDocumentService.route(cashControlDocument.getDocumentHeader().getWorkflowDocument(), "", null);
    }

    @Override
    public void processLettersOfCredit(String batchFileDirectoryName) {
        String runtimeStamp = dateTimeService.toDateTimeStringForFilename(new java.util.Date());

        String errOutputFile = batchFileDirectoryName + File.separator +
                ArConstants.BatchFileSystem.LOC_CREATION_BY_LOCF_ERROR_OUTPUT_FILE + "_" + runtimeStamp +
                ArConstants.BatchFileSystem.EXTENSION;
        File errOutPutFile = new File(errOutputFile);

        try (PrintWriter errorFile = new PrintWriter(errOutPutFile, StandardCharsets.UTF_8)) {
            Collection<ContractsGrantsInvoiceDocument> cgInvoices = retrieveLetterOfCreditInvoices();
            if (CollectionUtils.isNotEmpty(cgInvoices)) {
                CashControlDocument cashControlDoc = createCashControlDocument(errorFile);
                for (ContractsGrantsInvoiceDocument cgInvoice : cgInvoices) {
                    if (cgInvoice.getOpenAmount().isGreaterThan(KualiDecimal.ZERO)) {
                        processLetterOfCreditInvoice(cgInvoice, cashControlDoc, errorFile);
                    } else {
                        String errorString = configService.getPropertyValueAsString(ArKeyConstants.LOC_CREATION_ERROR_INVOICE_PAID);
                        errorFile.println(MessageFormat.format(errorString, cgInvoice.getDocumentNumber()));
                    }
                }
                routeCashControlDocument(cashControlDoc, errorFile);
            } else {
                String errorString = configService.getPropertyValueAsString(ArKeyConstants.LOC_CREATION_ERROR_NO_INVOICES_TO_PROCESS);
                errorFile.println(errorString);
            }
        } catch (IOException e) {
            throw new RuntimeException("Could not write to file in " + batchFileDirectoryName, e);
        }
    }

    @Override
    @Transactional
    public void processLetterOfCreditInvoice(ContractsGrantsInvoiceDocument cgInvoice,
            CashControlDocument cashControlDoc, PrintWriter errorFile) {
        CashControlDetail cashControlDetail = createCashControlDetail(cgInvoice);
        cashControlDocumentService.addNewCashControlDetail(
                configService.getPropertyValueAsString(ArKeyConstants.CREATED_BY_CASH_CTRL_DOC), cashControlDoc,
                cashControlDetail);
        String payAppDocNumber = cashControlDetail.getReferenceFinancialDocumentNumber();
        PaymentApplicationDocument payAppDoc;
        payAppDoc = (PaymentApplicationDocument) documentService.getByDocumentHeaderId(payAppDocNumber);
        payAppDoc = paymentApplicationDocumentService.createInvoicePaidAppliedsForEntireInvoiceDocument(cgInvoice,
                payAppDoc);
        getGeneralLedgerPendingEntryService().generateGeneralLedgerPendingEntries(payAppDoc);
        documentService.saveDocument(payAppDoc);

        LOG.info("Routing Payment Application document # {}.", payAppDoc::getDocumentNumber);
        documentService.prepareWorkflowDocument(payAppDoc);
        // calling workflow service to bypass business rule checks
        workflowDocumentService.route(payAppDoc.getDocumentHeader().getWorkflowDocument(), "", null);
    }

    @Override
    @Transactional
    public CashControlDocument createCashControlDocument(PrintWriter errorFile) {
        CashControlDocument cashControlDoc = null;

        cashControlDoc = (CashControlDocument) documentService.getNewDocument(CashControlDocument.class);
        cashControlDoc.getDocumentHeader().setDocumentDescription(
                configService.getPropertyValueAsString(ArKeyConstants.CASH_CTRL_DOC_CREATED_BY_BATCH));
        AccountsReceivableDocumentHeader accountsReceivableDocumentHeader = new AccountsReceivableDocumentHeader();
        accountsReceivableDocumentHeader.setDocumentNumber(cashControlDoc.getDocumentNumber());
        String defaultProcessingChartCode = parameterService.getParameterValueAsString(CashControlDocument.class,
                ArParameterConstants.DEFAULT_PROCESSING_CHART);
        String defaultProcessingOrgCode = parameterService.getParameterValueAsString(CashControlDocument.class,
                ArParameterConstants.DEFAULT_PROCESSING_ORG);
        accountsReceivableDocumentHeader.setProcessingChartOfAccountCode(defaultProcessingChartCode);
        accountsReceivableDocumentHeader.setProcessingOrganizationCode(defaultProcessingOrgCode);
        cashControlDoc.setAccountsReceivableDocumentHeader(accountsReceivableDocumentHeader);
        cashControlDoc.setCustomerPaymentMediumCode(ArConstants.PaymentMediumCode.WIRE_TRANSFER);
        documentService.saveDocument(cashControlDoc);

        return cashControlDoc;
    }

    /**
     * This method created cash control documents and payment application based on the loc creation type and loc value
     * passed.
     *
     * @param contractsGrantsInvoiceDocument
     * @return
     */
    protected CashControlDetail createCashControlDetail(ContractsGrantsInvoiceDocument contractsGrantsInvoiceDocument) {
        CashControlDetail cashControlDetail = new CashControlDetail();
        cashControlDetail.setCustomerNumber(contractsGrantsInvoiceDocument.getAccountsReceivableDocumentHeader().getCustomerNumber());
        cashControlDetail.setFinancialDocumentLineAmount(contractsGrantsInvoiceDocument.getOpenAmount());
        Timestamp ts = new Timestamp(new java.util.Date().getTime());
        java.sql.Date today = new java.sql.Date(ts.getTime());
        cashControlDetail.setCustomerPaymentDate(today);

        return cashControlDetail;
    }

    /**
     * Retrieves ContractsAndGrantsInvoiceDocument documents which are open and which have the given letter of credit
     * fund and letter of credit creation type (either fund or fund group)
     *
     * @return a Collection of matching letter of credit created Contracts & Grants Invoices
     */
    protected Collection<ContractsGrantsInvoiceDocument> retrieveLetterOfCreditInvoices() {
        Map<String, String> fieldValues = new HashMap<>();
        fieldValues.put(ArPropertyConstants.ContractsGrantsInvoiceDocumentFields.LETTER_OF_CREDIT_CREATION_TYPE,
                SearchOperator.NOT_NULL.op());
        fieldValues.put(ArPropertyConstants.OPEN_INVOICE_IND, KFSConstants.Booleans.TRUE);
        fieldValues.put(ArPropertyConstants.DOCUMENT_STATUS_CODE, KFSConstants.DocumentStatusCodes.APPROVED);
        return contractsGrantsInvoiceDocumentService.retrieveAllCGInvoicesByCriteria(fieldValues);
    }

    public DocumentService getDocumentService() {
        return documentService;
    }

    public void setDocumentService(DocumentService documentService) {
        this.documentService = documentService;
    }

    public CashControlDocumentService getCashControlDocumentService() {
        return cashControlDocumentService;
    }

    public void setCashControlDocumentService(CashControlDocumentService cashControlDocumentService) {
        this.cashControlDocumentService = cashControlDocumentService;
    }

    public WorkflowDocumentService getWorkflowDocumentService() {
        return workflowDocumentService;
    }

    public void setWorkflowDocumentService(WorkflowDocumentService workflowDocumentService) {
        this.workflowDocumentService = workflowDocumentService;
    }

    public FinancialSystemDocumentService getFinancialSystemDocumentService() {
        return financialSystemDocumentService;
    }

    public void setFinancialSystemDocumentService(FinancialSystemDocumentService financialSystemDocumentService) {
        this.financialSystemDocumentService = financialSystemDocumentService;
    }

    public DateTimeService getDateTimeService() {
        return dateTimeService;
    }

    public void setDateTimeService(DateTimeService dateTimeService) {
        this.dateTimeService = dateTimeService;
    }

    public ContractsGrantsInvoiceDocumentService getContractsGrantsInvoiceDocumentService() {
        return contractsGrantsInvoiceDocumentService;
    }

    public void setContractsGrantsInvoiceDocumentService(ContractsGrantsInvoiceDocumentService contractsGrantsInvoiceDocumentService) {
        this.contractsGrantsInvoiceDocumentService = contractsGrantsInvoiceDocumentService;
    }

    public GeneralLedgerPendingEntryService getGeneralLedgerPendingEntryService() {
        return generalLedgerPendingEntryService;
    }

    public void setGeneralLedgerPendingEntryService(GeneralLedgerPendingEntryService generalLedgerPendingEntryService) {
        this.generalLedgerPendingEntryService = generalLedgerPendingEntryService;
    }

    public KualiModuleService getKualiModuleService() {
        return kualiModuleService;
    }

    public void setKualiModuleService(KualiModuleService kualiModuleService) {
        this.kualiModuleService = kualiModuleService;
    }

    public ConfigurationService getConfigService() {
        return configService;
    }

    public void setConfigService(ConfigurationService configService) {
        this.configService = configService;
    }

    public ParameterService getParameterService() {
        return parameterService;
    }

    public void setParameterService(ParameterService parameterService) {
        this.parameterService = parameterService;
    }

    public PaymentApplicationDocumentService getPaymentApplicationDocumentService() {
        return paymentApplicationDocumentService;
    }

    public void setPaymentApplicationDocumentService(PaymentApplicationDocumentService paymentApplicationDocumentService) {
        this.paymentApplicationDocumentService = paymentApplicationDocumentService;
    }
}
