/*
 * The Kuali Financial System, a comprehensive financial management system for higher education.
 *
 * Copyright 2005-2022 Kuali, Inc.
 *
 * This program is free software: you can redistribute it and/or modify
 * it under the terms of the GNU Affero General Public License as published by
 * the Free Software Foundation, either version 3 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU Affero General Public License for more details.
 *
 * You should have received a copy of the GNU Affero General Public License
 * along with this program.  If not, see <http://www.gnu.org/licenses/>.
 */
package org.kuali.kfs.module.ar.businessobject;

import org.apache.commons.lang3.StringUtils;
import org.kuali.kfs.core.api.mo.common.active.MutableInactivatable;
import org.kuali.kfs.core.api.util.type.KualiDecimal;
import org.kuali.kfs.integration.cg.ContractsAndGrantsBillingAward;
import org.kuali.kfs.integration.cg.ContractsAndGrantsModuleBillingService;
import org.kuali.kfs.krad.util.ObjectUtils;
import org.kuali.kfs.sys.context.SpringContext;

import java.sql.Date;
import java.util.List;
import java.util.Objects;

/**
 * Milestone to be used for Milestone Schedule under Contracts & Grants
 */
public class Milestone extends MilestoneBase implements MutableInactivatable {

    private String proposalNumber;
    private String chartOfAccountsCode;
    private String accountNumber;
    private Date milestoneExpectedCompletionDate;
    private boolean billed = false;
    private boolean active;
    private List<InvoiceMilestone> invoiceMilestones;

    private ContractsAndGrantsBillingAward award;
    private transient ContractsAndGrantsModuleBillingService contractsAndGrantsModuleBillingService;

    public Milestone() {
    }

    private Milestone(String proposalNumber, String chartOfAccountsCode, String accountNumber,
            Date milestoneExpectedCompletionDate, boolean billed, boolean active, String milestoneNumber,
            String milestoneDescription, KualiDecimal milestoneAmount, Date milestoneActualCompletionDate) {
        this.proposalNumber = proposalNumber;
        this.chartOfAccountsCode = chartOfAccountsCode;
        this.accountNumber = accountNumber;
        this.milestoneExpectedCompletionDate = milestoneExpectedCompletionDate;
        this.billed = billed;
        this.active = active;
        this.milestoneNumber = milestoneNumber;
        this.milestoneDescription = milestoneDescription;
        this.milestoneAmount = milestoneAmount;
        this.milestoneActualCompletionDate = milestoneActualCompletionDate;
    }

    public String getProposalNumber() {
        return proposalNumber;
    }

    public void setProposalNumber(String proposalNumber) {
        this.proposalNumber = proposalNumber;
    }

    public String getChartOfAccountsCode() {
        return chartOfAccountsCode;
    }

    public void setChartOfAccountsCode(String chartOfAccountsCode) {
        this.chartOfAccountsCode = chartOfAccountsCode;
    }

    public String getAccountNumber() {
        return accountNumber;
    }

    public void setAccountNumber(String accountNumber) {
        this.accountNumber = accountNumber;
    }

    public Date getMilestoneExpectedCompletionDate() {
        return milestoneExpectedCompletionDate;
    }

    public void setMilestoneExpectedCompletionDate(Date milestoneExpectedCompletionDate) {
        this.milestoneExpectedCompletionDate = milestoneExpectedCompletionDate;
    }

    public boolean isBilled() {
        return billed;
    }

    public void setBilled(boolean billed) {
        this.billed = billed;
    }

    @Override
    public boolean isActive() {
        return active;
    }

    @Override
    public void setActive(boolean active) {
        this.active = active;
    }

    public List<InvoiceMilestone> getInvoiceMilestones() {
        return invoiceMilestones;
    }

    public void setInvoiceMilestones(List<InvoiceMilestone> invoiceMilestones) {
        this.invoiceMilestones = invoiceMilestones;
    }

    public ContractsAndGrantsBillingAward getAward() {
        final ContractsAndGrantsBillingAward updatedAward = getContractsAndGrantsModuleBillingService().
                updateAwardIfNecessary(proposalNumber, award);

        // If the updated award is null, we return the original award instead so that we don't get an NPE in the case
        // where it has been set by PojoPropertyUtilsBean.
        if (ObjectUtils.isNull(updatedAward)) {
            return award;
        }
        return updatedAward;
    }

    public void setAward(ContractsAndGrantsBillingAward award) {
        this.award = award;
    }

    @Override
    public boolean equals(Object o) {
        if (this == o) {
            return true;
        }
        if (!(o instanceof Milestone)) {
            return false;
        }
        if (!super.equals(o)) {
            return false;
        }
        Milestone milestone = (Milestone) o;
        return billed == milestone.billed &&
                active == milestone.active &&
                Objects.equals(proposalNumber, milestone.proposalNumber) &&
                Objects.equals(chartOfAccountsCode, milestone.chartOfAccountsCode) &&
                Objects.equals(accountNumber, milestone.accountNumber) &&
                Objects.equals(milestoneExpectedCompletionDate, milestone.milestoneExpectedCompletionDate);
    }

    @Override
    public int hashCode() {
        return Objects.hash(super.hashCode(), proposalNumber, chartOfAccountsCode, accountNumber,
                milestoneExpectedCompletionDate, billed, active);
    }

    protected ContractsAndGrantsModuleBillingService getContractsAndGrantsModuleBillingService() {
        if (contractsAndGrantsModuleBillingService == null) {
            contractsAndGrantsModuleBillingService = SpringContext.getBean(ContractsAndGrantsModuleBillingService.class);
        }
        return contractsAndGrantsModuleBillingService;
    }

    protected void setContractsAndGrantsModuleBillingService(
            ContractsAndGrantsModuleBillingService contractsAndGrantsModuleBillingService) {
        this.contractsAndGrantsModuleBillingService = contractsAndGrantsModuleBillingService;
    }

    public static class MilestoneBuilder {
        private String proposalNumber;
        private String chartOfAccountsCode;
        private String accountNumber;
        private Date milestoneExpectedCompletionDate;
        private boolean billed;
        private boolean active;
        private String milestoneNumber;
        private String milestoneDescription;
        private KualiDecimal milestoneAmount;
        private Date milestoneActualCompletionDate;

        public MilestoneBuilder setProposalNumber(String proposalNumber) {
            this.proposalNumber = proposalNumber;
            return this;
        }

        public MilestoneBuilder setChartOfAccountsCode(String chartOfAccountsCode) {
            this.chartOfAccountsCode = chartOfAccountsCode;
            return this;
        }

        public MilestoneBuilder setAccountNumber(String accountNumber) {
            this.accountNumber = accountNumber;
            return this;
        }

        public MilestoneBuilder setMilestoneExpectedCompletionDate(Date milestoneExpectedCompletionDate) {
            this.milestoneExpectedCompletionDate = milestoneExpectedCompletionDate;
            return this;
        }

        public MilestoneBuilder setBilled(boolean billed) {
            this.billed = billed;
            return this;
        }

        public MilestoneBuilder setActive(boolean active) {
            this.active = active;
            return this;
        }

        public MilestoneBuilder setMilestoneNumber(String milestoneNumber) {
            this.milestoneNumber = milestoneNumber;
            return this;
        }

        public MilestoneBuilder setMilestoneDescription(String milestoneDescription) {
            this.milestoneDescription = milestoneDescription;
            return this;
        }

        public MilestoneBuilder setMilestoneAmount(KualiDecimal milestoneAmount) {
            this.milestoneAmount = milestoneAmount;
            return this;
        }

        public MilestoneBuilder setMilestoneActualCompletionDate(Date milestoneActualCompletionDate) {
            this.milestoneActualCompletionDate = milestoneActualCompletionDate;
            return this;
        }

        public Milestone build() throws IllegalStateException {
            validate();
            return new Milestone(proposalNumber, chartOfAccountsCode, accountNumber, milestoneExpectedCompletionDate,
                    billed, active, milestoneNumber, milestoneDescription, milestoneAmount,
                    milestoneActualCompletionDate);
        }

        private void validate() {
            if (StringUtils.isBlank(proposalNumber)) {
                throw new IllegalStateException("Proposal Number is required.");
            }
            if (StringUtils.isBlank(chartOfAccountsCode)) {
                throw new IllegalStateException("Chart of Accounts Code is required.");
            }
            if (StringUtils.isBlank(accountNumber)) {
                throw new IllegalStateException("Account Number is required.");
            }
            if (StringUtils.isBlank(milestoneNumber)) {
                throw new IllegalStateException("Milestone Number is required.");
            }
            if (milestoneExpectedCompletionDate == null) {
                throw new IllegalStateException("Milestone Expected Completion Date is required.");
            }
        }
    }
}
