/*
 * The Kuali Financial System, a comprehensive financial management system for higher education.
 *
 * Copyright 2005-2022 Kuali, Inc.
 *
 * This program is free software: you can redistribute it and/or modify
 * it under the terms of the GNU Affero General Public License as published by
 * the Free Software Foundation, either version 3 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU Affero General Public License for more details.
 *
 * You should have received a copy of the GNU Affero General Public License
 * along with this program.  If not, see <http://www.gnu.org/licenses/>.
 */
package org.kuali.kfs.module.ar.businessobject.actions;

import org.kuali.kfs.datadictionary.Action;
import org.kuali.kfs.datadictionary.ActionType;
import org.kuali.kfs.datadictionary.BusinessObjectAdminService;
import org.kuali.kfs.kim.impl.identity.Person;
import org.kuali.kfs.kns.datadictionary.BusinessObjectEntry;
import org.kuali.kfs.krad.bo.BusinessObjectBase;
import org.kuali.kfs.krad.bo.ModuleConfiguration;
import org.kuali.kfs.krad.service.KualiModuleService;
import org.kuali.kfs.krad.util.KRADConstants;
import org.kuali.kfs.krad.util.UrlFactory;
import org.kuali.kfs.module.ar.ArConstants;
import org.kuali.kfs.module.ar.businessobject.TemplateBase;
import org.kuali.kfs.sys.FinancialSystemModuleConfiguration;
import org.kuali.kfs.sys.KFSConstants;
import org.kuali.kfs.sys.KFSPropertyConstants;
import org.kuali.kfs.sys.businessobject.actions.BusinessObjectActionsProvider;

import java.io.File;
import java.util.HashMap;
import java.util.List;
import java.util.Map;

public class TemplateActionsProvider extends BusinessObjectActionsProvider {
    private KualiModuleService kualiModuleService;
    private String baseUrl;
    private String templateCodeKey;

    @Override
    public List<Action> getActionLinks(BusinessObjectBase businessObject, Person user) {
        final List<Action> actionLinks = super.getActionLinks(businessObject, user);

        BusinessObjectAdminService businessObjectAdminService = businessObjectDictionaryService
                .getBusinessObjectAdminService(businessObject.getClass());

        BusinessObjectEntry businessObjectEntry = businessObjectDictionaryService.getBusinessObjectEntry(
                businessObject.getClass().getName());

        if (businessObjectEntry.supportsAction(ActionType.EDIT)
                && businessObjectAdminService.allowsEdit(businessObject, user)) {
            final String uploadUrl = generateUploadUrl(businessObject);
            final Action upAction = new Action("Upload", "GET", uploadUrl);
            actionLinks.add(upAction);
            if (templateFileExists(((TemplateBase) businessObject).getFilename())) {
                final String downloadUrl = generateDownloadUrl(businessObject);
                final Action downAction = new Action("Download", "GET", downloadUrl);
                actionLinks.add(downAction);
            }
        }

        return actionLinks;
    }

    private String generateUploadUrl(BusinessObjectBase businessObject) {
        Map<String, String> parameters = new HashMap<>();
        parameters.put(KFSConstants.DISPATCH_REQUEST_PARAMETER, KFSConstants.START_METHOD);
        parameters.put(templateCodeKey, ((TemplateBase) businessObject).getTemplateCode());
        return UrlFactory.parameterizeUrl(baseUrl, parameters);
    }

    private String generateDownloadUrl(BusinessObjectBase businessObject) {
        Map<String, String> parameters = new HashMap<>();
        parameters.put(KRADConstants.DISPATCH_REQUEST_PARAMETER, ArConstants.DOWNLOAD_METHOD);
        parameters.put(KFSPropertyConstants.FILE_NAME, ((TemplateBase) businessObject).getFilename());
        return UrlFactory.parameterizeUrl(baseUrl, parameters);
    }

    /**
     * Check if the template file actually exists, even though we have a fileName
     *
     * @param fileName name of template file
     * @return true if template file exists, false otherwise
     */
    private boolean templateFileExists(String fileName) {
        ModuleConfiguration systemConfiguration = kualiModuleService
                .getModuleServiceByNamespaceCode(KFSConstants.OptionalModuleNamespaces.ACCOUNTS_RECEIVABLE)
                .getModuleConfiguration();
        String templateFolderPath = ((FinancialSystemModuleConfiguration) systemConfiguration)
                .getTemplateFileDirectories().get(KFSConstants.TEMPLATES_DIRECTORY_KEY);
        String filePath = templateFolderPath + File.separator + fileName;
        return new File(filePath).isFile();
    }

    public KualiModuleService getKualiModuleService() {
        return kualiModuleService;
    }

    public void setKualiModuleService(KualiModuleService kualiModuleService) {
        this.kualiModuleService = kualiModuleService;
    }

    public void setBaseUrl(String baseUrl) {
        this.baseUrl = baseUrl;
    }

    public void setTemplateCodeKey(String templateCodeKey) {
        this.templateCodeKey = templateCodeKey;
    }
}
