/*
 * The Kuali Financial System, a comprehensive financial management system for higher education.
 *
 * Copyright 2005-2022 Kuali, Inc.
 *
 * This program is free software: you can redistribute it and/or modify
 * it under the terms of the GNU Affero General Public License as published by
 * the Free Software Foundation, either version 3 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU Affero General Public License for more details.
 *
 * You should have received a copy of the GNU Affero General Public License
 * along with this program.  If not, see <http://www.gnu.org/licenses/>.
 */
package org.kuali.kfs.module.ar.businessobject.lookup;

import org.apache.commons.lang3.StringUtils;
import org.kuali.kfs.core.web.format.DateFormatter;
import org.kuali.kfs.core.web.format.Formatter;
import org.kuali.kfs.kns.lookup.CollectionIncomplete;
import org.kuali.kfs.kns.lookup.HtmlData.AnchorHtmlData;
import org.kuali.kfs.kns.lookup.HtmlData.MultipleAnchorHtmlData;
import org.kuali.kfs.kns.lookup.KualiLookupableHelperServiceImpl;
import org.kuali.kfs.kns.web.comparator.CellComparatorHelper;
import org.kuali.kfs.kns.web.struts.form.LookupForm;
import org.kuali.kfs.kns.web.ui.Column;
import org.kuali.kfs.kns.web.ui.ResultRow;
import org.kuali.kfs.krad.bo.BusinessObject;
import org.kuali.kfs.krad.bo.PersistableBusinessObject;
import org.kuali.kfs.krad.util.KRADConstants;
import org.kuali.kfs.krad.util.ObjectUtils;
import org.kuali.kfs.krad.util.UrlFactory;
import org.kuali.kfs.module.ar.ArConstants;
import org.kuali.kfs.module.ar.businessobject.ContractsGrantsAgingOpenInvoicesReport;
import org.kuali.kfs.module.ar.businessobject.TicklersReport;
import org.kuali.kfs.module.ar.document.ContractsGrantsInvoiceDocument;
import org.kuali.kfs.module.ar.document.service.ContractsGrantsAgingOpenInvoicesReportService;
import org.kuali.kfs.module.ar.report.service.ContractsGrantsReportHelperService;
import org.kuali.kfs.sys.KFSConstants;
import org.kuali.kfs.sys.KFSPropertyConstants;

import java.util.Collection;
import java.util.Date;
import java.util.HashMap;
import java.util.List;
import java.util.Map;

// still used in string reference in ContractsGrantsAgingReportLookupableHelperServiceImpl
public class ContractsGrantsAgingOpenInvoicesReportLookupableHelperServiceImpl extends KualiLookupableHelperServiceImpl {

    protected ContractsGrantsAgingOpenInvoicesReportService contractsGrantsAgingOpenInvoicesReportService;
    protected ContractsGrantsReportHelperService contractsGrantsReportHelperService;

    /**
     * Get the search results that meet the input search criteria.
     *
     * @param fieldValues Map containing prop name keys and search values
     * @return a List of found business objects
     */
    @Override
    public List<? extends BusinessObject> getSearchResults(Map<String, String> fieldValues) {
        setBackLocation(fieldValues.get(KFSConstants.BACK_LOCATION));
        setDocFormKey(fieldValues.get(KFSConstants.DOC_FORM_KEY));
        List<? extends BusinessObject> results = getContractsGrantsAgingOpenInvoicesReportService()
                .getPopulatedReportDetails(getParameters());
        return new CollectionIncomplete(results, (long) results.size());
    }

    /**
     * @return a List of the names of fields which are marked in data dictionary as return fields.
     */
    @Override
    protected Map<String, String> getParameters(BusinessObject bo, Map<String, String> fieldConversions,
            String lookupImpl, List returnKeys) {
        Map<String, String> parameters = new HashMap<>();
        parameters.put(KRADConstants.DISPATCH_REQUEST_PARAMETER, KRADConstants.RETURN_METHOD_TO_CALL);
        parameters.put(KRADConstants.DOC_FORM_KEY, getDocFormKey());
        parameters.put(KRADConstants.REFRESH_CALLER, lookupImpl);
        if (getReferencesToRefresh() != null) {
            parameters.put(KRADConstants.REFERENCES_TO_REFRESH, getReferencesToRefresh());
        }

        for (Object entry : getReturnKeys()) {
            String fieldNm = (String) entry;
            Object fieldVal = ObjectUtils.getPropertyValue(bo, fieldNm);
            if (fieldVal == null) {
                fieldVal = KFSConstants.EMPTY_STRING;
            }

            // Encrypt value if it is a secure field
            // TODO: revisit authorization checks in this class

            if (fieldConversions.containsKey(fieldNm)) {
                fieldNm = fieldConversions.get(fieldNm);
            }

            // need to format date in url
            if (fieldVal instanceof Date) {
                DateFormatter dateFormatter = new DateFormatter();
                fieldVal = dateFormatter.format(fieldVal);
            }

            parameters.put(fieldNm, fieldVal.toString());
        }

        return parameters;
    }

    @Override
    public Collection<? extends BusinessObject> performLookup(LookupForm lookupForm, Collection<ResultRow> resultTable,
            boolean bounded) {
        List<? extends BusinessObject> displayList;
        // call search method to get results
        if (bounded) {
            displayList = getSearchResults(lookupForm.getFieldsForLookup());
        } else {
            displayList = getSearchResultsUnbounded(lookupForm.getFieldsForLookup());
        }
        // MJM get resultTable populated here

        HashMap<String, Class> propertyTypes = new HashMap<>();

        boolean hasReturnableRow = false;

        // iterate through result list and wrap rows with return url and action urls
        for (BusinessObject element : displayList) {
            List<Column> columns = getColumns();
            for (Column col : columns) {
                Formatter formatter = col.getFormatter();

                // pick off result column from result list, do formatting
                String propValue = KFSConstants.EMPTY_STRING;
                Object prop = ObjectUtils.getPropertyValue(element, col.getPropertyName());

                // formatters
                if (prop != null) {
                    propValue = getContractsGrantsReportHelperService().formatByType(prop, formatter);
                }

                // comparator
                Class propClass = propertyTypes.get(col.getPropertyName());
                col.setComparator(CellComparatorHelper.getAppropriateComparatorForPropertyClass(propClass));
                col.setValueComparator(CellComparatorHelper.getAppropriateValueComparatorForPropertyClass(propClass));

                col.setPropertyValue(propValue);

                if (StringUtils.isNotBlank(propValue)) {
                    if (StringUtils.equals(KFSConstants.CustomerOpenItemReport.DOCUMENT_NUMBER,
                            col.getPropertyName())) {
                        String propertyURL = contractsGrantsReportHelperService.getDocSearchUrl(propValue);
                        col.setPropertyURL(propertyURL);
                    } else if (StringUtils.equals("Actions", col.getColumnTitle())) {
                        ContractsGrantsAgingOpenInvoicesReport detail = (ContractsGrantsAgingOpenInvoicesReport) element;
                        String proposalNumber = getProposalNumber(detail);

                        if (StringUtils.isNotBlank(proposalNumber)) {
                            AnchorHtmlData collectionActivity = new AnchorHtmlData(
                                    getContractsGrantsReportHelperService().getInitiateCollectionActivityDocumentUrl(
                                            proposalNumber, detail.getDocumentNumber()),
                                    KFSConstants.EMPTY_STRING, col.getPropertyValue());
                            collectionActivity.setAppendDisplayText(ACTION_URLS_SEPARATOR);

                            AnchorHtmlData tickler = new AnchorHtmlData(getTicklerUrl(proposalNumber),
                                    KFSConstants.EMPTY_STRING, ArConstants.TICKLER_REPORT);

                            List<AnchorHtmlData> anchorLists = List.of(collectionActivity, tickler);
                            col.setColumnAnchor(new MultipleAnchorHtmlData(anchorLists));
                        }
                    } else {
                        col.setPropertyURL("");
                    }
                }
            }

            ResultRow row = new ResultRow(columns, KFSConstants.EMPTY_STRING, KFSConstants.EMPTY_STRING);
            if (element instanceof PersistableBusinessObject) {
                row.setObjectId(((PersistableBusinessObject) element).getObjectId());
            }

            boolean rowReturnable = isResultReturnable(element);
            row.setRowReturnable(rowReturnable);
            if (rowReturnable) {
                hasReturnableRow = true;
            }

            resultTable.add(row);
        }
        lookupForm.setHasReturnableRow(hasReturnableRow);

        return displayList;
    }

    /**
     * This method returns the proposal number extracted from the invoice report detail
     *
     * @param ContractsGrantsAgingOpenInvoicesReport
     * @return Proposal Number
     */
    protected String getProposalNumber(ContractsGrantsAgingOpenInvoicesReport detail) {
        ContractsGrantsInvoiceDocument cgInvoice = getBusinessObjectService()
                .findBySinglePrimaryKey(ContractsGrantsInvoiceDocument.class, detail.getDocumentNumber());
        return (ObjectUtils.isNotNull(cgInvoice.getInvoiceGeneralDetail())
                && ObjectUtils.isNotNull(cgInvoice.getInvoiceGeneralDetail().getProposalNumber()))
                    ? cgInvoice.getInvoiceGeneralDetail().getProposalNumber() : null;
    }

    /**
     * This method returns the Tickler Report URL
     *
     * @param proposalNumber
     * @return Returns the url for the Tickler Report
     */
    protected String getTicklerUrl(String proposalNumber) {
        Map<String, String> parameters = new HashMap<>();
        parameters.put(KRADConstants.DISPATCH_REQUEST_PARAMETER, KFSConstants.SEARCH_METHOD);
        parameters.put(KFSPropertyConstants.PROPOSAL_NUMBER, proposalNumber);
        parameters.put(KFSConstants.BUSINESS_OBJECT_CLASS_ATTRIBUTE, TicklersReport.class.getName());
        parameters.put(KFSConstants.DOC_FORM_KEY, "88888888");
        parameters.put(KFSConstants.HIDE_LOOKUP_RETURN_LINK, "true");
        return UrlFactory.parameterizeUrl(ArConstants.UrlActions.TICKLER_REPORT, parameters);
    }

    public ContractsGrantsAgingOpenInvoicesReportService getContractsGrantsAgingOpenInvoicesReportService() {
        return contractsGrantsAgingOpenInvoicesReportService;
    }

    public void setContractsGrantsAgingOpenInvoicesReportService(
            ContractsGrantsAgingOpenInvoicesReportService contractsGrantsAgingOpenInvoicesReportService) {
        this.contractsGrantsAgingOpenInvoicesReportService = contractsGrantsAgingOpenInvoicesReportService;
    }

    public ContractsGrantsReportHelperService getContractsGrantsReportHelperService() {
        return contractsGrantsReportHelperService;
    }

    public void setContractsGrantsReportHelperService(
            ContractsGrantsReportHelperService contractsGrantsReportHelperService) {
        this.contractsGrantsReportHelperService = contractsGrantsReportHelperService;
    }
}
