/*
 * The Kuali Financial System, a comprehensive financial management system for higher education.
 *
 * Copyright 2005-2022 Kuali, Inc.
 *
 * This program is free software: you can redistribute it and/or modify
 * it under the terms of the GNU Affero General Public License as published by
 * the Free Software Foundation, either version 3 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU Affero General Public License for more details.
 *
 * You should have received a copy of the GNU Affero General Public License
 * along with this program.  If not, see <http://www.gnu.org/licenses/>.
 */
package org.kuali.kfs.module.ar.document.service;

import org.kuali.kfs.integration.cg.ContractsAndGrantsBillingAward;
import org.kuali.kfs.integration.cg.ContractsAndGrantsBillingAwardAccount;
import org.kuali.kfs.module.ar.businessobject.Bill;
import org.kuali.kfs.module.ar.businessobject.Milestone;

import java.util.Collection;
import java.util.List;

/**
 * Methods which test various aspects of awards, verifying that the award can be used to create a contracts & grants invoice
 */
public interface ContractsGrantsBillingAwardVerificationService {

    /**
     * Check if the value of Billing Frequency Code is in the value set.
     *
     * @param award
     * @return
     */
    boolean isValueOfBillingFrequencyValid(ContractsAndGrantsBillingAward award);

    /**
     * Check if the final Invoice for all accounts in the invoice have already been built.
     *
     * @param award
     * @return
     */
    boolean isAwardFinalInvoiceAlreadyBuilt(ContractsAndGrantsBillingAward award);


    /**
     * Checks if the awardAccount has valid milestones to invoice if the award is MS billing or if the awardAccount has
     * valid bills to invoice if the award is PDBS billing.
     *
     * @param awardAccount to check for valid milestones/bills to invoice
     * @return true if the awardAccount is MS and has valid milestones to invoice, is PDBS and has valid milestones or
     * the award is not MS or PDBS billing, false if not.
     */
    boolean isAwardAccountValidToInvoiceBasedOnSchedule(ContractsAndGrantsBillingAwardAccount awardAccount);

    /**
     * Checks if the award has valid milestones to invoice.
     *
     * @param award to check for valid milestones to invoice
     * @return true if has valid milestones to invoice. false if not.
     */
    boolean hasMilestonesToInvoice(ContractsAndGrantsBillingAward award);

    /**
     * For the list of milestones passed in determine which ones are eligible to be invoiced and return a list of those.
     * @param milestones collection of milestones to check to see if they can be invoiced
     * @return list of milestones that can be invoiced
     */
    List<Milestone> determineMilestonesToInvoice(Collection<Milestone> milestones);

    /**
     * Checks if the award has valid bills to invoice.
     *
     * @param award to check for valid bills to invoice
     * @return true if has valid bills to invoice. false if not.
     */
    boolean hasBillsToInvoice(ContractsAndGrantsBillingAward award);

    /**
     * For the list of bills passed in determine which ones are eligible to be invoiced and return a list of those.
     * @param bills collection of bills to check to see if they can be invoiced
     * @return list of bills that can be invoiced
     */
    List<Bill> determineBillsToInvoice(Collection<Bill> bills);

    /**
     * Check if agency owning award has no customer record
     *
     * @param award
     * @return
     */
    boolean owningAgencyHasCustomerRecord(ContractsAndGrantsBillingAward award);

    /**
     * This method checks if the System Information and Organization Accounting Default are setup for the Chart Code and Org Code
     * from the award accounts.
     *
     * @param award
     * @return
     */
    boolean isChartAndOrgSetupForInvoicing(ContractsAndGrantsBillingAward award);

    /**
     * this method checks If all accounts of award has invoices in progress.
     *
     * @param award
     * @return
     */
    boolean isInvoiceInProgress(ContractsAndGrantsBillingAward award);

    /**
     * @param award award to validate
     * @return true if Income and AR object codes are defined in the Contracts & Grants Invoice Object Code table for
     * this award, false otherwise
     */
    boolean hasIncomeAndReceivableObjectCodes(ContractsAndGrantsBillingAward award);
}
