/*
 * The Kuali Financial System, a comprehensive financial management system for higher education.
 *
 * Copyright 2005-2022 Kuali, Inc.
 *
 * This program is free software: you can redistribute it and/or modify
 * it under the terms of the GNU Affero General Public License as published by
 * the Free Software Foundation, either version 3 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU Affero General Public License for more details.
 *
 * You should have received a copy of the GNU Affero General Public License
 * along with this program.  If not, see <http://www.gnu.org/licenses/>.
 */
package org.kuali.kfs.module.ar.document.service.impl;

import org.kuali.kfs.core.api.util.type.KualiDecimal;
import org.kuali.kfs.kew.api.document.WorkflowDocumentService;
import org.kuali.kfs.kew.routeheader.DocumentRouteHeaderValue;
import org.kuali.kfs.krad.service.BusinessObjectService;
import org.kuali.kfs.module.ar.businessobject.AppliedPayment;
import org.kuali.kfs.module.ar.businessobject.InvoicePaidApplied;
import org.kuali.kfs.module.ar.document.CustomerInvoiceDocument;
import org.kuali.kfs.module.ar.document.service.InvoicePaidAppliedService;
import org.kuali.kfs.sys.service.UniversityDateService;
import org.springframework.transaction.annotation.Transactional;

import java.util.Collection;
import java.util.HashMap;
import java.util.List;
import java.util.Map;
import java.util.stream.Collectors;

@Transactional
public class InvoicePaidAppliedServiceImpl implements InvoicePaidAppliedService<AppliedPayment> {
    private BusinessObjectService businessObjectService;
    private UniversityDateService universityDateService;
    private WorkflowDocumentService workflowDocumentService;

    @Override
    public void clearDocumentPaidAppliedsFromDatabase(String documentNumber) {
        Map<String, String> fields = new HashMap<>();
        fields.put("documentNumber", documentNumber);
        businessObjectService.deleteMatching(InvoicePaidApplied.class, fields);
    }

    @Override
    public Integer getNumberOfInvoicePaidAppliedsForInvoiceDetail(String financialDocumentReferenceInvoiceNumber,
            Integer invoiceItemNumber) {
        Map<String, Object> criteria = new HashMap<>();
        criteria.put("financialDocumentReferenceInvoiceNumber", financialDocumentReferenceInvoiceNumber);
        criteria.put("invoiceItemNumber", invoiceItemNumber);

        return businessObjectService.countMatching(InvoicePaidApplied.class, criteria);
    }

    public Collection<InvoicePaidApplied> getInvoicePaidAppliedsFromSpecificDocument(String documentNumber,
            String referenceCustomerInvoiceDocumentNumber) {
        Map<String, String> criteria = new HashMap<>();
        criteria.put("financialDocumentReferenceInvoiceNumber", referenceCustomerInvoiceDocumentNumber);
        criteria.put("documentNumber", documentNumber);
        return businessObjectService.findMatching(InvoicePaidApplied.class, criteria);
    }

    @Override
    public boolean doesInvoiceHaveAppliedAmounts(CustomerInvoiceDocument document) {
        Collection<InvoicePaidApplied> results = getActiveInvoicePaidAppliedsForInvoice(document);

        for (InvoicePaidApplied invoicePaidApplied : results) {
            // don't include discount (the doc num and the ref num are the same document number)
            // or where applied amount is zero (could have been adjusted)
            if (!invoicePaidApplied.getDocumentNumber().equals(document.getDocumentNumber())
                    && invoicePaidApplied.getInvoiceItemAppliedAmount().isGreaterThan(KualiDecimal.ZERO)) {
                return true;
            }
        }
        return false;
    }

    @Override
    public Collection<InvoicePaidApplied> getInvoicePaidAppliedsForInvoice(String documentNumber) {
        Map<String, String> criteria = new HashMap<>();
        criteria.put("financialDocumentReferenceInvoiceNumber", documentNumber);
        return businessObjectService.findMatching(InvoicePaidApplied.class, criteria);
    }

    @Override
    public Collection<InvoicePaidApplied> getInvoicePaidAppliedsForInvoice(CustomerInvoiceDocument invoice) {
        return getInvoicePaidAppliedsForInvoice(invoice.getDocumentNumber());
    }

    @Override
    public List<InvoicePaidApplied> getActiveInvoicePaidAppliedsForInvoice(CustomerInvoiceDocument invoice) {
        return filterInvoicePaidAppliedsToOnlyActive((List<InvoicePaidApplied>) getInvoicePaidAppliedsForInvoice(invoice));
    }

    @Override
    public List<InvoicePaidApplied> filterInvoicePaidAppliedsToOnlyActive(List<InvoicePaidApplied> invoicePaidApplieds) {
        return invoicePaidApplieds.stream()
                .filter(invoicePaidApplied -> !invoicePaidApplied.isAdjusted())
                .filter(this::docIsNotCancelled)
                .collect(Collectors.toList());
    }

    private boolean docIsNotCancelled(final InvoicePaidApplied invoicePaidApplied) {
        final DocumentRouteHeaderValue document =
                workflowDocumentService.getDocument(invoicePaidApplied.getDocumentNumber());
        return !document.isCanceled();
    }

    public BusinessObjectService getBusinessObjectService() {
        return businessObjectService;
    }

    public void setBusinessObjectService(BusinessObjectService businessObjectService) {
        this.businessObjectService = businessObjectService;
    }

    public UniversityDateService getUniversityDateService() {
        return universityDateService;
    }

    public void setUniversityDateService(UniversityDateService universityDateService) {
        this.universityDateService = universityDateService;
    }

    public void setWorkflowDocumentService(final WorkflowDocumentService workflowDocumentService) {
        this.workflowDocumentService = workflowDocumentService;
    }
}
