/*
 * The Kuali Financial System, a comprehensive financial management system for higher education.
 *
 * Copyright 2005-2022 Kuali, Inc.
 *
 * This program is free software: you can redistribute it and/or modify
 * it under the terms of the GNU Affero General Public License as published by
 * the Free Software Foundation, either version 3 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU Affero General Public License for more details.
 *
 * You should have received a copy of the GNU Affero General Public License
 * along with this program.  If not, see <http://www.gnu.org/licenses/>.
 */
package org.kuali.kfs.module.ar.document.service.impl;

import org.apache.commons.lang3.StringUtils;
import org.kuali.kfs.core.api.datetime.DateTimeService;
import org.kuali.kfs.coreservice.framework.parameter.ParameterService;
import org.kuali.kfs.kew.api.WorkflowDocument;
import org.kuali.kfs.krad.service.BusinessObjectService;
import org.kuali.kfs.krad.service.DocumentService;
import org.kuali.kfs.krad.util.ObjectUtils;
import org.kuali.kfs.module.ar.ArParameterConstants;
import org.kuali.kfs.module.ar.businessobject.CustomerAddress;
import org.kuali.kfs.module.ar.businessobject.CustomerInvoiceDetail;
import org.kuali.kfs.module.ar.businessobject.InvoiceRecurrence;
import org.kuali.kfs.module.ar.businessobject.OrganizationOptions;
import org.kuali.kfs.module.ar.document.CustomerInvoiceDocument;
import org.kuali.kfs.module.ar.document.CustomerInvoiceRecurrenceUtils;
import org.kuali.kfs.module.ar.document.service.CustomerAddressService;
import org.kuali.kfs.module.ar.document.service.InvoiceRecurrenceDocumentService;
import org.kuali.kfs.sys.service.impl.KfsParameterConstants;

import java.sql.Date;
import java.sql.Timestamp;
import java.time.LocalDate;
import java.util.ArrayList;
import java.util.HashMap;
import java.util.List;
import java.util.Map;

public class InvoiceRecurrenceDocumentServiceImpl implements InvoiceRecurrenceDocumentService {

    private ParameterService parameterService;
    private BusinessObjectService businessObjectService;
    private CustomerAddressService customerAddressService;
    private DocumentService documentService;
    private DateTimeService dateTimeService;

    @Override
    public boolean isCustomerInvoiceDetailTaxable(CustomerInvoiceDocument customerInvoiceDocument,
            CustomerInvoiceDetail customerInvoiceDetail) {
        //check if sales tax is enabled
        if (!parameterService.getParameterValueAsBoolean(KfsParameterConstants.ACCOUNTS_RECEIVABLE_DOCUMENT.class, ArParameterConstants.SALES_TAX_IND)) {
            return false;
        }

        //check if customer is tax exempt
        if (ObjectUtils.isNotNull(customerInvoiceDocument.getCustomer())) {
            if (customerInvoiceDocument.getCustomer().isCustomerTaxExemptIndicator()) {
                return false;
            }
        }

        //check item if the taxable indicator is checked
        return customerInvoiceDetail.isTaxableIndicator();
    }

    @Override
    public String getPostalCodeForTaxation(CustomerInvoiceDocument document) {

        String postalCode = null;
        String customerNumber = document.getAccountsReceivableDocumentHeader().getCustomerNumber();
        Integer shipToAddressIdentifier = document.getCustomerShipToAddressIdentifier();

        //if customer number or ship to address id isn't provided, go to org options
        if (ObjectUtils.isNotNull(shipToAddressIdentifier) && StringUtils.isNotEmpty(customerNumber)) {

            CustomerAddress customerShipToAddress = customerAddressService.getByPrimaryKey(customerNumber, shipToAddressIdentifier);
            if (ObjectUtils.isNotNull(customerShipToAddress)) {
                postalCode = customerShipToAddress.getCustomerZipCode();
            }
        } else {
            Map<String, String> criteria = new HashMap<>();
            criteria.put("chartOfAccountsCode", document.getBillByChartOfAccountCode());
            criteria.put("organizationCode", document.getBilledByOrganizationCode());
            OrganizationOptions organizationOptions = businessObjectService.findByPrimaryKey(OrganizationOptions.class,
                    criteria);

            if (ObjectUtils.isNotNull(organizationOptions)) {
                postalCode = organizationOptions.getOrganizationPostalZipCode();
            }
        }
        return postalCode;
    }

    @Override
    public boolean isInvoiceApproved(String invoiceNumber) {
        boolean isSuccess = true;

        if (ObjectUtils.isNull(invoiceNumber)) {
            return true;
        }

        CustomerInvoiceDocument customerInvoiceDocument = (CustomerInvoiceDocument) documentService
                .getByDocumentHeaderId(invoiceNumber);
        if (ObjectUtils.isNotNull(customerInvoiceDocument)) {
            WorkflowDocument workflowDocument = customerInvoiceDocument.getDocumentHeader().getWorkflowDocument();
            if (!workflowDocument.isApproved()) {
                isSuccess = false;
            }
        } else {
            isSuccess = false;
        }
        return isSuccess;
    }

    @Override
    public boolean isValidRecurrenceBeginDate(Date beginDate) {
        if (ObjectUtils.isNull(beginDate)) {
            return true;
        }
        Timestamp currentDate = new Timestamp(dateTimeService.getCurrentDate().getTime());
        Timestamp beginDateTimestamp = new Timestamp(beginDate.getTime());
        return !beginDateTimestamp.before(currentDate) && !beginDateTimestamp.equals(currentDate);
    }

    @Override
    public boolean isValidRecurrenceEndDate(Date beginDate, Date endDate) {
        if (ObjectUtils.isNull(beginDate) ||
            ObjectUtils.isNull(endDate)) {
            return true;
        }
        Timestamp beginDateTimestamp = new Timestamp(beginDate.getTime());
        Timestamp endDateTimestamp = new Timestamp(endDate.getTime());
        return ObjectUtils.isNull(endDateTimestamp)
               || !endDateTimestamp.before(beginDateTimestamp) && !endDateTimestamp.equals(beginDateTimestamp);
    }

    @Override
    public boolean isValidEndDateAndTotalRecurrenceNumber(Date recurrenceBeginDate, Date recurrenceEndDate,
            Integer totalRecurrenceNumber, String recurrenceIntervalCode) {
        if (ObjectUtils.isNull(recurrenceBeginDate) ||
            ObjectUtils.isNull(recurrenceIntervalCode) ||
            ObjectUtils.isNull(recurrenceEndDate) ||
            ObjectUtils.isNull(totalRecurrenceNumber)) {
            return true;
        }

        LocalDate beginDate = recurrenceBeginDate.toLocalDate();
        LocalDate endDate = recurrenceEndDate.toLocalDate();
        int totalRecurrences = CustomerInvoiceRecurrenceUtils.calculateTotalRecurrences(beginDate, endDate,
                recurrenceIntervalCode);
        return totalRecurrences == totalRecurrenceNumber;
    }

    @Override
    public boolean isValidEndDateOrTotalRecurrenceNumber(Date endDate, Integer totalRecurrenceNumber) {
        return ObjectUtils.isNotNull(endDate) || ObjectUtils.isNotNull(totalRecurrenceNumber);
    }

    @Override
    public boolean isValidMaximumNumberOfRecurrences(Integer totalRecurrenceNumber, String intervalCode) {
        if (ObjectUtils.isNull(intervalCode) ||
            ObjectUtils.isNull(totalRecurrenceNumber)) {
            return true;
        }
        Integer maximumRecurrencesByInterval;
        if (ObjectUtils.isNotNull(intervalCode)) {
            List<String> maximumRecurrences = new ArrayList<>(parameterService
                    .getSubParameterValuesAsString(InvoiceRecurrence.class,
                            ArParameterConstants.RECURRENCES, intervalCode));
            if (maximumRecurrences.size() > 0 && StringUtils.isNotBlank(maximumRecurrences.get(0))) {
                maximumRecurrencesByInterval = Integer.valueOf(maximumRecurrences.get(0));
                return totalRecurrenceNumber <= maximumRecurrencesByInterval;
            }
        }
        return true;
    }

    @Override
    public boolean isValidInitiator(String initiator) {
        return true;
    }

    public ParameterService getParameterService() {
        return parameterService;
    }

    public void setParameterService(ParameterService parameterService) {
        this.parameterService = parameterService;
    }

    public BusinessObjectService getBusinessObjectService() {
        return businessObjectService;
    }

    public void setBusinessObjectService(BusinessObjectService businessObjectService) {
        this.businessObjectService = businessObjectService;
    }

    public CustomerAddressService getCustomerAddressService() {
        return customerAddressService;
    }

    public void setCustomerAddressService(CustomerAddressService customerAddressService) {
        this.customerAddressService = customerAddressService;
    }

    public void setDocumentService(DocumentService documentService) {
        this.documentService = documentService;
    }

    public void setDateTimeService(DateTimeService dateTimeService) {
        this.dateTimeService = dateTimeService;
    }
}
