/*
 * The Kuali Financial System, a comprehensive financial management system for higher education.
 *
 * Copyright 2005-2022 Kuali, Inc.
 *
 * This program is free software: you can redistribute it and/or modify
 * it under the terms of the GNU Affero General Public License as published by
 * the Free Software Foundation, either version 3 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU Affero General Public License for more details.
 *
 * You should have received a copy of the GNU Affero General Public License
 * along with this program.  If not, see <http://www.gnu.org/licenses/>.
 */
package org.kuali.kfs.module.ar.document.service.impl;

import org.apache.commons.collections4.ListUtils;
import org.apache.commons.lang3.StringUtils;
import org.apache.commons.lang3.Validate;
import org.apache.commons.lang3.tuple.Pair;
import org.apache.logging.log4j.LogManager;
import org.apache.logging.log4j.Logger;
import org.kuali.kfs.coa.businessobject.Account;
import org.kuali.kfs.coa.businessobject.Chart;
import org.kuali.kfs.coa.businessobject.ObjectCode;
import org.kuali.kfs.coa.businessobject.ObjectCodeCurrent;
import org.kuali.kfs.coa.businessobject.OffsetDefinition;
import org.kuali.kfs.coa.businessobject.ProjectCode;
import org.kuali.kfs.coa.businessobject.SubAccount;
import org.kuali.kfs.coa.businessobject.SubObjectCode;
import org.kuali.kfs.coa.service.AccountService;
import org.kuali.kfs.coa.service.ChartService;
import org.kuali.kfs.coa.service.ObjectCodeService;
import org.kuali.kfs.coa.service.OffsetDefinitionService;
import org.kuali.kfs.coa.service.ProjectCodeService;
import org.kuali.kfs.coa.service.SubAccountService;
import org.kuali.kfs.coa.service.SubObjectCodeService;
import org.kuali.kfs.core.api.datetime.DateTimeService;
import org.kuali.kfs.core.api.util.type.KualiDecimal;
import org.kuali.kfs.datadictionary.legacy.DataDictionaryService;
import org.kuali.kfs.kim.impl.identity.Person;
import org.kuali.kfs.krad.bo.Note;
import org.kuali.kfs.krad.document.Document;
import org.kuali.kfs.krad.service.BusinessObjectService;
import org.kuali.kfs.krad.service.DocumentService;
import org.kuali.kfs.krad.service.NoteService;
import org.kuali.kfs.krad.util.GlobalVariables;
import org.kuali.kfs.krad.util.ObjectUtils;
import org.kuali.kfs.module.ar.ArConstants;
import org.kuali.kfs.module.ar.businessobject.AccountsReceivableDocumentHeader;
import org.kuali.kfs.module.ar.businessobject.CustomerInvoiceDetail;
import org.kuali.kfs.module.ar.businessobject.InvoicePaidApplied;
import org.kuali.kfs.module.ar.businessobject.NonAppliedHolding;
import org.kuali.kfs.module.ar.businessobject.ReceivableCustomerInvoiceDetail;
import org.kuali.kfs.module.ar.businessobject.SystemInformation;
import org.kuali.kfs.module.ar.document.CustomerInvoiceDocument;
import org.kuali.kfs.module.ar.document.PaymentApplicationAdjustmentDocument;
import org.kuali.kfs.module.ar.document.PaymentApplicationDocument;
import org.kuali.kfs.module.ar.document.service.CustomerInvoiceDetailService;
import org.kuali.kfs.module.ar.document.service.CustomerInvoiceDocumentService;
import org.kuali.kfs.module.ar.document.service.SystemInformationService;
import org.kuali.kfs.module.ar.rest.resource.requests.PaymentApplicationAdjustmentRequest;
import org.kuali.kfs.sys.KFSConstants;
import org.kuali.kfs.sys.KFSPropertyConstants;
import org.kuali.kfs.sys.businessobject.ChartOrgHolder;
import org.kuali.kfs.sys.businessobject.DocumentHeader;
import org.kuali.kfs.sys.businessobject.GeneralLedgerPendingEntry;
import org.kuali.kfs.sys.businessobject.GeneralLedgerPendingEntrySequenceHelper;
import org.kuali.kfs.sys.businessobject.SourceAccountingLine;
import org.kuali.kfs.sys.businessobject.UniversityDate;
import org.kuali.kfs.sys.context.SpringContext;
import org.kuali.kfs.sys.service.FinancialSystemUserService;
import org.kuali.kfs.sys.service.GeneralLedgerPendingEntryService;
import org.kuali.kfs.sys.service.UniversityDateService;

import java.sql.Date;
import java.util.ArrayList;
import java.util.Collection;
import java.util.Comparator;
import java.util.Iterator;
import java.util.LinkedList;
import java.util.List;
import java.util.Locale;
import java.util.Map;
import java.util.Objects;
import java.util.Optional;
import java.util.concurrent.atomic.AtomicInteger;
import java.util.function.BiFunction;
import java.util.function.Predicate;
import java.util.stream.Collectors;
import java.util.stream.IntStream;
import java.util.stream.Stream;

import static java.util.Map.entry;

public class PaymentApplicationAdjustmentDocumentService {
    private static final Logger LOG = LogManager.getLogger();

    private final AccountService accountService;
    private final BusinessObjectService businessObjectService;
    private final ChartService chartService;
    private final CustomerInvoiceDetailService customerInvoiceDetailService;
    private final CustomerInvoiceDocumentService customerInvoiceDocumentService;
    private final DataDictionaryService dataDictionaryService;
    private final DocumentService documentService;
    private final FinancialSystemUserService financialSystemUserService;
    private final GeneralLedgerPendingEntryService generalLedgerPendingEntryService;
    private final NoteService noteService;
    private final ObjectCodeService objectCodeService;
    private final OffsetDefinitionService offsetDefinitionService;
    private final ProjectCodeService projectCodeService;
    private final SubAccountService subAccountService;
    private final SubObjectCodeService subObjectCodeService;
    private final SystemInformationService systemInformationService;
    private final UniversityDateService universityDateService;

    public PaymentApplicationAdjustmentDocumentService(
            final AccountService accountService,
            final BusinessObjectService businessObjectService,
            final ChartService chartService,
            final CustomerInvoiceDetailService customerInvoiceDetailService,
            final CustomerInvoiceDocumentService customerInvoiceDocumentService,
            final DataDictionaryService dataDictionaryService,
            final DocumentService documentService,
            final FinancialSystemUserService financialSystemUserService,
            final GeneralLedgerPendingEntryService generalLedgerPendingEntryService,
            final NoteService noteService,
            final ObjectCodeService objectCodeService,
            final OffsetDefinitionService offsetDefinitionService,
            final ProjectCodeService projectCodeService,
            final SubAccountService subAccountService,
            final SubObjectCodeService subObjectCodeService,
            final SystemInformationService systemInformationService,
            final UniversityDateService universityDateService
    ) {
        Validate.isTrue(accountService != null, "accountService must be provided");
        Validate.isTrue(businessObjectService != null, "businessObjectService must be provided");
        Validate.isTrue(chartService != null, "chartService must be provided");
        Validate.isTrue(customerInvoiceDetailService != null, "customerInvoiceDetailService must be provided");
        Validate.isTrue(customerInvoiceDocumentService != null, "customerInvoiceDocumentService must be provided");
        Validate.isTrue(dataDictionaryService != null, "dataDictionaryService must be provided");
        Validate.isTrue(documentService != null, "documentService must be provided");
        Validate.isTrue(financialSystemUserService != null, "financialSystemUserService must be provided");
        Validate.isTrue(generalLedgerPendingEntryService != null, "generalLedgerPendingEntryService must be provided");
        Validate.isTrue(noteService != null, "noteService must be provided");
        Validate.isTrue(objectCodeService != null, "objectCodeService must be provided");
        Validate.isTrue(offsetDefinitionService != null, "offsetDefinitionService must be provided");
        Validate.isTrue(projectCodeService != null, "projectCodeService must be provided");
        Validate.isTrue(subAccountService != null, "subAccountService must be provided");
        Validate.isTrue(subObjectCodeService != null, "subObjectCodeService must be provided");
        Validate.isTrue(systemInformationService != null, "systemInformationService must be provided");
        Validate.isTrue(universityDateService != null, "universityDateService must be provided");

        this.accountService = accountService;
        this.businessObjectService = businessObjectService;
        this.chartService = chartService;
        this.customerInvoiceDetailService = customerInvoiceDetailService;
        this.customerInvoiceDocumentService = customerInvoiceDocumentService;
        this.dataDictionaryService = dataDictionaryService;
        this.documentService = documentService;
        this.financialSystemUserService = financialSystemUserService;
        this.generalLedgerPendingEntryService = generalLedgerPendingEntryService;
        this.noteService = noteService;
        this.objectCodeService = objectCodeService;
        this.offsetDefinitionService = offsetDefinitionService;
        this.projectCodeService = projectCodeService;
        this.subAccountService = subAccountService;
        this.subObjectCodeService = subObjectCodeService;
        this.systemInformationService = systemInformationService;
        this.universityDateService = universityDateService;
    }

    public PaymentApplicationAdjustmentDocument createPaymentApplicationAdjustment(
            final PaymentApplicationAdjustmentDocument adjusteeDocument
    ) {
        final PaymentApplicationAdjustmentDocument adjustmentDocument =
                (PaymentApplicationAdjustmentDocument) documentService
                        .getNewDocument(PaymentApplicationAdjustmentDocument.class);

        final DocumentHeader documentHeader = adjustmentDocument.getDocumentHeader();
        documentHeader.setDocumentDescription("Created by Application Adjustment");

        final AccountsReceivableDocumentHeader arDocumentHeader =
                adjusteeDocument.getAccountsReceivableDocumentHeader();
        final String processingChartCode = arDocumentHeader.getProcessingChartOfAccountCode();
        final String processingOrgCode = arDocumentHeader.getProcessingOrganizationCode();

        final String documentNumber = adjustmentDocument.getDocumentNumber();

        final AccountsReceivableDocumentHeader arDocHeader = new AccountsReceivableDocumentHeader();
        arDocHeader.setProcessingChartOfAccountCode(processingChartCode);
        arDocHeader.setProcessingOrganizationCode(processingOrgCode);
        arDocHeader.setDocumentNumber(documentNumber);
        arDocHeader.setCustomerNumber(arDocumentHeader.getCustomerNumber());
        adjustmentDocument.setAccountsReceivableDocumentHeader(arDocHeader);

        adjustmentDocument.setAdjusteeDocumentNumber(adjusteeDocument.getDocumentNumber());

        final List<InvoicePaidApplied> ipasWhoseInvoiceHasAnOpenAmount =
                adjusteeDocument.getInvoicePaidApplieds()
                        .stream()
                        .filter(ipa -> customerInvoiceDocumentService.getPaidAppliedTotalForInvoice(
                                ipa.getCustomerInvoiceDocument()).isNonZero())
                        .collect(Collectors.toList());
        adjustmentDocument.setInvoicePaidApplieds(ipasWhoseInvoiceHasAnOpenAmount);

        final List<NonAppliedHolding> nahsWithAnOpenAmount =
                adjusteeDocument.getNonAppliedHoldings()
                    .stream()
                    .filter(nah -> nah.getFinancialDocumentLineAmount().isPositive())
                    .collect(Collectors.toList());
        nahsWithAnOpenAmount
                .forEach(nah -> nah.setReferenceFinancialDocumentNumber(documentNumber));
        adjustmentDocument.setNonAppliedHoldings(nahsWithAnOpenAmount);

        documentService.saveDocument(adjustmentDocument);

        adjusteeDocument.refreshReferenceObject("nonAppliedHoldings");
        adjusteeDocument.setAdjustmentDocumentNumber(adjustmentDocument.getDocumentNumber());
        documentService.updateDocument(adjusteeDocument);

        return adjustmentDocument;
    }

    public void updateNonArAccountingLines(
            final PaymentApplicationAdjustmentDocument appaDoc,
            final List<PaymentApplicationAdjustmentRequest.AccountingLine> nonArAccountingLines
    ) {
        LOG.debug(
                "updateNonArAccountingLines(...) - Enter : appaDoc={}; nonArAccountingLines={}",
                appaDoc,
                nonArAccountingLines
        );

        removeNonArAccountingLinesThatExistInTheDocumentButNotTheRequest(appaDoc, nonArAccountingLines);
        updateExistingNonArAccountingLinesThatExistInBothTheRequestAndTheDocument(appaDoc, nonArAccountingLines);
        addNewNonArAccountingLinesThatExistInTheRequestButNotTheDocument(appaDoc, nonArAccountingLines);

        // If these updateDocument calls are removed or replaced with saveDocument (which does more processing than
        // update), then we can get an OptimisticLockException on AccountingLineBase (target accounting line) when
        // trying to do subsequent actions such as route after a save. I'm not exactly sure why or how best to address
        // that, but since these were originally removed to fix an issue with adhoc routing on APPA and I've found
        // another way to fix that, I'm inclined to leave this as is, at least until OJB is eliminated.
        documentService.updateDocument(appaDoc);

        LOG.debug("updateNonArAccountingLines(...) - Exit");
        documentService.updateDocument(appaDoc);
    }

    private void addNewNonArAccountingLinesThatExistInTheRequestButNotTheDocument(
            final PaymentApplicationAdjustmentDocument appaDoc,
            final List<PaymentApplicationAdjustmentRequest.AccountingLine> nonArAccountingLines
    ) {
        final String appaDocumentNumber = appaDoc.getDocumentNumber();

        final List<SourceAccountingLine> nonArSourceAccountingLines = appaDoc.getNonArAccountingLines();

        final AtomicInteger highestSequenceNumber = determineHighestSequenceNumber(nonArSourceAccountingLines);

        nonArAccountingLines
                .stream()
                .filter(nonArAccountingLine -> nonArAccountingLine.getSequenceNumber() == null)
                .map(nonArAccountingLine ->
                        createSourceAccountingLine(
                                nonArAccountingLine,
                                appaDocumentNumber,
                                highestSequenceNumber.incrementAndGet()
                        )
                )
                .forEach(nonArSourceAccountingLines::add);
    }

    private AtomicInteger determineHighestSequenceNumber(
            final List<SourceAccountingLine> nonArSourceAccountingLines
    ) {
        return nonArSourceAccountingLines
                .stream()
                .mapToInt(SourceAccountingLine::getSequenceNumber)
                .max()
                    .stream()
                    .mapToObj(max -> new AtomicInteger(max))
                    .findFirst()
                    .orElse(new AtomicInteger());
    }

    private void updateExistingNonArAccountingLinesThatExistInBothTheRequestAndTheDocument(
            final PaymentApplicationAdjustmentDocument appaDoc,
            final List<PaymentApplicationAdjustmentRequest.AccountingLine> nonArAccountingLines
    ) {
        final String appaDocumentNumber = appaDoc.getDocumentNumber();

        final List<SourceAccountingLine> nonArSourceAccountingLines = appaDoc.getNonArAccountingLines();

        nonArAccountingLines.forEach(nonArAccountingLine -> {
            final Integer sequenceNumber = nonArAccountingLine.getSequenceNumber();

            nonArSourceAccountingLines
                    .stream()
                    .filter(equivalentNonArSourceAccountingLineFromDocument(appaDocumentNumber, sequenceNumber))
                    .findFirst()
                    .ifPresent(nonArSourceAccountingLine -> updateSourceAccountingLine(nonArSourceAccountingLine, nonArAccountingLine));
        });
    }

    private static Predicate<SourceAccountingLine> equivalentNonArSourceAccountingLineFromDocument(
            final String appaDocumentNumber,
            final Integer sequenceNumber
    ) {
        return nonArSourceAccountingLine ->
                nonArSourceAccountingLine.getDocumentNumber().equals(appaDocumentNumber) &&
                        nonArSourceAccountingLine.getSequenceNumber().equals(sequenceNumber);
    }

    private static void removeNonArAccountingLinesThatExistInTheDocumentButNotTheRequest(
            final PaymentApplicationAdjustmentDocument appaDoc,
            final List<PaymentApplicationAdjustmentRequest.AccountingLine> nonArAccountingLines
    ) {
        final Iterator<SourceAccountingLine> iterator = appaDoc.getNonArAccountingLines().iterator();
        while (iterator.hasNext()) {
            final SourceAccountingLine nonArSourceAccountingLine = iterator.next();
            final Integer sequenceNumber = nonArSourceAccountingLine.getSequenceNumber();
            nonArAccountingLines
                    .stream()
                    .filter(nonArAccountingLine ->
                            Objects.equals(nonArAccountingLine.getSequenceNumber(), sequenceNumber)
                    )
                    .findFirst()
                    .ifPresentOrElse(
                            notUsed -> { /* It was found; it does not need to be removed; ignoring it */ },
                            iterator::remove
                );
        }
    }

    /**
     * Populate a {@link SourceAccountingLine} BO from an
     * {@link org.kuali.kfs.module.ar.rest.resource.requests.PaymentApplicationAdjustmentRequest.AccountingLine}.
     */
    public SourceAccountingLine createSourceAccountingLine(
            final PaymentApplicationAdjustmentRequest.AccountingLine accountingLine,
            final String appaDocumentNumber,
            final Integer sequenceNumber
    ) {
        LOG.debug(
                "createSourceAccountingLine(...) - Enter : " +
                        "accountingLine={}; appaDocumentNumber={}; sequenceNumber={}",
                accountingLine,
                appaDocumentNumber,
                sequenceNumber
        );

        final SourceAccountingLine sourceAccountingLine = new SourceAccountingLine();

        sourceAccountingLine.setDocumentNumber(appaDocumentNumber);
        sourceAccountingLine.setSequenceNumber(sequenceNumber);

        updateSourceAccountingLine(sourceAccountingLine, accountingLine);

        LOG.debug("createSourceAccountingLine(...) - Exit : sourceAccountingLine={}", sourceAccountingLine);
        return sourceAccountingLine;
    }

    private void updateSourceAccountingLine(
            final SourceAccountingLine sourceAccountingLine,
            final PaymentApplicationAdjustmentRequest.AccountingLine accountingLine
    ) {
        LOG.debug(
                "updateSourceAccountingLine(...) - Enter : sourceAccountingLine={}; accountingLine={}",
                sourceAccountingLine,
                accountingLine
        );

        sourceAccountingLine.setAmount(accountingLine.getAmount());
        sourceAccountingLine.setFinancialDocumentLineDescription(accountingLine.getDescription());
        sourceAccountingLine.setOrganizationReferenceId(accountingLine.getOrgRefId());

        final String chartCode = accountingLine.getChartCode().toUpperCase(Locale.US);
        final Chart chart = chartService.getByPrimaryId(chartCode);
        sourceAccountingLine.setChart(chart);
        sourceAccountingLine.setChartOfAccountsCode(chartCode);

        final Account account =
                accountService.getByPrimaryId(chartCode, accountingLine.getAccountNumber());
        sourceAccountingLine.setAccount(account);
        sourceAccountingLine.setAccountNumber(accountingLine.getAccountNumber());

        final ObjectCode objectCode =
                objectCodeService.getByPrimaryIdForCurrentYear(
                        chartCode,
                        accountingLine.getObject()
                );
        sourceAccountingLine.setObjectCode(objectCode);
        sourceAccountingLine.setFinancialObjectCode(accountingLine.getObject());

        if (StringUtils.isNotBlank(accountingLine.getSubAccountNumber())) {
            final SubAccount subAccount =
                    subAccountService.getByPrimaryId(
                            chartCode,
                            accountingLine.getAccountNumber(),
                            accountingLine.getSubAccountNumber()
                    );
            sourceAccountingLine.setSubAccount(subAccount);
            sourceAccountingLine.setSubAccountNumber(accountingLine.getSubAccountNumber());
        }

        if (StringUtils.isNotBlank(accountingLine.getSubObject())) {
            final SubObjectCode subObjectCode =
                    subObjectCodeService.getByPrimaryIdForCurrentYear(
                            chartCode,
                            accountingLine.getAccountNumber(),
                            accountingLine.getObject(),
                            accountingLine.getSubObject()
                    );
            sourceAccountingLine.setSubObjectCode(subObjectCode);
            sourceAccountingLine.setFinancialSubObjectCode(accountingLine.getSubObject());
        }

        if (StringUtils.isNotBlank(accountingLine.getProjectCode())) {
            final ProjectCode project = projectCodeService.getByPrimaryId(accountingLine.getProjectCode());
            sourceAccountingLine.setProject(project);
            sourceAccountingLine.setProjectCode(accountingLine.getProjectCode());
        }

        LOG.debug("updateSourceAccountingLine(...) - Exit : sourceAccountingLine={}", sourceAccountingLine);
    }

    /**
     * Given a document and a list of nonAppliedHoldings requests, this method will add any new holdings that exist
     * within the request but not in the doc. It will also update any pre-existing holdings amounts that have already
     * been added to the document. Lastly, any holdings that are in the document but not in the list of nonAppliedHoldings
     * request will be removed
     *
     * @param appaDoc
     * @param nonAppliedHoldings
     */
    public void updateNonAppliedHoldings(PaymentApplicationAdjustmentDocument appaDoc,
                                         List<PaymentApplicationAdjustmentRequest.NonAppliedHolding> nonAppliedHoldings) {
        List<String> customerNumbers = new ArrayList<>();
        nonAppliedHoldings.forEach(nonAppliedHolding -> {
            createOrUpdateNonAppliedHolding(appaDoc, nonAppliedHolding);
            customerNumbers.add(nonAppliedHolding.getCustomerNumber());
        });

        List<NonAppliedHolding> nonAppliedHoldingsToRemove = appaDoc.getNonAppliedHoldings().stream()
                .filter(nonAppliedHolding -> !customerNumbers.contains(nonAppliedHolding.getCustomerNumber()))
                .collect(Collectors.toList());
        appaDoc.getNonAppliedHoldings().removeAll(nonAppliedHoldingsToRemove);
    }

    /**
     * Given a Document and a List of InvoiceApplicationS, this method will
     *  - add any new InvoiceApplications that exist within the Request but not in the Document
     *  - update any pre-existing InvoicePaidApplied amounts that already exist on the document
     *  - remove any InvoicePaidAppliedS that are in the Document but not in the Request
     *  - remove any InvoicePaidAppliedS with a zero amount
     */
    public void updateInvoicePaidApplieds(
            final PaymentApplicationAdjustmentDocument appaDoc,
            final List<PaymentApplicationAdjustmentRequest.InvoiceApplication> invoiceApplications
    ) {
        LOG.debug("updateInvoicePaidApplieds(...) - Enter : appaDoc={}; invoiceApplications={}",
                appaDoc,
                invoiceApplications);

        final Integer universityFiscalYear = universityDateService.getCurrentFiscalYear();
        final String universityFiscalPeriodCode =
                universityDateService
                        .getCurrentUniversityDate()
                        .getAccountingPeriod()
                        .getUniversityFiscalPeriodCode();
        final int nextInvoicePaidAppliedItemNumber =
                determineNextInvoicePaidAppliedItemNumber(appaDoc.getInvoicePaidApplieds());

        updateDocumentWithInvoiceApplications(
                appaDoc,
                invoiceApplications,
                universityFiscalYear,
                universityFiscalPeriodCode,
                nextInvoicePaidAppliedItemNumber
        );

        removeIpasFromDocumentWhichAreNotInTheRequest(appaDoc, invoiceApplications);

        LOG.debug("updateInvoicePaidApplieds(...) - Exit");
    }

    /*
     * paidAppliedItemNumber is part of the primary key for InvoicePaidApplied. In order to prevent key collisions
     * we find the largest paidAppliedItemNumber on this document an increment it to ensure all InvoicePaidAppliedS
     * have unique item numbers
     */
    private static int determineNextInvoicePaidAppliedItemNumber(final Collection<InvoicePaidApplied> ipas) {
        return ipas.stream()
                .max(Comparator.comparing(InvoicePaidApplied::getPaidAppliedItemNumber))
                .map(InvoicePaidApplied::getPaidAppliedItemNumber)
                .orElse(0)
                + 1;
    }

    private void updateDocumentWithInvoiceApplications(
            final PaymentApplicationAdjustmentDocument appaDoc,
            final List<PaymentApplicationAdjustmentRequest.InvoiceApplication> invoiceApplications,
            final Integer universityFiscalYear,
            final String universityFiscalPeriodCode,
            final int nextInvoicePaidAppliedItemNumber
    ) {
        LOG.debug("updateDocumentWithInvoiceApplications(...) - Enter");

        invoiceApplications.forEach(invoiceApplication -> {
            final List<PaymentApplicationAdjustmentRequest.InvoiceApplicationDetail> detailApplications =
                    invoiceApplication.getDetailApplications();

            updateDocumentWithInvoiceApplicationDetails(
                    appaDoc,
                    universityFiscalYear,
                    universityFiscalPeriodCode,
                    nextInvoicePaidAppliedItemNumber,
                    invoiceApplication,
                    detailApplications
            );
        });

        LOG.debug("updateDocumentWithInvoiceApplications(...) - Exit");
    }

    private void updateDocumentWithInvoiceApplicationDetails(
            final PaymentApplicationAdjustmentDocument appaDoc,
            final Integer universityFiscalYear,
            final String universityFiscalPeriodCode,
            final int nextInvoicePaidAppliedItemNumber,
            final PaymentApplicationAdjustmentRequest.InvoiceApplication invoiceApplication,
            final List<PaymentApplicationAdjustmentRequest.InvoiceApplicationDetail> detailApplications
    ) {
        LOG.debug("updateDocumentWithInvoiceApplicationDetails(...) - Enter");

        IntStream.range(0, detailApplications.size())
                .forEach(index -> {
                    final PaymentApplicationAdjustmentRequest.InvoiceApplicationDetail detailApplication =
                            detailApplications.get(index);

                    final var invoicePaidAppliedItemNumber = nextInvoicePaidAppliedItemNumber + index;

                    updateExistingOrCreateNewInvoicePaidApplied(
                            appaDoc,
                            invoiceApplication.getDocumentNumber(),
                            detailApplication,
                            invoicePaidAppliedItemNumber,
                            universityFiscalYear,
                            universityFiscalPeriodCode
                    );
                });

        LOG.debug("updateDocumentWithInvoiceApplicationDetails(...) - Exit");
    }

    private void updateExistingOrCreateNewInvoicePaidApplied(
            final PaymentApplicationAdjustmentDocument appaDoc,
            final String invoiceNumber,
            final PaymentApplicationAdjustmentRequest.InvoiceApplicationDetail detailApplication,
            final Integer invoicePaidAppliedItemNumber,
            final Integer universityFiscalYear,
            final String universityFiscalPeriodCode
    ) {
        LOG.debug("updateExistingOrCreateNewInvoicePaidApplied(...) - Enter : " +
                "invoiceNumber={}; detailApplication={}; invoicePaidAppliedItemNumber={}; fiscalYear={}; " +
                "fiscalPeriod={}",
                invoiceNumber,
                detailApplication,
                invoicePaidAppliedItemNumber,
                universityFiscalYear,
                universityFiscalPeriodCode);

        final Integer invoiceItemNumber = detailApplication.getSequenceNumber();
        final KualiDecimal amountApplied = detailApplication.getAmountApplied();

        final Collection<InvoicePaidApplied> ipas = appaDoc.getInvoicePaidApplieds();

        final Optional<InvoicePaidApplied> ipaFromDocument =
                findMatchingInvoicePaidApplied(ipas, invoiceNumber, invoiceItemNumber);

        ipaFromDocument.ifPresentOrElse(
            ipa -> ipa.setInvoiceItemAppliedAmount(amountApplied),
            () -> {
                final Optional<CustomerInvoiceDetail> detailFromInvoice =
                        findOnInvoiceWithTheSameAmountApplied(invoiceNumber, invoiceItemNumber, amountApplied);
                detailFromInvoice.ifPresentOrElse(
                    d -> LOG.debug("updateExistingOrCreateNewInvoicePaidApplied(...) - detail exists on the " +
                            "Invoice with same amountApplied; not creating IPA : detail={}", d),
                    () -> {
                        final InvoicePaidApplied newIpa =
                                new InvoicePaidApplied(
                                        appaDoc.getDocumentNumber(),
                                        invoiceNumber,
                                        invoiceItemNumber,
                                        amountApplied,
                                        invoicePaidAppliedItemNumber,
                                        universityFiscalYear,
                                        universityFiscalPeriodCode
                                );
                        ipas.add(newIpa);
                    }
                );
            }
        );

        LOG.debug("updateExistingOrCreateNewInvoicePaidApplied(...) - Exit : ipas={}", ipas);
    }

    private Optional<CustomerInvoiceDetail> findOnInvoiceWithTheSameAmountApplied(
            final String invoiceNumber,
            final Integer invoiceItemNumber,
            final KualiDecimal amountApplied
    ) {
        final CustomerInvoiceDocument invoice =
                businessObjectService.findBySinglePrimaryKey(CustomerInvoiceDocument.class, invoiceNumber);
        final List<CustomerInvoiceDetail> invoiceDetails = invoice.getCustomerInvoiceDetailsWithoutDiscounts();
        return invoiceDetails
                .stream()
                .filter(itemNumberAndAmountAppliedAreTheSame(invoiceItemNumber, amountApplied))
                .findFirst();
    }

    private Predicate<CustomerInvoiceDetail> itemNumberAndAmountAppliedAreTheSame(
            final Integer invoiceItemNumber,
            final KualiDecimal amountApplied
    ) {
        return invoiceDetail ->
                invoiceItemNumber.equals(invoiceDetail.getInvoiceItemNumber()) &&
                        amountApplied.equals(invoiceDetail.getAmountApplied());
    }

    /*
     * Return an InvoicePaidApplied instance, from PaymentApplicationAdjustmentDocument's IPAs, which matches
     * invoiceNumber & invoiceItemNumber; else NULL.
     */
    private Optional<InvoicePaidApplied> findMatchingInvoicePaidApplied(
            final Collection<InvoicePaidApplied> ipas,
            final String invoiceNumber,
            final Integer invoiceItemNumber
    ) {
        return ipas
                .stream()
                .filter(ipa -> ipa.getFinancialDocumentReferenceInvoiceNumber().equalsIgnoreCase(invoiceNumber))
                .filter(ipa -> ipa.getInvoiceItemNumber().equals(invoiceItemNumber))
                .findFirst();
    }

    private void removeIpasFromDocumentWhichAreNotInTheRequest(
            final PaymentApplicationAdjustmentDocument appaDoc,
            final List<PaymentApplicationAdjustmentRequest.InvoiceApplication> invoiceApplications
    ) {
        final List<String> invoiceNumbersFromRequest =
                invoiceApplications
                        .stream()
                        .map(PaymentApplicationAdjustmentRequest.InvoiceApplication::getDocumentNumber)
                        .collect(Collectors.toList());

        final List<InvoicePaidApplied> ipas = appaDoc.getInvoicePaidApplieds();

        final List<InvoicePaidApplied> ipasInDocumentButNotInRequest =
                ipas.stream()
                        .filter(ipa ->
                                !invoiceNumbersFromRequest.contains(ipa.getFinancialDocumentReferenceInvoiceNumber()))
                        .collect(Collectors.toList());

        ipas.removeAll(ipasInDocumentButNotInRequest);
    }

    public void removeZeroAmountInvoicePaidAppliedsFromDocument(final PaymentApplicationAdjustmentDocument appaDoc) {
        appaDoc.getInvoicePaidApplieds()
                .removeIf(ipa -> ipa.getInvoiceItemAppliedAmount().isZero());
    }

    public void removeZeroAmountNonAppliedHoldingsFromDocument(final PaymentApplicationAdjustmentDocument appaDoc) {
        appaDoc.getNonAppliedHoldings()
                .removeIf(nonApplied -> nonApplied.getFinancialDocumentLineAmount().isZero());
    }

    private NonAppliedHolding createNonAppliedHolding(
            PaymentApplicationAdjustmentDocument appaDoc,
            PaymentApplicationAdjustmentRequest.NonAppliedHolding nonAppliedHolding) {
        NonAppliedHolding documentNonAppliedHolding = new NonAppliedHolding();
        documentNonAppliedHolding.setCustomerNumber(nonAppliedHolding.getCustomerNumber());
        documentNonAppliedHolding.setReferenceFinancialDocumentNumber(appaDoc.getDocumentNumber());
        documentNonAppliedHolding.setFinancialDocumentLineAmount(nonAppliedHolding.getAmount());
        return documentNonAppliedHolding;
    }

    /**
     * Searches a PaymentApplicationAdjustmentDocument for a matching nonAppliedHolding that matches by customer number.
     * If found, the financialDocumentLineAmount is updated to match the request amount. If not found, a new holding
     * is created and added to the document
     *
     * @param appaDoc
     * @param nonAppliedHolding
     */
    private void createOrUpdateNonAppliedHolding(PaymentApplicationAdjustmentDocument appaDoc,
                                                 PaymentApplicationAdjustmentRequest.NonAppliedHolding nonAppliedHolding) {
        NonAppliedHolding matchingDocumentNonAppliedHolding = appaDoc.getNonAppliedHoldings().stream()
                .filter(docNonAppliedHolding ->
                        docNonAppliedHolding.getCustomerNumber().equalsIgnoreCase(nonAppliedHolding.getCustomerNumber()))
                .findFirst()
                .orElse(null);

        if (matchingDocumentNonAppliedHolding != null) {
            matchingDocumentNonAppliedHolding.setFinancialDocumentLineAmount(nonAppliedHolding.getAmount());
        } else {
            appaDoc.getNonAppliedHoldings().add(createNonAppliedHolding(appaDoc, nonAppliedHolding));
        }
    }

    /**
     * @param adjusteeDocument   The {@link Document} being adjusted.
     * @param adjustmentDocument The {@code Document} doing the adjusting.
     * @param postingYear        The posting year.
     * @param sequenceHelper     The GLPE sequence number helper.
     * @return A {@link List} of {@link GeneralLedgerPendingEntry}, containing the appropriate actual/offset
     * debits/credits.
     */
    public List<GeneralLedgerPendingEntry> createPendingEntries(
            final Document adjusteeDocument,
            final PaymentApplicationAdjustmentDocument adjustmentDocument,
            final Integer postingYear,
            final GeneralLedgerPendingEntrySequenceHelper sequenceHelper
    ) {
        final String documentTypeCode =
                dataDictionaryService.getDocumentTypeNameByClass(PaymentApplicationAdjustmentDocument.class);

        final var glpesForNonAppliedHoldings =
                createPendingEntriesForNonAppliedHoldings(
                        adjusteeDocument,
                        adjustmentDocument,
                        sequenceHelper,
                        documentTypeCode,
                        postingYear
                );

        final var glpesForInvoicePaidApplieds =
                createPendingEntriesForInvoicePaidApplieds(
                        adjusteeDocument,
                        adjustmentDocument,
                        sequenceHelper,
                        documentTypeCode,
                        postingYear
                );

        final var glpesForNonArAccountingLines =
                createPendingEntriesForNonArAccountingLines(
                        adjustmentDocument,
                        sequenceHelper,
                        documentTypeCode,
                        postingYear
                );

        final var glpes =
                ListUtils.union(
                    ListUtils.union(glpesForInvoicePaidApplieds, glpesForNonAppliedHoldings),
                    glpesForNonArAccountingLines
                );
        LOG.debug("createPendingEntries(...) - Return : glpes={}", glpes);
        return glpes;
    }

    /*
     * NonArAccountingLines differ from InvoicePaidAppliedS & NonAppliedHoldings in that they are not brought over
     * from the Document being adjusted. Whatever is on the adjustment Document is "new" and should generate GLPEs
     * (as long as the amount is greater than zero).
     */
    private List<GeneralLedgerPendingEntry> createPendingEntriesForNonArAccountingLines(
            final PaymentApplicationAdjustmentDocument adjustmentDocument,
            final GeneralLedgerPendingEntrySequenceHelper sequenceHelper,
            final String documentTypeCode,
            final Integer postingYear
    ) {
        final Collection<SourceAccountingLine> adjustmentNonArAccountingLines =
                getNonArAccountingLines(adjustmentDocument);

        final List<SourceAccountingLine> nonArAccountingLines =
                filterOutNonArAccountingLineWhoseAmountIsZero(adjustmentNonArAccountingLines);

        final List<GeneralLedgerPendingEntry> glpes = nonArCredits(
                sequenceHelper,
                nonArAccountingLines,
                adjustmentDocument.getDocumentHeader(),
                documentTypeCode,
                postingYear
        );

        return glpes;
    }

    private static Collection<SourceAccountingLine> getNonArAccountingLines(final Document document) {
        if (document instanceof PaymentApplicationAdjustmentDocument) {
            return ((PaymentApplicationAdjustmentDocument) document).getNonArAccountingLines();
        }
        return List.of();
    }

    private static List<SourceAccountingLine> filterOutNonArAccountingLineWhoseAmountIsZero(
            final Collection<SourceAccountingLine> adjustmentNonArAccountingLines
    ) {
        return adjustmentNonArAccountingLines
                .stream()
                .filter(nonArAccountingLine -> nonArAccountingLine.getAmount().isPositive())
                .collect(Collectors.toList());
    }

    private List<GeneralLedgerPendingEntry> nonArCredits(
            final GeneralLedgerPendingEntrySequenceHelper sequenceHelper,
            final List<SourceAccountingLine> nonArAccountingLines,
            final DocumentHeader documentHeader,
            final String documentTypeCode,
            final Integer postingYear
    ) {
        if (nonArAccountingLines.isEmpty()) {
            return List.of();
        }

        final List<GeneralLedgerPendingEntry> generatedEntries = new LinkedList<>();

        final String organizationDocumentNumber = documentHeader.getOrganizationDocumentNumber();

        nonArAccountingLines
                .forEach(sourceAccountingLine -> {

                    // Actual
                    final GeneralLedgerPendingEntry actualCredit =
                            createBasicGeneralLedgerPendingEntry(
                                    KFSConstants.GL_CREDIT_CODE,
                                    documentTypeCode,
                                    organizationDocumentNumber
                            );
                    actualCredit.setAccountNumber(sourceAccountingLine.getAccountNumber());
                    actualCredit.setChartOfAccountsCode(sourceAccountingLine.getChartOfAccountsCode());
                    actualCredit.setFinancialObjectCode(sourceAccountingLine.getFinancialObjectCode());
                    sourceAccountingLine.refreshReferenceObject("financialObject");
                    actualCredit.setFinancialObjectTypeCode(sourceAccountingLine.getObjectTypeCode());
                    final String financialSubObjectCode =
                            StringUtils.isBlank(sourceAccountingLine.getFinancialSubObjectCode())
                                    ? getDashFinancialSubObjectCode()
                                    : sourceAccountingLine.getFinancialSubObjectCode();
                    actualCredit.setFinancialSubObjectCode(financialSubObjectCode);
                    final String projectCode =
                            StringUtils.isBlank(sourceAccountingLine.getProjectCode())
                                    ? getDashProjectCode()
                                    : sourceAccountingLine.getProjectCode();
                    actualCredit.setProjectCode(projectCode);
                    final String subAccountNumber =
                            StringUtils.isBlank(sourceAccountingLine.getSubAccountNumber())
                                    ? getDashSubAccountNumber()
                                    : sourceAccountingLine.getSubAccountNumber();
                    actualCredit.setSubAccountNumber(subAccountNumber);
                    actualCredit.setTransactionLedgerEntryAmount(sourceAccountingLine.getAmount());
                    final String transactionLedgerEntryDescription =
                            StringUtils.isBlank(sourceAccountingLine.getFinancialDocumentLineDescription())
                                    ? documentHeader.getDocumentDescription()
                                    : sourceAccountingLine.getFinancialDocumentLineDescription();
                    actualCredit.setTransactionLedgerEntryDescription(transactionLedgerEntryDescription);
                    actualCredit.setTransactionLedgerEntrySequenceNumber(sequenceHelper.getSequenceCounter());
                    actualCredit.setUniversityFiscalYear(postingYear);
                    generatedEntries.add(actualCredit);
                    sequenceHelper.increment();

                    // Offset
                    final OffsetDefinition offsetEntryDefinition =
                            offsetDefinitionService.getByPrimaryId(
                                    postingYear,
                                    sourceAccountingLine.getChartOfAccountsCode(),
                                    documentTypeCode,
                                    KFSConstants.BALANCE_TYPE_ACTUAL
                            );
                    offsetEntryDefinition.refreshReferenceObject("financialObject");

                    final GeneralLedgerPendingEntry offsetDebit =
                            createBasicGeneralLedgerPendingEntry(
                                    KFSConstants.GL_DEBIT_CODE,
                                    documentTypeCode,
                                    organizationDocumentNumber
                            );
                    offsetDebit.setAccountNumber(actualCredit.getAccountNumber());
                    offsetDebit.setChartOfAccountsCode(actualCredit.getChartOfAccountsCode());
                    final String financialObjectCode = offsetEntryDefinition.getFinancialObjectCode();
                    offsetDebit.setFinancialObjectCode(financialObjectCode);
                    final ObjectCode offsetObjectCode = offsetEntryDefinition.getFinancialObject();
                    final String offsetFinancialObjectTypeCode = offsetObjectCode.getFinancialObjectTypeCode();
                    offsetDebit.setFinancialObjectTypeCode(offsetFinancialObjectTypeCode);
                    offsetDebit.setFinancialSubObjectCode(getDashFinancialSubObjectCode());
                    offsetDebit.setProjectCode(getDashProjectCode());
                    offsetDebit.setSubAccountNumber(getDashSubAccountNumber());
                    offsetDebit.setTransactionLedgerEntryAmount(actualCredit.getTransactionLedgerEntryAmount());
                    offsetDebit.setTransactionLedgerEntryDescription(KFSConstants.GL_PE_OFFSET_STRING);
                    offsetDebit.setTransactionLedgerEntrySequenceNumber(sequenceHelper.getSequenceCounter());
                    offsetDebit.setUniversityFiscalYear(actualCredit.getUniversityFiscalYear());
                    generatedEntries.add(offsetDebit);
                    sequenceHelper.increment();
                });
        return generatedEntries;
    }

    private List<GeneralLedgerPendingEntry> createPendingEntriesForNonAppliedHoldings(
            final Document adjusteeDocument,
            final Document adjustmentDocument,
            final GeneralLedgerPendingEntrySequenceHelper sequenceHelper,
            final String documentTypeCode,
            final Integer postingYear
    ) {
        final Collection<NonAppliedHolding> adjusteeNonAppliedHoldings = getNonAppliedHoldings(adjusteeDocument);
        final Collection<NonAppliedHolding> adjustmentNonAppliedHoldings = getNonAppliedHoldings(adjustmentDocument);

        final List<Pair<NonAppliedHolding, KualiDecimal>> newNahs =
                determineNewNonAppliedHoldings(
                        adjusteeNonAppliedHoldings,
                        adjustmentNonAppliedHoldings
                )
                        .stream()
                        .map(newNah -> Pair.of(newNah, newNah.getFinancialDocumentLineAmount().abs()))
                        .collect(Collectors.toList());
        final List<Pair<NonAppliedHolding, KualiDecimal>> increasedNahs =
                determineIncreasedNonAppliedHoldings(
                        adjusteeNonAppliedHoldings,
                        adjustmentNonAppliedHoldings
                );
        final var glpesForNewOrIncreased =
                creditTheAccountsReceivableClearingAccountAndObjectCode(
                        sequenceHelper,
                        ListUtils.union(newNahs, increasedNahs),
                        adjustmentDocument.getDocumentHeader(),
                        documentTypeCode,
                        postingYear
                );

        final List<Pair<NonAppliedHolding, KualiDecimal>> removedNahs =
                determineRemovedNonAppliedHoldings(
                        adjusteeNonAppliedHoldings,
                        adjustmentNonAppliedHoldings
                )
                        .stream()
                        .map(removedNah -> Pair.of(removedNah, removedNah.getFinancialDocumentLineAmount().abs()))
                        .collect(Collectors.toList());
        final List<Pair<NonAppliedHolding, KualiDecimal>> decreasedNahs =
                determineDecreasedNonAppliedHoldings(
                        adjusteeNonAppliedHoldings,
                        adjustmentNonAppliedHoldings
                );
        final var glpesForRemovedOrDecreased =
                debitTheAccountsReceivableClearingAccountAndObjectCode(
                        sequenceHelper,
                        ListUtils.union(removedNahs, decreasedNahs),
                        adjusteeDocument,
                        adjustmentDocument.getDocumentHeader(),
                        documentTypeCode,
                        postingYear
                );

        return ListUtils.union(glpesForNewOrIncreased, glpesForRemovedOrDecreased);
    }

    private static Collection<NonAppliedHolding> getNonAppliedHoldings(final Document adjustee) {
        Collection<NonAppliedHolding> nonAppliedHoldings = null;
        if (adjustee instanceof PaymentApplicationDocument) {
            nonAppliedHoldings = ((PaymentApplicationDocument) adjustee).getNonAppliedHoldings();
        } else if (adjustee instanceof PaymentApplicationAdjustmentDocument) {
            nonAppliedHoldings = ((PaymentApplicationAdjustmentDocument) adjustee).getNonAppliedHoldings();
        }
        return Objects.requireNonNullElseGet(nonAppliedHoldings, List::of);
    }

    /*
     * A {@link NonAppliedHolding} (NAH) will be considered "new" if it exists, based on 'customerNumber', in this
     * adjustment document but does not exist in the document being adjusted.
     */
    private static List<NonAppliedHolding> determineNewNonAppliedHoldings(
            final Collection<NonAppliedHolding> adjusteeNahs,
            final Collection<NonAppliedHolding> adjustmentNahs
    ) {
        final var adjusteeNahCustomerNumbers =
                adjusteeNahs.stream()
                        .map(adjusteeNah -> adjusteeNah.getCustomerNumber())
                        .collect(Collectors.toList());
        final var newNahs =
                adjustmentNahs.stream()
                        .filter(adjustmentNah -> !adjusteeNahCustomerNumbers
                                .contains(adjustmentNah.getCustomerNumber()))
                        .collect(Collectors.toList());
        LOG.debug("determineNewNonAppliedHoldings(...) - Exit : newNahs={}", newNahs);
        return newNahs;
    }

    /*
     * A {@link NonAppliedHolding} (NAH) will be considered "removed" if it exists, based on 'customerNumber', in the
     * document being adjusted but does not exist in this adjustment document.
     */
    private static List<NonAppliedHolding> determineRemovedNonAppliedHoldings(
            final Collection<NonAppliedHolding> adjusteeNahs,
            final Collection<NonAppliedHolding> adjustmentNahs
    ) {
        final var adjustmentNahCustomerNumbers =
                adjustmentNahs.stream()
                        .map(adjustmentNah -> adjustmentNah.getCustomerNumber())
                        .collect(Collectors.toList());
        final var removedNahs =
                adjusteeNahs.stream()
                        .filter(adjusteeNah -> !adjustmentNahCustomerNumbers
                                .contains(adjusteeNah.getCustomerNumber()))
                        .collect(Collectors.toList());
        LOG.debug("determineRemovedNonAppliedHoldings(...) - Exit : removedNahs={}", removedNahs);
        return removedNahs;
    }

    /*
     * A {@link NonAppliedHolding} (NAH) will be considered "increased" if it exists, based on 'customerNumber', in
     * both this adjustment document and in the document being adjusted, but the one in this document has a larger
     * "financialDocumentLineAmount".
     *
     * @return A List of {@link Pair}s of the increased NAH & the amount it was increased.
     */
    private static List<Pair<NonAppliedHolding, KualiDecimal>> determineIncreasedNonAppliedHoldings(
            final Collection<NonAppliedHolding> adjusteeNahs,
            final Collection<NonAppliedHolding> adjustmentNahs
    ) {
        final BiFunction<KualiDecimal, KualiDecimal, Boolean> increasedNahFunction =
            (adjustmentAmount, adjusteeAmount) -> adjustmentAmount.isGreaterThan(adjusteeAmount);
        final BiFunction<KualiDecimal, KualiDecimal, KualiDecimal> differenceFunction =
            (adjustmentAmount, adjusteeAmount) -> adjustmentAmount.subtract(adjusteeAmount).abs();

        final List<Pair<NonAppliedHolding, KualiDecimal>> increasedNahs =
                determineNonAppliedHoldingsByFunction(
                        adjusteeNahs,
                        adjustmentNahs,
                        increasedNahFunction,
                        differenceFunction
                );

        LOG.debug("determineIncreasedNonAppliedHoldings(...) - Exit - increasedNahs={}", increasedNahs);
        return increasedNahs;
    }

    /*
     * A {@link NonAppliedHolding} (NAH) will be considered "decreased" if it exists, based on 'customerNumber', in
     * both this adjustment document and in the document being adjusted, but the one in this document has a smaller
     * "financialDocumentLineAmount".
     *
     * @return A List of {@link Pair}s of the decreased NAH & the amount it was decreased.
     */
    private static List<Pair<NonAppliedHolding, KualiDecimal>> determineDecreasedNonAppliedHoldings(
            final Collection<NonAppliedHolding> adjusteeNahs,
            final Collection<NonAppliedHolding> adjustmentNahs
    ) {
        final BiFunction<KualiDecimal, KualiDecimal, Boolean> decreasedNahFunction =
            (adjustmentAmount, adjusteeAmount) -> adjustmentAmount.isLessThan(adjusteeAmount);

        final BiFunction<KualiDecimal, KualiDecimal, KualiDecimal> differenceFunction =
            (adjustmentAmount, adjusteeAmount) -> adjusteeAmount.subtract(adjustmentAmount).abs();

        final List<Pair<NonAppliedHolding, KualiDecimal>> decreasedNahs =
                determineNonAppliedHoldingsByFunction(
                        adjusteeNahs,
                        adjustmentNahs,
                        decreasedNahFunction,
                        differenceFunction);

        LOG.debug("determineDecreasedNonAppliedHoldings(...) - Exit - decreasedNahs={}", decreasedNahs);
        return decreasedNahs;
    }

    private static List<Pair<NonAppliedHolding, KualiDecimal>> determineNonAppliedHoldingsByFunction(
            final Collection<NonAppliedHolding> adjusteeNahs,
            final Collection<NonAppliedHolding> adjustmentNahs,
            final BiFunction<KualiDecimal, KualiDecimal, Boolean> compareFunction,
            final BiFunction<KualiDecimal, KualiDecimal, KualiDecimal> differenceFunction
    ) {
        final List<Pair<NonAppliedHolding, KualiDecimal>> nahPairs = new LinkedList<>();

        for (final NonAppliedHolding adjustmentNah : adjustmentNahs) {
            for (final NonAppliedHolding adjusteeNah : adjusteeNahs) {
                final var adjustmentCustomerNumber = adjustmentNah.getCustomerNumber();
                final var adjusteeCustomerNumber = adjusteeNah.getCustomerNumber();
                if (!adjustmentCustomerNumber.equalsIgnoreCase(adjusteeCustomerNumber)) {
                    // This is not the NAH you're looking for
                    continue;
                }

                final var adjustmentAmount = adjustmentNah.getFinancialDocumentLineAmount();
                final var adjusteeAmount = adjusteeNah.getFinancialDocumentLineAmount();

                if (compareFunction.apply(adjustmentAmount, adjusteeAmount)) {
                    final Pair<NonAppliedHolding, KualiDecimal> pair =
                            Pair.of(adjustmentNah, differenceFunction.apply(adjustmentAmount, adjusteeAmount));
                    nahPairs.add(pair);
                }

                break;
            }
        }
        return nahPairs;
    }

    private List<GeneralLedgerPendingEntry> creditTheAccountsReceivableClearingAccountAndObjectCode(
            final GeneralLedgerPendingEntrySequenceHelper sequenceHelper,
            final List<Pair<NonAppliedHolding, KualiDecimal>> nonAppliedHoldings,
            final DocumentHeader documentHeader,
            final String documentTypeCode,
            final Integer postingYear
    ) {
        if (nonAppliedHoldings.isEmpty()) {
            return List.of();
        }

        final SystemInformation unappliedSystemInformation =
                retrieveSystemInformationForInitiator(documentHeader.getInitiatorPrincipalId());

        // Get the university clearing account
        unappliedSystemInformation.refreshReferenceObject("universityClearingAccount");
        final Account universityClearingAccount = unappliedSystemInformation.getUniversityClearingAccount();

        final OffsetDefinition offsetEntryDefinition =
                offsetDefinitionService.getByPrimaryId(
                        postingYear,
                        universityClearingAccount.getChartOfAccountsCode(),
                        documentTypeCode,
                        KFSConstants.BALANCE_TYPE_ACTUAL
                );
        offsetEntryDefinition.refreshReferenceObject("financialObject");

        // Get the university clearing object, object type and sub-object code
        final String unappliedSubAccountNumber = unappliedSystemInformation.getUniversityClearingSubAccountNumber();
        final String unappliedObjectCode = unappliedSystemInformation.getUniversityClearingObjectCode();
        final ObjectCode unappliedUniversityClearingObject = unappliedSystemInformation.getUniversityClearingObject();
        final String unappliedObjectTypeCode = unappliedUniversityClearingObject.getFinancialObjectTypeCode();
        final String unappliedSubObjectCode = unappliedSystemInformation.getUniversityClearingSubObjectCode();

        final List<GeneralLedgerPendingEntry> generatedEntries = new LinkedList<>();

        nonAppliedHoldings
                .stream()
                .filter(pair -> ObjectUtils.isNotNull(pair.getLeft()))
                .map(Pair::getRight)
                .forEach((KualiDecimal creditAmount) -> {

                    final String organizationDocumentNumber = documentHeader.getOrganizationDocumentNumber();

                    final GeneralLedgerPendingEntry actualCredit =
                            createBasicGeneralLedgerPendingEntry(
                                    KFSConstants.GL_CREDIT_CODE,
                                    documentTypeCode,
                                    organizationDocumentNumber
                            );
                    actualCredit.setAccountNumber(universityClearingAccount.getAccountNumber());
                    actualCredit.setChartOfAccountsCode(universityClearingAccount.getChartOfAccountsCode());
                    actualCredit.setFinancialObjectCode(unappliedObjectCode);
                    actualCredit.setFinancialObjectTypeCode(unappliedObjectTypeCode);
                    final String financialSubObjectCode =
                            StringUtils.isBlank(unappliedSubObjectCode)
                                    ? getDashFinancialSubObjectCode()
                                    : unappliedSubObjectCode;
                    actualCredit.setFinancialSubObjectCode(financialSubObjectCode);
                    actualCredit.setProjectCode(getDashProjectCode());
                    final String subAccountNumber =
                            StringUtils.isBlank(unappliedSubAccountNumber)
                                    ? getDashSubAccountNumber()
                                    : unappliedSubAccountNumber;
                    actualCredit.setSubAccountNumber(subAccountNumber);
                    actualCredit.setTransactionLedgerEntryAmount(creditAmount);
                    actualCredit.setTransactionLedgerEntryDescription(documentHeader.getDocumentDescription());
                    actualCredit.setTransactionLedgerEntrySequenceNumber(sequenceHelper.getSequenceCounter());
                    sequenceHelper.increment();
                    actualCredit.setUniversityFiscalYear(postingYear);
                    generatedEntries.add(actualCredit);

                    final GeneralLedgerPendingEntry offsetDebit =
                            createBasicGeneralLedgerPendingEntry(
                                    KFSConstants.GL_DEBIT_CODE,
                                    documentTypeCode,
                                    organizationDocumentNumber
                            );
                    offsetDebit.setAccountNumber(actualCredit.getAccountNumber());
                    offsetDebit.setChartOfAccountsCode(actualCredit.getChartOfAccountsCode());
                    final String financialObjectCode = offsetEntryDefinition.getFinancialObjectCode();
                    offsetDebit.setFinancialObjectCode(financialObjectCode);
                    final ObjectCode financialObject = offsetEntryDefinition.getFinancialObject();
                    final String financialObjectTypeCode = financialObject.getFinancialObjectTypeCode();
                    offsetDebit.setFinancialObjectTypeCode(financialObjectTypeCode);
                    offsetDebit.setFinancialSubObjectCode(getDashFinancialSubObjectCode());
                    offsetDebit.setProjectCode(getDashProjectCode());
                    offsetDebit.setSubAccountNumber(getDashSubAccountNumber());
                    offsetDebit.setTransactionLedgerEntryAmount(actualCredit.getTransactionLedgerEntryAmount());
                    offsetDebit.setTransactionLedgerEntryDescription(KFSConstants.GL_PE_OFFSET_STRING);
                    offsetDebit.setTransactionLedgerEntrySequenceNumber(sequenceHelper.getSequenceCounter());
                    sequenceHelper.increment();
                    offsetDebit.setUniversityFiscalYear(actualCredit.getUniversityFiscalYear());
                    generatedEntries.add(offsetDebit);
                });
        return generatedEntries;
    }

    private SystemInformation retrieveSystemInformationForInitiator(final String principalId) {
        final ChartOrgHolder userOrg =
                financialSystemUserService.getPrimaryOrganization(principalId, ArConstants.AR_NAMESPACE_CODE);
        final String processingChartCode = userOrg.getChartOfAccountsCode();
        final String processingOrganizationCode = userOrg.getOrganizationCode();

        final Integer currentFiscalYear = universityDateService.getCurrentFiscalYear();

        return systemInformationService
                .getByProcessingChartOrgAndFiscalYear(
                        processingChartCode,
                        processingOrganizationCode,
                        currentFiscalYear
                );
    }

    private List<GeneralLedgerPendingEntry> debitTheAccountsReceivableClearingAccountAndObjectCode(
            final GeneralLedgerPendingEntrySequenceHelper sequenceHelper,
            final List<Pair<NonAppliedHolding, KualiDecimal>> nonAppliedHoldings,
            final Document adjusteeDocument,
            final DocumentHeader documentHeader,
            final String documentTypeCode,
            final Integer postingYear
    ) {
        if (nonAppliedHoldings.isEmpty()) {
            return List.of();
        }

        final SystemInformation unappliedSystemInformation =
                retrieveSystemInformationForInitiator(adjusteeDocument.getDocumentHeader().getInitiatorPrincipalId());

        // Get the university clearing account
        unappliedSystemInformation.refreshReferenceObject("universityClearingAccount");
        final Account universityClearingAccount = unappliedSystemInformation.getUniversityClearingAccount();

        final OffsetDefinition offsetEntryDefinition =
                offsetDefinitionService.getByPrimaryId(
                        postingYear,
                        universityClearingAccount.getChartOfAccountsCode(),
                        documentTypeCode,
                        KFSConstants.BALANCE_TYPE_ACTUAL
                );
        offsetEntryDefinition.refreshReferenceObject("financialObject");

        // Get the university clearing object, object type and sub-object code
        final String unappliedSubAccountNumber = unappliedSystemInformation.getUniversityClearingSubAccountNumber();
        final String unappliedObjectCode = unappliedSystemInformation.getUniversityClearingObjectCode();
        final ObjectCode universityClearingObject = unappliedSystemInformation.getUniversityClearingObject();
        final String unappliedObjectTypeCode = universityClearingObject.getFinancialObjectTypeCode();

        final List<GeneralLedgerPendingEntry> generatedEntries = new LinkedList<>();

        nonAppliedHoldings
                .stream()
                .filter(pair -> ObjectUtils.isNotNull(pair.getLeft()))
                .map(Pair::getRight)
                .forEach((KualiDecimal debitAmount) -> {

                    final String organizationDocumentNumber = documentHeader.getOrganizationDocumentNumber();

                    final GeneralLedgerPendingEntry actualDebit =
                            createBasicGeneralLedgerPendingEntry(
                                    KFSConstants.GL_DEBIT_CODE,
                                    documentTypeCode,
                                    organizationDocumentNumber
                            );
                    actualDebit.setAccountNumber(universityClearingAccount.getAccountNumber());
                    actualDebit.setChartOfAccountsCode(universityClearingAccount.getChartOfAccountsCode());
                    actualDebit.setFinancialObjectCode(unappliedObjectCode);
                    actualDebit.setFinancialObjectTypeCode(unappliedObjectTypeCode);
                    actualDebit.setFinancialSubObjectCode(getDashFinancialSubObjectCode());
                    actualDebit.setProjectCode(getDashProjectCode());
                    final String subAccountNumber =
                            StringUtils.isBlank(unappliedSubAccountNumber)
                                    ? getDashSubAccountNumber()
                                    : unappliedSubAccountNumber;
                    actualDebit.setSubAccountNumber(subAccountNumber);
                    actualDebit.setTransactionLedgerEntryAmount(debitAmount);
                    actualDebit.setTransactionLedgerEntryDescription(documentHeader.getDocumentDescription());
                    actualDebit.setTransactionLedgerEntrySequenceNumber(sequenceHelper.getSequenceCounter());
                    sequenceHelper.increment();
                    actualDebit.setUniversityFiscalYear(postingYear);
                    generatedEntries.add(actualDebit);

                    final GeneralLedgerPendingEntry offsetCredit =
                            createBasicGeneralLedgerPendingEntry(
                                    KFSConstants.GL_CREDIT_CODE,
                                    documentTypeCode,
                                    organizationDocumentNumber
                            );
                    offsetCredit.setAccountNumber(actualDebit.getAccountNumber());
                    offsetCredit.setChartOfAccountsCode(actualDebit.getChartOfAccountsCode());
                    offsetCredit.setFinancialObjectCode(offsetEntryDefinition.getFinancialObjectCode());
                    final String financialObjectTypeCode =
                            offsetEntryDefinition.getFinancialObject().getFinancialObjectTypeCode();
                    offsetCredit.setFinancialObjectTypeCode(financialObjectTypeCode);
                    offsetCredit.setFinancialSubObjectCode(getDashFinancialSubObjectCode());
                    offsetCredit.setProjectCode(getDashProjectCode());
                    offsetCredit.setSubAccountNumber(getDashSubAccountNumber());
                    offsetCredit.setTransactionLedgerEntryAmount(actualDebit.getTransactionLedgerEntryAmount());
                    offsetCredit.setTransactionLedgerEntryDescription(KFSConstants.GL_PE_OFFSET_STRING);
                    offsetCredit.setTransactionLedgerEntrySequenceNumber(sequenceHelper.getSequenceCounter());
                    sequenceHelper.increment();
                    offsetCredit.setUniversityFiscalYear(actualDebit.getUniversityFiscalYear());
                    generatedEntries.add(offsetCredit);
                });
        return generatedEntries;
    }

    private List<GeneralLedgerPendingEntry> createPendingEntriesForInvoicePaidApplieds(
            final Document adjusteeDocument,
            final Document adjustmentDocument,
            final GeneralLedgerPendingEntrySequenceHelper sequenceHelper,
            final String documentTypeCode,
            final Integer postingYear
    ) {
        final Collection<InvoicePaidApplied> adjusteeInvoicePaidApplieds = getInvoicePaidApplieds(adjusteeDocument);
        final Collection<InvoicePaidApplied> adjustmentInvoicePaidApplieds =
                getInvoicePaidApplieds(adjustmentDocument);

        final List<Pair<InvoicePaidApplied, KualiDecimal>> newIpas =
                determineNewInvoicePaidApplieds(
                        adjusteeInvoicePaidApplieds,
                        adjustmentInvoicePaidApplieds
                )
                        .stream()
                        .map(ipa -> Pair.of(ipa, ipa.getInvoiceItemAppliedAmount()))
                        .collect(Collectors.toList());
        final List<Pair<InvoicePaidApplied, KualiDecimal>> ipasWhoseAppliedAmountIncreased =
                determineIpasWhoseAppliedAmountIncreased(
                        adjusteeInvoicePaidApplieds,
                        adjustmentInvoicePaidApplieds
                );
        final var glpesForNewAndIncreased =
                creditTheAccountAndObjectCodeFromTheInvoice(
                        sequenceHelper,
                        ListUtils.union(newIpas, ipasWhoseAppliedAmountIncreased),
                        adjustmentDocument.getDocumentHeader(),
                        documentTypeCode,
                        postingYear
                );

        final List<Pair<InvoicePaidApplied, KualiDecimal>> removedIpas =
                determineRemovedInvoicePaidApplieds(
                        adjusteeInvoicePaidApplieds,
                        adjustmentInvoicePaidApplieds
                )
                        .stream()
                        .map(ipa -> Pair.of(ipa, ipa.getInvoiceItemAppliedAmount()))
                        .collect(Collectors.toList());
        final List<Pair<InvoicePaidApplied, KualiDecimal>> ipasWhoseAppliedAmountDecreased =
                determineIpasWhoseAppliedAmountDecreased(
                        adjusteeInvoicePaidApplieds,
                        adjustmentInvoicePaidApplieds
                );
        final var glpesForRemovedAndDecreased =
                debitTheAccountAndObjectCodeFromTheInvoice(
                        sequenceHelper,
                        ListUtils.union(removedIpas, ipasWhoseAppliedAmountDecreased),
                        adjustmentDocument.getDocumentHeader(),
                        documentTypeCode,
                        postingYear
                );

        return ListUtils.union(glpesForNewAndIncreased, glpesForRemovedAndDecreased);
    }

    private static List<InvoicePaidApplied> getInvoicePaidApplieds(final Document adjustee) {
        List<InvoicePaidApplied> adjusteeInvoicePainApplieds = null;
        if (adjustee instanceof PaymentApplicationDocument) {
            adjusteeInvoicePainApplieds = ((PaymentApplicationDocument) adjustee).getInvoicePaidApplieds();
        } else if (adjustee instanceof PaymentApplicationAdjustmentDocument) {
            adjusteeInvoicePainApplieds = ((PaymentApplicationAdjustmentDocument) adjustee).getInvoicePaidApplieds();
        }
        return Objects.requireNonNullElseGet(adjusteeInvoicePainApplieds, List::of);
    }

    /*
     * An {@link InvoicePaidApplied} (IPA) will be considered "new" if it exists, based on
     * 'financialDocumentReferenceInvoiceNumber', in this adjustment document but does not exist in the document
     * being adjusted.
     */
    private static List<InvoicePaidApplied> determineNewInvoicePaidApplieds(
            final Collection<InvoicePaidApplied> adjusteeIpas,
            final Collection<InvoicePaidApplied> adjustmentIpas
    ) {
        final var adjusteeIpaInvoiceNumbers =
                adjusteeIpas.stream()
                        .map(adjusteeIpa -> adjusteeIpa.getFinancialDocumentReferenceInvoiceNumber())
                        .collect(Collectors.toList());
        final var newIpas =
                adjustmentIpas.stream()
                        .filter(adjustmentIpa -> {
                            final String adjustmentIpaInvoiceNumber =
                                    adjustmentIpa.getFinancialDocumentReferenceInvoiceNumber();
                            return !adjusteeIpaInvoiceNumbers.contains(adjustmentIpaInvoiceNumber);
                        })
                        .collect(Collectors.toList());
        LOG.debug("determineNewInvoicePaidApplieds(...) - Exit : newIpas={}", newIpas);
        return newIpas;
    }

    /*
     * An {@link InvoicePaidApplied} (IPA) will be considered "removed" if it exists, based on
     * 'financialDocumentReferenceInvoiceNumber', in the document being adjusted but does not exist in this
     * adjustment document.
     */
    private static List<InvoicePaidApplied> determineRemovedInvoicePaidApplieds(
            final Collection<InvoicePaidApplied> adjusteeIpas,
            final Collection<InvoicePaidApplied> adjustmentIpas
    ) {
        final var adjustmentIpaInvoiceNumbers =
                adjustmentIpas.stream()
                        .map(InvoicePaidApplied::getFinancialDocumentReferenceInvoiceNumber)
                        .collect(Collectors.toList());
        final var removedIpas =
                adjusteeIpas.stream()
                        .filter(adjusteeIpa -> {
                            final String adjusteeIpaInvoiceNumber =
                                    adjusteeIpa.getFinancialDocumentReferenceInvoiceNumber();
                            return !adjustmentIpaInvoiceNumbers.contains(adjusteeIpaInvoiceNumber);
                        })
                        .collect(Collectors.toList());
        LOG.debug("determineRemovedInvoicePaidApplieds(...) - Exit : removedIpas={}", removedIpas);
        return removedIpas;
    }

    /*
     * An {@link InvoicePaidApplied} (IPA) will be considered "increased" if it exists, based on
     * 'financialDocumentReferenceInvoiceNumber', in both this adjustment document and in the document being
     * adjusted, but the one in this document has a larger "invoiceItemAppliedAmount".
     */
    private static List<Pair<InvoicePaidApplied, KualiDecimal>> determineIpasWhoseAppliedAmountIncreased(
            final Collection<InvoicePaidApplied> adjusteeIpas,
            final Collection<InvoicePaidApplied> adjustmentIpas
    ) {
        final BiFunction<KualiDecimal, KualiDecimal, Boolean> increasedIpaFunction =
            (adjustmentAmount, adjusteeAmount) -> adjustmentAmount.isGreaterThan(adjusteeAmount);

        final BiFunction<KualiDecimal, KualiDecimal, KualiDecimal> differenceFunction =
            (adjustmentAmount, adjusteeAmount) -> adjusteeAmount.subtract(adjustmentAmount).abs();

        final List<Pair<InvoicePaidApplied, KualiDecimal>> increasedIpas =
                determineInvoicePaidAppliedsByFunction(
                        adjusteeIpas,
                        adjustmentIpas,
                        increasedIpaFunction,
                        differenceFunction
                );

        LOG.debug("determineIpasWhoseAppliedAmountIncreased(...) - Exit : increasedIpas={}", increasedIpas);
        return increasedIpas;
    }

    /*
     * An {@link InvoicePaidApplied} (IPA) will be considered "decreased" if it exists, based on
     * 'financialDocumentReferenceInvoiceNumber', in both this adjustment document and in the document being
     * adjusted, but the one in this document has a smaller "invoiceItemAppliedAmount".
     */
    private static List<Pair<InvoicePaidApplied, KualiDecimal>> determineIpasWhoseAppliedAmountDecreased(
            final Collection<InvoicePaidApplied> adjusteeIpas,
            final Collection<InvoicePaidApplied> adjustmentIpas
    ) {
        final BiFunction<KualiDecimal, KualiDecimal, Boolean> decreasedIpaFunction =
            (adjustmentAmount, adjusteeAmount) -> adjustmentAmount.isLessThan(adjusteeAmount);

        final BiFunction<KualiDecimal, KualiDecimal, KualiDecimal> differenceFunction =
            (adjustmentAmount, adjusteeAmount) -> adjustmentAmount.subtract(adjusteeAmount).abs();

        final List<Pair<InvoicePaidApplied, KualiDecimal>> decreasedIpas =
                determineInvoicePaidAppliedsByFunction(
                        adjusteeIpas,
                        adjustmentIpas,
                        decreasedIpaFunction,
                        differenceFunction
                );

        LOG.debug("determineIpasWhoseAppliedAmountDecreased(...) - Exit : decreasedIpas={}", decreasedIpas);
        return decreasedIpas;
    }

    private static List<Pair<InvoicePaidApplied, KualiDecimal>> determineInvoicePaidAppliedsByFunction(
            final Collection<InvoicePaidApplied> adjusteeIpas,
            final Collection<InvoicePaidApplied> adjustmentIpas,
            final BiFunction<KualiDecimal, KualiDecimal, Boolean> compareFunction,
            final BiFunction<KualiDecimal, KualiDecimal, KualiDecimal> differenceFunction
    ) {
        final List<Pair<InvoicePaidApplied, KualiDecimal>> ipas = new LinkedList<>();

        for (final InvoicePaidApplied adjustmentIpa : adjustmentIpas) {
            for (final InvoicePaidApplied adjusteeIpa : adjusteeIpas) {
                if (!InvoicePaidApplied.referToSameInvoiceItem(adjustmentIpa, adjusteeIpa)) {
                    // This is not the IPA you're looking for
                    continue;
                }

                final var adjustmentAmount = adjustmentIpa.getInvoiceItemAppliedAmount();
                final var adjusteeAmount = adjusteeIpa.getInvoiceItemAppliedAmount();
                if (compareFunction.apply(adjustmentAmount, adjusteeAmount)) {
                    final Pair pair =
                            Pair.of(adjustmentIpa, differenceFunction.apply(adjustmentAmount, adjusteeAmount));
                    ipas.add(pair);
                }

                break;
            }
        }
        return ipas;
    }

    // Creates an offset debit too.
    private List<GeneralLedgerPendingEntry> creditTheAccountAndObjectCodeFromTheInvoice(
            final GeneralLedgerPendingEntrySequenceHelper sequenceHelper,
            final List<Pair<InvoicePaidApplied, KualiDecimal>> invoicePaidApplieds,
            final DocumentHeader documentHeader,
            final String documentTypeCode,
            final Integer postingYear
    ) {

        if (invoicePaidApplieds.isEmpty()) {
            return List.of();
        }

        final List<GeneralLedgerPendingEntry> generatedEntries = new LinkedList<>();

        for (final Pair<InvoicePaidApplied, KualiDecimal> pair : invoicePaidApplieds) {

            final InvoicePaidApplied ipa = pair.getLeft();
            final KualiDecimal creditAmount = pair.getRight();

            ipa.refreshNonUpdateableReferences();

            final CustomerInvoiceDetail invoiceDetail = ipa.getInvoiceDetail();
            final Account invoiceAccount = invoiceDetail.getAccount();

            final ObjectCode invoiceReceivableObjectCode = getInvoiceReceivableObjectCode(ipa);

            final String organizationDocumentNumber = documentHeader.getOrganizationDocumentNumber();

            final GeneralLedgerPendingEntry actualCredit =
                    createBasicGeneralLedgerPendingEntry(
                            KFSConstants.GL_CREDIT_CODE,
                            documentTypeCode,
                            organizationDocumentNumber
                    );
            actualCredit.setAccountNumber(invoiceAccount.getAccountNumber());
            actualCredit.setChartOfAccountsCode(invoiceAccount.getChartOfAccountsCode());
            final String projectCode =
                    StringUtils.isBlank(invoiceDetail.getProjectCode())
                            ? getDashProjectCode()
                            : invoiceDetail.getProjectCode();
            actualCredit.setProjectCode(projectCode);
            final String subAccountNumber =
                    StringUtils.isBlank(invoiceDetail.getSubAccountNumber())
                            ? getDashSubAccountNumber()
                            : invoiceDetail.getSubAccountNumber();
            actualCredit.setSubAccountNumber(subAccountNumber);
            actualCredit.setTransactionLedgerEntryAmount(creditAmount);
            actualCredit.setUniversityFiscalYear(postingYear);
            actualCredit.setFinancialObjectCode(invoiceReceivableObjectCode.getFinancialObjectCode());
            actualCredit.setFinancialObjectTypeCode(invoiceReceivableObjectCode.getFinancialObjectTypeCode());
            final String financialSubObjectCode =
                    StringUtils.isBlank(invoiceDetail.getFinancialSubObjectCode())
                            ? getDashFinancialSubObjectCode()
                            : invoiceDetail.getFinancialSubObjectCode();
            actualCredit.setFinancialSubObjectCode(financialSubObjectCode);
            actualCredit.setTransactionLedgerEntryDescription(documentHeader.getDocumentDescription());
            actualCredit.setTransactionLedgerEntrySequenceNumber(sequenceHelper.getSequenceCounter());
            sequenceHelper.increment();
            generatedEntries.add(actualCredit);

            final GeneralLedgerPendingEntry offsetDebit =
                    createBasicGeneralLedgerPendingEntry(
                            KFSConstants.GL_DEBIT_CODE,
                            documentTypeCode,
                            organizationDocumentNumber
                    );
            offsetDebit.setAccountNumber(actualCredit.getAccountNumber());
            offsetDebit.setChartOfAccountsCode(actualCredit.getChartOfAccountsCode());
            offsetDebit.setProjectCode(actualCredit.getProjectCode());
            offsetDebit.setSubAccountNumber(actualCredit.getSubAccountNumber());
            offsetDebit.setTransactionLedgerEntryAmount(actualCredit.getTransactionLedgerEntryAmount());
            offsetDebit.setUniversityFiscalYear(actualCredit.getUniversityFiscalYear());
            generalLedgerPendingEntryService.populateOffsetGeneralLedgerPendingEntry(
                    actualCredit.getUniversityFiscalYear(),
                    actualCredit,
                    sequenceHelper,
                    offsetDebit
            );
            sequenceHelper.increment();
            generatedEntries.add(offsetDebit);
        }
        return generatedEntries;
    }

    // Creates an offset credit too.
    private List<GeneralLedgerPendingEntry> debitTheAccountAndObjectCodeFromTheInvoice(
            final GeneralLedgerPendingEntrySequenceHelper sequenceHelper,
            final List<Pair<InvoicePaidApplied, KualiDecimal>> invoicePaidApplieds,
            final DocumentHeader documentHeader,
            final String documentTypeCode,
            final Integer postingYear
    ) {

        if (invoicePaidApplieds.isEmpty()) {
            return List.of();
        }

        final List<GeneralLedgerPendingEntry> generatedEntries = new LinkedList<>();

        for (final Pair<InvoicePaidApplied, KualiDecimal> pair : invoicePaidApplieds) {

            final InvoicePaidApplied ipa = pair.getLeft();
            final KualiDecimal debitAmount = pair.getRight();

            ipa.refreshNonUpdateableReferences();

            final CustomerInvoiceDetail invoiceDetail = ipa.getInvoiceDetail();
            final Account invoiceAccount = invoiceDetail.getAccount();
            final ObjectCode invoiceReceivableObjectCode = getInvoiceReceivableObjectCode(ipa);

            final String organizationDocumentNumber = documentHeader.getOrganizationDocumentNumber();

            final GeneralLedgerPendingEntry actualDebit =
                    createBasicGeneralLedgerPendingEntry(
                            KFSConstants.GL_DEBIT_CODE,
                            documentTypeCode,
                            organizationDocumentNumber
                    );
            actualDebit.setAccountNumber(invoiceAccount.getAccountNumber());
            actualDebit.setChartOfAccountsCode(invoiceAccount.getChartOfAccountsCode());
            actualDebit.setFinancialObjectCode(invoiceReceivableObjectCode.getFinancialObjectCode());
            actualDebit.setFinancialObjectTypeCode(invoiceReceivableObjectCode.getFinancialObjectTypeCode());
            final String financialSubObjectCode =
                    StringUtils.isBlank(invoiceDetail.getFinancialSubObjectCode())
                            ? getDashFinancialSubObjectCode()
                            : invoiceDetail.getFinancialSubObjectCode();
            actualDebit.setFinancialSubObjectCode(financialSubObjectCode);
            final String projectCode =
                    StringUtils.isBlank(invoiceDetail.getProjectCode())
                            ? getDashProjectCode()
                            : invoiceDetail.getProjectCode();
            actualDebit.setProjectCode(projectCode);
            final String subAccountNumber =
                    StringUtils.isBlank(invoiceDetail.getSubAccountNumber())
                            ? getDashSubAccountNumber()
                            : invoiceDetail.getSubAccountNumber();
            actualDebit.setSubAccountNumber(subAccountNumber);
            actualDebit.setTransactionLedgerEntryAmount(debitAmount);
            actualDebit.setTransactionLedgerEntryDescription(documentHeader.getDocumentDescription());
            actualDebit.setTransactionLedgerEntrySequenceNumber(sequenceHelper.getSequenceCounter());
            sequenceHelper.increment();
            actualDebit.setUniversityFiscalYear(postingYear);
            generatedEntries.add(actualDebit);

            final GeneralLedgerPendingEntry offsetCredit =
                    createBasicGeneralLedgerPendingEntry(
                            KFSConstants.GL_CREDIT_CODE,
                            documentTypeCode,
                            organizationDocumentNumber
                    );
            offsetCredit.setAccountNumber(actualDebit.getAccountNumber());
            offsetCredit.setChartOfAccountsCode(actualDebit.getChartOfAccountsCode());
            offsetCredit.setSubAccountNumber(actualDebit.getSubAccountNumber());
            offsetCredit.setProjectCode(actualDebit.getProjectCode());
            offsetCredit.setTransactionLedgerEntryAmount(actualDebit.getTransactionLedgerEntryAmount());
            offsetCredit.setUniversityFiscalYear(actualDebit.getUniversityFiscalYear());
            generalLedgerPendingEntryService.populateOffsetGeneralLedgerPendingEntry(
                    actualDebit.getUniversityFiscalYear(),
                    actualDebit,
                    sequenceHelper,
                    offsetCredit
            );
            sequenceHelper.increment();
            generatedEntries.add(offsetCredit);
        }
        return generatedEntries;
    }

    private static GeneralLedgerPendingEntry createBasicGeneralLedgerPendingEntry(
            final String transactionDebitCreditCode,
            final String documentTypeCode,
            final String organizationDocumentNumber
    ) {
        final GeneralLedgerPendingEntry glpe = new GeneralLedgerPendingEntry();

        glpe.setFinancialBalanceTypeCode(KFSConstants.BALANCE_TYPE_ACTUAL);
        glpe.setFinancialDocumentTypeCode(documentTypeCode);
        glpe.setFinancialSystemOriginationCode(KFSConstants.ORIGIN_CODE_KUALI);
        glpe.setOrganizationDocumentNumber(organizationDocumentNumber);
        glpe.setTransactionDebitCreditCode(transactionDebitCreditCode);

        return glpe;
    }

    private ObjectCode getInvoiceReceivableObjectCode(final InvoicePaidApplied invoicePaidApplied) {
        ObjectCode objectCode = null;

        final CustomerInvoiceDocument customerInvoiceDocument = invoicePaidApplied.getCustomerInvoiceDocument();
        final CustomerInvoiceDetail customerInvoiceDetail = invoicePaidApplied.getInvoiceDetail();
        final ReceivableCustomerInvoiceDetail receivableInvoiceDetail =
                new ReceivableCustomerInvoiceDetail(customerInvoiceDetail, customerInvoiceDocument);

        if (ObjectUtils.isNotNull(receivableInvoiceDetail)
                && ObjectUtils.isNotNull(receivableInvoiceDetail.getFinancialObjectCode())) {
            objectCode = receivableInvoiceDetail.getObjectCode();
            if (ObjectUtils.isNull(objectCode)) {
                final Map<String, Object> fieldKeys = Map.ofEntries(
                    entry(KFSPropertyConstants.CHART_OF_ACCOUNTS_CODE, receivableInvoiceDetail.getChartOfAccountsCode()),
                    entry(KFSPropertyConstants.FINANCIAL_OBJECT_CODE, receivableInvoiceDetail.getFinancialObjectCode())
                );
                objectCode = businessObjectService.findByPrimaryKey(ObjectCodeCurrent.class, fieldKeys);
            }
        }

        return objectCode;
    }

    // This exists so tests can avoid static mocking
    String getDashFinancialSubObjectCode() {
        return KFSConstants.getDashFinancialSubObjectCode();
    }

    // This exists so tests can avoid static mocking
    String getDashProjectCode() {
        return KFSConstants.getDashProjectCode();
    }

    // This exists so tests can avoid static mocking
    String getDashSubAccountNumber() {
        return KFSConstants.getDashSubAccountNumber();
    }

    // This exists so tests can avoid static mocking
    Person getPerson() {
        return GlobalVariables.getUserSession().getPerson();
    }

    /**
     * 1. When an invoice is removed or if the applied amount is changed to zero:
     *     - set the Open Invoice Indicator on the Invoice to Yes
     *     - blank out the Close Date on the Invoice
     *     - add a note to the Invoice: Reopened by <user who initiated the APPA> with APPA <doc number>
     * 2. When the applied amount results in taking the open amount for an invoice down to zero:
     *     - set the Open Invoice Indicator on the Invoice to No
     *     - set the Close Date on the Invoice to today’s date
     *     - add a note to the Invoice: Closed by <user who initiated the APPA> with APPA <doc number>
     */
    public void postProcess(final Document adjusteeDocument, final Document adjustmentDocument) {
        final Collection<InvoicePaidApplied> adjusteeInvoicePaidApplieds =
                getInvoicePaidApplieds(adjusteeDocument);

        final Collection<InvoicePaidApplied> adjustmentInvoicePaidApplieds =
                getInvoicePaidApplieds(adjustmentDocument);

        final String documentNumber = adjustmentDocument.getDocumentNumber();

        final List<InvoicePaidApplied> removedIpas =
                determineRemovedInvoicePaidApplieds(adjusteeInvoicePaidApplieds, adjustmentInvoicePaidApplieds);
        // Removing an InvoicePaidApplied from the adjustment document is an implicit adjustment of the adjustee's
        // InvoicePaidApplied. In other words, its the same as setting the InvoicePaidApplied amount to zero on the
        // adjustment document
        markInvoicePaidAppliedsAdjusted(removedIpas);

        final List<Pair<InvoicePaidApplied, KualiDecimal>> ipasWhoseAppliedAmountDecreased =
                determineIpasWhoseAppliedAmountDecreased(adjusteeInvoicePaidApplieds, adjustmentInvoicePaidApplieds);

        final List<InvoicePaidApplied> ipasWhoseAppliedAmountDecreasedToZero =
                ipasWhoseAppliedAmountDecreased.stream()
                        .map(pair -> pair.getLeft())
                        .filter(ipa -> ipa.getInvoiceItemAppliedAmount().isZero())
                        .collect(Collectors.toList());

        final List<InvoicePaidApplied> newIpas =
                determineNewInvoicePaidApplieds(adjusteeInvoicePaidApplieds, adjustmentInvoicePaidApplieds);
        final List<InvoicePaidApplied> newIpasWhoseOpenAmountIsZero =
                newIpas.stream()
                        .filter(ipa ->
                                Objects.requireNonNullElse(ipa.getInvoiceItemOpenAmount(), KualiDecimal.ZERO)
                                        .isZero()
                        )
                        .collect(Collectors.toList());

        final List<Pair<InvoicePaidApplied, KualiDecimal>> ipasWhoseAppliedAmountIncreased =
                determineIpasWhoseAppliedAmountIncreased(adjusteeInvoicePaidApplieds, adjustmentInvoicePaidApplieds);
        final List<InvoicePaidApplied> ipasWhoseAppliedAmountIncreasedAndOpenAmountIsNowZero =
                ipasWhoseAppliedAmountIncreased.stream()
                        .map(pair -> pair.getLeft())
                        .filter(ipa ->
                                Objects.requireNonNullElse(ipa.getInvoiceItemOpenAmount(), KualiDecimal.ZERO)
                                        .isZero()
                        )
                        .collect(Collectors.toList());

        final List<InvoicePaidApplied> adjustedInvoicePaidApplieds =
                findAdjustedInvoicePaidApplieds(adjusteeInvoicePaidApplieds, adjustmentInvoicePaidApplieds);
        markInvoicePaidAppliedsAdjusted(adjustedInvoicePaidApplieds);

        updateInvoicePaidAppliedsWithCurrentInvoiceOpenAmounts(adjustmentInvoicePaidApplieds);

        processInvoiceWasRemovedOrItsAppliedAmountDecreasedToZero(
                documentNumber,
                removedIpas,
                ipasWhoseAppliedAmountDecreasedToZero
        );

        processInvoiceAppliedAmountIncreasedAndOpenAmountIsNowZero(
                documentNumber,
                newIpasWhoseOpenAmountIsZero,
                ipasWhoseAppliedAmountIncreasedAndOpenAmountIsNowZero
        );
    }

    // returns any InvoicePaidApplieds from the adjusteeInvoicePaidApplieds that share an invoice and item number with
    // an InvoicePaidApplied in the adjustmentInvoicePaidApplieds list
    private List<InvoicePaidApplied> findAdjustedInvoicePaidApplieds(
            Collection<InvoicePaidApplied> adjusteeInvoicePaidApplieds,
            Collection<InvoicePaidApplied> adjustmentInvoicePaidApplieds
    ) {
        final List<InvoicePaidApplied> adjustedInvoicePaidApplieds = new LinkedList<>();
        for (final InvoicePaidApplied adjustmentInvoicePaidApplied : adjustmentInvoicePaidApplieds) {
            for (final InvoicePaidApplied adjusteeInvoicePaidApplied: adjusteeInvoicePaidApplieds) {
                if (InvoicePaidApplied.referToSameInvoiceItem(adjustmentInvoicePaidApplied, adjusteeInvoicePaidApplied)) {
                    adjustedInvoicePaidApplieds.add(adjusteeInvoicePaidApplied);
                }
            }
        }
        return adjustedInvoicePaidApplieds;
    }

    private void markInvoicePaidAppliedsAdjusted(List<InvoicePaidApplied> invoicePaidApplieds) {
        invoicePaidApplieds.forEach(invoicePaidApplied -> {
            invoicePaidApplied.setAdjusted(true);
            businessObjectService.save(invoicePaidApplied);
        });
    }

    private void updateInvoicePaidAppliedsWithCurrentInvoiceOpenAmounts(
            Collection<InvoicePaidApplied> invoicePaidApplieds
    ) {
        invoicePaidApplieds.forEach(invoicePaidApplied -> {
            final CustomerInvoiceDetail invoiceDetail =
                    customerInvoiceDetailService.getCustomerInvoiceDetail(
                            invoicePaidApplied.getFinancialDocumentReferenceInvoiceNumber(),
                            invoicePaidApplied.getInvoiceItemNumber());
            if (invoiceDetail != null) {
                invoicePaidApplied.setInvoiceItemOpenAmount(invoiceDetail.getAmountOpen());
            }
            businessObjectService.save(invoicePaidApplied);
        });
    }

    private void processInvoiceWasRemovedOrItsAppliedAmountDecreasedToZero(
            final String documentNumber,
            final Collection<InvoicePaidApplied> removedIpas,
            final Collection<InvoicePaidApplied> decreasedAndAppliedAmountIsNowZero
    ) {
        Stream.of(removedIpas, decreasedAndAppliedAmountIsNowZero)
                .flatMap(Collection::stream)
                .forEach(ipa -> {
                    LOG.debug("processInvoiceWasRemovedOrItsAppliedAmountDecreasedToZero(...) - Processing : ipa={}",
                            ipa);
                    final CustomerInvoiceDocument invoiceDocument = ipa.getCustomerInvoiceDocument();

                    invoiceDocument.setOpenInvoiceIndicator(true);
                    invoiceDocument.setClosedDate(null);

                    final String note =
                            String.format("Reopened by %s with APPA %s", getPrincipalName(), documentNumber);
                    final Note noteObj = documentService.createNoteFromDocument(invoiceDocument, note);
                    invoiceDocument.addNote(noteObj);
                    noteService.save(noteObj);

                    documentService.updateDocument(invoiceDocument);
                });
    }

    private void processInvoiceAppliedAmountIncreasedAndOpenAmountIsNowZero(
            final String documentNumber,
            final Collection<InvoicePaidApplied> newIpasWhoseOpenAmountIsZero,
            final Collection<InvoicePaidApplied> increasedAndOpenAmountIsNowZero
    ) {
        Stream.of(newIpasWhoseOpenAmountIsZero, increasedAndOpenAmountIsNowZero)
                .flatMap(Collection::stream)
                .forEach(ipa -> {
                    LOG.debug("processInvoiceAppliedAmountIncreasedAndOpenAmountIsNowZero(...) - Processing : ipa={}",
                            ipa);
                    final CustomerInvoiceDocument invoiceDocument = ipa.getCustomerInvoiceDocument();

                    invoiceDocument.setOpenInvoiceIndicator(false);

                    final Date now = determineNow();
                    invoiceDocument.setClosedDate(now);

                    final String note =
                            String.format("Closed by %s with APPA %s", getPrincipalName(), documentNumber);
                    final Note noteObj = documentService.createNoteFromDocument(invoiceDocument, note);
                    invoiceDocument.addNote(noteObj);
                    noteService.save(noteObj);

                    documentService.updateDocument(invoiceDocument);
                });
    }

    // non-private for testing purposes.
    String getPrincipalName() {
        return GlobalVariables.getUserSession().getPerson().getPrincipalName();
    }

    // non-private for testing purposes.
    Date determineNow() {
        final DateTimeService dateTimeService = SpringContext.getBean(DateTimeService.class);
        return new Date(dateTimeService.getCurrentDate().getTime());
    }

    /**
     * For a given {@link GeneralLedgerPendingEntry}, if EITHER it's 'universityFiscalPeriodCode' or it's
     * 'universityFiscalYear' are not populated then the 'currentUniversityDate' will be used to populate BOTH
     * pieces of data, to ensure they match.
     *
     * TODO: Handle year end documents
     */
    public void fillInFiscalPeriodYear(final Collection<GeneralLedgerPendingEntry> glpes) {
        LOG.debug("fillInFiscalPeriodYear(...) - Enter");

        if (glpes.isEmpty()) {
            LOG.debug("fillInFiscalPeriodYear(...) - Exit; no GLPEs were provided");
            return;
        }

        final UniversityDate currentUniversityDate = universityDateService.getCurrentUniversityDate();
        final String currentFiscalPeriod = currentUniversityDate.getUniversityFiscalAccountingPeriod();
        final Integer currentFiscalYear = currentUniversityDate.getUniversityFiscalYear();

        glpes.forEach(glpe -> {
            final String glpeFiscalPeriod = glpe.getUniversityFiscalPeriodCode();
            final Integer glpeFiscalYear = glpe.getUniversityFiscalYear();
            if (StringUtils.isBlank(glpeFiscalPeriod) || glpeFiscalYear == null) {
                glpe.setUniversityFiscalPeriodCode(currentFiscalPeriod);
                glpe.setUniversityFiscalYear(currentFiscalYear);
            }
        });

        LOG.debug("fillInFiscalPeriodYear(...) - Exit");
    }

}
