/*
 * The Kuali Financial System, a comprehensive financial management system for higher education.
 *
 * Copyright 2005-2022 Kuali, Inc.
 *
 * This program is free software: you can redistribute it and/or modify
 * it under the terms of the GNU Affero General Public License as published by
 * the Free Software Foundation, either version 3 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU Affero General Public License for more details.
 *
 * You should have received a copy of the GNU Affero General Public License
 * along with this program.  If not, see <http://www.gnu.org/licenses/>.
 */
package org.kuali.kfs.module.ar.document.validation.impl;

import java.util.HashMap;
import java.util.Map;

import org.apache.commons.lang3.StringUtils;
import org.kuali.kfs.kns.document.MaintenanceDocument;
import org.kuali.kfs.kns.maintenance.rules.MaintenanceDocumentRuleBase;
import org.kuali.kfs.module.ar.ArKeyConstants;
import org.kuali.kfs.module.ar.ArPropertyConstants;
import org.kuali.kfs.module.ar.ArPropertyConstants.ContractsGrantsInvoiceObjectCodeFields;
import org.kuali.kfs.module.ar.businessobject.ContractsGrantsInvoiceObjectCode;
import org.kuali.kfs.sys.KFSPropertyConstants;

public class ContractsGrantsInvoiceObjectCodeRule extends MaintenanceDocumentRuleBase {

    protected ContractsGrantsInvoiceObjectCode cgiObjectCode;

    @Override
    public void setupConvenienceObjects() {
        super.setupConvenienceObjects();
        cgiObjectCode = (ContractsGrantsInvoiceObjectCode) super.getNewBo();
    }

    @Override
    protected boolean processCustomRouteDocumentBusinessRules(MaintenanceDocument document) {
        boolean result = super.processCustomRouteDocumentBusinessRules(document);
        result &= checkFundFields();
        return result;
    }

    protected boolean checkFundFields() {
        boolean result = true;
        int fundFieldCount = 0;
        String lastPopulatedFundField = null;
        String chartOfAccountsCode = cgiObjectCode.getChartOfAccountsCode();
        Integer fiscalYear = cgiObjectCode.getUniversityFiscalYear();
        if (StringUtils.isNotEmpty(cgiObjectCode.getFundGroupCode())) {
            result = checkUnique(ContractsGrantsInvoiceObjectCodeFields.FUND_GROUP_CODE,
                    cgiObjectCode.getFundGroupCode(), chartOfAccountsCode, fiscalYear);
            lastPopulatedFundField = ContractsGrantsInvoiceObjectCodeFields.FUND_GROUP_CODE;
            fundFieldCount++;
        }
        if (StringUtils.isNotEmpty(cgiObjectCode.getSubFundGroupTypeCode())) {
            result &= checkUnique(ContractsGrantsInvoiceObjectCodeFields.SUB_FUND_GROUP_TYPE_CODE,
                    cgiObjectCode.getSubFundGroupTypeCode(), chartOfAccountsCode, fiscalYear);
            lastPopulatedFundField = ContractsGrantsInvoiceObjectCodeFields.SUB_FUND_GROUP_TYPE_CODE;
            fundFieldCount++;
        }
        if (StringUtils.isNotEmpty(cgiObjectCode.getSubFundGroupCode())) {
            result &= checkUnique(ContractsGrantsInvoiceObjectCodeFields.SUB_FUND_GROUP_CODE,
                    cgiObjectCode.getSubFundGroupCode(), chartOfAccountsCode, fiscalYear);
            lastPopulatedFundField = ContractsGrantsInvoiceObjectCodeFields.SUB_FUND_GROUP_CODE;
            fundFieldCount++;
        }

        if (fundFieldCount == 0) {
            putFieldError(ContractsGrantsInvoiceObjectCodeFields.FUND_GROUP_CODE,
                    ArKeyConstants.ERROR_CONTRACTS_GRANTS_INVOICE_OBJECT_ONE_FUND_FIELD_REQUIRED);
            result = false;
        } else if (fundFieldCount > 1) {
            putFieldError(lastPopulatedFundField,
                    ArKeyConstants.ERROR_CONTRACTS_GRANTS_INVOICE_OBJECT_ONLY_ONE_FUND_FIELD_ALLOWED);
            result = false;
        }
        return result;
    }

    private boolean checkUnique(String propertyName, String propertyValue, String chartOfAccountsCode,
            Integer fiscalYear) {
        Map<String, Object> criteria = new HashMap<>();
        criteria.put(propertyName, propertyValue);
        criteria.put(KFSPropertyConstants.CHART_OF_ACCOUNTS_CODE, chartOfAccountsCode);
        criteria.put(KFSPropertyConstants.UNIVERSITY_FISCAL_YEAR, fiscalYear);
        criteria.put(ArPropertyConstants.ContractsGrantsInvoiceObjectCodeFields.ACTIVE, true);

        Map<String, Object> negativeCriteria = new HashMap<>();
        if (cgiObjectCode.getInvoiceObjectCodeIdentifier() != null) {
            negativeCriteria.put(
                    ArPropertyConstants.ContractsGrantsInvoiceObjectCodeFields.INVOICE_OBJECT_CODE_IDENTIFIER,
                    cgiObjectCode.getInvoiceObjectCodeIdentifier());
        }
        int existingCount = getBoService().countMatching(ContractsGrantsInvoiceObjectCode.class, criteria,
                negativeCriteria);
        if (existingCount > 0) {
            String propertyLabel = getDataDictionaryService().getAttributeLabel(ContractsGrantsInvoiceObjectCode.class,
                    propertyName);
            putFieldError(propertyName, ArKeyConstants.ERROR_CONTRACTS_GRANTS_INVOICE_OBJECT_CODE_ALREADY_USED,
                    new String[] {propertyLabel, propertyValue, chartOfAccountsCode});
            return false;
        }
        return true;
    }
}
