/*
 * The Kuali Financial System, a comprehensive financial management system for higher education.
 *
 * Copyright 2005-2022 Kuali, Inc.
 *
 * This program is free software: you can redistribute it and/or modify
 * it under the terms of the GNU Affero General Public License as published by
 * the Free Software Foundation, either version 3 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU Affero General Public License for more details.
 *
 * You should have received a copy of the GNU Affero General Public License
 * along with this program.  If not, see <http://www.gnu.org/licenses/>.
 */
package org.kuali.kfs.module.ar.document.validation.impl;

import org.kuali.kfs.kns.rules.DocumentRuleBase;
import org.kuali.kfs.krad.document.Document;
import org.kuali.kfs.krad.util.GlobalVariables;
import org.kuali.kfs.module.ar.ArKeyConstants;
import org.kuali.kfs.module.ar.businessobject.InvoicePaidApplied;
import org.kuali.kfs.module.ar.document.PaymentApplicationAdjustmentDocument;
import org.kuali.kfs.sys.KFSConstants;

import java.util.List;
import java.util.function.Predicate;
import java.util.stream.Collectors;

public class PaymentApplicationAdjustmentDocumentRule extends DocumentRuleBase {
    @Override
    protected boolean processCustomRouteDocumentBusinessRules(Document document) {
        boolean isValid = super.processCustomRouteDocumentBusinessRules(document);

        final PaymentApplicationAdjustmentDocument applicationAdjustmentDocument =
                (PaymentApplicationAdjustmentDocument) document;
        final List<String> invalidInvoicePaths =
                getInvalidInvoicePaths(applicationAdjustmentDocument.getInvoicePaidApplieds());
        boolean invoiceAmountsValid = invalidInvoicePaths.isEmpty();
        if (!invoiceAmountsValid) {
            invalidInvoicePaths.forEach(path ->
                    GlobalVariables.getMessageMap().putError(path,
                            ArKeyConstants.PaymentApplicationAdjustmentDocumentErrors.AMOUNT_TO_BE_APPLIED_EXCEEDS_AMOUNT_OUTSTANDING));
            GlobalVariables.getMessageMap().putError(KFSConstants.MODAL_TITLE_KEY,
                    ArKeyConstants.PaymentApplicationAdjustmentDocumentErrors.AMOUNT_TO_BE_APPLIED_EXCEEDS_AMOUNT_OUTSTANDING_TITLE);
            GlobalVariables.getMessageMap().putError(KFSConstants.MODAL_MESSAGE_KEY,
                    ArKeyConstants.PaymentApplicationAdjustmentDocumentErrors.AMOUNT_TO_BE_APPLIED_EXCEEDS_AMOUNT_OUTSTANDING_MESSAGE);
        }

        boolean documentAmountIsValid = validateDocumentAmount(applicationAdjustmentDocument);

        return isValid && documentAmountIsValid && invoiceAmountsValid;
    }

    /**
     * Performs similar validation as the route event but does not fail if validations fail. The purpose of this
     * behavior is to provide the user with feedback about potential submission problems in the fucutre
     *
     * @param document
     * @return
     */
    @Override
    protected boolean processCustomSaveDocumentBusinessRules(Document document) {
        boolean isValid = super.processCustomSaveDocumentBusinessRules(document);
        List<String> invalidInvoicePaths = getInvalidInvoicePaths(((PaymentApplicationAdjustmentDocument) document)
                .getInvoicePaidApplieds());
        invalidInvoicePaths.forEach(path -> GlobalVariables.getMessageMap().putWarning(path,
                ArKeyConstants.PaymentApplicationAdjustmentDocumentErrors.AMOUNT_TO_BE_APPLIED_EXCEEDS_AMOUNT_OUTSTANDING));
        return isValid;
    }

    private boolean validateDocumentAmount(PaymentApplicationAdjustmentDocument applicationAdjustmentDocument) {
        boolean balanceIsValid = applicationAdjustmentDocument.getUnallocatedBalance().isZero();
        if (!balanceIsValid) {
            GlobalVariables.getMessageMap().putError(KFSConstants.MODAL_TITLE_KEY,
                    ArKeyConstants.PaymentApplicationAdjustmentDocumentErrors.OPEN_AMOUNT_IS_NON_ZERO_TITLE);
            GlobalVariables.getMessageMap().putError(KFSConstants.MODAL_MESSAGE_KEY,
                    ArKeyConstants.PaymentApplicationAdjustmentDocumentErrors.OPEN_AMOUNT_IS_NON_ZERO_MESSAGE);
        }
        return balanceIsValid;
    }

    private List<String> getInvalidInvoicePaths(List<InvoicePaidApplied> invoicePaidApplieds) {
        return invoicePaidApplieds.stream()
                // make sure each invoice detail is up to date
                .map(invoicePaidApplied -> {
                    invoicePaidApplied.refreshReferenceObject("invoiceDetail");
                    return invoicePaidApplied;
                })
                // don't validate invoice paid applieds that haven't been updated.
                // They've already been applied in a previous document
                .filter(PaymentApplicationAdjustmentDocumentRule::invoiceAmountHasIncreased)
                .filter(Predicate.not(PaymentApplicationAdjustmentDocumentRule::increaseInvoiceIsValid))
                .map(invoicePaidApplied -> {
                    // the invoice item indices start at 1
                    final var invoiceDetailIndex = invoicePaidApplied.getInvoiceItemNumber() - 1;
                    return String.format("invoiceApplications.%s.detailApplications.%d.amountApplied",
                    invoicePaidApplied.getFinancialDocumentReferenceInvoiceNumber(), invoiceDetailIndex);
                })
                .collect(Collectors.toList());
    }

    /**
     * Checks to see if an invoicePaidApplied applied amount has increased since the last time it was processed on the
     * invoice. The invoice detail contains the previous amount applied and our invoicePaidApplied represents the new
     * amount we want to apply. If the new amount is greater than the old invoice detail amount, we know this is an
     * increase in amount applied
     *
     * @param invoicePaidApplied the InvoicePaidApplied in question
     * @return true if the new amountApplied is greater than the invoiceDetails amount. Otherwise, false
     */
    private static boolean invoiceAmountHasIncreased(InvoicePaidApplied invoicePaidApplied) {
        return invoicePaidApplied.getInvoiceItemAppliedAmount()
                .isGreaterThan(invoicePaidApplied.getInvoiceDetail().getAmountApplied());
    }

    /**
     * Given an invoicePaidApplied that should have increased its amount applied, this compares the increase delta to
     * the current amount open on the invoice. The amount applied can only increase by the available open amount. To
     * compute how much the applied amount has been increased, we look at he new applied amount and subtract the old
     * applied amount that is stored on the invoice detail.
     *
     * @param invoicePaidApplied the InvoicePaidApplied inquestion
     * @return true if the increased amount applied is less than or equal to the invoice open amount. Otherwise, false
     */
    private static boolean increaseInvoiceIsValid(InvoicePaidApplied invoicePaidApplied) {
        final var amountIncreased = invoicePaidApplied.getInvoiceItemAppliedAmount()
                .subtract(invoicePaidApplied.getInvoiceDetail().getAmountApplied());
        return amountIncreased.isLessEqual(invoicePaidApplied.getInvoiceDetail().getAmountOpen());
    }
}
