/*
 * The Kuali Financial System, a comprehensive financial management system for higher education.
 *
 * Copyright 2005-2022 Kuali, Inc.
 *
 * This program is free software: you can redistribute it and/or modify
 * it under the terms of the GNU Affero General Public License as published by
 * the Free Software Foundation, either version 3 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU Affero General Public License for more details.
 *
 * You should have received a copy of the GNU Affero General Public License
 * along with this program.  If not, see <http://www.gnu.org/licenses/>.
 */
package org.kuali.kfs.module.ar.rest.resource.requests;

import com.fasterxml.jackson.annotation.JsonCreator;
import com.fasterxml.jackson.annotation.JsonIgnoreProperties;
import com.fasterxml.jackson.annotation.JsonProperty;
import org.kuali.kfs.core.api.util.type.KualiDecimal;
import org.apache.commons.collections4.CollectionUtils;
import org.apache.commons.lang3.StringUtils;
import org.apache.commons.lang3.Validate;

import java.util.List;
import java.util.Objects;
import java.util.StringJoiner;

@JsonIgnoreProperties(ignoreUnknown = true)
public class PaymentApplicationAdjustmentRequest extends DocumentRequest {
    private final List<InvoiceApplication> invoiceApplications;
    private final List<NonAppliedHolding> nonAppliedHoldings;
    private final List<AccountingLine> nonArAccountingLines;

    @JsonCreator
    public PaymentApplicationAdjustmentRequest(
            @JsonProperty("description") final String description,
            @JsonProperty("explanation") final String explanation,
            @JsonProperty("orgDocNumber") final String orgDocNumber,
            @JsonProperty("adHocRouting") final AdHocRoutingRequest adHocRoutingRequest,
            @JsonProperty("invoiceApplications") final List<InvoiceApplication> invoiceApplications,
            @JsonProperty("nonAppliedHoldings") final List<NonAppliedHolding> nonAppliedHoldings,
            @JsonProperty("accountingLines") final List<AccountingLine> nonArAccountingLines
    ) {
        super(description, explanation, orgDocNumber, adHocRoutingRequest);
        Validate.isTrue(invoiceApplications != null, "invoiceApplications must be supplied");
        Validate.isTrue(nonAppliedHoldings != null, "nonAppliedHoldings must be supplied");
        Validate.isTrue(nonArAccountingLines != null, "accountingLines must be supplied");
        this.invoiceApplications = invoiceApplications;
        this.nonAppliedHoldings = nonAppliedHoldings;
        this.nonArAccountingLines = nonArAccountingLines;
    }

    public List<NonAppliedHolding> getNonAppliedHoldings() {
        return nonAppliedHoldings;
    }

    public List<InvoiceApplication> getInvoiceApplications() {
        return invoiceApplications;
    }

    public List<AccountingLine> getNonArAccountingLines() {
        return nonArAccountingLines;
    }

    @Override
    public boolean equals(final Object o) {
        if (this == o) {
            return true;
        }
        if (o == null || getClass() != o.getClass()) {
            return false;
        }
        final PaymentApplicationAdjustmentRequest that = (PaymentApplicationAdjustmentRequest) o;
        return invoiceApplications.equals(that.invoiceApplications) &&
                nonAppliedHoldings.equals(that.nonAppliedHoldings) &&
                nonArAccountingLines.equals(that.nonArAccountingLines);
    }

    @Override
    public int hashCode() {
        return Objects.hash(invoiceApplications, nonAppliedHoldings, nonArAccountingLines);
    }

    @Override
    public String toString() {
        return new StringJoiner(", ", PaymentApplicationAdjustmentRequest.class.getSimpleName() + "[", "]")
                .add("invoiceApplications=" + invoiceApplications)
                .add("nonAppliedHoldings=" + nonAppliedHoldings)
                .add("nonArAccountingLines=" + nonArAccountingLines)
                .toString();
    }

    @JsonIgnoreProperties(ignoreUnknown = true)
    public static class InvoiceApplication {
        private final String customerNumber;
        private final String documentNumber;
        private final List<InvoiceApplicationDetail> detailApplications;

        @JsonCreator
        public InvoiceApplication(
                @JsonProperty("customerNumber") final String customerNumber,
                @JsonProperty("documentNumber") final String documentNumber,
                @JsonProperty("detailApplications") final List<InvoiceApplicationDetail> detailApplications
        ) {
            Validate.isTrue(StringUtils.isNotBlank(customerNumber), "customerNumber must be supplied");
            Validate.isTrue(StringUtils.isNotBlank(documentNumber), "documentNumber must be supplied");
            Validate.isTrue(CollectionUtils.isNotEmpty(detailApplications), "detailApplications must be supplied");
            this.customerNumber = customerNumber;
            this.documentNumber = documentNumber;
            this.detailApplications = detailApplications;
        }

        public String getCustomerNumber() {
            return customerNumber;
        }

        public String getDocumentNumber() {
            return documentNumber;
        }

        public List<InvoiceApplicationDetail> getDetailApplications() {
            return detailApplications;
        }

        @Override
        public boolean equals(final Object o) {
            if (this == o) {
                return true;
            }
            if (o == null || getClass() != o.getClass()) {
                return false;
            }
            final InvoiceApplication that = (InvoiceApplication) o;
            return customerNumber.equals(that.customerNumber) &&
                    documentNumber.equals(that.documentNumber) &&
                    detailApplications.equals(that.detailApplications);
        }

        @Override
        public int hashCode() {
            return Objects.hash(customerNumber, documentNumber, detailApplications);
        }

        @Override
        public String toString() {
            return new StringJoiner(", ",
                    InvoiceApplication.class.getSimpleName() + "[", "]")
                    .add("customerNumber='" + customerNumber + "'")
                    .add("detailApplications=" + detailApplications)
                    .add("documentNumber='" + documentNumber + "'")
                    .toString();
        }

    }

    @JsonIgnoreProperties(ignoreUnknown = true)
    public static class InvoiceApplicationDetail {
        private final KualiDecimal amountApplied;
        private final Integer sequenceNumber;

        @JsonCreator
        public InvoiceApplicationDetail(
                @JsonProperty("amountApplied") final KualiDecimal amountApplied,
                @JsonProperty("sequenceNumber") final Integer sequenceNumber
        ) {
            Validate.isTrue(amountApplied != null, "amountApplied must be supplied");
            Validate.isTrue(sequenceNumber != null, "sequenceNumber must be supplied");
            Validate.isTrue(sequenceNumber >= 0, "sequenceNumber must not be negative");
            this.amountApplied = amountApplied;
            this.sequenceNumber = sequenceNumber;
        }

        public KualiDecimal getAmountApplied() {
            return amountApplied;
        }

        public Integer getSequenceNumber() {
            return sequenceNumber;
        }

        @Override
        public boolean equals(final Object o) {
            if (this == o) {
                return true;
            }
            if (o == null || getClass() != o.getClass()) {
                return false;
            }
            final InvoiceApplicationDetail that =
                    (InvoiceApplicationDetail) o;
            return amountApplied.equals(that.amountApplied) &&
                    sequenceNumber.equals(that.sequenceNumber);
        }

        @Override
        public int hashCode() {
            return Objects.hash(amountApplied, sequenceNumber);
        }

        @Override
        public String toString() {
            return new StringJoiner(", ",
                    InvoiceApplicationDetail.class.getSimpleName() + "[", "]")
                    .add("amountApplied=" + amountApplied)
                    .add("sequenceNumber=" + sequenceNumber)
                    .toString();
        }

    }

    @JsonIgnoreProperties(ignoreUnknown = true)
    public static class NonAppliedHolding {
        private final KualiDecimal amount;
        private final String customerNumber;

        @JsonCreator
        public NonAppliedHolding(
                @JsonProperty("amount") final KualiDecimal amount,
                @JsonProperty("customerNumber") final String customerNumber
        ) {
            Validate.isTrue(amount != null, "amount must be supplied");
            Validate.isTrue(StringUtils.isNotBlank(customerNumber), "customerNumber must be supplied");
            this.amount = amount;
            this.customerNumber = customerNumber;
        }

        public KualiDecimal getAmount() {
            return amount;
        }

        public String getCustomerNumber() {
            return customerNumber;
        }

        @Override
        public boolean equals(final Object o) {
            if (this == o) {
                return true;
            }
            if (o == null || getClass() != o.getClass()) {
                return false;
            }
            final NonAppliedHolding that = (NonAppliedHolding) o;
            return amount.equals(that.amount) &&
                    customerNumber.equals(that.customerNumber);
        }

        @Override
        public int hashCode() {
            return Objects.hash(amount, customerNumber);
        }

        @Override
        public String toString() {
            return new StringJoiner(", ", NonAppliedHolding.class.getSimpleName() + "[", "]")
                    .add("amount=" + amount)
                    .add("customerNumber='" + customerNumber + "'")
                    .toString();
        }

    }

    @JsonIgnoreProperties(ignoreUnknown = true)
    public static class AccountingLine {
        private final Integer sequenceNumber;
        private final String chartCode;
        private final String accountNumber;
        private final String subAccountNumber;
        private final String object;
        private final String subObject;
        private final String projectCode;
        private final String orgRefId;
        private final String description;
        private final KualiDecimal amount;

        @JsonCreator
        public AccountingLine(
                @JsonProperty("sequenceNumber") final Integer sequenceNumber,
                @JsonProperty("chartOfAccountsCode") final String chartCode,
                @JsonProperty("accountNumber") final String accountNumber,
                @JsonProperty("subAccountNumber") final String subAccountNumber,
                @JsonProperty("financialObjectCode") final String object,
                @JsonProperty("financialSubObjectCode") final String subObject,
                @JsonProperty("projectCode") final String projectCode,
                @JsonProperty("organizationReferenceId") final String orgRefId,
                @JsonProperty("description") final String description,
                @JsonProperty("amount") final KualiDecimal amount
        ) {
            Validate.isTrue(StringUtils.isNotBlank(chartCode), "chartOfAccountsCode must be supplied");
            Validate.isTrue(StringUtils.isNotBlank(accountNumber), "accountNumber must be supplied");
            Validate.isTrue(StringUtils.isNotBlank(object), "financialObjectCode must be supplied");
            Validate.isTrue(amount != null, "amount must be supplied");

            this.sequenceNumber = sequenceNumber;
            this.chartCode = chartCode;
            this.accountNumber = accountNumber;
            this.subAccountNumber = subAccountNumber;
            this.object = object;
            this.subObject = subObject;
            this.projectCode = projectCode;
            this.orgRefId = orgRefId;
            this.description = description;
            this.amount = amount;
        }

        public Integer getSequenceNumber() {
            return sequenceNumber;
        }

        public String getChartCode() {
            return chartCode;
        }

        public String getAccountNumber() {
            return accountNumber;
        }

        public String getSubAccountNumber() {
            return subAccountNumber;
        }

        public String getObject() {
            return object;
        }

        public String getSubObject() {
            return subObject;
        }

        public String getProjectCode() {
            return projectCode;
        }

        public String getOrgRefId() {
            return orgRefId;
        }

        public String getDescription() {
            return description;
        }

        public KualiDecimal getAmount() {
            return amount;
        }

        @Override
        public boolean equals(final Object o) {
            if (this == o) {
                return true;
            }
            if (!(o instanceof AccountingLine)) {
                return false;
            }
            final AccountingLine that = (AccountingLine) o;
            return Objects.equals(sequenceNumber, that.sequenceNumber) &&
                    chartCode.equals(that.chartCode) &&
                    accountNumber.equals(that.accountNumber) &&
                    Objects.equals(subAccountNumber, that.subAccountNumber) &&
                    object.equals(that.object) &&
                    Objects.equals(subObject, that.subObject) &&
                    Objects.equals(projectCode, that.projectCode) &&
                    Objects.equals(orgRefId, that.orgRefId) &&
                    Objects.equals(description, that.description) &&
                    amount.equals(that.amount);
        }

        @Override
        public int hashCode() {
            return Objects.hash(
                    sequenceNumber,
                    chartCode,
                    accountNumber,
                    subAccountNumber,
                    object,
                    subObject,
                    projectCode,
                    orgRefId,
                    description,
                    amount
            );
        }

        @Override
        public String toString() {
            return new StringJoiner(", ", AccountingLine.class.getSimpleName() + "[", "]")
                    .add("sequenceNumber='" + sequenceNumber + "'")
                    .add("accountNumber='" + accountNumber + "'")
                    .add("amount=" + amount)
                    .add("chartOfAccountsCode='" + chartCode + "'")
                    .add("description='" + description + "'")
                    .add("financialObjectCode='" + object + "'")
                    .add("organizationReferenceId='" + orgRefId + "'")
                    .add("projectCode='" + projectCode + "'")
                    .add("subAccountNumber='" + subAccountNumber + "'")
                    .add("financialSubObjectCode='" + subObject + "'")
                    .toString();
        }

    }
}

