/*
 * The Kuali Financial System, a comprehensive financial management system for higher education.
 *
 * Copyright 2005-2022 Kuali, Inc.
 *
 * This program is free software: you can redistribute it and/or modify
 * it under the terms of the GNU Affero General Public License as published by
 * the Free Software Foundation, either version 3 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU Affero General Public License for more details.
 *
 * You should have received a copy of the GNU Affero General Public License
 * along with this program.  If not, see <http://www.gnu.org/licenses/>.
 */
package org.kuali.kfs.module.ar.rest.resource.responses;

import org.apache.commons.lang3.StringUtils;
import org.apache.commons.lang3.Validate;
import org.kuali.kfs.coa.businessobject.Account;
import org.kuali.kfs.coa.service.AccountService;
import org.kuali.kfs.core.api.util.type.KualiDecimal;
import org.kuali.kfs.module.ar.businessobject.CustomerInvoiceDetail;
import org.kuali.kfs.module.ar.businessobject.InvoicePaidApplied;
import org.kuali.kfs.module.ar.document.CustomerInvoiceDocument;
import org.kuali.kfs.sys.context.SpringContext;
import org.kuali.kfs.sys.rest.presentation.Link;

import java.util.Date;
import java.util.List;
import java.util.Optional;
import java.util.stream.Collectors;

public class PaymentApplicationAdjustmentInvoiceResponse {
    private final String paymentApplicationAdjustmentDocumentNumber;
    private final List<InvoiceDetailResponse> detailApplications;
    private final CustomerInvoiceDocument invoice;
    private final String documentUrl;
    private final String customerUrl;
    private AccountService accountService;

    public PaymentApplicationAdjustmentInvoiceResponse(
            final String paymentApplicationAdjustmentDocumentNumber,
            final CustomerInvoiceDocument invoice,
            final List<InvoicePaidApplied> invoicePaidApplieds,
            final String documentUrl,
            final String customerUrl
    ) {
        Validate.isTrue(StringUtils.isNotBlank(paymentApplicationAdjustmentDocumentNumber), "Document number must be provided");
        Validate.isTrue(invoice != null, "Invoice cannot be null");
        Validate.isTrue(invoicePaidApplieds != null, "Invoice Paid Applieds cannot be null");
        Validate.isTrue(StringUtils.isNotBlank(documentUrl), "Document url is required");
        Validate.isTrue(StringUtils.isNotBlank(customerUrl), "Customer url is required");
        this.paymentApplicationAdjustmentDocumentNumber = paymentApplicationAdjustmentDocumentNumber;
        this.invoice = invoice;
        this.detailApplications = generateDetailResponses(invoice, invoicePaidApplieds);
        this.customerUrl = customerUrl;
        this.documentUrl = documentUrl;
    }

    public List<InvoiceDetailResponse> getDetailApplications() {
        return detailApplications;
    }

    public String getDocumentNumber() {
        return invoice.getDocumentNumber();
    }

    public String getCustomerNumber() {
        return invoice.getCustomer().getCustomerNumber();
    }

    public Date getInvoiceBillingDate() {
        return invoice.getBillingDate();
    }

    public String getHeaderText() {
        return invoice.getInvoiceHeaderText();
    }

    public String getCustomerName() {
        return invoice.getCustomer().getCustomerName();
    }

    public KualiDecimal getTotalAmount() {
        return invoice.getTotalDollarAmount();
    }

    public KualiDecimal getOpenAmount() {
        return getTotalAmount().subtract(getAmountToApply());
    }

    public KualiDecimal getAmountToApply() {
        return detailApplications.stream()
            .map(InvoiceDetailResponse::getAmountApplied)
            .reduce(KualiDecimal.ZERO, (total, nextAmount) -> total.add(nextAmount));
    }

    public Link getCustomerLink() {
        return new Link(getCustomerNumber(), customerUrl);
    }

    public Link getDocumentLink() {
        return new Link(getDocumentNumber(), documentUrl);
    }

    private List<InvoiceDetailResponse> generateDetailResponses(
            final CustomerInvoiceDocument invoice,
            final List<InvoicePaidApplied> invoicePaidApplieds
    ) {
        return invoice.getCustomerInvoiceDetailsWithoutDiscounts().stream()
                .map(detail -> {
                    Optional<InvoicePaidApplied> matchingInvoicePaidApplied =
                            findInvoicePaidAppliedForDetail(invoicePaidApplieds, detail);
                    return new InvoiceDetailResponse(paymentApplicationAdjustmentDocumentNumber, detail, matchingInvoicePaidApplied);
                })
                .collect(Collectors.toList());
    }

    private Optional<InvoicePaidApplied> findInvoicePaidAppliedForDetail(
            final List<InvoicePaidApplied> invoicePaidApplieds,
            final CustomerInvoiceDetail detail
    ) {
        return invoicePaidApplieds.stream()
                .filter(invoicePaidApplied -> invoicePaidApplied.getInvoiceItemNumber().equals(detail.getInvoiceItemNumber()))
                .findFirst();
    }

    private AccountService getAccountService() {
        if (accountService == null) {
            accountService = SpringContext.getBean(AccountService.class);
        }
        return accountService;
    }

    // To allow testing
    public void setAccountService(AccountService accountService) {
        this.accountService = accountService;
    }

    private class InvoiceDetailResponse {
        private final String paymentApplicationAdjustmentDocumentNumber;
        private final CustomerInvoiceDetail invoiceDetail;
        private final Optional<InvoicePaidApplied> invoicePaidApplied;

        InvoiceDetailResponse(
                final String paymentApplicationAdjustmentDocumentNumber,
                final CustomerInvoiceDetail invoiceDetail,
                final Optional<InvoicePaidApplied> invoicePaidApplied
        ) {
            Validate.isTrue(StringUtils.isNotBlank(paymentApplicationAdjustmentDocumentNumber), "Document number must be provided");
            Validate.isTrue(invoiceDetail != null, "Invoice detail cannot be null");
            this.paymentApplicationAdjustmentDocumentNumber = paymentApplicationAdjustmentDocumentNumber;
            this.invoiceDetail = invoiceDetail;
            this.invoicePaidApplied = invoicePaidApplied;
        }

        public String getChartOfAccountsCode() {
            return invoiceDetail.getChartOfAccountsCode();
        }

        public String getAccountNumber() {
            return invoiceDetail.getAccountNumber();
        }

        public boolean getIsAccountClosed() {
            final Account account = getAccountService().getByPrimaryId(invoiceDetail.getChartOfAccountsCode(), invoiceDetail.getAccountNumber());
            return account.isClosed();
        }

        public Integer getSequenceNumber() {
            return invoiceDetail.getInvoiceItemNumber();
        }

        public String getInvoiceItemDescription() {
            return invoiceDetail.getInvoiceItemDescription();
        }

        public KualiDecimal getAmount() {
            return invoiceDetail.getAmount();
        }

        public KualiDecimal getAmountApplied() {
            if (invoicePaidApplied.isPresent()) {
                return invoicePaidApplied.get().getInvoiceItemAppliedAmount();
            }
            return KualiDecimal.ZERO;
        }

        public KualiDecimal getAmountOpen() {
            // invoiceItemOpenAmount should only be set on documents that have been processed to final, until then
            // we want to defer to the invoice on how much is open on the invoice
            if (invoicePaidApplied.isPresent() && invoicePaidApplied.get().getInvoiceItemOpenAmount() != null) {
                return invoicePaidApplied.get().getInvoiceItemOpenAmount();
            }
            return getAmount().subtract(getAmountApplied());
        }
    }
}
