/**
 * The Kuali Financial System, a comprehensive financial management system for higher education.
 *
 * Copyright 2005-2018 Kuali, Inc.
 *
 * This program is free software: you can redistribute it and/or modify
 * it under the terms of the GNU Affero General Public License as published by
 * the Free Software Foundation, either version 3 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU Affero General Public License for more details.
 *
 * You should have received a copy of the GNU Affero General Public License
 * along with this program.  If not, see <http://www.gnu.org/licenses/>.
 */
package org.kuali.kfs.module.cam.document.validation.impl;

import org.apache.commons.lang3.StringUtils;
import org.kuali.kfs.coreservice.framework.parameter.ParameterService;
import org.kuali.kfs.kns.rules.PromptBeforeValidationBase;
import org.kuali.kfs.krad.document.Document;
import org.kuali.kfs.krad.service.KRADServiceLocatorWeb;
import org.kuali.kfs.krad.util.ObjectUtils;
import org.kuali.kfs.module.cam.CamsConstants;
import org.kuali.kfs.module.cam.CamsKeyConstants;
import org.kuali.kfs.module.cam.CamsPropertyConstants;
import org.kuali.kfs.module.cam.businessobject.Asset;
import org.kuali.kfs.module.cam.businessobject.AssetGlobal;
import org.kuali.kfs.module.cam.businessobject.AssetPayment;
import org.kuali.kfs.module.cam.businessobject.AssetPaymentAssetDetail;
import org.kuali.kfs.module.cam.businessobject.AssetPaymentDetail;
import org.kuali.kfs.module.cam.businessobject.AssetPaymentInProcessPayment;
import org.kuali.kfs.module.cam.document.AssetPaymentDocument;
import org.kuali.kfs.module.cam.document.service.AssetService;
import org.kuali.kfs.sys.KFSConstants;
import org.kuali.kfs.sys.context.SpringContext;
import org.kuali.rice.core.api.config.property.ConfigurationService;
import org.kuali.rice.core.api.util.type.KualiDecimal;

import java.util.ArrayList;
import java.util.Arrays;
import java.util.List;
import java.util.Map;

/**
 * This prerule is ex..
 */
public class AssetPaymentDocumentPreRules extends PromptBeforeValidationBase {
    protected static final org.apache.log4j.Logger LOG = org.apache.log4j.Logger.getLogger(AssetPaymentDocumentPreRules.class);

    @Override
    public boolean doPrompts(Document document) {
        boolean preRulesOK = true;
        AssetPaymentDocument assetPaymentDocument = (AssetPaymentDocument) document;

        if (hasDifferentObjectSubTypes(assetPaymentDocument)) {
            if (!isOkHavingDifferentObjectSubTypes()) {
                event.setActionForwardName(KFSConstants.MAPPING_BASIC);
                preRulesOK &= false;
            }
        }

        if (hasNoPaymentsSelectedWhenCanMakeSelection(assetPaymentDocument.getAssetPaymentAssetDetail(), assetPaymentDocument.getSourceAccountingLines(), assetPaymentDocument.getAssetPaymentDistributor().getAssetPaymentDistributions(), assetPaymentDocument.getAssetPaymentInProcessPayments())) {
            if (isOkNoPaymentsSelectedWhenCanMakeSelection()) {
                event.setActionForwardName(KFSConstants.MAPPING_BASIC);
                preRulesOK &= false;
            }
        }

        return preRulesOK;
    }

    /**
     * This method determines whether or not an asset has different object sub type codes in its documents.
     *
     * @return true when the asset has payments with object codes that point to different object sub type codes
     */
    public boolean hasDifferentObjectSubTypes(AssetPaymentDocument document) {
        //This method will only execute if the document is being submitted.
        if (!(document.getDocumentHeader().getWorkflowDocument().isSaved() || document.getDocumentHeader().getWorkflowDocument().isInitiated())) {
            return false;
        }

        List<String> subTypes = new ArrayList<String>();
        subTypes = new ArrayList<String>(SpringContext.getBean(ParameterService.class).getParameterValuesAsString(Asset.class, CamsConstants.Parameters.OBJECT_SUB_TYPE_GROUPS));

        List<AssetPaymentDetail> assetPaymentDetails = document.getSourceAccountingLines();
        List<String> validObjectSubTypes = new ArrayList<String>();

        String objectSubTypeCode = null;

        /*
         * Expected system parameter elements (object sub types). [BD,BF] [CM,CF,CO] [UC,UF,UO] [LI,LF]
         */

        // Getting the list of object sub type codes from the asset payments on the jsp.
        List<String> objectSubTypeList = new ArrayList<String>();
        for (AssetPaymentDetail assetPaymentDetail : assetPaymentDetails) {
            assetPaymentDetail.refreshReferenceObject(CamsPropertyConstants.AssetPaymentDetail.OBJECT_CODE);
            if (ObjectUtils.isNull(assetPaymentDetail.getObjectCode())) {
                return false;
            }
            objectSubTypeList.add(assetPaymentDetail.getObjectCode().getFinancialObjectSubTypeCode());
        }

        if (!getAssetService().isObjectSubTypeCompatible(objectSubTypeList)) {
            return true;
        }

        List<AssetPaymentAssetDetail> assetPaymentAssetDetails = document.getAssetPaymentAssetDetail();
        for (AssetPaymentAssetDetail assetPaymentAssetDetail : assetPaymentAssetDetails) {
            assetPaymentAssetDetail.getAsset().refreshReferenceObject(CamsPropertyConstants.Asset.ASSET_PAYMENTS);
            List<AssetPayment> assetPayments = assetPaymentAssetDetail.getAsset().getAssetPayments();

            // Comparing against the already approved asset payments
            if (!assetPayments.isEmpty()) {
                for (AssetPayment assetPayment : assetPayments) {
                    String paymentSubObjectType = assetPayment.getFinancialObject().getFinancialObjectSubTypeCode();

                    validObjectSubTypes = new ArrayList<String>();
                    for (String subType : subTypes) {
                        validObjectSubTypes = Arrays.asList(StringUtils.split(subType, ","));
                        if (validObjectSubTypes.contains(paymentSubObjectType)) {
                            break;
                        }
                        validObjectSubTypes = new ArrayList<String>();
                    }
                    if (validObjectSubTypes.isEmpty())
                        validObjectSubTypes.add(paymentSubObjectType);

                    // Comparing the same asset payment document
                    for (AssetPaymentDetail assetPaymentDetail : assetPaymentDetails) {
                        if (!validObjectSubTypes.contains(assetPaymentDetail.getObjectCode().getFinancialObjectSubTypeCode())) {
                            // Differences where found.
                            return true;
                        }
                    }
                }
            }
        }
        // If none object sub types are different...
        return false;
    }

    protected boolean isOkHavingDifferentObjectSubTypes() {
        String parameterDetail = "(module:" + KRADServiceLocatorWeb.getKualiModuleService().getNamespaceCode(AssetGlobal.class) + "/component:" + AssetGlobal.class.getSimpleName() + ")";

        ConfigurationService kualiConfiguration = SpringContext.getBean(ConfigurationService.class);

        String continueQuestion = kualiConfiguration.getPropertyValueAsString(CamsKeyConstants.CONTINUE_QUESTION);
        String warningMessage = kualiConfiguration.getPropertyValueAsString(CamsKeyConstants.Payment.WARNING_NOT_SAME_OBJECT_SUB_TYPES) + " " + CamsConstants.Parameters.OBJECT_SUB_TYPE_GROUPS + " " + parameterDetail + ". " + continueQuestion;
        return super.askOrAnalyzeYesNoQuestion(CamsConstants.AssetPayment.ASSET_PAYMENT_DIFFERENT_OBJECT_SUB_TYPE_CONFIRMATION_QUESTION, warningMessage);
    }

    /**
     * This models the same logical structure as from viewPaymentInProcessByAsset.tag. We are trying to determine if the
     * user did not make a selection if they could have made one for an in process payment.
     * @param assetPaymentAssetDetails are the asset selections
     * @param assetPaymentDetails the source accounting lines
     * @param assetPaymentDistribution the calculated distribution. We need the amount from this
     * @param assetPaymentInProcessPayments the payment selections the user made on the distribution
     * @return an elegible selection or not
     */
    protected boolean hasNoPaymentsSelectedWhenCanMakeSelection(List<AssetPaymentAssetDetail> assetPaymentAssetDetails, List<AssetPaymentDetail> assetPaymentDetails, Map<String, Map<AssetPaymentAssetDetail, KualiDecimal>> assetPaymentDistribution, List<AssetPaymentInProcessPayment> assetPaymentInProcessPayments) {
        // Loop over all selected assets
        for (AssetPaymentAssetDetail assetDetail : assetPaymentAssetDetails) {
            // And all selected source accounting lines
            for (AssetPaymentDetail payment : assetPaymentDetails) {
                boolean canMakePaymentSelection = false;
                boolean madePaymentSelection = false;

                // Determine if the user made a selection
                for (AssetPaymentInProcessPayment assetPaymentInProcessPayment : assetPaymentInProcessPayments) {
                    if (assetPaymentInProcessPayment.getCapitalAssetNumber().equals(assetDetail.getCapitalAssetNumber()) && assetPaymentInProcessPayment.getSequenceNumber().equals(payment.getSequenceNumber())) {
                        madePaymentSelection = true;
                        break;
                    }
                }

                // Determine if the user can make a selection
                for (AssetPayment assetPayment : assetDetail.getAsset().getAssetPayments()) {
                    if (assetPayment.getChartOfAccountsCode().equals(payment.getChartOfAccountsCode()) &&
                        assetPayment.getAccountNumber().equals(payment.getAccountNumber()) &&
                        assetPayment.getFinancialObjectCode().equals(payment.getFinancialObjectCode()) &&
                        assetPayment.getTransferPaymentCode().equals(CamsConstants.AssetPayment.TRANSFER_PAYMENT_CODE_N)) {

                        canMakePaymentSelection = true;
                        break;
                    }
                }

                if (canMakePaymentSelection && !madePaymentSelection) {
                    // Present the question page
                    return true;
                }
            }
        }

        return false;
    }

    protected boolean isOkNoPaymentsSelectedWhenCanMakeSelection() {
        ConfigurationService kualiConfiguration = SpringContext.getBean(ConfigurationService.class);

        String question = kualiConfiguration.getPropertyValueAsString(CamsKeyConstants.Payment.QUESTION_NO_PAYMENTS_SELECTED_WHEN_CAN_MAKE_SELECTION);
        return super.askOrAnalyzeYesNoQuestion(CamsConstants.AssetPayment.ASSET_PAYMENT_NO_PAYMENTS_SELECTED_WHEN_CAN_MAKE_SELECTION_CONFIRMATION_QUESTION, question);
    }

    protected AssetService getAssetService() {
        return SpringContext.getBean(AssetService.class);
    }

    protected ParameterService getParameterService() {
        return SpringContext.getBean(ParameterService.class);
    }
}
