/**
 * The Kuali Financial System, a comprehensive financial management system for higher education.
 *
 * Copyright 2005-2018 Kuali, Inc.
 *
 * This program is free software: you can redistribute it and/or modify
 * it under the terms of the GNU Affero General Public License as published by
 * the Free Software Foundation, either version 3 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU Affero General Public License for more details.
 *
 * You should have received a copy of the GNU Affero General Public License
 * along with this program.  If not, see <http://www.gnu.org/licenses/>.
 */
package org.kuali.kfs.module.cam.document;

import org.apache.commons.lang3.StringUtils;
import org.apache.log4j.Logger;
import org.kuali.kfs.coreservice.framework.parameter.ParameterService;
import org.kuali.kfs.integration.cam.CapitalAssetManagementModuleService;
import org.kuali.kfs.krad.document.Copyable;
import org.kuali.kfs.krad.exception.ValidationException;
import org.kuali.kfs.krad.rules.rule.event.KualiDocumentEvent;
import org.kuali.kfs.krad.rules.rule.event.SaveDocumentEvent;
import org.kuali.kfs.krad.util.ObjectUtils;
import org.kuali.kfs.module.cam.CamsConstants;
import org.kuali.kfs.module.cam.CamsPropertyConstants;
import org.kuali.kfs.module.cam.batch.AssetDepreciationStep;
import org.kuali.kfs.module.cam.businessobject.Asset;
import org.kuali.kfs.module.cam.businessobject.AssetObjectCode;
import org.kuali.kfs.module.cam.businessobject.AssetPayment;
import org.kuali.kfs.module.cam.businessobject.AssetPaymentAllocationType;
import org.kuali.kfs.module.cam.businessobject.AssetPaymentAssetDetail;
import org.kuali.kfs.module.cam.businessobject.AssetPaymentDetail;
import org.kuali.kfs.module.cam.businessobject.AssetPaymentInProcessPayment;
import org.kuali.kfs.module.cam.document.service.AssetObjectCodeService;
import org.kuali.kfs.module.cam.document.service.AssetPaymentService;
import org.kuali.kfs.module.cam.document.validation.event.AssetPaymentManuallyAddAccountingLineEvent;
import org.kuali.kfs.module.cam.util.distribution.AssetDistribution;
import org.kuali.kfs.module.cam.util.distribution.AssetDistributionEvenly;
import org.kuali.kfs.module.cam.util.distribution.AssetDistributionManual;
import org.kuali.kfs.module.cam.util.distribution.AssetDistributionPercent;
import org.kuali.kfs.module.cam.util.distribution.AssetPaymentDistributionByTotalCost;
import org.kuali.kfs.sys.KFSConstants;
import org.kuali.kfs.sys.businessobject.AccountingLine;
import org.kuali.kfs.sys.businessobject.GeneralLedgerPendingEntry;
import org.kuali.kfs.sys.businessobject.GeneralLedgerPendingEntrySequenceHelper;
import org.kuali.kfs.sys.businessobject.GeneralLedgerPendingEntrySourceDetail;
import org.kuali.kfs.sys.context.SpringContext;
import org.kuali.kfs.sys.document.AccountingDocumentBase;
import org.kuali.kfs.sys.document.AmountTotaling;
import org.kuali.kfs.sys.document.validation.impl.AccountingDocumentRuleBaseConstants;
import org.kuali.rice.core.api.util.type.KualiDecimal;
import org.kuali.rice.kew.api.WorkflowDocument;
import org.kuali.rice.kew.framework.postprocessor.DocumentRouteStatusChange;

import java.util.ArrayList;
import java.util.Calendar;
import java.util.Collection;
import java.util.Iterator;
import java.util.List;

/**
 * Capital assets document class for the asset payment document
 */
public class AssetPaymentDocument extends AccountingDocumentBase implements Copyable, AmountTotaling {
    protected static Logger LOG = Logger.getLogger(AssetPaymentDocument.class);

    protected List<AssetPaymentAssetDetail> assetPaymentAssetDetail;
    protected Long capitalAssetNumber;
    protected boolean capitalAssetBuilderOriginIndicator;
    protected AssetPaymentAllocationType assetPaymentAllocationType;
    protected String assetPaymentAllocationTypeCode;
    protected boolean allocationFromFPDocuments;
    protected List<AssetPaymentInProcessPayment> assetPaymentInProcessPayments;

    // helper transients for GLPE generation. Used to avoid having to modify the method arguments in super class
    protected transient AssetPayment apipAssetPaymentHelper;
    protected transient KualiDecimal apipAssetPaymentTotalDepreciationHelper;

    public AssetPaymentDocument() {
        super();
        this.setAllocationFromFPDocuments(false);
        assetPaymentAllocationTypeCode = CamsPropertyConstants.AssetPaymentAllocation.ASSET_DISTRIBUTION_DEFAULT_CODE;
        this.setAssetPaymentAssetDetail(new ArrayList<AssetPaymentAssetDetail>());
        this.setAssetPaymentInProcessPayments(new ArrayList<AssetPaymentInProcessPayment>());
    }

    /**
     * Remove asset from collection for deletion
     *
     * @see org.kuali.kfs.sys.document.AccountingDocumentBase#buildListOfDeletionAwareLists()
     */
    @Override
    public List buildListOfDeletionAwareLists() {
        List<List> deletionAwareList = super.buildListOfDeletionAwareLists();
        deletionAwareList.add(this.getAssetPaymentAssetDetail());
        deletionAwareList.add(this.getAssetPaymentInProcessPayments());
        return deletionAwareList;
    }

    /**
     * When document save, AddAccountingLineEvent is added by the framework.
     * Also, we need to add AssetPaymentManuallyAddAccountingLineEvent manually
     * to run all relating validations.
     *
     * @see org.kuali.kfs.sys.document.AccountingDocumentBase#generateSaveEvents()
     */
    @Override
    public List generateSaveEvents() {
        List subEvents = new ArrayList();
        // keep the order of events as for validation will run in the same
        // order.
        if (!isCapitalAssetBuilderOriginIndicator()) {
            // Add AssetPaymentManuallyAddAccountingLineEvent for each manually
            // added accounting line.
            String errorPathPrefix = KFSConstants.DOCUMENT_PROPERTY_NAME + "." + KFSConstants.EXISTING_SOURCE_ACCT_LINE_PROPERTY_NAME;
            int index = 0;
            for (Iterator i = getSourceAccountingLines().iterator(); i.hasNext(); index++) {
                String indexedErrorPathPrefix = errorPathPrefix + "[" + index + "]";
                AccountingLine currentLine = (AccountingLine) i.next();
                AssetPaymentManuallyAddAccountingLineEvent newSubEvent = new AssetPaymentManuallyAddAccountingLineEvent(indexedErrorPathPrefix, this, currentLine);
                subEvents.add(newSubEvent);
            }
        }

        subEvents.addAll(super.generateSaveEvents());

        return subEvents;
    }

    /**
     * Lock on purchase order document since post processor will update PO
     * document by adding notes.
     *
     * @see org.kuali.rice.krad.document.DocumentBase#getWorkflowEngineDocumentIdsToLock()
     */
    @Override
    public List<String> getWorkflowEngineDocumentIdsToLock() {
        List<String> documentIds = null;
        if (this.isCapitalAssetBuilderOriginIndicator()) {
            String poDocId = SpringContext.getBean(CapitalAssetManagementModuleService.class).getCurrentPurchaseOrderDocumentNumber(this.getDocumentNumber());
            if (StringUtils.isNotBlank(poDocId)) {
                documentIds = new ArrayList<String>();
                documentIds.add(poDocId);
            }
        }
        return documentIds;
    }

    /**
     * Determines if the given AccountingLine (as a GeneralLedgerPostable) is a
     * credit or a debit, in terms of GLPE generation
     */
    @Override
    public boolean isDebit(GeneralLedgerPendingEntrySourceDetail postable) {
        return false;
    }

    public boolean isCapitalAssetBuilderOriginIndicator() {
        return capitalAssetBuilderOriginIndicator;
    }

    public void setCapitalAssetBuilderOriginIndicator(boolean capitalAssetBuilderOriginIndicator) {
        this.capitalAssetBuilderOriginIndicator = capitalAssetBuilderOriginIndicator;
    }

    /**
     * This method...
     *
     * @param assetPaymentAssetDetail
     */
    public void addAssetPaymentAssetDetail(AssetPaymentAssetDetail assetPaymentAssetDetail) {
        this.getAssetPaymentAssetDetail().add(assetPaymentAssetDetail);
    }

    @Override
    public void postProcessSave(KualiDocumentEvent event) {
        super.postProcessSave(event);

        if (!(event instanceof SaveDocumentEvent)) { // don't lock until they
            // route
            ArrayList<Long> capitalAssetNumbers = new ArrayList<Long>();
            for (AssetPaymentAssetDetail assetPaymentAssetDetail : this.getAssetPaymentAssetDetail()) {
                if (assetPaymentAssetDetail.getCapitalAssetNumber() != null) {
                    capitalAssetNumbers.add(assetPaymentAssetDetail.getCapitalAssetNumber());
                }
            }

            String documentTypeForLocking = CamsConstants.DocumentTypeName.ASSET_PAYMENT;
            if (this.isCapitalAssetBuilderOriginIndicator()) {
                documentTypeForLocking = CamsConstants.DocumentTypeName.ASSET_PAYMENT_FROM_CAB;
            }

            if (!this.getCapitalAssetManagementModuleService().storeAssetLocks(capitalAssetNumbers, this.getDocumentNumber(), documentTypeForLocking, null)) {
                throw new ValidationException("Asset " + capitalAssetNumbers.toString() + " is being locked by other documents.");
            }
        }
    }

    protected CapitalAssetManagementModuleService getCapitalAssetManagementModuleService() {
        return SpringContext.getBean(CapitalAssetManagementModuleService.class);
    }

    @Override
    public void doRouteStatusChange(DocumentRouteStatusChange statusChangeEvent) {
        super.doRouteStatusChange(statusChangeEvent);
        WorkflowDocument workflowDocument = getDocumentHeader().getWorkflowDocument();

        // Update asset payment table with the approved asset detail records.
        if (workflowDocument.isProcessed()) {
            SpringContext.getBean(AssetPaymentService.class).processApprovedAssetPayment(this);
        }

        // Remove asset lock when doc status change. We don't include
        // isFinal since document always go to 'processed' first.
        if (workflowDocument.isCanceled() || workflowDocument.isDisapproved() || workflowDocument.isProcessed()) {
            this.getCapitalAssetManagementModuleService().deleteAssetLocks(this.getDocumentNumber(), null);
        }

        if (isCapitalAssetBuilderOriginIndicator()) {
            SpringContext.getBean(CapitalAssetManagementModuleService.class).notifyRouteStatusChange(getDocumentHeader());
        }
    }

    @Override
    public void prepareForSave(KualiDocumentEvent event) {
        for (AssetPaymentAssetDetail assetDetail : this.getAssetPaymentAssetDetail()) {
            assetDetail.refreshReferenceObject(CamsPropertyConstants.AssetPaymentAssetDetail.ASSET);
            if (ObjectUtils.isNotNull(assetDetail.getAsset()) && assetDetail.getAsset().getTotalCostAmount() != null) {
                assetDetail.setPreviousTotalCostAmount(assetDetail.getAsset().getTotalCostAmount());
            }
            // CSU 6702 BEGIN Inferred change
            List<AssetPaymentDetail> apdList = assetDetail.getAssetPaymentDetails();
            for (AssetPaymentDetail apd : apdList) {
                String accountingPeriodCompositeString = getAccountingPeriodCompositeString();
                apd.setPostingYear(new Integer(StringUtils.right(accountingPeriodCompositeString, 4)));
                apd.setPostingPeriodCode(StringUtils.left(accountingPeriodCompositeString, 2));
            }
            // CSU 6702 END Inferred change
        }

        super.prepareForSave(event);
    }

    @Override
    public void clearAnyGeneralLedgerPendingEntries() {
        /*Reset the transfer payment indicators.*/
        List<GeneralLedgerPendingEntrySourceDetail> pendingEntrySourceDetails = getGeneralLedgerPendingEntrySourceDetails();
        for (GeneralLedgerPendingEntrySourceDetail sourceDetail : pendingEntrySourceDetails) {
            AssetPaymentDetail detail = (AssetPaymentDetail) sourceDetail;
            detail.setTransferPaymentIndicator(false);
        }
        super.clearAnyGeneralLedgerPendingEntries();
    }

    /**
     * For source accounting lines that have asset payments selected, we set payment transfer codes to Y. Once that is
     * done, when the payments are created (AssetPaymentServiceImpl.processPayments) accumulated depreciation and
     * current period depreciation will also be set. Furthermore we prepare two variables that are needed for
     * custom GLPE generation
     */
    @Override
    public boolean generateGeneralLedgerPendingEntries(GeneralLedgerPendingEntrySourceDetail glpeSourceDetail, GeneralLedgerPendingEntrySequenceHelper sequenceHelper) {
        LOG.debug("processGenerateGeneralLedgerPendingEntries(AccountingDocument, AccountingLine, GeneralLedgerPendingEntrySequenceHelper) - start");

        boolean success = true;
        AssetPaymentDetail assetPaymentDetail = (AssetPaymentDetail) glpeSourceDetail;

        // Loop over all selected assets
        for (AssetPaymentAssetDetail assetDetail : getAssetPaymentAssetDetail()) {
            // Reset the amount -- GLPEs are created per asset
            apipAssetPaymentTotalDepreciationHelper = new KualiDecimal(0);

            // Loop over the payment selections the user made for each asset
            for (AssetPaymentInProcessPayment assetPaymentInProcessPayment : assetPaymentInProcessPayments) {
                if (assetPaymentInProcessPayment.getCapitalAssetNumber().equals(assetDetail.getCapitalAssetNumber())
                    && assetPaymentInProcessPayment.getSequenceNumber().equals(assetPaymentDetail.getSequenceNumber())) {

                    assetPaymentInProcessPayment.refreshReferenceObject(CamsPropertyConstants.AssetPaymentInProcessPayments.ASSET_PAYMENTS);
                    AssetPayment apipAssetPayment = assetPaymentInProcessPayment.getAssetPayment();

                    if (ObjectUtils.isNotNull(apipAssetPayment.getAccumulatedPrimaryDepreciationAmount())) {
                        apipAssetPaymentTotalDepreciationHelper = apipAssetPaymentTotalDepreciationHelper.add(apipAssetPayment.getAccumulatedPrimaryDepreciationAmount());
                    }

                    // This will be repeated on each iteration, that's fine -- the accounting key will be the same
                    // on all of these per the UI selection restrictions
                    apipAssetPaymentHelper = apipAssetPayment;
                    assetPaymentDetail.setTransferPaymentIndicator(true);
                }
            }

            // Make sure we don't get 0$ GLPE lines
            if (apipAssetPaymentTotalDepreciationHelper.isNonZero()) {
                success &= super.generateGeneralLedgerPendingEntries(glpeSourceDetail, sequenceHelper);
                sequenceHelper.increment();
            }
        }

        return success;
    }

    /**
     * Sets custom plant fund account, fields in support of AssetObjectCode, transaction amount, and debitCreditCode per depreciation requirements
     */
    @Override
    public void customizeExplicitGeneralLedgerPendingEntry(GeneralLedgerPendingEntrySourceDetail postable, GeneralLedgerPendingEntry explicitEntry) {
        determinePlantFund(postable.getObjectCode().getFinancialObjectSubTypeCode(), explicitEntry);

        AssetObjectCodeService assetObjectCodeService = SpringContext.getBean(AssetObjectCodeService.class);
        AssetObjectCode assetObjectCode = assetObjectCodeService.findAssetObjectCode(apipAssetPaymentHelper.getChartOfAccountsCode(), apipAssetPaymentHelper.getFinancialObject().getFinancialObjectSubTypeCode());
        explicitEntry.setFinancialObjectCode(assetObjectCode.getAccumulatedDepreciationFinancialObjectCode());
        explicitEntry.setFinancialObjectTypeCode(assetObjectCode.getAccumulatedDepreciationFinancialObject().getFinancialObjectTypeCode());

        explicitEntry.setTransactionLedgerEntryAmount(apipAssetPaymentTotalDepreciationHelper.abs());
        explicitEntry.setTransactionLedgerEntryDescription(CamsConstants.Depreciation.CORRECTION_TRANSACTION_DESCRIPTION + apipAssetPaymentHelper.getCapitalAssetNumber());

        if (apipAssetPaymentTotalDepreciationHelper.isPositive()) {
            explicitEntry.setTransactionDebitCreditCode(KFSConstants.GL_DEBIT_CODE);
        } else {
            explicitEntry.setTransactionDebitCreditCode(KFSConstants.GL_CREDIT_CODE);
        }

        explicitEntry.setSubAccountNumber(AccountingDocumentRuleBaseConstants.GENERAL_LEDGER_PENDING_ENTRY_CODE.getBlankSubAccountNumber());
        explicitEntry.setFinancialSubObjectCode(AccountingDocumentRuleBaseConstants.GENERAL_LEDGER_PENDING_ENTRY_CODE.getBlankFinancialSubObjectCode());
    }

    /**
     * Sets custom plant fund account, fields in support of AssetObjectCode, transaction amount, and debitCreditCode per depreciation requirements. This is for the offset entry.
     */
    @Override
    public boolean customizeOffsetGeneralLedgerPendingEntry(GeneralLedgerPendingEntrySourceDetail accountingLine, GeneralLedgerPendingEntry explicitEntry, GeneralLedgerPendingEntry offsetEntry) {
        // on the explicitEntry we executed the plant fund account determination. No need to do that again
        offsetEntry.setAccountNumber(explicitEntry.getAccountNumber());
        offsetEntry.setChartOfAccountsCode(explicitEntry.getChartOfAccountsCode());

        AssetObjectCodeService assetObjectCodeService = SpringContext.getBean(AssetObjectCodeService.class);
        AssetObjectCode assetObjectCode = assetObjectCodeService.findAssetObjectCode(apipAssetPaymentHelper.getChartOfAccountsCode(), apipAssetPaymentHelper.getFinancialObject().getFinancialObjectSubTypeCode());
        offsetEntry.setFinancialObjectCode(assetObjectCode.getDepreciationExpenseFinancialObjectCode());
        offsetEntry.setFinancialObjectTypeCode(assetObjectCode.getDepreciationExpenseFinancialObject().getFinancialObjectTypeCode());

        offsetEntry.setTransactionLedgerEntryAmount(explicitEntry.getTransactionLedgerEntryAmount());
        offsetEntry.setTransactionLedgerEntryDescription(explicitEntry.getTransactionLedgerEntryDescription());

        if (explicitEntry.getTransactionDebitCreditCode().equals(KFSConstants.GL_CREDIT_CODE)) {
            offsetEntry.setTransactionDebitCreditCode(KFSConstants.GL_DEBIT_CODE);
        } else {
            offsetEntry.setTransactionDebitCreditCode(KFSConstants.GL_CREDIT_CODE);
        }

        offsetEntry.setSubAccountNumber(AccountingDocumentRuleBaseConstants.GENERAL_LEDGER_PENDING_ENTRY_CODE.getBlankSubAccountNumber());
        offsetEntry.setFinancialSubObjectCode(AccountingDocumentRuleBaseConstants.GENERAL_LEDGER_PENDING_ENTRY_CODE.getBlankFinancialSubObjectCode());

        return true;
    }

    /**
     * Logic for determining how plant fund account should be set.
     * @param postableFinancialObjectSubTypeCode used to determined what the plant fund account should be per AssetObjectCode reference table
     * @param explicitEntry will have the accountNumber and chartOfAccountsCode set ot the plant fund account values
     * @see org.kuali.kfs.module.cam.batch.service.impl.AssetDepreciationServiceImpl#calculateDepreciation(Integer, Integer, Collection, Calendar, Collection) has similar logic
     */
    protected void determinePlantFund(String postableFinancialObjectSubTypeCode, GeneralLedgerPendingEntry explicitEntry) {
        String plantAccount = "";
        String plantCOA = "";

        ParameterService parameterService = SpringContext.getBean(ParameterService.class);
        Collection<String> organizationPlantFundObjectSubType = new ArrayList<String>();
        Collection<String> campusPlantFundObjectSubType = new ArrayList<String>();
        if (parameterService.parameterExists(AssetDepreciationStep.class, CamsConstants.Parameters.DEPRECIATION_ORGANIZATION_PLANT_FUND_OBJECT_SUB_TYPE)) {
            organizationPlantFundObjectSubType = new ArrayList<>(parameterService.getParameterValuesAsString(AssetDepreciationStep.class, CamsConstants.Parameters.DEPRECIATION_ORGANIZATION_PLANT_FUND_OBJECT_SUB_TYPE));
        }
        if (parameterService.parameterExists(AssetDepreciationStep.class, CamsConstants.Parameters.DEPRECIATION_CAMPUS_PLANT_FUND_OBJECT_SUB_TYPES)) {
            campusPlantFundObjectSubType = new ArrayList<>(parameterService.getParameterValuesAsString(AssetDepreciationStep.class, CamsConstants.Parameters.DEPRECIATION_CAMPUS_PLANT_FUND_OBJECT_SUB_TYPES));
        }

        Asset apipAssetPaymentAssetHelper = apipAssetPaymentHelper.getAsset();
        // getting the right Plant Fund Chart code & Plant Fund Account
        if (organizationPlantFundObjectSubType.contains(postableFinancialObjectSubTypeCode)) {
            plantAccount = apipAssetPaymentAssetHelper.getOrganizationOwnerAccount().getOrganization().getOrganizationPlantAccountNumber();
            plantCOA = apipAssetPaymentAssetHelper.getOrganizationOwnerAccount().getOrganization().getOrganizationPlantChartCode();
        } else if (campusPlantFundObjectSubType.contains(postableFinancialObjectSubTypeCode)) {
            plantAccount = apipAssetPaymentAssetHelper.getOrganizationOwnerAccount().getOrganization().getCampusPlantAccountNumber();
            plantCOA = apipAssetPaymentAssetHelper.getOrganizationOwnerAccount().getOrganization().getCampusPlantChartCode();
        }
        if (StringUtils.isBlank(plantCOA) || StringUtils.isBlank(plantAccount)) {
            LOG.error("Asset Manual Payment Plant COA is " + plantCOA + " and plant account is " + plantAccount + " for Financial Object SubType Code = " + apipAssetPaymentAssetHelper.getFinancialObjectSubTypeCode() + " so Asset payment will use postables account# and coaCd " + apipAssetPaymentAssetHelper.getCapitalAssetNumber() + " - " + apipAssetPaymentHelper.getPaymentSequenceNumber());
        } else {
            explicitEntry.setAccountNumber(plantAccount);
            explicitEntry.setChartOfAccountsCode(plantCOA);
        }
    }

    public List<AssetPaymentAssetDetail> getAssetPaymentAssetDetail() {
        return assetPaymentAssetDetail;
    }

    public void setAssetPaymentAssetDetail(List<AssetPaymentAssetDetail> assetPaymentAssetDetail) {
        this.assetPaymentAssetDetail = assetPaymentAssetDetail;
    }

    public Long getCapitalAssetNumber() {
        return capitalAssetNumber;
    }

    public void setCapitalAssetNumber(Long capitalAssetNumber) {
        this.capitalAssetNumber = capitalAssetNumber;
    }

    /**
     * calculates the total previous cost amount of all the assets in the
     * document
     *
     * @return KualiDecimal
     */
    public KualiDecimal getAssetsTotalHistoricalCost() {
        KualiDecimal total = new KualiDecimal(0);
        if (this.getAssetPaymentAssetDetail().isEmpty()) {
            return new KualiDecimal(0);
        }

        for (AssetPaymentAssetDetail detail : this.getAssetPaymentAssetDetail()) {
            KualiDecimal amount = (detail.getPreviousTotalCostAmount() == null ? new KualiDecimal(0) : detail.getPreviousTotalCostAmount());
            total = total.add(amount);
        }
        return total;
    }

    /**
     * Get the asset payment distributor built by AssetPaymentDetails,
     * AssetPaymentAssetDetail and totalHistoricalCost
     *
     * @return AssetPaymentDistributor
     */
    public AssetDistribution getAssetPaymentDistributor() {
        if (CamsPropertyConstants.AssetPaymentAllocation.ASSET_DISTRIBUTION_BY_AMOUNT_CODE.equals(getAssetPaymentAllocationTypeCode())) {
            return new AssetDistributionManual(this);
        } else if (CamsPropertyConstants.AssetPaymentAllocation.ASSET_DISTRIBUTION_BY_PERCENTAGE_CODE.equals(getAssetPaymentAllocationTypeCode())) {
            return new AssetDistributionPercent(this);
        } else if (CamsPropertyConstants.AssetPaymentAllocation.ASSET_DISTRIBUTION_BY_TOTAL_COST_CODE.equals(getAssetPaymentAllocationTypeCode())) {
            return new AssetPaymentDistributionByTotalCost(this);
        }
        return new AssetDistributionEvenly(this);

    }

    /**
     * Get the allocation code value
     */
    public String getAssetPaymentAllocationTypeCode() {
        return assetPaymentAllocationTypeCode;
    }

    /**
     * Set the allocation code value
     */
    public void setAssetPaymentAllocationTypeCode(String code) {
        assetPaymentAllocationTypeCode = code;
        refreshReferenceObject("assetPaymentAllocationType");
    }

    /**
     * Sets the asset allocation type
     */
    public void setAssetPaymentAllocationType(AssetPaymentAllocationType assetPaymentAllocationType) {
        this.assetPaymentAllocationType = assetPaymentAllocationType;
    }

    /**
     * Get the asset allocation type
     */
    public AssetPaymentAllocationType getAssetPaymentAllocationType() {
        return assetPaymentAllocationType;
    }

    /**
     * Gets the allocationFromFPDocuments attribute.
     *
     * @return Returns the allocationFromFPDocuments
     */

    public boolean isAllocationFromFPDocuments() {
        return allocationFromFPDocuments;
    }

    /**
     * Sets the allocationFromFPDocuments attribute.
     *
     * @param allocationFromFPDocuments The allocationFromFPDocuments to set.
     */
    public void setAllocationFromFPDocuments(boolean allocationFromFPDocuments) {
        this.allocationFromFPDocuments = allocationFromFPDocuments;
    }

    public List<AssetPaymentInProcessPayment> getAssetPaymentInProcessPayments() {
        return assetPaymentInProcessPayments;
    }

    public void setAssetPaymentInProcessPayments(List<AssetPaymentInProcessPayment> assetPaymentInProcessPayments) {
        this.assetPaymentInProcessPayments = assetPaymentInProcessPayments;
    }
}
