/**
 * The Kuali Financial System, a comprehensive financial management system for higher education.
 *
 * Copyright 2005-2019 Kuali, Inc.
 *
 * This program is free software: you can redistribute it and/or modify
 * it under the terms of the GNU Affero General Public License as published by
 * the Free Software Foundation, either version 3 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU Affero General Public License for more details.
 *
 * You should have received a copy of the GNU Affero General Public License
 * along with this program.  If not, see <http://www.gnu.org/licenses/>.
 */
package org.kuali.kfs.module.cam.businessobject;

import org.apache.commons.lang3.StringUtils;
import org.kuali.kfs.coa.businessobject.Account;
import org.kuali.kfs.coa.businessobject.Chart;
import org.kuali.kfs.coa.businessobject.ObjectCode;
import org.kuali.kfs.coa.businessobject.ObjectType;
import org.kuali.kfs.coa.businessobject.SubAccount;
import org.kuali.kfs.coa.businessobject.SubObjectCode;
import org.kuali.kfs.gl.businessobject.Entry;
import org.kuali.kfs.krad.bo.PersistableBusinessObjectBase;
import org.kuali.kfs.module.cam.CamsConstants;
import org.kuali.kfs.sys.KFSConstants;
import org.kuali.rice.core.api.util.type.KualiDecimal;
import org.kuali.rice.kew.api.KewApiServiceLocator;
import org.kuali.rice.kew.api.doctype.DocumentType;
import org.kuali.rice.kew.doctype.bo.DocumentTypeEBO;

import java.sql.Date;
import java.sql.Timestamp;
import java.util.ArrayList;
import java.util.List;

public class GeneralLedgerEntry extends PersistableBusinessObjectBase {
    private Long generalLedgerAccountIdentifier;
    private Integer universityFiscalYear;
    private String chartOfAccountsCode;
    private String accountNumber;
    private String subAccountNumber;
    private String financialObjectCode;
    private String financialSubObjectCode;
    private String financialBalanceTypeCode;
    private String financialObjectTypeCode;
    private String universityFiscalPeriodCode;
    private String financialDocumentTypeCode;
    private String financialSystemOriginationCode;
    private String documentNumber;
    private Integer transactionLedgerEntrySequenceNumber;
    private String transactionLedgerEntryDescription;
    private KualiDecimal transactionLedgerEntryAmount;
    private KualiDecimal transactionLedgerSubmitAmount;
    private String organizationReferenceId;
    private String referenceFinancialSystemOriginationCode;
    private String referenceFinancialDocumentNumber;
    private String transactionDebitCreditCode;
    private String organizationDocumentNumber;
    private String projectCode;
    private Date transactionPostingDate;
    private Date transactionDate;
    private Timestamp transactionDateTimeStamp;
    private String activityStatusCode;

    // References
    private Account account;
    private Chart chart;
    private ObjectCode financialObject;
    private SubAccount subAccount;
    private SubObjectCode financialSubObject;
    private ObjectType objectType;
    private DocumentTypeEBO financialSystemDocumentTypeCode;

    private List<GeneralLedgerEntryAsset> generalLedgerEntryAssets;
    private List<PurchasingAccountsPayableLineAssetAccount> purApLineAssetAccounts;
    // non-db fields
    private boolean selected;
    private KualiDecimal amount;
    private boolean active;

    public GeneralLedgerEntry() {
        this.generalLedgerEntryAssets = new ArrayList<>();
        this.purApLineAssetAccounts = new ArrayList<>();
    }

    /**
     * Constructs a GeneralLedgerEntry item from GL line entry
     *
     * @param entry GL Lines
     */
    public GeneralLedgerEntry(Entry entry) {
        this.generalLedgerEntryAssets = new ArrayList<>();
        this.setUniversityFiscalYear(entry.getUniversityFiscalYear());
        this.setChartOfAccountsCode(entry.getChartOfAccountsCode());
        this.setAccountNumber(entry.getAccountNumber());
        this.setSubAccountNumber(entry.getSubAccountNumber());
        this.setFinancialObjectCode(entry.getFinancialObjectCode());
        this.setFinancialSubObjectCode(entry.getFinancialSubObjectCode());
        this.setFinancialBalanceTypeCode(entry.getFinancialBalanceTypeCode());
        this.setFinancialObjectTypeCode(entry.getFinancialObjectTypeCode());
        this.setUniversityFiscalPeriodCode(entry.getUniversityFiscalPeriodCode());
        this.setFinancialDocumentTypeCode(entry.getFinancialDocumentTypeCode());
        this.setFinancialSystemOriginationCode(entry.getFinancialSystemOriginationCode());
        this.setDocumentNumber(entry.getDocumentNumber());
        this.setTransactionLedgerEntrySequenceNumber(entry.getTransactionLedgerEntrySequenceNumber());
        this.setTransactionLedgerEntryDescription(entry.getTransactionLedgerEntryDescription());
        this.setTransactionLedgerEntryAmount(entry.getTransactionLedgerEntryAmount());
        this.setOrganizationReferenceId(entry.getOrganizationReferenceId());
        this.setReferenceFinancialSystemOriginationCode(entry.getReferenceFinancialSystemOriginationCode());
        this.setReferenceFinancialDocumentNumber(entry.getReferenceFinancialDocumentNumber());
        this.setTransactionDebitCreditCode(entry.getTransactionDebitCreditCode());
        this.setOrganizationDocumentNumber(entry.getOrganizationDocumentNumber());
        this.setProjectCode(entry.getProjectCode());
        this.setTransactionDate(entry.getTransactionDate());
        this.setTransactionPostingDate(entry.getTransactionPostingDate());
        this.setTransactionDateTimeStamp(entry.getTransactionDateTimeStamp());
        this.setActivityStatusCode(CamsConstants.ActivityStatusCode.NEW);
    }

    public Long getGeneralLedgerAccountIdentifier() {
        return generalLedgerAccountIdentifier;
    }

    public void setGeneralLedgerAccountIdentifier(Long glAccountId) {
        this.generalLedgerAccountIdentifier = glAccountId;
    }

    public Integer getUniversityFiscalYear() {
        return universityFiscalYear;
    }

    public void setUniversityFiscalYear(Integer universityFiscalYear) {
        this.universityFiscalYear = universityFiscalYear;
    }

    public String getChartOfAccountsCode() {
        return chartOfAccountsCode;
    }

    public void setChartOfAccountsCode(String chartOfAccountsCode) {
        this.chartOfAccountsCode = chartOfAccountsCode;
    }

    public String getAccountNumber() {
        return accountNumber;
    }

    public void setAccountNumber(String accountNumber) {
        this.accountNumber = accountNumber;
    }

    public String getSubAccountNumber() {
        return subAccountNumber;
    }

    public void setSubAccountNumber(String subAccountNumber) {
        this.subAccountNumber = subAccountNumber;
    }

    public String getFinancialObjectCode() {
        return financialObjectCode;
    }

    public void setFinancialObjectCode(String financialObjectCode) {
        this.financialObjectCode = financialObjectCode;
    }

    public String getFinancialSubObjectCode() {
        return financialSubObjectCode;
    }

    public void setFinancialSubObjectCode(String financialSubObjectCode) {
        this.financialSubObjectCode = financialSubObjectCode;
    }

    public String getFinancialBalanceTypeCode() {
        return financialBalanceTypeCode;
    }

    public void setFinancialBalanceTypeCode(String financialBalanceTypeCode) {
        this.financialBalanceTypeCode = financialBalanceTypeCode;
    }

    public String getFinancialObjectTypeCode() {
        return financialObjectTypeCode;
    }

    public void setFinancialObjectTypeCode(String financialObjectTypeCode) {
        this.financialObjectTypeCode = financialObjectTypeCode;
    }

    public String getUniversityFiscalPeriodCode() {
        return universityFiscalPeriodCode;
    }

    public void setUniversityFiscalPeriodCode(String universityFiscalPeriodCode) {
        this.universityFiscalPeriodCode = universityFiscalPeriodCode;
    }

    public String getFinancialDocumentTypeCode() {
        return financialDocumentTypeCode;
    }

    public void setFinancialDocumentTypeCode(String financialDocumentTypeCode) {
        this.financialDocumentTypeCode = financialDocumentTypeCode;
    }

    public String getFinancialSystemOriginationCode() {
        return financialSystemOriginationCode;
    }

    public void setFinancialSystemOriginationCode(String financialSystemOriginationCode) {
        this.financialSystemOriginationCode = financialSystemOriginationCode;
    }

    public String getDocumentNumber() {
        return documentNumber;
    }

    public void setDocumentNumber(String documentNumber) {
        this.documentNumber = documentNumber;
    }

    public Integer getTransactionLedgerEntrySequenceNumber() {
        return transactionLedgerEntrySequenceNumber;
    }

    public void setTransactionLedgerEntrySequenceNumber(Integer transactionLedgerEntrySequenceNumber) {
        this.transactionLedgerEntrySequenceNumber = transactionLedgerEntrySequenceNumber;
    }

    public String getTransactionLedgerEntryDescription() {
        return transactionLedgerEntryDescription;
    }

    public void setTransactionLedgerEntryDescription(String transactionLedgerEntryDescription) {
        this.transactionLedgerEntryDescription = transactionLedgerEntryDescription;
    }

    public KualiDecimal getTransactionLedgerEntryAmount() {
        return transactionLedgerEntryAmount;
    }

    public void setTransactionLedgerEntryAmount(KualiDecimal transactionLedgerEntryAmount) {
        this.transactionLedgerEntryAmount = transactionLedgerEntryAmount;
    }

    public String getOrganizationReferenceId() {
        return organizationReferenceId;
    }

    public void setOrganizationReferenceId(String organizationReferenceId) {
        this.organizationReferenceId = organizationReferenceId;
    }

    public String getReferenceFinancialSystemOriginationCode() {
        return referenceFinancialSystemOriginationCode;
    }

    public void setReferenceFinancialSystemOriginationCode(String referenceFinancialSystemOriginationCode) {
        this.referenceFinancialSystemOriginationCode = referenceFinancialSystemOriginationCode;
    }

    public String getReferenceFinancialDocumentNumber() {
        return referenceFinancialDocumentNumber;
    }

    public void setReferenceFinancialDocumentNumber(String referenceFinancialDocumentNumber) {
        this.referenceFinancialDocumentNumber = referenceFinancialDocumentNumber;
    }

    public String getTransactionDebitCreditCode() {
        return transactionDebitCreditCode;
    }

    public void setTransactionDebitCreditCode(String transactionDebitCreditCode) {
        this.transactionDebitCreditCode = transactionDebitCreditCode;
    }

    public String getOrganizationDocumentNumber() {
        return organizationDocumentNumber;
    }

    public void setOrganizationDocumentNumber(String organizationDocumentNumber) {
        this.organizationDocumentNumber = organizationDocumentNumber;
    }

    public String getProjectCode() {
        return projectCode;
    }

    public void setProjectCode(String projectCode) {
        this.projectCode = projectCode;
    }

    public Date getTransactionPostingDate() {
        return transactionPostingDate;
    }

    public void setTransactionPostingDate(Date transactionPostingDate) {
        this.transactionPostingDate = transactionPostingDate;
    }

    public Timestamp getTransactionDateTimeStamp() {
        return transactionDateTimeStamp;
    }

    public void setTransactionDateTimeStamp(Timestamp transactionDateTimeStamp) {
        this.transactionDateTimeStamp = transactionDateTimeStamp;
    }

    public Account getAccount() {
        return account;
    }

    public void setAccount(Account account) {
        this.account = account;
    }

    public SubAccount getSubAccount() {
        return subAccount;
    }

    public void setSubAccount(SubAccount subAccount) {
        this.subAccount = subAccount;
    }

    public Chart getChart() {
        return chart;
    }

    public void setChart(Chart chart) {
        this.chart = chart;
    }

    public ObjectCode getFinancialObject() {
        return financialObject;
    }

    public void setFinancialObject(ObjectCode financialObject) {
        this.financialObject = financialObject;
    }

    public SubObjectCode getFinancialSubObject() {
        return financialSubObject;
    }

    public void setFinancialSubObject(SubObjectCode financialSubObject) {
        this.financialSubObject = financialSubObject;
    }

    public ObjectType getObjectType() {
        return objectType;
    }

    public void setObjectType(ObjectType objectType) {
        this.objectType = objectType;
    }

    public DocumentTypeEBO getFinancialSystemDocumentTypeCode() {
        if (financialSystemDocumentTypeCode == null || !StringUtils.equals(financialSystemDocumentTypeCode.getName(), financialDocumentTypeCode)) {
            financialSystemDocumentTypeCode = null;
            if (StringUtils.isNotBlank(financialDocumentTypeCode)) {
                DocumentType docType = KewApiServiceLocator.getDocumentTypeService().getDocumentTypeByName(financialDocumentTypeCode);
                if (docType != null) {
                    financialSystemDocumentTypeCode = org.kuali.rice.kew.doctype.bo.DocumentType.from(docType);
                }
            }
        }
        return financialSystemDocumentTypeCode;
    }

    public boolean isActive() {
        return CamsConstants.ActivityStatusCode.NEW.equalsIgnoreCase(this.getActivityStatusCode())
                || CamsConstants.ActivityStatusCode.MODIFIED.equalsIgnoreCase(this.getActivityStatusCode());
    }

    public String getActivityStatusCode() {
        return activityStatusCode;
    }

    public void setActivityStatusCode(String activityStatusCode) {
        this.activityStatusCode = activityStatusCode;
    }

    public List<GeneralLedgerEntryAsset> getGeneralLedgerEntryAssets() {
        return generalLedgerEntryAssets;
    }

    public void setGeneralLedgerEntryAssets(List<GeneralLedgerEntryAsset> generalLedgerEntryAssets) {
        this.generalLedgerEntryAssets = generalLedgerEntryAssets;
    }

    public Date getTransactionDate() {
        return transactionDate;
    }

    public void setTransactionDate(Date transactionDate) {
        this.transactionDate = transactionDate;
    }

    public List<PurchasingAccountsPayableLineAssetAccount> getPurApLineAssetAccounts() {
        return purApLineAssetAccounts;
    }

    public void setPurApLineAssetAccounts(List<PurchasingAccountsPayableLineAssetAccount> purApLineAssetAccounts) {
        this.purApLineAssetAccounts = purApLineAssetAccounts;
    }

    public KualiDecimal computePayment() {
        KualiDecimal absAmount = getTransactionLedgerEntryAmount();
        if (absAmount == null) {
            return null;
        }
        return KFSConstants.GL_CREDIT_CODE.equals(getTransactionDebitCreditCode()) ? absAmount.negated() : absAmount;
    }

    public boolean isSelected() {
        return selected;
    }

    public void setSelected(boolean selected) {
        this.selected = selected;
    }

    public KualiDecimal getAmount() {
        if (getTransactionLedgerEntryAmount() != null && KFSConstants.GL_CREDIT_CODE.equals(getTransactionDebitCreditCode())) {
            setAmount(getTransactionLedgerEntryAmount().negated());
        } else {
            setAmount(getTransactionLedgerEntryAmount());
        }
        return amount;
    }

    public void setAmount(KualiDecimal absAmount) {
        this.amount = absAmount;
    }

    public KualiDecimal getTransactionLedgerSubmitAmount() {
        return transactionLedgerSubmitAmount;
    }

    public void setTransactionLedgerSubmitAmount(KualiDecimal transactionLedgerSubmitAmount) {
        this.transactionLedgerSubmitAmount = transactionLedgerSubmitAmount;
    }
}
