/**
 * The Kuali Financial System, a comprehensive financial management system for higher education.
 *
 * Copyright 2005-2019 Kuali, Inc.
 *
 * This program is free software: you can redistribute it and/or modify
 * it under the terms of the GNU Affero General Public License as published by
 * the Free Software Foundation, either version 3 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU Affero General Public License for more details.
 *
 * You should have received a copy of the GNU Affero General Public License
 * along with this program.  If not, see <http://www.gnu.org/licenses/>.
 */
package org.kuali.kfs.module.cam.document;

import org.apache.commons.lang3.StringUtils;
import org.kuali.kfs.integration.cam.CapitalAssetManagementModuleService;
import org.kuali.kfs.krad.exception.ValidationException;
import org.kuali.kfs.krad.rules.rule.event.KualiDocumentEvent;
import org.kuali.kfs.krad.rules.rule.event.SaveDocumentEvent;
import org.kuali.kfs.krad.service.KualiModuleService;
import org.kuali.kfs.krad.service.ModuleService;
import org.kuali.kfs.module.cam.CamsConstants;
import org.kuali.kfs.module.cam.businessobject.Asset;
import org.kuali.kfs.module.cam.document.service.EquipmentLoanOrReturnService;
import org.kuali.kfs.sys.KFSConstants;
import org.kuali.kfs.sys.context.SpringContext;
import org.kuali.kfs.sys.document.FinancialSystemTransactionalDocumentBase;
import org.kuali.rice.core.api.datetime.DateTimeService;
import org.kuali.rice.kew.api.WorkflowDocument;
import org.kuali.rice.kew.framework.postprocessor.DocumentRouteStatusChange;
import org.kuali.rice.kim.api.identity.Person;
import org.kuali.rice.location.api.LocationConstants;
import org.kuali.rice.location.framework.country.CountryEbo;
import org.kuali.rice.location.framework.postalcode.PostalCodeEbo;
import org.kuali.rice.location.framework.state.StateEbo;

import java.sql.Date;
import java.util.ArrayList;
import java.util.HashMap;
import java.util.Map;

public class EquipmentLoanOrReturnDocument extends FinancialSystemTransactionalDocumentBase {
    protected String hiddenFieldForError;
    protected String documentNumber;
    protected Date loanDate;
    protected Date expectedReturnDate;
    protected Date loanReturnDate;
    protected String borrowerUniversalIdentifier;
    protected String borrowerAddress;
    protected String borrowerCityName;
    protected String borrowerStateCode;
    protected String borrowerZipCode;
    protected String borrowerCountryCode;
    protected String borrowerPhoneNumber;
    protected String borrowerStorageAddress;
    protected String borrowerStorageCityName;
    protected String borrowerStorageStateCode;
    protected String borrowerStorageZipCode;
    protected String borrowerStorageCountryCode;
    protected String borrowerStoragePhoneNumber;
    protected Long capitalAssetNumber;

    protected StateEbo borrowerState;
    protected StateEbo borrowerStorageState;
    protected CountryEbo borrowerCountry;
    protected CountryEbo borrowerStorageCountry;
    protected Person borrowerPerson;
    protected Asset asset;
    protected PostalCodeEbo borrowerPostalZipCode;
    protected PostalCodeEbo borrowerStoragePostalZipCode;

    // sets document status (i.e. new loan, return, or renew)
    protected boolean newLoan;
    protected boolean returnLoan;

    public EquipmentLoanOrReturnDocument() {
        super();
    }

    public Asset getAsset() {
        return asset;
    }

    public void setAsset(Asset asset) {
        this.asset = asset;
    }

    public CountryEbo getBorrowerCountry() {
        if (StringUtils.isBlank(borrowerCountryCode)) {
            borrowerCountry = null;
        } else {
            if (borrowerCountry == null || !StringUtils.equals(borrowerCountry.getCode(), borrowerCountryCode)) {
                ModuleService moduleService = SpringContext.getBean(KualiModuleService.class).getResponsibleModuleService(CountryEbo.class);
                if (moduleService != null) {
                    Map<String, Object> keys = new HashMap<>(1);
                    keys.put(LocationConstants.PrimaryKeyConstants.CODE, borrowerCountryCode);
                    borrowerCountry = moduleService.getExternalizableBusinessObject(CountryEbo.class, keys);
                } else {
                    throw new RuntimeException("CONFIGURATION ERROR: No responsible module found for EBO class.  Unable to proceed.");
                }
            }
        }
        return borrowerCountry;
    }

    public void setBorrowerCountry(CountryEbo borrowerCountry) {
        this.borrowerCountry = borrowerCountry;
    }

    public StateEbo getBorrowerState() {
        if (StringUtils.isBlank(KFSConstants.COUNTRY_CODE_UNITED_STATES) || StringUtils.isBlank(borrowerStateCode)) {
            borrowerState = null;
        } else {
            if (borrowerState == null || !StringUtils.equals(borrowerState.getCode(), borrowerStateCode)
                    || !StringUtils.equals(borrowerState.getCountryCode(), KFSConstants.COUNTRY_CODE_UNITED_STATES)) {
                ModuleService moduleService = SpringContext.getBean(KualiModuleService.class).getResponsibleModuleService(StateEbo.class);
                if (moduleService != null) {
                    Map<String, Object> keys = new HashMap<>(2);
                    keys.put(LocationConstants.PrimaryKeyConstants.COUNTRY_CODE, borrowerCountryCode);
                    keys.put(LocationConstants.PrimaryKeyConstants.CODE, borrowerStateCode);
                    borrowerState = moduleService.getExternalizableBusinessObject(StateEbo.class, keys);
                } else {
                    throw new RuntimeException("CONFIGURATION ERROR: No responsible module found for EBO class.  Unable to proceed.");
                }
            }
        }
        return borrowerState;
    }

    public void setBorrowerState(StateEbo borrowerState) {
        this.borrowerState = borrowerState;
    }

    public CountryEbo getBorrowerStorageCountry() {
        if (StringUtils.isBlank(borrowerStorageCountryCode)) {
            borrowerStorageCountry = null;
        } else {
            if (borrowerStorageCountry == null || !StringUtils.equals(borrowerStorageCountry.getCode(), borrowerStorageCountryCode)) {
                ModuleService moduleService = SpringContext.getBean(KualiModuleService.class).getResponsibleModuleService(CountryEbo.class);
                if (moduleService != null) {
                    Map<String, Object> keys = new HashMap<>(1);
                    keys.put(LocationConstants.PrimaryKeyConstants.CODE, borrowerStorageCountryCode);
                    borrowerStorageCountry = moduleService.getExternalizableBusinessObject(CountryEbo.class, keys);
                } else {
                    throw new RuntimeException("CONFIGURATION ERROR: No responsible module found for EBO class.  Unable to proceed.");
                }
            }
        }
        return borrowerStorageCountry;
    }

    public void setBorrowerStorageCountry(CountryEbo borrowerStorageCountry) {
        this.borrowerStorageCountry = borrowerStorageCountry;
    }

    public StateEbo getBorrowerStorageState() {
        if (StringUtils.isBlank(KFSConstants.COUNTRY_CODE_UNITED_STATES)
                || StringUtils.isBlank(borrowerStorageStateCode)) {
            borrowerStorageState = null;
        } else {
            if (borrowerStorageState == null
                    || !StringUtils.equals(borrowerStorageState.getCountryCode(), borrowerStorageCountryCode)
                    || !StringUtils.equals(borrowerStorageState.getCode(), borrowerStorageStateCode)) {
                ModuleService moduleService = SpringContext.getBean(KualiModuleService.class).getResponsibleModuleService(StateEbo.class);
                if (moduleService != null) {
                    Map<String, Object> keys = new HashMap<>(2);
                    keys.put(LocationConstants.PrimaryKeyConstants.COUNTRY_CODE, borrowerStorageCountryCode);
                    keys.put(LocationConstants.PrimaryKeyConstants.CODE, borrowerStorageStateCode);
                    borrowerStorageState = moduleService.getExternalizableBusinessObject(StateEbo.class, keys);
                } else {
                    throw new RuntimeException("CONFIGURATION ERROR: No responsible module found for EBO class.  Unable to proceed.");
                }
            }
        }

        return borrowerStorageState;
    }

    public void setBorrowerStorageState(StateEbo borrowerStorageState) {
        this.borrowerStorageState = borrowerStorageState;
    }

    public Person getBorrowerPerson() {
        borrowerPerson = SpringContext.getBean(org.kuali.rice.kim.api.identity.PersonService.class).updatePersonIfNecessary(borrowerUniversalIdentifier, borrowerPerson);
        return borrowerPerson;
    }

    public void setBorrowerPerson(Person borrowerPerson) {
        this.borrowerPerson = borrowerPerson;
    }

    public String getBorrowerAddress() {
        return borrowerAddress;
    }

    public void setBorrowerAddress(String borrowerAddress) {
        this.borrowerAddress = borrowerAddress;
    }

    public String getBorrowerCityName() {
        return borrowerCityName;
    }

    public void setBorrowerCityName(String borrowerCityName) {
        this.borrowerCityName = borrowerCityName;
    }

    public String getBorrowerCountryCode() {
        return borrowerCountryCode;
    }

    public void setBorrowerCountryCode(String borrowerCountryCode) {
        this.borrowerCountryCode = borrowerCountryCode;
    }

    public String getBorrowerPhoneNumber() {
        return borrowerPhoneNumber;
    }

    public void setBorrowerPhoneNumber(String borrowerPhoneNumber) {
        this.borrowerPhoneNumber = borrowerPhoneNumber;
    }

    public String getBorrowerStateCode() {
        return borrowerStateCode;
    }

    public void setBorrowerStateCode(String borrowerStateCode) {
        this.borrowerStateCode = borrowerStateCode;
    }

    public String getBorrowerStorageAddress() {
        return borrowerStorageAddress;
    }

    public void setBorrowerStorageAddress(String borrowerStorageAddress) {
        this.borrowerStorageAddress = borrowerStorageAddress;
    }

    public String getBorrowerStorageCityName() {
        return borrowerStorageCityName;
    }

    public void setBorrowerStorageCityName(String borrowerStorageCityName) {
        this.borrowerStorageCityName = borrowerStorageCityName;
    }

    public String getBorrowerStorageCountryCode() {
        return borrowerStorageCountryCode;
    }

    public void setBorrowerStorageCountryCode(String borrowerStorageCountryCode) {
        this.borrowerStorageCountryCode = borrowerStorageCountryCode;
    }

    public String getBorrowerStoragePhoneNumber() {
        return borrowerStoragePhoneNumber;
    }

    public void setBorrowerStoragePhoneNumber(String borrowerStoragePhoneNumber) {
        this.borrowerStoragePhoneNumber = borrowerStoragePhoneNumber;
    }

    public String getBorrowerStorageStateCode() {
        return borrowerStorageStateCode;
    }

    public void setBorrowerStorageStateCode(String borrowerStorageStateCode) {
        this.borrowerStorageStateCode = borrowerStorageStateCode;
    }

    public String getBorrowerStorageZipCode() {
        return borrowerStorageZipCode;
    }

    public void setBorrowerStorageZipCode(String borrowerStorageZipCode) {
        this.borrowerStorageZipCode = borrowerStorageZipCode;
    }

    public PostalCodeEbo getBorrowerPostalZipCode() {
        if (StringUtils.isBlank(KFSConstants.COUNTRY_CODE_UNITED_STATES) || StringUtils.isBlank(borrowerZipCode)) {
            borrowerPostalZipCode = null;
        } else {
            if (borrowerPostalZipCode == null
                    || !StringUtils.equals(borrowerPostalZipCode.getCountryCode(), borrowerCountryCode)
                    || !StringUtils.equals(borrowerPostalZipCode.getCode(), borrowerZipCode)) {
                ModuleService moduleService = SpringContext.getBean(KualiModuleService.class).getResponsibleModuleService(PostalCodeEbo.class);
                if (moduleService != null) {
                    Map<String, Object> keys = new HashMap<>(2);
                    keys.put(LocationConstants.PrimaryKeyConstants.COUNTRY_CODE, borrowerCountryCode);
                    keys.put(LocationConstants.PrimaryKeyConstants.CODE, borrowerZipCode);
                    borrowerPostalZipCode = moduleService.getExternalizableBusinessObject(PostalCodeEbo.class, keys);
                } else {
                    throw new RuntimeException("CONFIGURATION ERROR: No responsible module found for EBO class.  Unable to proceed.");
                }
            }
        }
        return borrowerPostalZipCode;
    }

    public void setBorrowerPostalZipCode(PostalCodeEbo borrowerPostalZipCode) {
        this.borrowerPostalZipCode = borrowerPostalZipCode;
    }

    public PostalCodeEbo getBorrowerStoragePostalZipCode() {
        if (StringUtils.isBlank(KFSConstants.COUNTRY_CODE_UNITED_STATES)
                || StringUtils.isBlank(borrowerStorageZipCode)) {
            borrowerStoragePostalZipCode = null;
        } else {
            if (borrowerStoragePostalZipCode == null
                    || !StringUtils.equals(borrowerStoragePostalZipCode.getCountryCode(), borrowerStorageCountryCode)
                    || !StringUtils.equals(borrowerStoragePostalZipCode.getCode(), borrowerStorageZipCode)) {
                ModuleService moduleService = SpringContext.getBean(KualiModuleService.class).getResponsibleModuleService(PostalCodeEbo.class);
                if (moduleService != null) {
                    Map<String, Object> keys = new HashMap<>(2);
                    keys.put(LocationConstants.PrimaryKeyConstants.COUNTRY_CODE, borrowerStorageCountryCode);
                    keys.put(LocationConstants.PrimaryKeyConstants.CODE, borrowerStorageZipCode);
                    borrowerStoragePostalZipCode = moduleService.getExternalizableBusinessObject(PostalCodeEbo.class, keys);
                } else {
                    throw new RuntimeException("CONFIGURATION ERROR: No responsible module found for EBO class.  Unable to proceed.");
                }
            }
        }

        return borrowerStoragePostalZipCode;
    }

    public void setborrowerStoragePostalZipCode(PostalCodeEbo borrowerStoragePostalZipCode) {
        this.borrowerStoragePostalZipCode = borrowerStoragePostalZipCode;
    }

    public String getBorrowerUniversalIdentifier() {
        return borrowerUniversalIdentifier;
    }

    public void setBorrowerUniversalIdentifier(String borrowerUniversalIdentifier) {
        this.borrowerUniversalIdentifier = borrowerUniversalIdentifier;
    }

    public String getBorrowerZipCode() {
        return borrowerZipCode;
    }

    public void setBorrowerZipCode(String borrowerZipCode) {
        this.borrowerZipCode = borrowerZipCode;
    }

    public String getDocumentNumber() {
        return documentNumber;
    }

    public void setDocumentNumber(String documentNumber) {
        this.documentNumber = documentNumber;
    }

    public Date getExpectedReturnDate() {
        return expectedReturnDate;
    }

    public void setExpectedReturnDate(Date expectedReturnDate) {
        this.expectedReturnDate = expectedReturnDate;
    }

    public Date getLoanDate() {
        if (loanDate != null) {
            return loanDate;
        } else {
            return SpringContext.getBean(DateTimeService.class).getCurrentSqlDate();
        }
    }

    public void setLoanDate(Date loanDate) {
        this.loanDate = loanDate;
    }

    public Date getLoanReturnDate() {
        return loanReturnDate;
    }

    public void setLoanReturnDate(Date loanReturnDate) {
        this.loanReturnDate = loanReturnDate;
    }

    public void postProcessSave(KualiDocumentEvent event) {
        super.postProcessSave(event);

        if (!(event instanceof SaveDocumentEvent)) { // don't lock until they route
            ArrayList capitalAssetNumbers = new ArrayList<Long>();
            if (this.getCapitalAssetNumber() != null) {
                capitalAssetNumbers.add(this.getCapitalAssetNumber());
            }
            // check and lock on asset numbers exclude approve event.
            if (!this.getCapitalAssetManagementModuleService().storeAssetLocks(capitalAssetNumbers, this.getDocumentNumber(), CamsConstants.DocumentTypeName.ASSET_EQUIPMENT_LOAN_OR_RETURN, null)) {
                throw new ValidationException("Asset " + capitalAssetNumbers.toString() + " is being locked by other documents.");
            }
        }
    }

    protected CapitalAssetManagementModuleService getCapitalAssetManagementModuleService() {
        return SpringContext.getBean(CapitalAssetManagementModuleService.class);
    }

    /**
     * If the document final, unlock the document
     */
    @Override
    public void doRouteStatusChange(DocumentRouteStatusChange statusChangeEvent) {
        super.doRouteStatusChange(statusChangeEvent);

        WorkflowDocument workflowDocument = getDocumentHeader().getWorkflowDocument();

        if (workflowDocument.isProcessed()) {
            SpringContext.getBean(EquipmentLoanOrReturnService.class).processApprovedEquipmentLoanOrReturn(this);
        }

        // Remove asset lock when doc status change. We don't include isFinal since document always go to 'processed' first.
        if (workflowDocument.isCanceled() || workflowDocument.isDisapproved() || workflowDocument.isProcessed()) {
            this.getCapitalAssetManagementModuleService().deleteAssetLocks(this.getDocumentNumber(), null);
        }
    }

    @Override
    public boolean answerSplitNodeQuestion(String nodeName) throws UnsupportedOperationException {
        if (CamsConstants.RouteLevelNames.ORGANIZATION_INACTIVE.equals(nodeName)) {
            return AssetTransferDocument.isRequiresOrganizationInactiveRouteNode(this, asset);
        }

        return super.answerSplitNodeQuestion(nodeName);
    }

    public Long getCapitalAssetNumber() {
        return capitalAssetNumber;
    }

    public void setCapitalAssetNumber(Long capitalAssetNumber) {
        this.capitalAssetNumber = capitalAssetNumber;
    }

    public boolean isNewLoan() {
        return newLoan;
    }

    public void setNewLoan(boolean newLoan) {
        this.newLoan = newLoan;
    }

    public boolean isReturnLoan() {
        return returnLoan;
    }

    public void setReturnLoan(boolean returnLoan) {
        this.returnLoan = returnLoan;
    }

    public String getHiddenFieldForError() {
        return hiddenFieldForError;
    }

    public void setHiddenFieldForError(String hiddenFieldForError) {
        this.hiddenFieldForError = hiddenFieldForError;
    }

}
