/**
 * The Kuali Financial System, a comprehensive financial management system for higher education.
 *
 * Copyright 2005-2019 Kuali, Inc.
 *
 * This program is free software: you can redistribute it and/or modify
 * it under the terms of the GNU Affero General Public License as published by
 * the Free Software Foundation, either version 3 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU Affero General Public License for more details.
 *
 * You should have received a copy of the GNU Affero General Public License
 * along with this program.  If not, see <http://www.gnu.org/licenses/>.
 */
package org.kuali.kfs.module.cam.businessobject;

import org.apache.commons.lang3.StringUtils;
import org.kuali.kfs.integration.purap.ItemCapitalAsset;
import org.kuali.kfs.kns.service.DataDictionaryService;
import org.kuali.kfs.krad.bo.PersistableBusinessObjectBase;
import org.kuali.kfs.krad.service.BusinessObjectService;
import org.kuali.kfs.krad.util.ObjectUtils;
import org.kuali.kfs.krad.util.UrlFactory;
import org.kuali.kfs.module.cam.CamsConstants;
import org.kuali.kfs.module.cam.CamsPropertyConstants;
import org.kuali.kfs.module.cam.document.service.PurApLineService;
import org.kuali.kfs.module.purap.businessobject.PurchaseOrderItemCapitalAsset;
import org.kuali.kfs.sys.KFSConstants;
import org.kuali.kfs.sys.context.SpringContext;
import org.kuali.rice.core.api.util.type.KualiDecimal;

import java.util.ArrayList;
import java.util.Collection;
import java.util.HashMap;
import java.util.List;
import java.util.Map;
import java.util.Properties;

public class PurchasingAccountsPayableItemAsset extends PersistableBusinessObjectBase implements
        Comparable<PurchasingAccountsPayableItemAsset> {

    private String documentNumber;
    private Integer accountsPayableLineItemIdentifier;
    private Integer capitalAssetBuilderLineNumber;
    private String accountsPayableLineItemDescription;
    private KualiDecimal accountsPayableItemQuantity;
    private String capitalAssetManagementDocumentNumber;
    private String activityStatusCode;

    private PurchasingAccountsPayableDocument purchasingAccountsPayableDocument;
    private List<PurchasingAccountsPayableLineAssetAccount> purchasingAccountsPayableLineAssetAccounts;

    // non persistent fields
    private boolean active;
    private Integer itemLineNumber;
    private boolean additionalChargeNonTradeInIndicator;
    private boolean tradeInAllowance;
    private boolean itemAssignedToTradeInIndicator;
    private KualiDecimal unitCost;
    private KualiDecimal totalCost;
    private String firstFincialObjectCode;
    private KualiDecimal splitQty;
    private boolean selectedValue;
    private String itemTypeCode;
    private String lockingInformation;
    // used for Capital Asset Transaction
    private String capitalAssetTransactionTypeCode;
    private List<ItemCapitalAsset> purApItemAssets;
    private Integer capitalAssetSystemIdentifier;

    private Integer purchaseOrderItemIdentifier;
    // used to control "create asset" and "apply payment" button display
    private boolean createAssetIndicator;
    private boolean applyPaymentIndicator;

    private String preTagInquiryUrl;
    private List<Long> approvedAssetNumbers;

    private Integer paymentRequestIdentifier;

    public PurchasingAccountsPayableItemAsset() {
        this.purchasingAccountsPayableLineAssetAccounts = new ArrayList<>();
        this.selectedValue = false;
        this.createAssetIndicator = false;
        this.applyPaymentIndicator = false;
        this.purApItemAssets = new ArrayList<>();
    }

    // constructor used for split
    public PurchasingAccountsPayableItemAsset(PurchasingAccountsPayableItemAsset initialItemAsset) {
        this.documentNumber = initialItemAsset.documentNumber;
        this.accountsPayableLineItemIdentifier = initialItemAsset.getAccountsPayableLineItemIdentifier();
        this.accountsPayableLineItemDescription = initialItemAsset.getAccountsPayableLineItemDescription();
        this.itemLineNumber = initialItemAsset.getItemLineNumber();
        this.firstFincialObjectCode = initialItemAsset.getFirstFincialObjectCode();
        this.activityStatusCode = initialItemAsset.getActivityStatusCode();
        this.tradeInAllowance = initialItemAsset.isTradeInAllowance();
        this.itemAssignedToTradeInIndicator = initialItemAsset.isItemAssignedToTradeInIndicator();
        this.additionalChargeNonTradeInIndicator = initialItemAsset.isAdditionalChargeNonTradeInIndicator();
        this.purchasingAccountsPayableLineAssetAccounts = new ArrayList<>();
        this.selectedValue = false;
        this.createAssetIndicator = initialItemAsset.isCreateAssetIndicator();
        this.applyPaymentIndicator = initialItemAsset.isApplyPaymentIndicator();
        this.purchaseOrderItemIdentifier = initialItemAsset.getPurchaseOrderItemIdentifier();
        this.capitalAssetTransactionTypeCode = initialItemAsset.getCapitalAssetTransactionTypeCode();
        this.purApItemAssets = new ArrayList<>(initialItemAsset.getPurApItemAssets());
        this.capitalAssetSystemIdentifier = initialItemAsset.getCapitalAssetSystemIdentifier();
        this.purchasingAccountsPayableDocument = initialItemAsset.getPurchasingAccountsPayableDocument();
        this.lockingInformation = initialItemAsset.getLockingInformation();
    }

    public String getLockingInformation() {
        return lockingInformation;
    }

    public void setLockingInformation(String lockingInformation) {
        this.lockingInformation = lockingInformation;
    }

    public Integer getCapitalAssetSystemIdentifier() {
        return capitalAssetSystemIdentifier;
    }

    public void setCapitalAssetSystemIdentifier(Integer capitalAssetSystemIdentifier) {
        this.capitalAssetSystemIdentifier = capitalAssetSystemIdentifier;
    }

    public Integer getPurchaseOrderItemIdentifier() {
        return purchaseOrderItemIdentifier;
    }

    public void setPurchaseOrderItemIdentifier(Integer purchaseOrderItemIdentifier) {
        this.purchaseOrderItemIdentifier = purchaseOrderItemIdentifier;
    }

    public List<ItemCapitalAsset> getPurApItemAssets() {
        return purApItemAssets;
    }

    public void setPurApItemAssets(List<ItemCapitalAsset> capitalAssetNumbers) {
        this.purApItemAssets = capitalAssetNumbers;
    }

    public ItemCapitalAsset getPurApItemAsset(int index) {
        while (getPurApItemAssets().size() <= index) {
            getPurApItemAssets().add(new PurchaseOrderItemCapitalAsset());
        }
        return getPurApItemAssets().get(index);
    }

    public boolean isCreateAssetIndicator() {
        return createAssetIndicator;
    }

    public void setCreateAssetIndicator(boolean createAssetIndicator) {
        this.createAssetIndicator = createAssetIndicator;
    }

    public boolean isApplyPaymentIndicator() {
        return applyPaymentIndicator;
    }

    public void setApplyPaymentIndicator(boolean applyPaymentIndicator) {
        this.applyPaymentIndicator = applyPaymentIndicator;
    }

    public boolean isSelectedValue() {
        return selectedValue;
    }

    public void setSelectedValue(boolean selectedValue) {
        this.selectedValue = selectedValue;
    }

    public String getItemTypeCode() {
        return itemTypeCode;
    }

    public void setItemTypeCode(String itemTypeCode) {
        this.itemTypeCode = itemTypeCode;
    }

    public String getCapitalAssetTransactionTypeCode() {
        return capitalAssetTransactionTypeCode;
    }

    public void setCapitalAssetTransactionTypeCode(String capitalAssetTransactionTypeCode) {
        this.capitalAssetTransactionTypeCode = capitalAssetTransactionTypeCode;
    }

    public boolean isAdditionalChargeNonTradeInIndicator() {
        return additionalChargeNonTradeInIndicator;
    }

    public void setAdditionalChargeNonTradeInIndicator(boolean additionalChargeNonTradeInIndicator) {
        this.additionalChargeNonTradeInIndicator = additionalChargeNonTradeInIndicator;
    }

    public boolean isTradeInAllowance() {
        return tradeInAllowance;
    }

    public void setTradeInAllowance(boolean tradeInAllowance) {
        this.tradeInAllowance = tradeInAllowance;
    }

    public KualiDecimal getSplitQty() {
        return splitQty;
    }

    public void setSplitQty(KualiDecimal splitQty) {
        this.splitQty = splitQty;
    }

    public List<PurchasingAccountsPayableLineAssetAccount> getPurchasingAccountsPayableLineAssetAccounts() {
        return purchasingAccountsPayableLineAssetAccounts;
    }

    public void setPurchasingAccountsPayableLineAssetAccounts(
            List<PurchasingAccountsPayableLineAssetAccount> purchasingAccountsPayableLineAssetAccounts) {
        this.purchasingAccountsPayableLineAssetAccounts = purchasingAccountsPayableLineAssetAccounts;
    }

    public String getDocumentNumber() {
        return documentNumber;
    }

    public void setDocumentNumber(String documentNumber) {
        this.documentNumber = documentNumber;
    }

    public Integer getAccountsPayableLineItemIdentifier() {
        return accountsPayableLineItemIdentifier;
    }

    public void setAccountsPayableLineItemIdentifier(Integer accountsPayableLineItemIdentifier) {
        this.accountsPayableLineItemIdentifier = accountsPayableLineItemIdentifier;
    }

    public String getAccountsPayableLineItemDescription() {
        return accountsPayableLineItemDescription;
    }

    public void setAccountsPayableLineItemDescription(String accountsPayableLineItemDescription) {
        this.accountsPayableLineItemDescription = accountsPayableLineItemDescription;
    }

    public KualiDecimal getAccountsPayableItemQuantity() {
        return accountsPayableItemQuantity;
    }

    public void setAccountsPayableItemQuantity(KualiDecimal accountsPayableItemQuantity) {
        this.accountsPayableItemQuantity = accountsPayableItemQuantity;
    }

    public Integer getCapitalAssetBuilderLineNumber() {
        return capitalAssetBuilderLineNumber;
    }

    public void setCapitalAssetBuilderLineNumber(Integer capitalAssetBuilderLineNumber) {
        this.capitalAssetBuilderLineNumber = capitalAssetBuilderLineNumber;
    }

    public String getCapitalAssetManagementDocumentNumber() {
        return capitalAssetManagementDocumentNumber;
    }

    public void setCapitalAssetManagementDocumentNumber(String capitalAssetManagementDocumentNumber) {
        this.capitalAssetManagementDocumentNumber = capitalAssetManagementDocumentNumber;
    }

    public Integer getItemLineNumber() {
        return itemLineNumber;
    }

    public void setItemLineNumber(Integer itemLineNumber) {
        this.itemLineNumber = itemLineNumber;
    }

    public boolean isActive() {
        return CamsConstants.ActivityStatusCode.NEW.equalsIgnoreCase(this.getActivityStatusCode()) ||
            CamsConstants.ActivityStatusCode.MODIFIED.equalsIgnoreCase(this.getActivityStatusCode());
    }

    public String getActivityStatusCode() {
        return activityStatusCode;
    }

    public void setActivityStatusCode(String activityStatusCode) {
        this.activityStatusCode = activityStatusCode;
    }

    public PurchasingAccountsPayableDocument getPurchasingAccountsPayableDocument() {
        return purchasingAccountsPayableDocument;
    }

    public void setPurchasingAccountsPayableDocument(
            PurchasingAccountsPayableDocument purchasingAccountsPayableDocument) {
        this.purchasingAccountsPayableDocument = purchasingAccountsPayableDocument;
    }

    public boolean isItemAssignedToTradeInIndicator() {
        return itemAssignedToTradeInIndicator;
    }

    public void setItemAssignedToTradeInIndicator(boolean itemAssignedToTradeInIndicator) {
        this.itemAssignedToTradeInIndicator = itemAssignedToTradeInIndicator;
    }

    public KualiDecimal getUnitCost() {
        return unitCost;
    }

    public void setUnitCost(KualiDecimal unitCost) {
        this.unitCost = unitCost;
    }

    public KualiDecimal getTotalCost() {
        return totalCost;
    }

    public void setTotalCost(KualiDecimal totalCost) {
        this.totalCost = totalCost;
    }

    public String getFirstFincialObjectCode() {
        return firstFincialObjectCode;
    }

    public void setFirstFincialObjectCode(String firstFincialObjectCode) {
        this.firstFincialObjectCode = firstFincialObjectCode;
    }

    public String getPreTagInquiryUrl() {
        if (StringUtils.isNotBlank(this.preTagInquiryUrl)) {
            return preTagInquiryUrl;
        }

        if (ObjectUtils.isNotNull(this.getPurchasingAccountsPayableDocument())) {
            Integer purchaseOrderIdentifier = this.getPurchasingAccountsPayableDocument().getPurchaseOrderIdentifier();

            PurApLineService purApLineService = SpringContext.getBean(PurApLineService.class);
            if (purApLineService.isPretaggingExisting(purApLineService.getPreTagLineItem(purchaseOrderIdentifier,
                    this.getItemLineNumber()))) {
                String baseUrl = KFSConstants.INQUIRY_ACTION;
                Properties parameters = new Properties();
                parameters.put(KFSConstants.DISPATCH_REQUEST_PARAMETER, KFSConstants.START_METHOD);
                parameters.put(KFSConstants.BUSINESS_OBJECT_CLASS_ATTRIBUTE, Pretag.class.getName());
                parameters.put(CamsPropertyConstants.Pretag.PURCHASE_ORDER_NUMBER, purchaseOrderIdentifier.toString());
                parameters.put(CamsPropertyConstants.Pretag.ITEM_LINE_NUMBER, this.getItemLineNumber().toString());

                this.preTagInquiryUrl = UrlFactory.parameterizeUrl(baseUrl, parameters);

                return this.preTagInquiryUrl;
            }
        }
        return "";
    }

    @Override
    public int compareTo(PurchasingAccountsPayableItemAsset o) {
        boolean o1ItemTypeBelowTheLine = this.isAdditionalChargeNonTradeInIndicator() || this.isTradeInAllowance();
        boolean o2ItemTypeBelowTheLine = o.isAdditionalChargeNonTradeInIndicator() || o.isTradeInAllowance();
        if (o1ItemTypeBelowTheLine && !o2ItemTypeBelowTheLine) {
            return 1;
        } else if (o2ItemTypeBelowTheLine && !o1ItemTypeBelowTheLine) {
            return -1;
        }
        return 0;
    }

    public List<Long> getApprovedAssetNumbers() {
        if (this.approvedAssetNumbers != null && !this.approvedAssetNumbers.isEmpty()) {
            return this.approvedAssetNumbers;
        } else {
            this.approvedAssetNumbers = new ArrayList<>();
            if (!StringUtils.isEmpty(this.getCapitalAssetManagementDocumentNumber())) {
                Map<String, String> fieldValues = new HashMap<>();
                if (CamsConstants.ActivityStatusCode.PROCESSED_IN_CAMS.equalsIgnoreCase(this.getActivityStatusCode())) {
                    // get asset number from asset global add doc
                    fieldValues.put(CamsPropertyConstants.AssetGlobalDetail.DOCUMENT_NUMBER,
                            this.getCapitalAssetManagementDocumentNumber());
                    Collection<AssetGlobalDetail> assetGlobalDetails = SpringContext.getBean(
                            BusinessObjectService.class).findMatching(AssetGlobalDetail.class, fieldValues);
                    for (AssetGlobalDetail detail : assetGlobalDetails) {
                        this.approvedAssetNumbers.add(detail.getCapitalAssetNumber());
                    }
                    if (assetGlobalDetails.isEmpty()) {
                        // get asset number from asset payment doc
                        fieldValues.clear();
                        fieldValues.put(CamsPropertyConstants.DOCUMENT_NUMBER,
                                this.getCapitalAssetManagementDocumentNumber());
                        Collection<AssetPaymentAssetDetail> paymentAssetDetails = SpringContext.getBean(
                                BusinessObjectService.class).findMatching(AssetPaymentAssetDetail.class, fieldValues);
                        for (AssetPaymentAssetDetail detail : paymentAssetDetails) {
                            this.approvedAssetNumbers.add(detail.getCapitalAssetNumber());
                        }
                    }
                }
            }
            return this.approvedAssetNumbers;
        }
    }

    private DataDictionaryService getDataDictionaryService() {
        return SpringContext.getBean(DataDictionaryService.class);
    }

    public PurchasingAccountsPayableLineAssetAccount getPurchasingAccountsPayableLineAssetAccount(int index) {
        int size = getPurchasingAccountsPayableLineAssetAccounts().size();
        while (size <= index || getPurchasingAccountsPayableLineAssetAccounts().get(index) == null) {
            getPurchasingAccountsPayableLineAssetAccounts().add(size++, new PurchasingAccountsPayableLineAssetAccount());
        }
        return getPurchasingAccountsPayableLineAssetAccounts().get(index);

    }

    public Integer getPaymentRequestIdentifier() {
        return paymentRequestIdentifier;
    }

    public void setPaymentRequestIdentifier(Integer paymentRequestIdentifier) {
        this.paymentRequestIdentifier = paymentRequestIdentifier;
    }

    /**
     * KFSCNTRB-1676/FSKD-5487
     * Returns true if it is an active TRDI additional charge asset line.
     */
    public boolean isActiveAdditionalTRDI() {
        return isTradeInAllowance() && isActive();
    }

    /**
     * KFSCNTRB-1676/FSKD-5487
     * Returns true if it is an active non-TRDI additional charge asset line.
     */
    public boolean isActiveAdditionalNonTRDI() {
        return isAdditionalChargeNonTradeInIndicator() && isActive();
    }

    /**
     * KFSCNTRB-1676/FSKD-5487
     * Returns true if it is an active trade-in ITEM asset line.
     */
    public boolean isActiveItemTradeIn() {
        return !isAdditionalChargeNonTradeInIndicator() && !isTradeInAllowance() && isItemAssignedToTradeInIndicator()
                && isActive();
    }

    /**
     * KFSCNTRB-1676/FSKD-5487
     * Returns true if it is an active ITEM asset line.
     */
    public boolean isActiveItemLine() {
        return !isAdditionalChargeNonTradeInIndicator() && !isTradeInAllowance() && isActive();
    }

}
