/**
 * The Kuali Financial System, a comprehensive financial management system for higher education.
 *
 * Copyright 2005-2019 Kuali, Inc.
 *
 * This program is free software: you can redistribute it and/or modify
 * it under the terms of the GNU Affero General Public License as published by
 * the Free Software Foundation, either version 3 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU Affero General Public License for more details.
 *
 * You should have received a copy of the GNU Affero General Public License
 * along with this program.  If not, see <http://www.gnu.org/licenses/>.
 */
package org.kuali.kfs.module.cam.util.distribution;

import org.kuali.kfs.module.cam.businessobject.AssetPaymentAssetDetail;
import org.kuali.kfs.module.cam.businessobject.AssetPaymentDetail;
import org.kuali.kfs.module.cam.document.AssetPaymentDocument;
import org.kuali.kfs.sys.KFSPropertyConstants;
import org.kuali.rice.core.api.util.type.KualiDecimal;

import java.util.HashMap;
import java.util.Map;

/**
 * No operation needed. This implementation does no distribution since the user will perform this manually.
 */
public class AssetDistributionManual extends AssetDistribution {

    /**
     * @param doc
     */
    public AssetDistributionManual(AssetPaymentDocument doc) {
        super(doc);
    }

    public Map<String, Map<AssetPaymentAssetDetail, KualiDecimal>> getAssetPaymentDistributions() {
        Map<String, Map<AssetPaymentAssetDetail, KualiDecimal>> distributionResult = new HashMap<>();

        KualiDecimal totalLineAmount = getTotalLineAmount();
        for (AssetPaymentDetail line : getAssetPaymentDetailLines()) {
            KualiDecimal lineAmount = line.getAmount();
            KualiDecimal remainingAmount = lineAmount;
            Map<AssetPaymentAssetDetail, KualiDecimal> apadMap = new HashMap<>();
            int size = doc.getAssetPaymentAssetDetail().size();
            for (int i = 0; i < size; i++) {
                AssetPaymentAssetDetail apad = doc.getAssetPaymentAssetDetail().get(i);

                if (i < size - 1) {
                    double allocationPercentage = determineAllocationPercentage(totalLineAmount, lineAmount,
                            apad.getAllocatedUserValue());
                    KualiDecimal amount = new KualiDecimal(allocationPercentage * lineAmount.doubleValue());
                    apadMap.put(apad, amount);
                    remainingAmount = remainingAmount.subtract(amount);
                } else {
                    apadMap.put(apad, remainingAmount);
                }
            }

            String assetPaymentDetailKey = line.getAssetPaymentDetailKey();
            Map<AssetPaymentAssetDetail, KualiDecimal> existingMap = findExistingMap(distributionResult,
                    assetPaymentDetailKey);
            if (existingMap == null) {
                distributionResult.put(assetPaymentDetailKey, apadMap);
            } else {
                addAmountToExistingAssetPaymentAssetDetail(apadMap, existingMap);
            }
        }

        return distributionResult;
    }

    /**
     * Determine if there is an existing result with the same account string as a key, disregarding the sequence.
     * This is important for when there are multiple accounting lines with the same account string and amounts that
     * add up to zero.
     *
     * @param distributionResult
     * @param assetPaymentDetailKey
     * @return
     */
    private Map<AssetPaymentAssetDetail, KualiDecimal> findExistingMap(Map<String,
            Map<AssetPaymentAssetDetail, KualiDecimal>> distributionResult, String assetPaymentDetailKey) {
        assetPaymentDetailKey = escapeRegexAndGenerifySequence(assetPaymentDetailKey);

        for (String key: distributionResult.keySet()) {
            if (key.matches(assetPaymentDetailKey)) {
                return distributionResult.get(key);
            }
        }

        return null;
    }

    private String escapeRegexAndGenerifySequence(String assetPaymentDetailKey) {
        assetPaymentDetailKey = assetPaymentDetailKey.replace("{", "\\{");
        assetPaymentDetailKey = assetPaymentDetailKey.replace("}", "\\}");
        assetPaymentDetailKey = assetPaymentDetailKey.replaceFirst(KFSPropertyConstants.SEQUENCE_NUMBER + "=\\d+", KFSPropertyConstants.SEQUENCE_NUMBER + "=\\\\d+");
        return assetPaymentDetailKey;
    }

    protected double determineAllocationPercentage(KualiDecimal totalLineAmount, KualiDecimal lineAmount,
            KualiDecimal allocatedUserValue) {
        double allocationPercentage = 0d;

        if (totalLineAmount.isNonZero()) {
            allocationPercentage = allocatedUserValue.doubleValue() / totalLineAmount.doubleValue();
        } else if (allocatedUserValue.equals(lineAmount)) {
            allocationPercentage = 1d;
        }

        return allocationPercentage;
    }

    protected void addAmountToExistingAssetPaymentAssetDetail(Map<AssetPaymentAssetDetail, KualiDecimal> newMap,
            Map<AssetPaymentAssetDetail, KualiDecimal> existingMap) {
        for (AssetPaymentAssetDetail existingApad : existingMap.keySet()) {
            KualiDecimal newAmount = newMap.get(existingApad);
            KualiDecimal existingAmount = existingMap.get(existingApad);

            existingMap.put(existingApad, newAmount.add(existingAmount));
        }
    }

    /**
     * Get the total amount from all accounting lines.
     *
     * @return
     */
    private KualiDecimal getTotalLineAmount() {
        KualiDecimal result = KualiDecimal.ZERO;
        for (AssetPaymentDetail sourceLine : getAssetPaymentDetailLines()) {
            result = result.add(sourceLine.getAmount());
        }
        return result;
    }

    public Map<AssetPaymentAssetDetail, KualiDecimal> getTotalAssetAllocations() {
        Map<AssetPaymentAssetDetail, KualiDecimal> assetTotalAllocationMap = new HashMap<>();

        for (AssetPaymentAssetDetail apad : doc.getAssetPaymentAssetDetail()) {
            assetTotalAllocationMap.put(apad, apad.getAllocatedAmount());
        }

        return assetTotalAllocationMap;
    }

}
