/*
 * The Kuali Financial System, a comprehensive financial management system for higher education.
 *
 * Copyright 2005-2021 Kuali, Inc.
 *
 * This program is free software: you can redistribute it and/or modify
 * it under the terms of the GNU Affero General Public License as published by
 * the Free Software Foundation, either version 3 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU Affero General Public License for more details.
 *
 * You should have received a copy of the GNU Affero General Public License
 * along with this program.  If not, see <http://www.gnu.org/licenses/>.
 */
package org.kuali.kfs.module.cam.businessobject;

import org.kuali.kfs.krad.bo.PersistableBusinessObjectBase;
import org.kuali.kfs.krad.util.ObjectUtils;
import org.kuali.kfs.sys.businessobject.FinancialSystemDocumentHeader;
import org.kuali.rice.core.api.util.type.KualiDecimal;

import java.math.BigDecimal;
import java.util.ArrayList;
import java.util.List;
import java.util.Objects;

public class AssetPaymentAssetDetail extends PersistableBusinessObjectBase {

    private String documentNumber;
    private Long capitalAssetNumber;
    private KualiDecimal previousTotalCostAmount;
    private KualiDecimal allocatedAmount = KualiDecimal.ZERO;
    private KualiDecimal allocatedUserValue = KualiDecimal.ZERO;
    private BigDecimal allocatedUserValuePct = BigDecimal.ZERO;

    private Asset asset;
    private List<AssetPaymentDetail> assetPaymentDetails;
    private FinancialSystemDocumentHeader documentHeader;

    public AssetPaymentAssetDetail() {
        this.assetPaymentDetails = new ArrayList<>();
        this.documentHeader = new FinancialSystemDocumentHeader();
    }

    public Long getCapitalAssetNumber() {
        return capitalAssetNumber;
    }

    public void setCapitalAssetNumber(Long capitalAssetNumber) {
        this.capitalAssetNumber = capitalAssetNumber;
    }

    public String getDocumentNumber() {
        return documentNumber;
    }

    public void setDocumentNumber(String documentNumber) {
        this.documentNumber = documentNumber;
    }

    public KualiDecimal getPreviousTotalCostAmount() {
        return previousTotalCostAmount;
    }

    public void setPreviousTotalCostAmount(KualiDecimal previousTotalCostAmount) {
        this.previousTotalCostAmount = previousTotalCostAmount;
    }

    public Asset getAsset() {
        return asset;
    }

    public void setAsset(Asset asset) {
        this.asset = asset;
    }

    public List<AssetPaymentDetail> getAssetPaymentDetails() {
        return assetPaymentDetails;
    }

    public void setAssetPaymentDetails(List<AssetPaymentDetail> assetPaymentDetails) {
        this.assetPaymentDetails = assetPaymentDetails;
    }

    public FinancialSystemDocumentHeader getDocumentHeader() {
        return documentHeader;
    }

    public void setDocumentHeader(FinancialSystemDocumentHeader documentHeader) {
        this.documentHeader = documentHeader;
    }

    public KualiDecimal getAllocatedAmount() {
        return allocatedAmount;
    }

    public void setAllocatedAmount(KualiDecimal allocatedAmount) {
        this.allocatedAmount = Objects.requireNonNullElse(allocatedAmount, KualiDecimal.ZERO);
    }

    /**
     * Set the value the user allocates when editable
     */
    public void setAllocatedUserValue(KualiDecimal allocatedUserValue) {
        this.allocatedUserValue = Objects.requireNonNullElse(allocatedUserValue, KualiDecimal.ZERO);
        setAllocatedAmount(allocatedUserValue);
    }

    /**
     * Get the value the user allocates when editable
     */
    public KualiDecimal getAllocatedUserValue() {
        return allocatedUserValue;
    }

    /**
     * @return the New total allocation amount
     */
    public KualiDecimal getNewTotal() {
        KualiDecimal previousCostAmount = getPreviousTotalCostAmount();
        // KFSCNTRB-1667: if previous cost doesn't exist, regard the previous total as 0 amount
        if (ObjectUtils.isNull(previousCostAmount)) {
            previousCostAmount = new KualiDecimal(0);
        }
        return getAllocatedAmount().add(previousCostAmount);
    }

    /**
     * @return the percent invariant value if percentages are used
     */
    public BigDecimal getAllocatedUserValuePct() {
        return allocatedUserValuePct;
    }

    /**
     * Sets the percent invariant value if percentages are used
     */
    public void setAllocatedUserValuePct(BigDecimal allocatedUserValuePct) {
        this.allocatedUserValuePct = allocatedUserValuePct;
    }
}
