/*
 * The Kuali Financial System, a comprehensive financial management system for higher education.
 *
 * Copyright 2005-2021 Kuali, Inc.
 *
 * This program is free software: you can redistribute it and/or modify
 * it under the terms of the GNU Affero General Public License as published by
 * the Free Software Foundation, either version 3 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU Affero General Public License for more details.
 *
 * You should have received a copy of the GNU Affero General Public License
 * along with this program.  If not, see <http://www.gnu.org/licenses/>.
 */
package org.kuali.kfs.module.cam.businessobject;

import org.apache.commons.lang3.StringUtils;
import org.kuali.kfs.krad.bo.PersistableBusinessObjectBase;
import org.kuali.kfs.krad.service.BusinessObjectService;
import org.kuali.kfs.krad.util.ObjectUtils;
import org.kuali.kfs.module.cam.CamsConstants;
import org.kuali.kfs.module.cam.CamsPropertyConstants;
import org.kuali.kfs.module.purap.document.PaymentRequestDocument;
import org.kuali.kfs.module.purap.document.VendorCreditMemoDocument;
import org.kuali.kfs.sys.businessobject.FinancialSystemDocumentHeader;
import org.kuali.kfs.sys.context.SpringContext;
import org.kuali.rice.kew.api.KewApiServiceLocator;
import org.kuali.rice.kew.api.doctype.DocumentType;
import org.kuali.rice.kew.doctype.bo.DocumentTypeEBO;

import java.util.ArrayList;
import java.util.HashMap;
import java.util.List;
import java.util.Map;

public class PurchasingAccountsPayableDocument extends PersistableBusinessObjectBase {

    private String documentNumber;
    private Integer purapDocumentIdentifier;
    private Integer purchaseOrderIdentifier;
    private String documentTypeCode;
    private String activityStatusCode;

    // References
    private DocumentTypeEBO financialSystemDocumentTypeCode;
    private FinancialSystemDocumentHeader documentHeader;
    private List<PurchasingAccountsPayableItemAsset> purchasingAccountsPayableItemAssets;

    // non-persistent
    private boolean active;
    private String purApContactEmailAddress;
    private String purApContactPhoneNumber;
    private String statusDescription;
    private String capitalAssetSystemTypeCodeFromPurAp;

    public PurchasingAccountsPayableDocument() {
        this.purchasingAccountsPayableItemAssets = new ArrayList<>();
    }

    public String getCapitalAssetSystemTypeCodeFromPurAp() {
        return capitalAssetSystemTypeCodeFromPurAp;
    }

    public void setCapitalAssetSystemTypeCodeFromPurAp(String capitalAssetSystemTypeCodeFromPurAp) {
        this.capitalAssetSystemTypeCodeFromPurAp = capitalAssetSystemTypeCodeFromPurAp;
    }

    public String getDocumentNumber() {
        return documentNumber;
    }

    public void setDocumentNumber(String documentNumber) {
        this.documentNumber = documentNumber;
    }

    public Integer getPurapDocumentIdentifier() {
        return purapDocumentIdentifier;
    }

    public void setPurapDocumentIdentifier(Integer purapDocumentIdentifier) {
        this.purapDocumentIdentifier = purapDocumentIdentifier;
    }

    public Integer getPurchaseOrderIdentifier() {
        return purchaseOrderIdentifier;
    }

    public void setPurchaseOrderIdentifier(Integer purchaseOrderIdentifier) {
        this.purchaseOrderIdentifier = purchaseOrderIdentifier;
    }

    public String getDocumentTypeCode() {
        return documentTypeCode;
    }

    public void setDocumentTypeCode(String documentTypeCode) {
        this.documentTypeCode = documentTypeCode;
    }

    public boolean isActive() {
        return CamsConstants.ActivityStatusCode.NEW.equalsIgnoreCase(this.getActivityStatusCode())
                || CamsConstants.ActivityStatusCode.MODIFIED.equalsIgnoreCase(this.getActivityStatusCode());
    }

    public String getActivityStatusCode() {
        return activityStatusCode;
    }

    public void setActivityStatusCode(String activityStatusCode) {
        this.activityStatusCode = activityStatusCode;
    }

    public DocumentTypeEBO getFinancialSystemDocumentTypeCode() {
        if (financialSystemDocumentTypeCode == null || !StringUtils.equals(financialSystemDocumentTypeCode.getName(),
                documentTypeCode)) {
            financialSystemDocumentTypeCode = null;
            if (StringUtils.isNotBlank(documentTypeCode)) {
                DocumentType docType = KewApiServiceLocator.getDocumentTypeService()
                        .getDocumentTypeByName(documentTypeCode);
                if (docType != null) {
                    financialSystemDocumentTypeCode = org.kuali.rice.kew.doctype.bo.DocumentType.from(docType);
                }
            }
        }
        return financialSystemDocumentTypeCode;
    }

    public FinancialSystemDocumentHeader getDocumentHeader() {
        return documentHeader;
    }

    public void setDocumentHeader(FinancialSystemDocumentHeader documentHeader) {
        this.documentHeader = documentHeader;
    }

    public List<PurchasingAccountsPayableItemAsset> getPurchasingAccountsPayableItemAssets() {
        return purchasingAccountsPayableItemAssets;
    }

    public void setPurchasingAccountsPayableItemAssets(
            List<PurchasingAccountsPayableItemAsset> purchasingAccountsPayableItemAssets) {
        this.purchasingAccountsPayableItemAssets = purchasingAccountsPayableItemAssets;
    }

    public String getPurApContactEmailAddress() {
        return purApContactEmailAddress;
    }

    public void setPurApContactEmailAddress(String purApContactEmailAddress) {
        this.purApContactEmailAddress = purApContactEmailAddress;
    }

    public String getPurApContactPhoneNumber() {
        return purApContactPhoneNumber;
    }

    public void setPurApContactPhoneNumber(String purApContactPhoneNumber) {
        this.purApContactPhoneNumber = purApContactPhoneNumber;
    }

    public String getStatusDescription() {
        if (StringUtils.isNotBlank(this.statusDescription)) {
            return this.statusDescription;
        } else {
            Map<String, Object> objectKeys = new HashMap<>();
            objectKeys.put(CamsPropertyConstants.PurchasingAccountsPayableDocument.PURAP_DOCUMENT_IDENTIFIER,
                    this.getPurapDocumentIdentifier());

            if (CamsConstants.PREQ.equals(this.documentTypeCode)) {
                PaymentRequestDocument paymentRequestDocument = SpringContext.getBean(BusinessObjectService.class)
                        .findByPrimaryKey(PaymentRequestDocument.class, objectKeys);
                if (ObjectUtils.isNotNull(paymentRequestDocument)) {
                    statusDescription = paymentRequestDocument.getApplicationDocumentStatus();
                }
            } else {
                VendorCreditMemoDocument vendorCreditMemoDocument = SpringContext.getBean(BusinessObjectService.class)
                        .findByPrimaryKey(VendorCreditMemoDocument.class, objectKeys);
                if (ObjectUtils.isNotNull(vendorCreditMemoDocument)) {
                    statusDescription = vendorCreditMemoDocument.getApplicationDocumentStatus();
                }
            }
        }

        return statusDescription;
    }

    public void setStatusDescription(String statusDescription) {
        this.statusDescription = statusDescription;
    }

    /**
     * Need to override this method, so we can save item assets, the framework can delete the allocated item assets.
     */
    @Override
    public List buildListOfDeletionAwareLists() {
        List<List> managedLists = new ArrayList<>();
        managedLists.add(getPurchasingAccountsPayableItemAssets());
        return managedLists;
    }

    public PurchasingAccountsPayableItemAsset getPurchasingAccountsPayableItemAsset(int index) {
        int size = getPurchasingAccountsPayableItemAssets().size();
        while (size <= index || getPurchasingAccountsPayableItemAssets().get(index) == null) {
            getPurchasingAccountsPayableItemAssets().add(size++, new PurchasingAccountsPayableItemAsset());
        }
        return getPurchasingAccountsPayableItemAssets().get(index);
    }

}
