/*
 * The Kuali Financial System, a comprehensive financial management system for higher education.
 *
 * Copyright 2005-2022 Kuali, Inc.
 *
 * This program is free software: you can redistribute it and/or modify
 * it under the terms of the GNU Affero General Public License as published by
 * the Free Software Foundation, either version 3 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU Affero General Public License for more details.
 *
 * You should have received a copy of the GNU Affero General Public License
 * along with this program.  If not, see <http://www.gnu.org/licenses/>.
 */
package org.kuali.kfs.module.cam.businessobject;

import org.apache.commons.lang3.builder.ToStringBuilder;
import org.apache.commons.lang3.builder.ToStringStyle;
import org.kuali.kfs.coa.businessobject.Account;
import org.kuali.kfs.coa.businessobject.AccountingPeriod;
import org.kuali.kfs.core.api.util.type.KualiDecimal;
import org.kuali.kfs.kew.doctype.bo.DocumentType;
import org.kuali.kfs.kew.doctype.service.DocumentTypeService;
import org.kuali.kfs.module.cam.CamsPropertyConstants;
import org.kuali.kfs.sys.businessobject.OriginationCode;
import org.kuali.kfs.sys.businessobject.SourceAccountingLine;
import org.kuali.kfs.sys.context.SpringContext;

import java.sql.Date;
import java.util.LinkedHashMap;
import java.util.Map;

/**
 * Accounting line for the asset payment document.
 */
public class AssetPaymentDetail extends SourceAccountingLine {

    private String expenditureFinancialSystemOriginationCode;
    private Date expenditureFinancialDocumentPostedDate;
    private boolean transferPaymentIndicator;

    private String expenditureFinancialDocumentNumber;
    private String expenditureFinancialDocumentTypeCode;
    private String postingPeriodCode;
    private String purchaseOrderNumber;
    private String requisitionNumber;
    private KualiDecimal amount;

    // bo references
    private AccountingPeriod financialDocumentPostingPeriod;
    private DocumentType expenditureFinancialSystemDocumentType;
    private OriginationCode expenditureFinancialSystemOrigination;
    private Account account;

    public AssetPaymentDetail() {
        super();
    }

    public AssetPaymentDetail(AssetPayment assetPayment) {
        // populate AssetPaymentDetail with AssetPayment data
        this.setSequenceNumber(assetPayment.getPaymentSequenceNumber());
        this.setChartOfAccountsCode(assetPayment.getChartOfAccountsCode());
        this.setAccountNumber(assetPayment.getAccountNumber());
        this.setSubAccountNumber(assetPayment.getSubAccountNumber());
        this.setFinancialObjectCode(assetPayment.getFinancialObjectCode());
        this.setFinancialSubObjectCode(assetPayment.getFinancialSubObjectCode());
        this.setProjectCode(assetPayment.getProjectCode());
        this.setOrganizationReferenceId(assetPayment.getOrganizationReferenceId());
        this.setExpenditureFinancialSystemOriginationCode(assetPayment.getFinancialSystemOriginationCode());
        this.setExpenditureFinancialDocumentNumber(assetPayment.getDocumentNumber());
        this.setExpenditureFinancialDocumentTypeCode(assetPayment.getFinancialDocumentTypeCode());
        this.setPurchaseOrderNumber(assetPayment.getPurchaseOrderNumber());
        this.setRequisitionNumber(assetPayment.getRequisitionNumber());
        this.setExpenditureFinancialDocumentPostedDate(assetPayment.getFinancialDocumentPostingDate());
        this.setPostingYear(assetPayment.getFinancialDocumentPostingYear());
        this.setPostingPeriodCode(assetPayment.getFinancialDocumentPostingPeriodCode());
        this.setAmount(assetPayment.getAccountChargeAmount());
    }

    protected LinkedHashMap<String, String> assetPaymentToStringMapper() {
        LinkedHashMap<String, String> m = new LinkedHashMap<>();
        m.put("documentNumber", this.getDocumentNumber());
        m.put("sequenceNumber", this.getSequenceNumber() == null ? "" : this.getSequenceNumber().toString());
        m.put("chartOfAccountsCode", getChartOfAccountsCode());
        m.put("accountNumber", getAccountNumber());
        m.put("subAccountNumber", getSubAccountNumber());
        m.put("financialObjectCode", getFinancialObjectCode());
        m.put("financialSubObjectCode", getFinancialSubObjectCode());
        m.put("projectCode", getProjectCode());
        m.put("postingYear", this.getPostingYear().toString());
        return m;
    }

    /**
     * Create a key including the
     * <li><b>expenditureFinancialDocumentNumber</b></li>
     * <li><b>expenditureFinancialDocumentTypeCode</b></li>
     * with accounting information for asset payment distribution
     * <p>
     * Make sure the full accounting line information is part of the key
     * chartOfAccount, accountNumber, subAccountNumber, objectCode, subObjectCode, projectCode
     *
     * @return
     */
    public String getAssetPaymentDetailKey() {
        LinkedHashMap<String, String> paymentMap = assetPaymentToStringMapper();
        paymentMap.put("expenditureFinancialDocumentTypeCode", this.getExpenditureFinancialDocumentTypeCode());
        paymentMap.put("expenditureFinancialDocumentNumber", this.getExpenditureFinancialDocumentNumber());

        //use SHORT_PREFIX_STYLE so that memory address is not part of the toString output
        ToStringBuilder builder = new ToStringBuilder(this, ToStringStyle.SHORT_PREFIX_STYLE);
        for (String key : paymentMap.keySet()) {
            builder.append(key, paymentMap.get(key));
        }
        return paymentMap.toString();
    }

    public String getExpenditureFinancialSystemOriginationCode() {
        return expenditureFinancialSystemOriginationCode;
    }

    public void setExpenditureFinancialSystemOriginationCode(String expenditureFinancialSystemOriginationCode) {
        this.expenditureFinancialSystemOriginationCode = expenditureFinancialSystemOriginationCode;
    }

    public Date getExpenditureFinancialDocumentPostedDate() {
        return expenditureFinancialDocumentPostedDate;
    }

    public void setExpenditureFinancialDocumentPostedDate(Date expenditureFinancialDocumentPostedDate) {
        this.expenditureFinancialDocumentPostedDate = expenditureFinancialDocumentPostedDate;
    }

    public boolean isTransferPaymentIndicator() {
        return transferPaymentIndicator;
    }

    public void setTransferPaymentIndicator(boolean transferPaymentIndicator) {
        this.transferPaymentIndicator = transferPaymentIndicator;
    }

    public String getExpenditureFinancialDocumentNumber() {
        return expenditureFinancialDocumentNumber;
    }

    public void setExpenditureFinancialDocumentNumber(String expenditureFinancialDocumentNumber) {
        this.expenditureFinancialDocumentNumber = expenditureFinancialDocumentNumber;
    }

    public String getExpenditureFinancialDocumentTypeCode() {
        return expenditureFinancialDocumentTypeCode;
    }

    public void setExpenditureFinancialDocumentTypeCode(String expenditureFinancialDocumentTypeCode) {
        this.expenditureFinancialDocumentTypeCode = expenditureFinancialDocumentTypeCode;
    }

    public String getPostingPeriodCode() {
        return postingPeriodCode;
    }

    public void setPostingPeriodCode(String postingPeriodCode) {
        this.postingPeriodCode = postingPeriodCode;
    }

    public String getPurchaseOrderNumber() {
        return purchaseOrderNumber;
    }

    public void setPurchaseOrderNumber(String purchaseOrderNumber) {
        this.purchaseOrderNumber = purchaseOrderNumber;
    }

    public String getRequisitionNumber() {
        return requisitionNumber;
    }

    public void setRequisitionNumber(String requisitionNumber) {
        this.requisitionNumber = requisitionNumber;
    }

    public AccountingPeriod getFinancialDocumentPostingPeriod() {
        return financialDocumentPostingPeriod;
    }

    public void setFinancialDocumentPostingPeriod(AccountingPeriod financialDocumentPostingPeriod) {
        this.financialDocumentPostingPeriod = financialDocumentPostingPeriod;
    }

    public DocumentType getExpenditureFinancialSystemDocumentType() {
        expenditureFinancialSystemDocumentType = SpringContext.getBean(DocumentTypeService.class)
                .updateDocumentTypeIfNecessary(expenditureFinancialDocumentTypeCode,
                        expenditureFinancialSystemDocumentType);
        return expenditureFinancialSystemDocumentType;
    }

    public OriginationCode getExpenditureFinancialSystemOrigination() {
        return expenditureFinancialSystemOrigination;
    }

    public void setExpenditureFinancialSystemOrigination(OriginationCode expenditureFinancialSystemOrigination) {
        this.expenditureFinancialSystemOrigination = expenditureFinancialSystemOrigination;
    }

    /**
     * Gets the account attribute.
     *
     * @return Returns the account
     */
    @Override
    public Account getAccount() {
        return account;
    }

    /**
     * Sets the account attribute.
     *
     * @param account The account to set.
     * @deprecated
     */
    @Deprecated
    @Override
    public void setAccount(Account account) {
        this.account = account;
    }

    @Override
    public KualiDecimal getAmount() {
        return amount;
    }

    @Override
    public void setAmount(KualiDecimal amount) {
        this.amount = amount;
    }

    @Override
    public Map getValuesMap() {
        Map simpleValues = super.getValuesMap();
        simpleValues.put(CamsPropertyConstants.AssetPaymentDetail.ORIGINATION_CODE,
                getExpenditureFinancialSystemOriginationCode());
        simpleValues.put(CamsPropertyConstants.AssetPaymentDetail.DOCUMENT_POSTING_DATE,
                getExpenditureFinancialDocumentPostedDate());
        return simpleValues;
    }
}
