/*
 * The Kuali Financial System, a comprehensive financial management system for higher education.
 *
 * Copyright 2005-2023 Kuali, Inc.
 *
 * This program is free software: you can redistribute it and/or modify
 * it under the terms of the GNU Affero General Public License as published by
 * the Free Software Foundation, either version 3 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU Affero General Public License for more details.
 *
 * You should have received a copy of the GNU Affero General Public License
 * along with this program.  If not, see <http://www.gnu.org/licenses/>.
 */
package org.kuali.kfs.module.cam.businessobject;

import org.kuali.kfs.coa.businessobject.Account;
import org.kuali.kfs.coa.businessobject.Chart;
import org.kuali.kfs.coa.businessobject.ObjectCode;
import org.kuali.kfs.coa.businessobject.ObjectType;
import org.kuali.kfs.coa.businessobject.SubAccount;
import org.kuali.kfs.coa.businessobject.SubObjectCode;
import org.kuali.kfs.core.api.util.type.KualiDecimal;
import org.kuali.kfs.gl.businessobject.Entry;
import org.kuali.kfs.kew.doctype.bo.DocumentType;
import org.kuali.kfs.kew.doctype.service.DocumentTypeService;
import org.kuali.kfs.krad.bo.PersistableBusinessObjectBase;
import org.kuali.kfs.module.cam.CamsConstants;
import org.kuali.kfs.sys.KFSConstants;
import org.kuali.kfs.sys.context.SpringContext;

import java.sql.Date;
import java.sql.Timestamp;
import java.util.ArrayList;
import java.util.List;

public class GeneralLedgerEntry extends PersistableBusinessObjectBase {

    private Long generalLedgerAccountIdentifier;
    private Integer universityFiscalYear;
    private String chartOfAccountsCode;
    private String accountNumber;
    private String subAccountNumber;
    private String financialObjectCode;
    private String financialSubObjectCode;
    private String financialBalanceTypeCode;
    private String financialObjectTypeCode;
    private String universityFiscalPeriodCode;
    private String financialDocumentTypeCode;
    private String financialSystemOriginationCode;
    private String documentNumber;
    private Integer transactionLedgerEntrySequenceNumber;
    private String transactionLedgerEntryDescription;
    private KualiDecimal transactionLedgerEntryAmount;
    private KualiDecimal transactionLedgerSubmitAmount;
    private String organizationReferenceId;
    private String referenceFinancialSystemOriginationCode;
    private String referenceFinancialDocumentNumber;
    private String transactionDebitCreditCode;
    private String organizationDocumentNumber;
    private String projectCode;
    private Date transactionPostingDate;
    private Date transactionDate;
    private Timestamp transactionDateTimeStamp;
    private String activityStatusCode;

    // References
    private Account account;
    private Chart chart;
    private ObjectCode financialObject;
    private SubAccount subAccount;
    private SubObjectCode financialSubObject;
    private ObjectType objectType;
    private DocumentType financialSystemDocumentType;

    private List<GeneralLedgerEntryAsset> generalLedgerEntryAssets;
    private List<PurchasingAccountsPayableLineAssetAccount> purApLineAssetAccounts;
    // non-db fields
    private boolean selected;
    private KualiDecimal amount;
    private boolean active;

    public GeneralLedgerEntry() {
        generalLedgerEntryAssets = new ArrayList<>();
        purApLineAssetAccounts = new ArrayList<>();
    }

    /**
     * Constructs a GeneralLedgerEntry item from GL line entry
     *
     * @param entry GL Lines
     */
    public GeneralLedgerEntry(final Entry entry) {
        generalLedgerEntryAssets = new ArrayList<>();
        setUniversityFiscalYear(entry.getUniversityFiscalYear());
        setChartOfAccountsCode(entry.getChartOfAccountsCode());
        setAccountNumber(entry.getAccountNumber());
        setSubAccountNumber(entry.getSubAccountNumber());
        setFinancialObjectCode(entry.getFinancialObjectCode());
        setFinancialSubObjectCode(entry.getFinancialSubObjectCode());
        setFinancialBalanceTypeCode(entry.getFinancialBalanceTypeCode());
        setFinancialObjectTypeCode(entry.getFinancialObjectTypeCode());
        setUniversityFiscalPeriodCode(entry.getUniversityFiscalPeriodCode());
        setFinancialDocumentTypeCode(entry.getFinancialDocumentTypeCode());
        setFinancialSystemOriginationCode(entry.getFinancialSystemOriginationCode());
        setDocumentNumber(entry.getDocumentNumber());
        setTransactionLedgerEntrySequenceNumber(entry.getTransactionLedgerEntrySequenceNumber());
        setTransactionLedgerEntryDescription(entry.getTransactionLedgerEntryDescription());
        setTransactionLedgerEntryAmount(entry.getTransactionLedgerEntryAmount());
        setOrganizationReferenceId(entry.getOrganizationReferenceId());
        setReferenceFinancialSystemOriginationCode(entry.getReferenceFinancialSystemOriginationCode());
        setReferenceFinancialDocumentNumber(entry.getReferenceFinancialDocumentNumber());
        setTransactionDebitCreditCode(entry.getTransactionDebitCreditCode());
        setOrganizationDocumentNumber(entry.getOrganizationDocumentNumber());
        setProjectCode(entry.getProjectCode());
        setTransactionDate(entry.getTransactionDate());
        setTransactionPostingDate(entry.getTransactionPostingDate());
        setTransactionDateTimeStamp(entry.getTransactionDateTimeStamp());
        setActivityStatusCode(CamsConstants.ActivityStatusCode.NEW);
    }

    public Long getGeneralLedgerAccountIdentifier() {
        return generalLedgerAccountIdentifier;
    }

    public void setGeneralLedgerAccountIdentifier(final Long glAccountId) {
        generalLedgerAccountIdentifier = glAccountId;
    }

    public Integer getUniversityFiscalYear() {
        return universityFiscalYear;
    }

    public void setUniversityFiscalYear(final Integer universityFiscalYear) {
        this.universityFiscalYear = universityFiscalYear;
    }

    public String getChartOfAccountsCode() {
        return chartOfAccountsCode;
    }

    public void setChartOfAccountsCode(final String chartOfAccountsCode) {
        this.chartOfAccountsCode = chartOfAccountsCode;
    }

    public String getAccountNumber() {
        return accountNumber;
    }

    public void setAccountNumber(final String accountNumber) {
        this.accountNumber = accountNumber;
    }

    public String getSubAccountNumber() {
        return subAccountNumber;
    }

    public void setSubAccountNumber(final String subAccountNumber) {
        this.subAccountNumber = subAccountNumber;
    }

    public String getFinancialObjectCode() {
        return financialObjectCode;
    }

    public void setFinancialObjectCode(final String financialObjectCode) {
        this.financialObjectCode = financialObjectCode;
    }

    public String getFinancialSubObjectCode() {
        return financialSubObjectCode;
    }

    public void setFinancialSubObjectCode(final String financialSubObjectCode) {
        this.financialSubObjectCode = financialSubObjectCode;
    }

    public String getFinancialBalanceTypeCode() {
        return financialBalanceTypeCode;
    }

    public void setFinancialBalanceTypeCode(final String financialBalanceTypeCode) {
        this.financialBalanceTypeCode = financialBalanceTypeCode;
    }

    public String getFinancialObjectTypeCode() {
        return financialObjectTypeCode;
    }

    public void setFinancialObjectTypeCode(final String financialObjectTypeCode) {
        this.financialObjectTypeCode = financialObjectTypeCode;
    }

    public String getUniversityFiscalPeriodCode() {
        return universityFiscalPeriodCode;
    }

    public void setUniversityFiscalPeriodCode(final String universityFiscalPeriodCode) {
        this.universityFiscalPeriodCode = universityFiscalPeriodCode;
    }

    public String getFinancialDocumentTypeCode() {
        return financialDocumentTypeCode;
    }

    public void setFinancialDocumentTypeCode(final String financialDocumentTypeCode) {
        this.financialDocumentTypeCode = financialDocumentTypeCode;
    }

    public String getFinancialSystemOriginationCode() {
        return financialSystemOriginationCode;
    }

    public void setFinancialSystemOriginationCode(final String financialSystemOriginationCode) {
        this.financialSystemOriginationCode = financialSystemOriginationCode;
    }

    public String getDocumentNumber() {
        return documentNumber;
    }

    public void setDocumentNumber(final String documentNumber) {
        this.documentNumber = documentNumber;
    }

    public Integer getTransactionLedgerEntrySequenceNumber() {
        return transactionLedgerEntrySequenceNumber;
    }

    public void setTransactionLedgerEntrySequenceNumber(final Integer transactionLedgerEntrySequenceNumber) {
        this.transactionLedgerEntrySequenceNumber = transactionLedgerEntrySequenceNumber;
    }

    public String getTransactionLedgerEntryDescription() {
        return transactionLedgerEntryDescription;
    }

    public void setTransactionLedgerEntryDescription(final String transactionLedgerEntryDescription) {
        this.transactionLedgerEntryDescription = transactionLedgerEntryDescription;
    }

    public KualiDecimal getTransactionLedgerEntryAmount() {
        return transactionLedgerEntryAmount;
    }

    public void setTransactionLedgerEntryAmount(final KualiDecimal transactionLedgerEntryAmount) {
        this.transactionLedgerEntryAmount = transactionLedgerEntryAmount;
    }

    public String getOrganizationReferenceId() {
        return organizationReferenceId;
    }

    public void setOrganizationReferenceId(final String organizationReferenceId) {
        this.organizationReferenceId = organizationReferenceId;
    }

    public String getReferenceFinancialSystemOriginationCode() {
        return referenceFinancialSystemOriginationCode;
    }

    public void setReferenceFinancialSystemOriginationCode(final String referenceFinancialSystemOriginationCode) {
        this.referenceFinancialSystemOriginationCode = referenceFinancialSystemOriginationCode;
    }

    public String getReferenceFinancialDocumentNumber() {
        return referenceFinancialDocumentNumber;
    }

    public void setReferenceFinancialDocumentNumber(final String referenceFinancialDocumentNumber) {
        this.referenceFinancialDocumentNumber = referenceFinancialDocumentNumber;
    }

    public String getTransactionDebitCreditCode() {
        return transactionDebitCreditCode;
    }

    public void setTransactionDebitCreditCode(final String transactionDebitCreditCode) {
        this.transactionDebitCreditCode = transactionDebitCreditCode;
    }

    public String getOrganizationDocumentNumber() {
        return organizationDocumentNumber;
    }

    public void setOrganizationDocumentNumber(final String organizationDocumentNumber) {
        this.organizationDocumentNumber = organizationDocumentNumber;
    }

    public String getProjectCode() {
        return projectCode;
    }

    public void setProjectCode(final String projectCode) {
        this.projectCode = projectCode;
    }

    public Date getTransactionPostingDate() {
        return transactionPostingDate;
    }

    public void setTransactionPostingDate(final Date transactionPostingDate) {
        this.transactionPostingDate = transactionPostingDate;
    }

    public Timestamp getTransactionDateTimeStamp() {
        return transactionDateTimeStamp;
    }

    public void setTransactionDateTimeStamp(final Timestamp transactionDateTimeStamp) {
        this.transactionDateTimeStamp = transactionDateTimeStamp;
    }

    public Account getAccount() {
        return account;
    }

    public void setAccount(final Account account) {
        this.account = account;
    }

    public SubAccount getSubAccount() {
        return subAccount;
    }

    public void setSubAccount(final SubAccount subAccount) {
        this.subAccount = subAccount;
    }

    public Chart getChart() {
        return chart;
    }

    public void setChart(final Chart chart) {
        this.chart = chart;
    }

    public ObjectCode getFinancialObject() {
        return financialObject;
    }

    public void setFinancialObject(final ObjectCode financialObject) {
        this.financialObject = financialObject;
    }

    public SubObjectCode getFinancialSubObject() {
        return financialSubObject;
    }

    public void setFinancialSubObject(final SubObjectCode financialSubObject) {
        this.financialSubObject = financialSubObject;
    }

    public ObjectType getObjectType() {
        return objectType;
    }

    public void setObjectType(final ObjectType objectType) {
        this.objectType = objectType;
    }

    public DocumentType getFinancialSystemDocumentType() {
        financialSystemDocumentType = SpringContext.getBean(DocumentTypeService.class)
                .updateDocumentTypeIfNecessary(financialDocumentTypeCode, financialSystemDocumentType);
        return financialSystemDocumentType;
    }

    public boolean isActive() {
        return CamsConstants.ActivityStatusCode.NEW.equalsIgnoreCase(getActivityStatusCode())
                || CamsConstants.ActivityStatusCode.MODIFIED.equalsIgnoreCase(getActivityStatusCode());
    }

    public String getActivityStatusCode() {
        return activityStatusCode;
    }

    public void setActivityStatusCode(final String activityStatusCode) {
        this.activityStatusCode = activityStatusCode;
    }

    public List<GeneralLedgerEntryAsset> getGeneralLedgerEntryAssets() {
        return generalLedgerEntryAssets;
    }

    public void setGeneralLedgerEntryAssets(final List<GeneralLedgerEntryAsset> generalLedgerEntryAssets) {
        this.generalLedgerEntryAssets = generalLedgerEntryAssets;
    }

    public Date getTransactionDate() {
        return transactionDate;
    }

    public void setTransactionDate(final Date transactionDate) {
        this.transactionDate = transactionDate;
    }

    public List<PurchasingAccountsPayableLineAssetAccount> getPurApLineAssetAccounts() {
        return purApLineAssetAccounts;
    }

    public void setPurApLineAssetAccounts(final List<PurchasingAccountsPayableLineAssetAccount> purApLineAssetAccounts) {
        this.purApLineAssetAccounts = purApLineAssetAccounts;
    }

    public KualiDecimal computePayment() {
        final KualiDecimal absAmount = getTransactionLedgerEntryAmount();
        if (absAmount == null) {
            return null;
        }
        return KFSConstants.GL_CREDIT_CODE.equals(getTransactionDebitCreditCode()) ? absAmount.negated() : absAmount;
    }

    public boolean isSelected() {
        return selected;
    }

    public void setSelected(final boolean selected) {
        this.selected = selected;
    }

    public KualiDecimal getAmount() {
        if (getTransactionLedgerEntryAmount() != null
                && KFSConstants.GL_CREDIT_CODE.equals(getTransactionDebitCreditCode())) {
            setAmount(getTransactionLedgerEntryAmount().negated());
        } else {
            setAmount(getTransactionLedgerEntryAmount());
        }
        return amount;
    }

    public void setAmount(final KualiDecimal absAmount) {
        amount = absAmount;
    }

    public KualiDecimal getTransactionLedgerSubmitAmount() {
        return transactionLedgerSubmitAmount;
    }

    public void setTransactionLedgerSubmitAmount(final KualiDecimal transactionLedgerSubmitAmount) {
        this.transactionLedgerSubmitAmount = transactionLedgerSubmitAmount;
    }
}
