/*
 * The Kuali Financial System, a comprehensive financial management system for higher education.
 *
 * Copyright 2005-2023 Kuali, Inc.
 *
 * This program is free software: you can redistribute it and/or modify
 * it under the terms of the GNU Affero General Public License as published by
 * the Free Software Foundation, either version 3 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU Affero General Public License for more details.
 *
 * You should have received a copy of the GNU Affero General Public License
 * along with this program.  If not, see <http://www.gnu.org/licenses/>.
 */
package org.kuali.kfs.module.cam.document.authorization;

import org.kuali.kfs.coreservice.framework.parameter.ParameterService;
import org.kuali.kfs.datadictionary.legacy.DocumentDictionaryService;
import org.kuali.kfs.kew.api.WorkflowDocument;
import org.kuali.kfs.kns.document.MaintenanceDocument;
import org.kuali.kfs.krad.bo.BusinessObject;
import org.kuali.kfs.krad.document.Document;
import org.kuali.kfs.krad.util.GlobalVariables;
import org.kuali.kfs.krad.util.ObjectUtils;
import org.kuali.kfs.module.cam.CamsConstants;
import org.kuali.kfs.module.cam.CamsParameterConstants;
import org.kuali.kfs.module.cam.CamsPropertyConstants;
import org.kuali.kfs.module.cam.businessobject.Asset;
import org.kuali.kfs.module.cam.businessobject.AssetComponent;
import org.kuali.kfs.module.cam.businessobject.AssetRepairHistory;
import org.kuali.kfs.module.cam.document.service.AssetService;
import org.kuali.kfs.sys.context.SpringContext;
import org.kuali.kfs.sys.document.authorization.FinancialSystemMaintenanceDocumentPresentationControllerBase;

import java.util.Arrays;
import java.util.List;
import java.util.Set;

/**
 * Presentation Controller for Asset Maintenance Documents
 */
public class AssetPresentationController extends FinancialSystemMaintenanceDocumentPresentationControllerBase {

    @Override
    public Set<String> getConditionallyHiddenPropertyNames(final BusinessObject businessObject) {
        final Set<String> fields = super.getConditionallyHiddenPropertyNames(businessObject);

        final MaintenanceDocument document = (MaintenanceDocument) businessObject;

        if (SpringContext.getBean(AssetService.class).isAssetFabrication(document)) {
            fields.addAll(Arrays.asList(CamsConstants.Asset.EDIT_DETAIL_INFORMATION_FIELDS));
            fields.addAll(Arrays.asList(CamsConstants.Asset.EDIT_ORGANIZATION_INFORMATION_FIELDS));
        }

        // Hide payment sequence numbers
        final Asset asset = (Asset) document.getNewMaintainableObject().getBusinessObject();
        hideInactiveAssetComponent(fields, asset);
        hideInactiveAssetRepairHistory(fields, asset);
        hideInactiveAssetWarranty(fields, asset);

        final int size = asset.getAssetPayments().size();
        for (int i = 0; i < size; i++) {
            fields.add(CamsPropertyConstants.Asset.ASSET_PAYMENTS + "[" + i + "]." +
                    CamsPropertyConstants.AssetPayment.PAYMENT_SEQ_NUMBER);
            fields.add(CamsPropertyConstants.Asset.ASSET_PAYMENTS + "[" + i + "]." +
                    CamsPropertyConstants.AssetPayment.CAPITAL_ASSET_NUMBER);
        }

        if (!CamsConstants.AssetRetirementReasonCode.MERGED.equals(asset.getRetirementReasonCode())) {
            // hide merge target capital asset number
            fields.add(CamsPropertyConstants.Asset.RETIREMENT_INFO_MERGED_TARGET);
        }

        return fields;
    }

    @Override
    public Set<String> getConditionallyReadOnlyPropertyNames(final MaintenanceDocument document) {
        final Set<String> fields = super.getConditionallyReadOnlyPropertyNames(document);

        if (SpringContext.getBean(AssetService.class).isAssetFabrication(document)) {
            fields.add(CamsPropertyConstants.Asset.ASSET_INVENTORY_STATUS);
            fields.add(CamsPropertyConstants.Asset.VENDOR_NAME);
            fields.add(CamsPropertyConstants.Asset.ACQUISITION_TYPE_CODE);
        } else {
            // acquisition type code is read-only during edit
            fields.add(CamsPropertyConstants.Asset.ACQUISITION_TYPE_CODE);
            // fabrication fields are read-only
            fields.addAll(Arrays.asList(CamsConstants.Asset.FABRICATION_INFORMATION_FIELDS));
        }

        final Asset asset = (Asset) document.getNewMaintainableObject().getBusinessObject();
        // if tag was created in a prior fiscal year, set tag number, asset type code and description as view only
        if (SpringContext.getBean(AssetService.class).isAssetTaggedInPriorFiscalYear(asset)) {
            final AssetAuthorizer documentAuthorizer = (AssetAuthorizer) SpringContext.getBean(
                    DocumentDictionaryService.class).getDocumentAuthorizer(document);
            final boolean isAuthorized = documentAuthorizer.isAuthorized(document, CamsConstants.CAM_MODULE_CODE,
                    CamsConstants.PermissionNames.EDIT_WHEN_TAGGED_PRIOR_FISCAL_YEAR,
                    GlobalVariables.getUserSession().getPerson().getPrincipalId());
            if (!isAuthorized) {
                fields.addAll(SpringContext.getBean(ParameterService.class).getParameterValuesAsString(Asset.class,
                        CamsParameterConstants.PRIOR_YEAR_RESTRICTED_FIELDS));
            }
        }

        return fields;
    }

    @Override
    public Set<String> getConditionallyHiddenSectionIds(final BusinessObject businessObject) {
        final Set<String> fields = super.getConditionallyHiddenSectionIds(businessObject);

        final MaintenanceDocument document = (MaintenanceDocument) businessObject;
        final Asset asset = (Asset) document.getNewMaintainableObject().getBusinessObject();

        if (asset.getAssetPayments().size() > CamsConstants.Asset.ASSET_MAXIMUM_NUMBER_OF_PAYMENT_DISPLAY) {
            // Hide the payment section if there are more then CamsConstants.ASSET_MAXIMUM_NUMBER_OF_PAYMENT_DISPLAY
            fields.add(CamsConstants.Asset.SECTION_ID_PAYMENT_INFORMATION);
        }

        if (SpringContext.getBean(AssetService.class).isAssetFabrication(document)) {
            // fabrication request asset creation. Hide sections that are only applicable to asset edit. For fields
            // that are to be hidden for asset edit, see AssetAuthorizer.addMaintenanceDocumentRestrictions
            fields.add(CamsConstants.Asset.SECTION_ID_LAND_INFORMATION);
            fields.add(CamsConstants.Asset.SECTION_ID_PAYMENT_INFORMATION);
            fields.add(CamsConstants.Asset.SECTION_ID_PAYMENT_LOOKUP);
            fields.add(CamsConstants.Asset.SECTION_ID_DOCUMENT_LOOKUP);
            fields.add(CamsConstants.Asset.SECTION_ID_DEPRECIATION_INFORMATION);
            fields.add(CamsConstants.Asset.SECTION_ID_HISTORY);
            fields.add(CamsConstants.Asset.SECTION_ID_RETIREMENT_INFORMATION);
            fields.add(CamsConstants.Asset.SECTION_ID_EQUIPMENT_LOAN_INFORMATION);
            fields.add(CamsConstants.Asset.SECTION_ID_WARRANTY);
            fields.add(CamsConstants.Asset.SECTION_ID_REPAIR_HISTORY);
            fields.add(CamsConstants.Asset.SECTION_ID_COMPONENTS);
            fields.add(CamsConstants.Asset.SECTION_ID_MERGE_HISTORY);
        } else {
            // Asset edit
            if (asset.getEstimatedFabricationCompletionDate() == null) {
                // hide fabrication only if asset is not fabricated through CAMS
                fields.add(CamsConstants.Asset.SECTION_ID_FABRICATION_INFORMATION);
            }

            if (!SpringContext.getBean(AssetService.class).isAssetRetired(asset)) {
                // if asset is not retired, hide retirement information
                fields.add(CamsConstants.Asset.SECTION_ID_RETIREMENT_INFORMATION);
            }

            if (asset.getExpectedReturnDate() == null || asset.getLoanReturnDate() != null) {
                // If asset is not loaned, hide the section
                fields.add(CamsConstants.Asset.SECTION_ID_LOAN_INFORMATION);
            }
            if (asset.getSeparateHistory() == null) {
                // If asset is not separated, hide the section
                fields.add(CamsConstants.Asset.SECTION_ID_HISTORY);
            }
        }

        return fields;
    }

    @Override
    public boolean canEdit(final Document document) {
        final AssetService assetService = SpringContext.getBean(AssetService.class);

        // for fabrication document, disallow edit when document routing to the 'Management' FYI users.
        if (assetService.isAssetFabrication((MaintenanceDocument) document)) {
            final WorkflowDocument workflowDocument = document.getDocumentHeader().getWorkflowDocument();
            if (workflowDocument.isEnroute()) {
                final Set<String> nodeNames = assetService.getCurrentRouteLevels(workflowDocument);

                if (nodeNames.contains(CamsConstants.RouteLevelNames.MANAGEMENT)
                        && !workflowDocument.isApprovalRequested()) {
                    return false;
                }
            }
        }
        // for retired asset, disable edit.
        final MaintenanceDocument maintDocument = (MaintenanceDocument) document;
        final Asset asset = (Asset) maintDocument.getOldMaintainableObject().getBusinessObject();
        return !getAssetService().isAssetRetired(asset) & super.canEdit(document);
    }

    @Override
    public boolean canBlanketApprove(final Document document) {
        return true;
    }

    @Override
    public boolean canRoute(final Document document) {
        final MaintenanceDocument maitDocument = (MaintenanceDocument) document;
        final Asset asset = (Asset) maitDocument.getOldMaintainableObject().getBusinessObject();
        return !getAssetService().isAssetRetired(asset) & super.canRoute(document);
    }

    @Override
    public boolean canSave(final Document document) {
        final MaintenanceDocument maitDocument = (MaintenanceDocument) document;
        final Asset asset = (Asset) maitDocument.getOldMaintainableObject().getBusinessObject();
        return !getAssetService().isAssetRetired(asset) & super.canSave(document);
    }

    protected AssetService getAssetService() {
        return SpringContext.getBean(AssetService.class);
    }

    private void hideInactiveAssetWarranty(final Set<String> fields, final Asset asset) {
        if (ObjectUtils.isNotNull(asset.getAssetWarranty())) {
            if (!asset.getAssetWarranty().isActive()) {
                fields.add(CamsPropertyConstants.Asset.ASSET_WARRANTY);
            }
        }
    }

    private void hideInactiveAssetComponent(final Set<String> fields, final Asset asset) {
        final List<AssetComponent> components = asset.getAssetComponents();
        int i = 0;
        for (final AssetComponent component : components) {
            if (!component.isActive()) {
                fields.add(CamsPropertyConstants.Asset.ASSET_COMPONENTS + "[" + i + "]");
            }
            i++;
        }
    }

    private void hideInactiveAssetRepairHistory(final Set<String> fields, final Asset asset) {
        int i = 0;
        for (final AssetRepairHistory assetRepairHistory : asset.getAssetRepairHistory()) {
            if (!assetRepairHistory.isActive()) {
                fields.add(CamsPropertyConstants.Asset.ASSET_REPAIR_HISTORY + "[" + i + "]");
            }
            i++;
        }
    }
}
