/*
 * The Kuali Financial System, a comprehensive financial management system for higher education.
 *
 * Copyright 2005-2023 Kuali, Inc.
 *
 * This program is free software: you can redistribute it and/or modify
 * it under the terms of the GNU Affero General Public License as published by
 * the Free Software Foundation, either version 3 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU Affero General Public License for more details.
 *
 * You should have received a copy of the GNU Affero General Public License
 * along with this program.  If not, see <http://www.gnu.org/licenses/>.
 */
package org.kuali.kfs.module.cam.document.service.impl;

import org.apache.commons.lang3.StringUtils;
import org.kuali.kfs.datadictionary.legacy.BusinessObjectDictionaryService;
import org.kuali.kfs.datadictionary.legacy.DataDictionaryService;
import org.kuali.kfs.datadictionary.legacy.DocumentDictionaryService;
import org.kuali.kfs.krad.bo.BusinessObject;
import org.kuali.kfs.krad.datadictionary.DataDictionaryEntry;
import org.kuali.kfs.krad.document.DocumentBase;
import org.kuali.kfs.krad.service.BusinessObjectService;
import org.kuali.kfs.krad.util.GlobalVariables;
import org.kuali.kfs.krad.util.ObjectUtils;
import org.kuali.kfs.module.cam.CamsConstants;
import org.kuali.kfs.module.cam.CamsKeyConstants;
import org.kuali.kfs.module.cam.businessobject.Asset;
import org.kuali.kfs.module.cam.businessobject.AssetLocation;
import org.kuali.kfs.module.cam.businessobject.AssetType;
import org.kuali.kfs.module.cam.document.service.AssetLocationService;
import org.kuali.kfs.sys.KFSConstants;
import org.kuali.kfs.sys.businessobject.State;
import org.kuali.kfs.sys.context.SpringContext;
import org.kuali.kfs.sys.service.LocationService;

import java.util.List;
import java.util.Map;

public class AssetLocationServiceImpl implements AssetLocationService {

    private BusinessObjectDictionaryService businessObjectDictionaryService;
    private BusinessObjectService businessObjectService;
    private DataDictionaryService DataDictionaryService;
    private DocumentDictionaryService documentDictionaryService;

    @Override
    public void setOffCampusLocation(final Asset asset) {
        final List<AssetLocation> assetLocations = asset.getAssetLocations();
        AssetLocation offCampusLocation = null;

        for (final AssetLocation location : assetLocations) {
            if (CamsConstants.AssetLocationTypeCode.OFF_CAMPUS.equalsIgnoreCase(location.getAssetLocationTypeCode())) {
                // We need a new instance for asset location. Otherwise, if we copy it from the assetLocations
                // collection, it could have newBO and oldBO pointing to the same AssetLocation instance which is bad.
                offCampusLocation = new AssetLocation(location);
                break;
            }
        }

        if (ObjectUtils.isNull(offCampusLocation)) {
            offCampusLocation = new AssetLocation(asset.getCapitalAssetNumber());
            offCampusLocation.setAssetLocationTypeCode(CamsConstants.AssetLocationTypeCode.OFF_CAMPUS);
        }
        asset.setOffCampusLocation(offCampusLocation);
    }

    /**
     * update existing offCampusLocation
     */
    @Override
    public void updateOffCampusLocation(final Asset asset) {
        final AssetLocation offLocation = asset.getOffCampusLocation();
        final boolean isOffCampusEmpty = isOffCampusLocationEmpty(offLocation);
        AssetLocation removableOffCampusLocation = null;

        for (final AssetLocation location : asset.getAssetLocations()) {
            if (CamsConstants.AssetLocationTypeCode.OFF_CAMPUS.equalsIgnoreCase(location.getAssetLocationTypeCode())) {
                if (isOffCampusEmpty) {
                    removableOffCampusLocation = location;
                } else {
                    location.setAssetLocationCityName(offLocation.getAssetLocationCityName());
                    location.setAssetLocationContactIdentifier(offLocation.getAssetLocationContactIdentifier());
                    location.setAssetLocationContactName(offLocation.getAssetLocationContactName());
                    location.setAssetLocationCountryCode(offLocation.getAssetLocationCountryCode());
                    location.setAssetLocationInstitutionName(offLocation.getAssetLocationInstitutionName());
                    location.setAssetLocationPhoneNumber(offLocation.getAssetLocationPhoneNumber());
                    location.setAssetLocationStateCode(offLocation.getAssetLocationStateCode());
                    location.setAssetLocationStreetAddress(offLocation.getAssetLocationStreetAddress());
                    location.setAssetLocationZipCode(offLocation.getAssetLocationZipCode());
                    return;
                }
            }
        }

        if (removableOffCampusLocation != null) {
            asset.getAssetLocations().remove(removableOffCampusLocation);
        } else if (!isOffCampusEmpty) {
            // new offCampusLocation, add it into assetLocation List
            asset.getAssetLocations().add(offLocation);
        }

    }

    @Override
    public boolean isOffCampusLocationExists(final AssetLocation offCampusLocation) {
        if (ObjectUtils.isNotNull(offCampusLocation)) {
            return CamsConstants.AssetLocationTypeCode.OFF_CAMPUS.equalsIgnoreCase(
                    offCampusLocation.getAssetLocationTypeCode());
        }
        return false;
    }

    @Override
    public boolean isOffCampusLocationEmpty(final AssetLocation offCampusLocation) {
        if (ObjectUtils.isNotNull(offCampusLocation)) {
            return StringUtils.isBlank(offCampusLocation.getAssetLocationCityName())
                    && StringUtils.isBlank(offCampusLocation.getAssetLocationContactIdentifier())
                    && StringUtils.isBlank(offCampusLocation.getAssetLocationContactName())
                    && StringUtils.isBlank(offCampusLocation.getAssetLocationCountryCode())
                    && StringUtils.isBlank(offCampusLocation.getAssetLocationInstitutionName())
                    && StringUtils.isBlank(offCampusLocation.getAssetLocationPhoneNumber())
                    && StringUtils.isBlank(offCampusLocation.getAssetLocationStateCode())
                    && StringUtils.isBlank(offCampusLocation.getAssetLocationStreetAddress())
                    && StringUtils.isBlank(offCampusLocation.getAssetLocationZipCode());
        }
        return true;
    }

    @Override
    public boolean validateLocation(
            final Map<LocationField, String> fieldMap, final BusinessObject businessObject,
            final boolean isCapital, final AssetType assetType) {
        final String campusCode = readPropertyValue(businessObject, fieldMap, LocationField.CAMPUS_CODE);
        final String buildingCode = readPropertyValue(businessObject, fieldMap, LocationField.BUILDING_CODE);
        final String roomNumber = readPropertyValue(businessObject, fieldMap, LocationField.ROOM_NUMBER);
        final String subRoomNumber = readPropertyValue(businessObject, fieldMap, LocationField.SUB_ROOM_NUMBER);
        final String contactName = readPropertyValue(businessObject, fieldMap, LocationField.CONTACT_NAME);
        final String streetAddress = readPropertyValue(businessObject, fieldMap, LocationField.STREET_ADDRESS);
        final String cityName = readPropertyValue(businessObject, fieldMap, LocationField.CITY_NAME);
        final String stateCode = readPropertyValue(businessObject, fieldMap, LocationField.STATE_CODE);
        final String zipCode = readPropertyValue(businessObject, fieldMap, LocationField.ZIP_CODE);
        final String countryCode = readPropertyValue(businessObject, fieldMap, LocationField.COUNTRY_CODE);

        // businessObject parameter could be BusinessObjectEntry or TransactionalDocumentEntry
        final DataDictionaryService ddService = getDataDictionaryService();
        final DataDictionaryEntry ddEntry;
        if (DocumentBase.class.isAssignableFrom(businessObject.getClass())) {
            final String docTypeName = ddService.getDocumentTypeNameByClass(businessObject.getClass());
            ddEntry = documentDictionaryService.getDocumentEntry(docTypeName);
        } else {
            ddEntry = businessObjectDictionaryService.getBusinessObjectEntry(businessObject.getClass().getName());
        }

        final boolean onCampus = StringUtils.isNotBlank(buildingCode)
                                 || StringUtils.isNotBlank(roomNumber)
                                 || StringUtils.isNotBlank(subRoomNumber);
        final boolean offCampus = StringUtils.isNotBlank(contactName)
                                  || StringUtils.isNotBlank(streetAddress)
                                  || StringUtils.isNotBlank(cityName)
                                  || StringUtils.isNotBlank(stateCode)
                                  || StringUtils.isNotBlank(zipCode)
                                  || StringUtils.isNotBlank(countryCode);

        final boolean valid;
        if (onCampus && offCampus) {
            putError(fieldMap, LocationField.BUILDING_CODE, CamsKeyConstants.AssetLocation.ERROR_CHOOSE_LOCATION_INFO);
            valid = false;
        } else {
            if (isCapital) {
                valid = validateCapitalAssetLocation(assetType, fieldMap, campusCode, buildingCode, roomNumber,
                        subRoomNumber, contactName, streetAddress, cityName, stateCode, zipCode, countryCode,
                        onCampus, offCampus, ddEntry);
            } else {
                valid = validateNonCapitalAssetLocation(fieldMap, contactName, streetAddress, cityName, stateCode,
                        zipCode, countryCode, onCampus, offCampus);
            }
        }
        return valid;
    }

    protected boolean validateCapitalAssetLocation(
            final AssetType assetType, final Map<LocationField, String> fieldMap,
            final String campusCode, final String buildingCode, final String roomNumber, final String subRoomNumber, final String contactName,
            final String streetAddress, final String cityName, final String stateCode, final String zipCode, final String countryCode,
            final boolean onCampus, final boolean offCampus, final DataDictionaryEntry businessObjectEntry) {
        boolean valid = true;
        if (ObjectUtils.isNull(assetType)) {
            GlobalVariables.getMessageMap().putErrorForSectionId(CamsConstants.LOCATION_INFORMATION_SECTION_ID,
                    CamsKeyConstants.AssetLocation.ERROR_CHOOSE_ASSET_TYPE);
            valid = false;
        } else {
            String label;
            if (assetType.isRequiredBuildingIndicator() && offCampus) {
                // off campus information not allowed
                if (StringUtils.isNotBlank(contactName)) {
                    label = businessObjectEntry.getAttributeDefinition(fieldMap.get(LocationField.CONTACT_NAME))
                            .getLabel();
                    putError(fieldMap, LocationField.CONTACT_NAME,
                            CamsKeyConstants.AssetLocation.ERROR_LOCATION_NOT_PERMITTED_ASSET_TYPE, label,
                            assetType.getCapitalAssetTypeDescription());
                    valid = false;
                }
                if (StringUtils.isNotBlank(streetAddress)) {
                    label = businessObjectEntry.getAttributeDefinition(fieldMap.get(LocationField.STREET_ADDRESS))
                            .getLabel();
                    putError(fieldMap, LocationField.STREET_ADDRESS,
                            CamsKeyConstants.AssetLocation.ERROR_LOCATION_NOT_PERMITTED_ASSET_TYPE, label,
                            assetType.getCapitalAssetTypeDescription());
                    valid = false;
                }

                if (StringUtils.isNotBlank(cityName)) {
                    label = businessObjectEntry.getAttributeDefinition(fieldMap.get(LocationField.CITY_NAME))
                            .getLabel();
                    putError(fieldMap, LocationField.CITY_NAME,
                            CamsKeyConstants.AssetLocation.ERROR_LOCATION_NOT_PERMITTED_ASSET_TYPE, label,
                            assetType.getCapitalAssetTypeDescription());
                    valid = false;
                }

                if (StringUtils.isNotBlank(stateCode)) {
                    label = businessObjectEntry.getAttributeDefinition(fieldMap.get(LocationField.STATE_CODE))
                            .getLabel();
                    putError(fieldMap, LocationField.STATE_CODE,
                            CamsKeyConstants.AssetLocation.ERROR_LOCATION_NOT_PERMITTED_ASSET_TYPE, label,
                            assetType.getCapitalAssetTypeDescription());
                    valid = false;
                }

                if (StringUtils.isNotBlank(zipCode)) {
                    label = businessObjectEntry.getAttributeDefinition(fieldMap.get(LocationField.ZIP_CODE))
                            .getLabel();
                    putError(fieldMap, LocationField.ZIP_CODE,
                            CamsKeyConstants.AssetLocation.ERROR_LOCATION_NOT_PERMITTED_ASSET_TYPE, label,
                            assetType.getCapitalAssetTypeDescription());
                    valid = false;
                }

                if (StringUtils.isNotBlank(countryCode)) {
                    label = businessObjectEntry.getAttributeDefinition(fieldMap.get(LocationField.COUNTRY_CODE))
                            .getLabel();
                    putError(fieldMap, LocationField.COUNTRY_CODE,
                            CamsKeyConstants.AssetLocation.ERROR_LOCATION_NOT_PERMITTED_ASSET_TYPE, label,
                            assetType.getCapitalAssetTypeDescription());
                    valid = false;
                }
            } else if (!assetType.isMovingIndicator() && !assetType.isRequiredBuildingIndicator() && onCampus) {
                // land information cannot have on-campus
                if (StringUtils.isNotBlank(buildingCode)) {
                    label = businessObjectEntry.getAttributeDefinition(fieldMap.get(LocationField.BUILDING_CODE))
                            .getLabel();
                    putError(fieldMap, LocationField.BUILDING_CODE,
                            CamsKeyConstants.AssetLocation.ERROR_LOCATION_NOT_PERMITTED_ASSET_TYPE, label,
                            assetType.getCapitalAssetTypeDescription());
                    valid = false;
                }

                if (StringUtils.isNotBlank(roomNumber)) {
                    label = businessObjectEntry.getAttributeDefinition(fieldMap.get(LocationField.ROOM_NUMBER))
                            .getLabel();
                    putError(fieldMap, LocationField.ROOM_NUMBER,
                            CamsKeyConstants.AssetLocation.ERROR_LOCATION_NOT_PERMITTED_ASSET_TYPE, label,
                            assetType.getCapitalAssetTypeDescription());
                    valid = false;
                }

                if (StringUtils.isNotBlank(subRoomNumber)) {
                    label = businessObjectEntry.getAttributeDefinition(fieldMap.get(LocationField.SUB_ROOM_NUMBER))
                            .getLabel();
                    putError(fieldMap, LocationField.SUB_ROOM_NUMBER,
                            CamsKeyConstants.AssetLocation.ERROR_LOCATION_NOT_PERMITTED_ASSET_TYPE, label,
                            assetType.getCapitalAssetTypeDescription());
                    valid = false;
                }
            } else if (onCampus) {
                valid = validateOnCampusLocation(fieldMap, assetType, campusCode, buildingCode, roomNumber,
                        subRoomNumber);
            } else if (offCampus) {
                valid = validateOffCampusLocation(fieldMap, contactName, streetAddress, cityName, stateCode, zipCode,
                        countryCode);
            } else if (assetType.isMovingIndicator() || assetType.isRequiredBuildingIndicator()) {
                putError(fieldMap, LocationField.BUILDING_CODE,
                        CamsKeyConstants.AssetLocation.ERROR_LOCATION_INFO_REQUIRED);
                valid = false;
            }
        }
        return valid;
    }

    protected boolean validateNonCapitalAssetLocation(
            final Map<LocationField, String> fieldMap, final String contactName,
            final String streetAddress, final String cityName, final String stateCode, final String zipCode, final String countryCode,
            final boolean onCampus, final boolean offCampus) {
        boolean valid = true;
        if (offCampus) {
            valid = validateOffCampusLocation(fieldMap, contactName, streetAddress, cityName, stateCode, zipCode,
                    countryCode);
        }
        return valid;
    }

    /**
     * Convenience method to append the path prefix
     */
    protected void putError(
            final Map<LocationField, String> fieldMap, final LocationField field, final String errorKey,
            final String... errorParameters) {
        GlobalVariables.getMessageMap().putError(fieldMap.get(field), errorKey, errorParameters);
    }

    protected boolean validateOnCampusLocation(
            final Map<LocationField, String> fieldMap, final AssetType assetType,
            final String campusCode, final String buildingCode, final String buildingRoomNumber, final String subRoomNumber) {
        boolean valid = true;
        if (assetType.isMovingIndicator()) {
            if (StringUtils.isBlank(buildingCode)) {
                putError(fieldMap, LocationField.BUILDING_CODE,
                        CamsKeyConstants.AssetLocation.ERROR_ON_CAMPUS_BUILDING_CODE_REQUIRED,
                        assetType.getCapitalAssetTypeDescription());
                valid = false;
            }
            if (StringUtils.isBlank(buildingRoomNumber)) {
                putError(fieldMap, LocationField.ROOM_NUMBER,
                        CamsKeyConstants.AssetLocation.ERROR_ON_CAMPUS_BUILDING_ROOM_NUMBER_REQUIRED,
                        assetType.getCapitalAssetTypeDescription());
                valid = false;
            }
        }
        if (assetType.isRequiredBuildingIndicator()) {
            if (StringUtils.isBlank(buildingCode)) {
                putError(fieldMap, LocationField.BUILDING_CODE,
                        CamsKeyConstants.AssetLocation.ERROR_ON_CAMPUS_BUILDING_CODE_REQUIRED,
                        assetType.getCapitalAssetTypeDescription());
                valid = false;
            }
            if (StringUtils.isNotBlank(buildingRoomNumber)) {
                putError(fieldMap, LocationField.ROOM_NUMBER,
                        CamsKeyConstants.AssetLocation.ERROR_ON_CAMPUS_BUILDING_ROOM_NUMBER_NOT_PERMITTED,
                        assetType.getCapitalAssetTypeDescription());
                valid = false;
            }
            if (StringUtils.isNotBlank(subRoomNumber)) {
                putError(fieldMap, LocationField.SUB_ROOM_NUMBER,
                        CamsKeyConstants.AssetLocation.ERROR_ON_CAMPUS_SUB_ROOM_NUMBER_NOT_PERMITTED,
                        assetType.getCapitalAssetTypeDescription());
                valid = false;
            }
        }
        return valid;
    }

    protected boolean validateOffCampusLocation(
            final Map<LocationField, String> fieldMap, final String contactName,
            final String streetAddress, final String cityName, final String stateCode, final String zipCode, final String countryCode) {
        boolean valid = true;
        boolean isCountryUS = false;
        if (isBlank(fieldMap, LocationField.COUNTRY_CODE, countryCode)) {
            putError(fieldMap, LocationField.COUNTRY_CODE,
                    CamsKeyConstants.AssetLocation.ERROR_OFF_CAMPUS_COUNTRY_REQUIRED);
            valid = false;
        } else {
            isCountryUS = countryCode.equals(KFSConstants.COUNTRY_CODE_UNITED_STATES);
        }

        if (isBlank(fieldMap, LocationField.CONTACT_NAME, contactName)) {
            putError(fieldMap, LocationField.CONTACT_NAME,
                    CamsKeyConstants.AssetLocation.ERROR_OFF_CAMPUS_CONTACT_REQUIRED);
            valid = false;
        }

        if (isBlank(fieldMap, LocationField.STREET_ADDRESS, streetAddress)) {
            putError(fieldMap, LocationField.STREET_ADDRESS,
                    CamsKeyConstants.AssetLocation.ERROR_OFF_CAMPUS_ADDRESS_REQUIRED);
            valid = false;
        }
        if (isBlank(fieldMap, LocationField.CITY_NAME, cityName)) {
            putError(fieldMap, LocationField.CITY_NAME, CamsKeyConstants.AssetLocation.ERROR_OFF_CAMPUS_CITY_REQUIRED);
            valid = false;
        }

        if (isCountryUS) {
            if (isBlank(fieldMap, LocationField.STATE_CODE, stateCode)) {
                putError(fieldMap, LocationField.STATE_CODE,
                        CamsKeyConstants.AssetLocation.ERROR_OFF_CAMPUS_STATE_REQUIRED);
                valid = false;
            }
            if (isBlank(fieldMap, LocationField.ZIP_CODE, zipCode)) {
                putError(fieldMap, LocationField.ZIP_CODE,
                        CamsKeyConstants.AssetLocation.ERROR_OFF_CAMPUS_ZIP_REQUIRED);
                valid = false;
            }
            if (!isBlank(fieldMap, LocationField.STATE_CODE, stateCode)) {
                final State locationState = getLocationService()
                        .getState(countryCode, stateCode);
                if (ObjectUtils.isNull(locationState)) {
                    putError(fieldMap, LocationField.STATE_CODE,
                            CamsKeyConstants.AssetLocation.ERROR_INVALID_OFF_CAMPUS_STATE, stateCode);
                    valid = false;
                }
            }
        }

        return valid;
    }

    protected boolean isBlank(final Map<LocationField, String> fieldMap, final LocationField field, final String countryCode) {
        return fieldMap.get(field) != null && StringUtils.isBlank(countryCode);
    }

    protected String readPropertyValue(
            final BusinessObject currObject, final Map<LocationField, String> fieldMap,
            final LocationField field) {
        String stringValue = null;
        try {
            final String propertyName = fieldMap.get(field);
            if (propertyName != null) {
                stringValue = (String) ObjectUtils.getNestedValue(currObject, propertyName);
            }
        } catch (final Exception e) {
            throw new RuntimeException(e);
        }
        return stringValue;
    }

    public void setBusinessObjectDictionaryService(
            final BusinessObjectDictionaryService businessObjectDictionaryService) {
        this.businessObjectDictionaryService = businessObjectDictionaryService;
    }

    public BusinessObjectService getBusinessObjectService() {
        return businessObjectService;
    }

    public void setBusinessObjectService(final BusinessObjectService businessObjectService) {
        this.businessObjectService = businessObjectService;
    }

    public DataDictionaryService getDataDictionaryService() {
        return DataDictionaryService;
    }

    public void setDataDictionaryService(final DataDictionaryService dataDictionaryService) {
        DataDictionaryService = dataDictionaryService;
    }

    public void setDocumentDictionaryService(final DocumentDictionaryService documentDictionaryService) {
        this.documentDictionaryService = documentDictionaryService;
    }

    /*
     * Wrapping static utility class in a method so tests can use a spy to mock this call; this way,
     * static mocking is not necessary.
     */
    LocationService getLocationService() {
        return SpringContext.getBean(LocationService.class, "locationService-fin");
    }

}
