/*
 * The Kuali Financial System, a comprehensive financial management system for higher education.
 *
 * Copyright 2005-2023 Kuali, Inc.
 *
 * This program is free software: you can redistribute it and/or modify
 * it under the terms of the GNU Affero General Public License as published by
 * the Free Software Foundation, either version 3 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU Affero General Public License for more details.
 *
 * You should have received a copy of the GNU Affero General Public License
 * along with this program.  If not, see <http://www.gnu.org/licenses/>.
 */
package org.kuali.kfs.module.cam.document.service.impl;

import org.apache.logging.log4j.LogManager;
import org.apache.logging.log4j.Logger;
import org.kuali.kfs.integration.purap.CapitalAssetSystem;
import org.kuali.kfs.integration.purap.ItemCapitalAsset;
import org.kuali.kfs.krad.service.BusinessObjectService;
import org.kuali.kfs.krad.util.ObjectUtils;
import org.kuali.kfs.module.cam.CamsConstants;
import org.kuali.kfs.module.cam.CamsPropertyConstants;
import org.kuali.kfs.module.cam.businessobject.Asset;
import org.kuali.kfs.module.cam.businessobject.PurchasingAccountsPayableDocument;
import org.kuali.kfs.module.cam.businessobject.PurchasingAccountsPayableItemAsset;
import org.kuali.kfs.module.cam.document.exception.PurApDocumentUnavailableException;
import org.kuali.kfs.module.cam.document.service.AssetService;
import org.kuali.kfs.module.cam.document.service.PurApInfoService;
import org.kuali.kfs.module.cam.web.struts.PurApLineForm;
import org.kuali.kfs.module.purap.PurapConstants;
import org.kuali.kfs.module.purap.PurapPropertyConstants;
import org.kuali.kfs.module.purap.businessobject.CreditMemoItem;
import org.kuali.kfs.module.purap.businessobject.PaymentRequestItem;
import org.kuali.kfs.module.purap.businessobject.PurApItem;
import org.kuali.kfs.module.purap.businessobject.PurchaseOrderItem;
import org.kuali.kfs.module.purap.businessobject.PurchasingCapitalAssetItem;
import org.kuali.kfs.module.purap.document.PurchaseOrderDocument;
import org.kuali.kfs.module.purap.document.VendorCreditMemoDocument;
import org.kuali.kfs.module.purap.document.service.PurapService;
import org.kuali.kfs.module.purap.document.service.PurchaseOrderService;
import org.kuali.kfs.module.purap.exception.PurError;

import java.util.ArrayList;
import java.util.Collection;
import java.util.HashMap;
import java.util.Iterator;
import java.util.List;
import java.util.Map;

public class PurApInfoServiceImpl implements PurApInfoService {

    private static final Logger LOG = LogManager.getLogger();
    protected static final String PURCHASE_ORDER_CURRENT_INDICATOR = "purchaseOrderCurrentIndicator";

    private AssetService assetService;
    private BusinessObjectService businessObjectService;
    private PurapService purapService;
    private PurchaseOrderService purchaseOrderService;

    @Override
    public PurchaseOrderDocument getCurrentDocumentForPurchaseOrderIdentifier(final Integer poId) {
        final Map<String, Object> fieldValues = new HashMap<>();

        fieldValues.put(CamsPropertyConstants.PurchasingAccountsPayableDocument.PURAP_DOCUMENT_IDENTIFIER, poId);
        fieldValues.put(PURCHASE_ORDER_CURRENT_INDICATOR, "Y");
        final Collection<PurchaseOrderDocument> poDocs = businessObjectService.findMatching(PurchaseOrderDocument.class,
                fieldValues);
        if (poDocs != null && !poDocs.isEmpty()) {
            final Iterator<PurchaseOrderDocument> poIterator = poDocs.iterator();
            if (poIterator.hasNext()) {
                return poIterator.next();
            }
        }

        return null;
    }

    @Override
    public void setPurchaseOrderFromPurAp(final PurApLineForm purApLineForm) {
        final PurchaseOrderDocument purchaseOrderDocument = getCurrentDocumentForPurchaseOrderIdentifier(
                purApLineForm.getPurchaseOrderIdentifier());

        if (ObjectUtils.isNull(purchaseOrderDocument)) {
            return;
        }
        // Set contact email address.
        if (purchaseOrderDocument.getInstitutionContactEmailAddress() != null) {
            purApLineForm.setPurApContactEmailAddress(purchaseOrderDocument.getInstitutionContactEmailAddress());
        } else if (purchaseOrderDocument.getRequestorPersonEmailAddress() != null) {
            purApLineForm.setPurApContactEmailAddress(purchaseOrderDocument.getRequestorPersonEmailAddress());
        }

        // Set contact phone number.
        if (purchaseOrderDocument.getInstitutionContactPhoneNumber() != null) {
            purApLineForm.setPurApContactPhoneNumber(purchaseOrderDocument.getInstitutionContactPhoneNumber());
        } else if (purchaseOrderDocument.getRequestorPersonPhoneNumber() != null) {
            purApLineForm.setPurApContactPhoneNumber(purchaseOrderDocument.getRequestorPersonPhoneNumber());
        }

        // set reqs_id
        purApLineForm.setRequisitionIdentifier(purchaseOrderDocument.getRequisitionIdentifier());
    }

    /**
     * Set CAMS transaction type code the user entered in PurAp
     *
     * @param purApDocs
     */
    @Override
    public void setCamsTransactionFromPurAp(final List<PurchasingAccountsPayableDocument> purApDocs) {
        if (ObjectUtils.isNull(purApDocs) || purApDocs.isEmpty()) {
            return;
        }
        final Integer poId = purApDocs.get(0).getPurchaseOrderIdentifier();
        final PurchaseOrderDocument purApdocument = getCurrentDocumentForPurchaseOrderIdentifier(poId);
        if (ObjectUtils.isNull(purApdocument)) {
            return;
        }

        final String capitalAssetSystemTypeCode = purApdocument.getCapitalAssetSystemTypeCode();
        final String capitalAssetSystemStateCode = purApdocument.getCapitalAssetSystemStateCode();
        boolean individualItemLock = false;

        if (PurapConstants.CapitalAssetTabStrings.INDIVIDUAL_ASSETS.equalsIgnoreCase(capitalAssetSystemTypeCode)) {
            // If PurAp sets the CAMS as INDIVIDUAL system
            setIndividualAssetsFromPurAp(poId, purApDocs, capitalAssetSystemStateCode);
            individualItemLock = true;
        } else if (PurapConstants.CapitalAssetTabStrings.ONE_SYSTEM.equalsIgnoreCase(capitalAssetSystemTypeCode)) {
            // If PurAp sets the CAMS as ONE system
            setOneSystemFromPurAp(poId, purApDocs, capitalAssetSystemStateCode);

        } else if (PurapConstants.CapitalAssetTabStrings.MULTIPLE_SYSTEMS.equalsIgnoreCase(capitalAssetSystemTypeCode)) {
            // If PurAp sets the CAMS as MULTIPLE system
            setMultipleSystemFromPurAp(poId, purApDocs, capitalAssetSystemStateCode);
        }

        // Setting locking information based on capital asset system type code. Currently, only individual system can
        // set asset numbers for each item
        for (final PurchasingAccountsPayableDocument purApDoc : purApDocs) {
            for (final PurchasingAccountsPayableItemAsset itemAsset : purApDoc.getPurchasingAccountsPayableItemAssets()) {
                itemAsset.setLockingInformation(individualItemLock ?
                        itemAsset.getAccountsPayableLineItemIdentifier().toString() :
                        CamsConstants.defaultLockingInformation);
            }
        }
    }

    /**
     * Set Multiple system capital asset transaction type code and asset numbers.
     *
     * @param poId
     * @param purApDocs
     */
    protected void setMultipleSystemFromPurAp(
            final Integer poId, final List<PurchasingAccountsPayableDocument> purApDocs,
            final String capitalAssetSystemStateCode) {
        final List<CapitalAssetSystem> capitalAssetSystems =
                purchaseOrderService.retrieveCapitalAssetSystemsForMultipleSystem(poId);
        if (ObjectUtils.isNotNull(capitalAssetSystems) && !capitalAssetSystems.isEmpty()) {
            // PurAp doesn't support multiple system asset information for KFS3.0. It works as one system for 3.0.
            final CapitalAssetSystem capitalAssetSystem = capitalAssetSystems.get(0);
            if (ObjectUtils.isNotNull(capitalAssetSystem)) {
                final String capitalAssetTransactionType = getCapitalAssetTransTypeForOneSystem(poId);
                // if modify existing asset, acquire the assets from Purap
                List<ItemCapitalAsset> purApCapitalAssets = null;
                if (PurapConstants.CapitalAssetSystemStates.MODIFY.equalsIgnoreCase(capitalAssetSystemStateCode)) {
                    purApCapitalAssets = getAssetsFromItemCapitalAsset(capitalAssetSystem.getItemCapitalAssets());
                }

                // set TransactionTypeCode, itemCapitalAssets and system identifier for each item
                for (final PurchasingAccountsPayableDocument purApDoc : purApDocs) {
                    setItemAssetsCamsTransaction(capitalAssetSystem.getCapitalAssetSystemIdentifier(),
                            capitalAssetTransactionType, purApCapitalAssets,
                            purApDoc.getPurchasingAccountsPayableItemAssets());
                }
            }
        }
    }

    /**
     * Set One System capital asset transaction type code and asset numbers.
     *
     * @param poId
     * @param purApDocs
     */
    protected void setOneSystemFromPurAp(
            final Integer poId, final List<PurchasingAccountsPayableDocument> purApDocs,
            final String capitalAssetSystemStateCode) {
        final CapitalAssetSystem capitalAssetSystem = purchaseOrderService.retrieveCapitalAssetSystemForOneSystem(poId);
        final String capitalAssetTransactionTypeCode = getCapitalAssetTransTypeForOneSystem(poId);
        List<ItemCapitalAsset> purApCapitalAssets = null;
        // if modify existing asset, acquire the assets from Purap
        if (PurapConstants.CapitalAssetSystemStates.MODIFY.equalsIgnoreCase(capitalAssetSystemStateCode)) {
            purApCapitalAssets = getAssetsFromItemCapitalAsset(capitalAssetSystem.getItemCapitalAssets());
        }

        for (final PurchasingAccountsPayableDocument purApDoc : purApDocs) {
            // set TransactionTypeCode, itemCapitalAssets and system identifier for each item
            setItemAssetsCamsTransaction(capitalAssetSystem.getCapitalAssetSystemIdentifier(),
                    capitalAssetTransactionTypeCode, purApCapitalAssets,
                    purApDoc.getPurchasingAccountsPayableItemAssets());
        }
    }

    /**
     * Update item assets by PurAp Cams Transaction setting
     *
     * @param capitalAssetSystemIdentifier
     * @param capitalAssetTransactionTypeCode
     * @param purApCapitalAssets
     * @param itemAssets
     */
    protected void setItemAssetsCamsTransaction(
            final Integer capitalAssetSystemIdentifier,
            final String capitalAssetTransactionTypeCode, final List<ItemCapitalAsset> purApCapitalAssets,
            final List<PurchasingAccountsPayableItemAsset> itemAssets) {
        for (final PurchasingAccountsPayableItemAsset item : itemAssets) {
            item.setCapitalAssetTransactionTypeCode(capitalAssetTransactionTypeCode);
            // set for item capital assets
            if (purApCapitalAssets != null && !purApCapitalAssets.isEmpty()) {
                item.getPurApItemAssets().addAll(purApCapitalAssets);
            }
            // set for capital asset system ID
            item.setCapitalAssetSystemIdentifier(capitalAssetSystemIdentifier);
        }
    }

    /**
     * Get capitalAssetTransactionTypeCode for one system from PurAp.
     *
     * @param poId
     * @return
     */
    protected String getCapitalAssetTransTypeForOneSystem(final Integer poId) {
        final PurchaseOrderDocument poDoc = getCurrentDocumentForPurchaseOrderIdentifier(poId);
        if (ObjectUtils.isNotNull(poDoc)) {
            final List<PurchasingCapitalAssetItem> capitalAssetItems = poDoc.getPurchasingCapitalAssetItems();

            if (ObjectUtils.isNotNull(capitalAssetItems) && capitalAssetItems.get(0) != null) {
                return capitalAssetItems.get(0).getCapitalAssetTransactionTypeCode();
            }
        }
        return null;
    }

    /**
     * Set Individual system asset transaction type and asset numbers.
     *
     * @param poId
     * @param purApDocs
     */
    protected void setIndividualAssetsFromPurAp(
            final Integer poId, final List<PurchasingAccountsPayableDocument> purApDocs,
            final String capitalAssetSystemStateCode) {
        final List<PurchasingCapitalAssetItem> capitalAssetItems =
                purchaseOrderService.retrieveCapitalAssetItemsForIndividual(poId);
        String capitalAssetTransactionTypeCode;
        List<ItemCapitalAsset> purApCapitalAssets = null;

        // For each capitalAssetItem from PurAp, we set it to all matching CAB items
        for (final PurchasingCapitalAssetItem purchasingCapitalAssetItem : capitalAssetItems) {
            capitalAssetTransactionTypeCode = purchasingCapitalAssetItem.getCapitalAssetTransactionTypeCode();
            // get matching CAB items origin from the same PO item.
            final List<PurchasingAccountsPayableItemAsset> matchingItems = getMatchingItems(
                    purchasingCapitalAssetItem.getItemIdentifier(), purApDocs);
            // if modify existing asset, acquire the assets from Puraps
            if (PurapConstants.CapitalAssetSystemStates.MODIFY.equalsIgnoreCase(capitalAssetSystemStateCode)) {
                purApCapitalAssets = getAssetsFromItemCapitalAsset(
                        purchasingCapitalAssetItem.getPurchasingCapitalAssetSystem().getItemCapitalAssets());
            }

            // set TransactionTypeCode, itemCapitalAssets and system identifier for each matching item
            setItemAssetsCamsTransaction(purchasingCapitalAssetItem.getCapitalAssetSystemIdentifier(),
                    capitalAssetTransactionTypeCode, purApCapitalAssets, matchingItems);
        }
    }

    /**
     * Get asset number list from ItemCapitalAsset list.
     *
     * @param itemCapitalAssets
     * @return
     */
    protected List<ItemCapitalAsset> getAssetsFromItemCapitalAsset(final List<ItemCapitalAsset> itemCapitalAssets) {
        final List<ItemCapitalAsset> assetNumbers = new ArrayList<>();

        for (final ItemCapitalAsset asset : itemCapitalAssets) {
            if (asset.getCapitalAssetNumber() != null && !isAssetNumberDuplicate(asset.getCapitalAssetNumber(),
                    assetNumbers)) {
                assetNumbers.add(asset);
            }
        }
        return assetNumbers;
    }

    /**
     * Check if given capitalAssetNumber is an duplicate number in assetNumbers list.
     *
     * @param candidateNumber
     * @param assetNumbers
     * @return
     */
    protected boolean isAssetNumberDuplicate(final Long candidateNumber, final List<ItemCapitalAsset> assetNumbers) {
        for (final ItemCapitalAsset existingNumber : assetNumbers) {
            if (existingNumber.getCapitalAssetNumber().equals(candidateNumber)) {
                return true;
            }
        }
        return false;
    }

    /**
     * Finding out the matching PREQ/CM items originating from the same PurchaseOrderItem.
     *
     * @param itemIdentifier
     * @param purApDocs
     * @return
     */
    protected List<PurchasingAccountsPayableItemAsset> getMatchingItems(
            final Integer itemIdentifier,
            final List<PurchasingAccountsPayableDocument> purApDocs) {
        final List<PurchasingAccountsPayableItemAsset> matchingItems = new ArrayList<>();

        if (itemIdentifier != null) {
            for (final PurchasingAccountsPayableDocument purApDoc : purApDocs) {
                for (final PurchasingAccountsPayableItemAsset item : purApDoc.getPurchasingAccountsPayableItemAssets()) {
                    if (itemIdentifier.equals(item.getPurchaseOrderItemIdentifier())) {
                        matchingItems.add(item);
                    }
                }
            }
        }
        return matchingItems;
    }

    /**
     * Set CAB line item information from PurAp PaymentRequestItem or CreditMemoItem.
     *
     * @param purchasingAccountsPayableItemAsset
     * @param docTypeCode
     */
    @Override
    public void setAccountsPayableItemsFromPurAp(
            final PurchasingAccountsPayableItemAsset purchasingAccountsPayableItemAsset, final String docTypeCode) {
        final Map<String, Object> pKeys = new HashMap<>();
        pKeys.put(PurapPropertyConstants.ITEM_IDENTIFIER,
                purchasingAccountsPayableItemAsset.getAccountsPayableLineItemIdentifier());

        // Access PurAp data based on item type(PREQ or CM).
        if (PurapConstants.PurapDocTypeCodes.PAYMENT_REQUEST_DOCUMENT.equalsIgnoreCase(docTypeCode)) {
            // for PREQ document
            final PaymentRequestItem item = businessObjectService.findByPrimaryKey(PaymentRequestItem.class, pKeys);
            if (ObjectUtils.isNull(item)) {
                throw new PurApDocumentUnavailableException("PaymentRequestItem with id = " +
                        purchasingAccountsPayableItemAsset.getAccountsPayableLineItemIdentifier() +
                        " doesn't exist in table.");
            }

            purchasingAccountsPayableItemAsset.setItemLineNumber(item.getItemLineNumber());
            if (item.getItemType() != null) {
                purchasingAccountsPayableItemAsset.setAdditionalChargeNonTradeInIndicator(
                        item.getItemType().isAdditionalChargeIndicator() & !CamsConstants.TRADE_IN_TYPE_CODE
                                .equalsIgnoreCase(item.getItemTypeCode()));
                purchasingAccountsPayableItemAsset.setTradeInAllowance(item.getItemType().isAdditionalChargeIndicator()
                        & CamsConstants.TRADE_IN_TYPE_CODE.equalsIgnoreCase(item.getItemTypeCode()));
                purchasingAccountsPayableItemAsset.setItemTypeCode(item.getItemTypeCode());
            }
            purchasingAccountsPayableItemAsset.setItemAssignedToTradeInIndicator(
                    item.getItemAssignedToTradeInIndicator());
            // PurchaseOrderItemIdentifier will be used when populate PurAp asset data
            final PurchaseOrderItem poi = item.getPurchaseOrderItem();
            if (poi != null) {
                purchasingAccountsPayableItemAsset.setPurchaseOrderItemIdentifier(poi.getItemIdentifier());
            }
        } else {
            // for CM document
            final CreditMemoItem item = businessObjectService.findByPrimaryKey(CreditMemoItem.class, pKeys);
            if (ObjectUtils.isNull(item)) {
                throw new PurApDocumentUnavailableException("CreditMemoItem with id = " +
                        purchasingAccountsPayableItemAsset.getAccountsPayableLineItemIdentifier() +
                        " doesn't exist in table.");
            }

            purchasingAccountsPayableItemAsset.setItemLineNumber(item.getItemLineNumber());
            if (item.getItemType() != null) {
                purchasingAccountsPayableItemAsset.setAdditionalChargeNonTradeInIndicator(
                        item.getItemType().isAdditionalChargeIndicator()
                        & !CamsConstants.TRADE_IN_TYPE_CODE.equalsIgnoreCase(item.getItemTypeCode()));
                purchasingAccountsPayableItemAsset.setTradeInAllowance(item.getItemType().isAdditionalChargeIndicator()
                        & CamsConstants.TRADE_IN_TYPE_CODE.equalsIgnoreCase(item.getItemTypeCode()));
                purchasingAccountsPayableItemAsset.setItemTypeCode(item.getItemTypeCode());
            }
            purchasingAccountsPayableItemAsset.setItemAssignedToTradeInIndicator(item.getItemAssignedToTradeInIndicator());
            // PurchaseOrderItemIdentifier will be used when populate PurAp asset data
            final PurchaseOrderItem poi = getPurchaseOrderItemfromCreditMemoItem(item);
            if (poi != null) {
                purchasingAccountsPayableItemAsset.setPurchaseOrderItemIdentifier(poi.getItemIdentifier());
            }
            // if PREQ Credit Memo document
            final VendorCreditMemoDocument cmDoc = item.getPurapDocument();
            if (ObjectUtils.isNotNull(cmDoc) && cmDoc.isSourceDocumentPaymentRequest()) {
                purchasingAccountsPayableItemAsset.setPaymentRequestIdentifier(cmDoc.getPaymentRequestIdentifier());
            }
        }
    }

    /**
     * Retrieves a purchase order item for a given CreditMemoItem by inspecting the item type to see if its above the
     * line or below the line and returns the appropriate type.
     *
     * @param item
     * @return
     */
    protected PurchaseOrderItem getPurchaseOrderItemfromCreditMemoItem(final CreditMemoItem item) {
        if (ObjectUtils.isNotNull(item.getPurapDocumentIdentifier())) {
            if (ObjectUtils.isNull(item.getPurapDocument())) {
                item.refreshReferenceObject(PurapPropertyConstants.PURAP_DOC);
            }
        }
        // ideally we should do this a different way - maybe move it all into the service or save this info somehow
        // (make sure and update though)
        if (item.getPurapDocument() != null) {
            final PurchaseOrderDocument po = ((VendorCreditMemoDocument) item.getPurapDocument()).getPurchaseOrderDocument();
            final PurchaseOrderItem poi;
            if (item.getItemType().isLineItemIndicator()) {
                if (po.getItems() == null || po.getItems().isEmpty()) {
                    po.refreshReferenceObject("items");
                }
                poi = (PurchaseOrderItem) po.getItem(item.getItemLineNumber() - 1);
            } else {
                // To get the purchaseOrderItem by given CreditMemoItem. Since the additional charge type may be
                // different in CM and PO, there could be no PO Item for a given CM item.
                poi = (PurchaseOrderItem) purapService.getBelowTheLineByType(po,
                        item.getItemType());
            }
            if (poi != null) {
                return poi;
            } else {
                LOG.debug(
                        "getPurchaseOrderItemfromCreditMemoItem() Returning null because PurchaseOrderItem "
                        + "object for line number{}or itemType {} is null",
                        item::getItemLineNumber,
                        item::getItemTypeCode
                );
                return null;
            }
        } else {

            LOG.error("getPurchaseOrderItemfromCreditMemoItem() Returning null because paymentRequest object is null");
            throw new PurError("Credit Memo Object in Purchase Order item line number " + item.getItemLineNumber() +
                    "or itemType " + item.getItemTypeCode() + " is null");
        }
    }

    @Override
    public List<Long> retrieveValidAssetNumberForLocking(
            final Integer poId, final String capitalAssetSystemTypeCode,
            final PurApItem purApItem) {
        final List<Long> capitalAssetNumbers = new ArrayList<>();
        CapitalAssetSystem capitalAssetSystem = null;

        if (PurapConstants.CapitalAssetTabStrings.INDIVIDUAL_ASSETS.equalsIgnoreCase(capitalAssetSystemTypeCode)) {
            // If PurAp sets the CAMS as INDIVIDUAL system
            capitalAssetSystem = getCapitalAssetSystemForIndividual(poId, purApItem);

        } else if (PurapConstants.CapitalAssetTabStrings.ONE_SYSTEM.equalsIgnoreCase(capitalAssetSystemTypeCode)) {
            capitalAssetSystem = purchaseOrderService.retrieveCapitalAssetSystemForOneSystem(poId);
        } else if (PurapConstants.CapitalAssetTabStrings.MULTIPLE_SYSTEMS.equalsIgnoreCase(
                capitalAssetSystemTypeCode)) {
            final List<CapitalAssetSystem> capitalAssetSystems =
                    purchaseOrderService.retrieveCapitalAssetSystemsForMultipleSystem(poId);
            if (ObjectUtils.isNotNull(capitalAssetSystems) && !capitalAssetSystems.isEmpty()) {
                // PurAp doesn't support multiple system asset information for KFS3.0. It works as One system.
                capitalAssetSystem = capitalAssetSystems.get(0);
            }
        }

        if (ObjectUtils.isNotNull(capitalAssetSystem) && capitalAssetSystem.getItemCapitalAssets() != null
                && !capitalAssetSystem.getItemCapitalAssets().isEmpty()) {
            for (final ItemCapitalAsset itemCapitalAsset : capitalAssetSystem.getItemCapitalAssets()) {
                if (itemCapitalAsset.getCapitalAssetNumber() != null) {
                    final Map<String, Object> pKeys = new HashMap<>();
                    // Asset must be valid and capital active 'A','C','S','U'
                    pKeys.put(CamsPropertyConstants.Asset.CAPITAL_ASSET_NUMBER,
                            itemCapitalAsset.getCapitalAssetNumber());

                    final Asset asset = businessObjectService.findByPrimaryKey(Asset.class, pKeys);
                    if (ObjectUtils.isNotNull(asset) && assetService.isCapitalAsset(asset)
                            && !assetService.isAssetRetired(asset)) {
                        capitalAssetNumbers.add(itemCapitalAsset.getCapitalAssetNumber());
                    }
                }
            }
        }
        return capitalAssetNumbers;
    }

    protected CapitalAssetSystem getCapitalAssetSystemForIndividual(final Integer poId, final PurApItem purApItem) {
        final List<PurchasingCapitalAssetItem> capitalAssetItems =
                purchaseOrderService.retrieveCapitalAssetItemsForIndividual(poId);
        if (capitalAssetItems == null || capitalAssetItems.isEmpty()) {
            return null;
        }

        Integer purchaseOrderItemIdentifier = null;
        PurchaseOrderItem poi = null;
        if (purApItem instanceof PaymentRequestItem) {
            poi = ((PaymentRequestItem) purApItem).getPurchaseOrderItem();

        } else if (purApItem instanceof CreditMemoItem) {
            poi = getPurchaseOrderItemfromCreditMemoItem((CreditMemoItem) purApItem);
        }

        if (poi != null) {
            purchaseOrderItemIdentifier = poi.getItemIdentifier();
        }
        for (final PurchasingCapitalAssetItem capitalAssetItem : capitalAssetItems) {
            if (capitalAssetItem.getItemIdentifier().equals(purchaseOrderItemIdentifier)) {
                return capitalAssetItem.getPurchasingCapitalAssetSystem();
            }
        }
        return null;
    }

    public void setAssetService(final AssetService assetService) {
        this.assetService = assetService;
    }

    public BusinessObjectService getBusinessObjectService() {
        return businessObjectService;
    }

    public void setBusinessObjectService(final BusinessObjectService businessObjectService) {
        this.businessObjectService = businessObjectService;
    }

    public void setPurapService(final PurapService purapService) {
        this.purapService = purapService;
    }

    public void setPurchaseOrderService(final PurchaseOrderService purchaseOrderService) {
        this.purchaseOrderService = purchaseOrderService;
    }
}
