/*
 * The Kuali Financial System, a comprehensive financial management system for higher education.
 *
 * Copyright 2005-2023 Kuali, Inc.
 *
 * This program is free software: you can redistribute it and/or modify
 * it under the terms of the GNU Affero General Public License as published by
 * the Free Software Foundation, either version 3 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU Affero General Public License for more details.
 *
 * You should have received a copy of the GNU Affero General Public License
 * along with this program.  If not, see <http://www.gnu.org/licenses/>.
 */
package org.kuali.kfs.module.cam.document.validation.impl;

import org.apache.commons.lang3.StringUtils;
import org.kuali.kfs.coa.businessobject.AccountingPeriod;
import org.kuali.kfs.coa.businessobject.ObjectCode;
import org.kuali.kfs.coa.service.AccountingPeriodService;
import org.kuali.kfs.coa.service.ObjectCodeService;
import org.kuali.kfs.core.api.datetime.DateTimeService;
import org.kuali.kfs.coreservice.framework.parameter.ParameterService;
import org.kuali.kfs.datadictionary.legacy.BusinessObjectDictionaryService;
import org.kuali.kfs.datadictionary.legacy.DocumentDictionaryService;
import org.kuali.kfs.integration.cam.CapitalAssetManagementModuleService;
import org.kuali.kfs.kew.api.WorkflowDocument;
import org.kuali.kfs.kns.document.MaintenanceDocument;
import org.kuali.kfs.kns.document.authorization.MaintenanceDocumentAuthorizer;
import org.kuali.kfs.kns.maintenance.rules.MaintenanceDocumentRuleBase;
import org.kuali.kfs.krad.bo.PersistableBusinessObject;
import org.kuali.kfs.krad.service.BusinessObjectService;
import org.kuali.kfs.krad.util.GlobalVariables;
import org.kuali.kfs.krad.util.KRADPropertyConstants;
import org.kuali.kfs.krad.util.ObjectUtils;
import org.kuali.kfs.module.cam.CamsConstants;
import org.kuali.kfs.module.cam.CamsConstants.DocumentTypeName;
import org.kuali.kfs.module.cam.CamsKeyConstants;
import org.kuali.kfs.module.cam.CamsParameterConstants;
import org.kuali.kfs.module.cam.CamsPropertyConstants;
import org.kuali.kfs.module.cam.businessobject.Asset;
import org.kuali.kfs.module.cam.businessobject.AssetObjectCode;
import org.kuali.kfs.module.cam.businessobject.AssetPayment;
import org.kuali.kfs.module.cam.businessobject.AssetRetirementGlobal;
import org.kuali.kfs.module.cam.businessobject.AssetRetirementGlobalDetail;
import org.kuali.kfs.module.cam.document.gl.AssetRetirementGeneralLedgerPendingEntrySource;
import org.kuali.kfs.module.cam.document.service.AssetObjectCodeService;
import org.kuali.kfs.module.cam.document.service.AssetPaymentService;
import org.kuali.kfs.module.cam.document.service.AssetRetirementService;
import org.kuali.kfs.module.cam.document.service.AssetService;
import org.kuali.kfs.sys.KFSKeyConstants;
import org.kuali.kfs.sys.businessobject.DocumentHeader;
import org.kuali.kfs.sys.context.SpringContext;
import org.kuali.kfs.sys.service.GeneralLedgerPendingEntryService;

import java.sql.Date;
import java.util.ArrayList;
import java.util.HashMap;
import java.util.List;
import java.util.Locale;
import java.util.Map;

/**
 * Business rules applicable to AssetLocationGlobal documents.
 */
public class AssetRetirementGlobalRule extends MaintenanceDocumentRuleBase {

    protected PersistableBusinessObject bo;

    @Override
    public void setupConvenienceObjects() {
        final AssetRetirementGlobal newRetirementGlobal = (AssetRetirementGlobal) super.getNewBo();
        newRetirementGlobal.refreshNonUpdateableReferences();
        for (final AssetRetirementGlobalDetail detail : newRetirementGlobal.getAssetRetirementGlobalDetails()) {
            detail.refreshNonUpdateableReferences();
        }
    }

    /**
     * Forces the processing of rules when saving.
     *
     * @param document MaintenanceDocument
     * @return boolean true when valid; Namely we need to enforce foreign key constraints else
     * processCustomSaveDocumentBusinessRules does not force user back to the document for error correction.
     */
    @Override
    protected boolean isDocumentValidForSave(final MaintenanceDocument document) {
        boolean valid = super.isDocumentValidForSave(document);

        if (valid) {
            final AssetRetirementGlobal assetRetirementGlobal = (AssetRetirementGlobal) document.getNewMaintainableObject()
                    .getBusinessObject();

            setupConvenienceObjects();
            valid = assetRetirementValidation(assetRetirementGlobal, document);
        }

        return valid;
    }

    /**
     * Processes rules when saving this global.
     *
     * @param document MaintenanceDocument type of document to be processed.
     * @return boolean true when success
     */
    @Override
    protected boolean processCustomSaveDocumentBusinessRules(final MaintenanceDocument document) {
        final AssetRetirementGlobal assetRetirementGlobal =
                (AssetRetirementGlobal) document.getNewMaintainableObject().getBusinessObject();

        setupConvenienceObjects();
        boolean valid = assetRetirementValidation(assetRetirementGlobal, document);

        if (valid && super.processCustomSaveDocumentBusinessRules(document)
            && !getAssetRetirementService().isAssetRetiredByMerged(assetRetirementGlobal)
            && !allPaymentsFederalOwned(assetRetirementGlobal)) {
            // Check if Asset Object Code and Object code exists and active.
            valid = validateObjectCodesForGLPosting(assetRetirementGlobal);
            if (valid) {
                // create poster
                final AssetRetirementGeneralLedgerPendingEntrySource assetRetirementGlPoster =
                        new AssetRetirementGeneralLedgerPendingEntrySource(document.getDocumentHeader());

                // if no value was selected on the doc, use the current period
                if (assetRetirementGlobal.getPostingPeriodCode() == null
                        && assetRetirementGlobal.getPostingYear() == null) {
                    final Date date = SpringContext.getBean(DateTimeService.class).getCurrentSqlDate();
                    final AccountingPeriod currentPeriod = SpringContext.getBean(AccountingPeriodService.class).getByDate(date);
                    assetRetirementGlobal.setAccountingPeriodCompositeString(currentPeriod);
                }
                assetRetirementGlPoster.setPostingYear(assetRetirementGlobal.getPostingYear());
                assetRetirementGlPoster.setPostingPeriodCode(assetRetirementGlobal.getPostingPeriodCode());

                // create postables
                getAssetRetirementService().createGLPostables(assetRetirementGlobal, assetRetirementGlPoster);

                if (SpringContext.getBean(GeneralLedgerPendingEntryService.class)
                        .generateGeneralLedgerPendingEntries(assetRetirementGlPoster)) {
                    assetRetirementGlobal.setGeneralLedgerPendingEntries(assetRetirementGlPoster.getPendingEntries());
                } else {
                    assetRetirementGlPoster.getPendingEntries().clear();
                }
            }
        }

        // add doc header description if retirement reason is "MERGED"
        if (CamsConstants.AssetRetirementReasonCode.MERGED.equals(assetRetirementGlobal.getRetirementReasonCode())) {
            if (!document.getDocumentHeader().getDocumentDescription().toLowerCase(Locale.US)
                    .contains(CamsConstants.AssetRetirementGlobal.MERGE_AN_ASSET_DESCRIPTION.toLowerCase(Locale.US))) {
                final Integer maxDocumentDescription = ddService.getAttributeMaxLength(DocumentHeader.class,
                        KRADPropertyConstants.DOCUMENT_DESCRIPTION);
                String documentDescription = CamsConstants.AssetRetirementGlobal.MERGE_AN_ASSET_DESCRIPTION + " "
                        + document.getDocumentHeader().getDocumentDescription();
                documentDescription = StringUtils.left(documentDescription, maxDocumentDescription);
                document.getDocumentHeader().setDocumentDescription(documentDescription);
            }
        }
        // get asset locks
        final List<Long> capitalAssetNumbers = retrieveAssetNumbersForLocking(assetRetirementGlobal);
        valid &= !getCapitalAssetManagementModuleService().isAssetLocked(capitalAssetNumbers,
                DocumentTypeName.ASSET_RETIREMENT_GLOBAL, document.getDocumentNumber());

        return valid;
    }

    protected List<Long> retrieveAssetNumbersForLocking(final AssetRetirementGlobal assetRetirementGlobal) {
        final List<Long> capitalAssetNumbers = new ArrayList<>();
        if (getAssetRetirementService().isAssetRetiredByMerged(assetRetirementGlobal)
                && assetRetirementGlobal.getMergedTargetCapitalAssetNumber() != null) {
            capitalAssetNumbers.add(assetRetirementGlobal.getMergedTargetCapitalAssetNumber());
        }

        for (final AssetRetirementGlobalDetail retirementDetail : assetRetirementGlobal.getAssetRetirementGlobalDetails()) {
            if (retirementDetail.getCapitalAssetNumber() != null) {
                capitalAssetNumbers.add(retirementDetail.getCapitalAssetNumber());
            }
        }
        return capitalAssetNumbers;
    }

    @Override
    protected boolean processCustomRouteDocumentBusinessRules(final MaintenanceDocument document) {
        boolean valid = super.processCustomRouteDocumentBusinessRules(document);

        final WorkflowDocument workflowDoc = document.getDocumentHeader().getWorkflowDocument();
        // adding asset locks
        if (!GlobalVariables.getMessageMap().hasErrors() && (workflowDoc.isInitiated() || workflowDoc.isSaved())) {
            valid &= setAssetLocks(document);

        }
        return valid;
    }

    protected boolean setAssetLocks(final MaintenanceDocument document) {
        final AssetRetirementGlobal assetRetirementGlobal = (AssetRetirementGlobal) document.getNewMaintainableObject()
                .getBusinessObject();
        // get asset locks
        return getCapitalAssetManagementModuleService().storeAssetLocks(
                retrieveAssetNumbersForLocking(assetRetirementGlobal), document.getDocumentNumber(),
                DocumentTypeName.ASSET_RETIREMENT_GLOBAL, null);
    }

    protected CapitalAssetManagementModuleService getCapitalAssetManagementModuleService() {
        return SpringContext.getBean(CapitalAssetManagementModuleService.class);
    }

    /**
     * Check if all asset payments are federal owned.
     *
     * @param assetRetirementGlobal
     * @return
     */
    protected boolean allPaymentsFederalOwned(final AssetRetirementGlobal assetRetirementGlobal) {
        final List<AssetRetirementGlobalDetail> assetRetirementGlobalDetails =
                assetRetirementGlobal.getAssetRetirementGlobalDetails();
        for (final AssetRetirementGlobalDetail assetRetirementGlobalDetail : assetRetirementGlobalDetails) {
            for (final AssetPayment assetPayment : assetRetirementGlobalDetail.getAsset().getAssetPayments()) {
                if (!getAssetPaymentService().isPaymentFederalOwned(assetPayment)) {
                    return false;
                }
            }
        }
        return true;
    }

    /**
     * Validate Asset Object Codes and Fin Object Codes eligible for GL Posting.
     *
     * @param assetRetirementGlobal
     * @return
     */
    protected boolean validateObjectCodesForGLPosting(final AssetRetirementGlobal assetRetirementGlobal) {
        boolean valid = true;
        final List<AssetRetirementGlobalDetail> assetRetirementGlobalDetails =
                assetRetirementGlobal.getAssetRetirementGlobalDetails();
        Asset asset;
        List<AssetPayment> assetPayments;

        for (final AssetRetirementGlobalDetail assetRetirementGlobalDetail : assetRetirementGlobalDetails) {
            asset = assetRetirementGlobalDetail.getAsset();
            assetPayments = asset.getAssetPayments();
            for (final AssetPayment assetPayment : assetPayments) {
                if (!getAssetPaymentService().isPaymentFederalOwned(assetPayment)) {
                    // validate Asset Object Code and Financial Object Codes respectively
                    valid &= validateAssetObjectCode(asset, assetPayment);
                    if (valid) {
                        valid = validateFinancialObjectCodes(asset, assetPayment);
                    }
                }
            }
        }

        return valid;
    }

    /**
     * Check Financial Object Code for GLPE.
     *
     * @param asset
     * @param assetPayment
     * @return
     */
    protected boolean validateFinancialObjectCodes(final Asset asset, final AssetPayment assetPayment) {
        final AssetPaymentService assetPaymentService = getAssetPaymentService();
        boolean valid = true;

        final ObjectCodeService objectCodeService = SpringContext.getBean(ObjectCodeService.class);
        final ObjectCode objectCode = objectCodeService.getByPrimaryIdForLatestValidYear(assetPayment.getChartOfAccountsCode(),
                assetPayment.getFinancialObjectCode());
        final AssetObjectCode assetObjectCode = getAssetObjectCodeService().findAssetObjectCode(
                asset.getOrganizationOwnerChartOfAccountsCode(), objectCode.getFinancialObjectSubTypeCode());
        if (assetPaymentService.isPaymentEligibleForCapitalizationGLPosting(assetPayment)) {
            // check for capitalization financial object code existing.
            assetObjectCode.refreshReferenceObject(
                    CamsPropertyConstants.AssetObjectCode.CAPITALIZATION_FINANCIAL_OBJECT);
            valid = validateFinObjectCodeForGLPosting(asset.getOrganizationOwnerChartOfAccountsCode(),
                    assetObjectCode.getCapitalizationFinancialObjectCode(),
                    assetObjectCode.getCapitalizationFinancialObject(),
                    CamsConstants.GLPostingObjectCodeType.CAPITALIZATION);
        }
        if (assetPaymentService.isPaymentEligibleForAccumDeprGLPosting(assetPayment)) {
            // check for accumulate depreciation financial Object Code existing
            assetObjectCode.refreshReferenceObject(
                    CamsPropertyConstants.AssetObjectCode.ACCUMULATED_DEPRECIATION_FINANCIAL_OBJECT);
            valid &= validateFinObjectCodeForGLPosting(asset.getOrganizationOwnerChartOfAccountsCode(),
                    assetObjectCode.getAccumulatedDepreciationFinancialObjectCode(),
                    assetObjectCode.getAccumulatedDepreciationFinancialObject(),
                    CamsConstants.GLPostingObjectCodeType.ACCUMULATE_DEPRECIATION);
        }
        if (assetPaymentService.isPaymentEligibleForOffsetGLPosting(assetPayment)) {
            // check for offset financial object code existing.
            valid &= validateFinObjectCodeForGLPosting(
                asset.getOrganizationOwnerChartOfAccountsCode(),
                SpringContext.getBean(ParameterService.class).getParameterValueAsString(AssetRetirementGlobal.class,
                        CamsParameterConstants.GAIN_LOSS_OBJECT_CODE
                ),
                    getAssetRetirementService().getOffsetFinancialObject(asset.getOrganizationOwnerChartOfAccountsCode()),
                    CamsConstants.GLPostingObjectCodeType.OFFSET_AMOUNT);
        }
        return valid;
    }

    /**
     * check existence and active status for given financial Object Code BO.
     *
     * @param chartOfAccountsCode
     * @param finObjectCode
     * @param finObject
     * @return
     */
    protected boolean validateFinObjectCodeForGLPosting(
            final String chartOfAccountsCode, final String finObjectCode,
            final ObjectCode finObject, final String glPostingType) {
        boolean valid = true;
        // not defined in Asset Object Code table
        if (StringUtils.isBlank(finObjectCode)) {
            GlobalVariables.getMessageMap().putErrorForSectionId(
                    CamsConstants.AssetRetirementGlobal.SECTION_ID_ASSET_DETAIL_INFORMATION,
                    CamsKeyConstants.GLPosting.ERROR_OBJECT_CODE_FROM_ASSET_OBJECT_CODE_NOT_FOUND,
                    glPostingType, chartOfAccountsCode);
            valid = false;
        } else if (ObjectUtils.isNull(finObject)) {
            // check Object Code existing
            GlobalVariables.getMessageMap().putErrorForSectionId(
                    CamsConstants.AssetRetirementGlobal.SECTION_ID_ASSET_DETAIL_INFORMATION,
                    CamsKeyConstants.GLPosting.ERROR_OBJECT_CODE_FROM_ASSET_OBJECT_CODE_INVALID,
                    glPostingType, finObjectCode, chartOfAccountsCode);
            valid = false;
        } else if (!finObject.isActive()) {
            // check Object Code active
            GlobalVariables.getMessageMap().putErrorForSectionId(
                    CamsConstants.AssetRetirementGlobal.SECTION_ID_ASSET_DETAIL_INFORMATION,
                    CamsKeyConstants.GLPosting.ERROR_OBJECT_CODE_FROM_ASSET_OBJECT_CODE_INACTIVE,
                    glPostingType, finObjectCode, chartOfAccountsCode);
            valid = false;
        }
        return valid;
    }

    /**
     * Asset Object Code must exist as an active status.
     *
     * @param asset
     * @param assetPayment
     * @return
     */
    protected boolean validateAssetObjectCode(final Asset asset, final AssetPayment assetPayment) {
        boolean valid = true;
        final ObjectCodeService objectCodeService = SpringContext.getBean(ObjectCodeService.class);
        final ObjectCode objectCode = objectCodeService.getByPrimaryIdForLatestValidYear(assetPayment.getChartOfAccountsCode(),
                assetPayment.getFinancialObjectCode());
        final AssetObjectCode assetObjectCode = getAssetObjectCodeService().findAssetObjectCode(
                asset.getOrganizationOwnerChartOfAccountsCode(), objectCode.getFinancialObjectSubTypeCode());
        // check Asset Object Code existing.
        if (ObjectUtils.isNull(assetObjectCode)) {
            GlobalVariables.getMessageMap().putErrorForSectionId(
                    CamsConstants.AssetRetirementGlobal.SECTION_ID_ASSET_DETAIL_INFORMATION,
                    CamsKeyConstants.GLPosting.ERROR_ASSET_OBJECT_CODE_NOT_FOUND,
                    asset.getOrganizationOwnerChartOfAccountsCode(), objectCode.getFinancialObjectSubTypeCode());
            valid = false;
        } else if (!assetObjectCode.isActive()) {
            // check Asset Object Code active
            GlobalVariables.getMessageMap().putErrorForSectionId(
                    CamsConstants.AssetRetirementGlobal.SECTION_ID_ASSET_DETAIL_INFORMATION,
                    CamsKeyConstants.GLPosting.ERROR_ASSET_OBJECT_CODE_INACTIVE,
                    asset.getOrganizationOwnerChartOfAccountsCode(), objectCode.getFinancialObjectSubTypeCode());
            valid = false;
        }

        return valid;
    }

    @Override
    public boolean processCustomAddCollectionLineBusinessRules(
            final MaintenanceDocument document, final String collectionName,
            final PersistableBusinessObject line) {
        boolean success = true;
        final AssetRetirementGlobalDetail assetRetirementGlobalDetail = (AssetRetirementGlobalDetail) line;
        final AssetRetirementGlobal assetRetirementGlobal = (AssetRetirementGlobal) document.getDocumentBusinessObject();

        if (!checkEmptyValue(assetRetirementGlobalDetail.getCapitalAssetNumber())) {
            GlobalVariables.getMessageMap().putError(
                    CamsPropertyConstants.AssetRetirementGlobalDetail.CAPITAL_ASSET_NUMBER,
                    CamsKeyConstants.Retirement.ERROR_BLANK_CAPITAL_ASSET_NUMBER);
            return false;
        }
        assetRetirementGlobalDetail.refreshReferenceObject(CamsPropertyConstants.AssetLocationGlobalDetail.ASSET);

        if (ObjectUtils.isNull(assetRetirementGlobalDetail.getAsset())) {
            GlobalVariables.getMessageMap().putError(
                    CamsPropertyConstants.AssetRetirementGlobalDetail.CAPITAL_ASSET_NUMBER,
                    CamsKeyConstants.Retirement.ERROR_INVALID_CAPITAL_ASSET_NUMBER,
                    assetRetirementGlobalDetail.getCapitalAssetNumber().toString());
            return false;
        }

        if (getAssetService().isAssetLoaned(assetRetirementGlobalDetail.getAsset())) {
            GlobalVariables.getMessageMap().putError(
                    CamsPropertyConstants.AssetRetirementGlobalDetail.CAPITAL_ASSET_NUMBER,
                    CamsKeyConstants.Retirement.ERROR_LOANED_ASSET_CANNOT_RETIRED);
            success = false;
        } else if (!getAssetService().isDocumentEnrouting(document)) {
            success = checkRetirementDetailOneLine(assetRetirementGlobalDetail, assetRetirementGlobal, document);
            success &= checkRetireMultipleAssets(assetRetirementGlobal.getRetirementReasonCode(),
                    assetRetirementGlobal.getAssetRetirementGlobalDetails(), 0, document);
        }

        // Calculate summary fields in order to show the values even though add new line fails.
        for (final AssetRetirementGlobalDetail detail : assetRetirementGlobal.getAssetRetirementGlobalDetails()) {
            getAssetService().setAssetSummaryFields(detail.getAsset());
        }
        return success & super.processCustomAddCollectionLineBusinessRules(document, collectionName, line);
    }

    /**
     * Check if only single asset is allowed to retire.
     *
     * @param retirementReasonCode
     * @param assetRetirementDetails
     * @param maxNumber
     * @param maintenanceDocument
     * @return
     */
    protected boolean checkRetireMultipleAssets(
            final String retirementReasonCode,
            final List<AssetRetirementGlobalDetail> assetRetirementDetails, final Integer maxNumber,
            final MaintenanceDocument maintenanceDocument) {
        boolean success = true;

        if (assetRetirementDetails.size() > maxNumber
                && !getAssetRetirementService().isAllowedRetireMultipleAssets(maintenanceDocument)) {
            GlobalVariables.getMessageMap().putErrorForSectionId(
                    CamsConstants.AssetRetirementGlobal.SECTION_ID_ASSET_DETAIL_INFORMATION,
                    CamsKeyConstants.Retirement.ERROR_MULTIPLE_ASSET_RETIRED);
            success = false;
        }

        return success;
    }

    /**
     * This method validates each asset to be retired.
     *
     * @param assetRetirementGlobal
     * @param maintenanceDocument
     * @return
     */
    protected boolean validateRetirementDetails(
            final AssetRetirementGlobal assetRetirementGlobal,
            final MaintenanceDocument maintenanceDocument) {
        boolean success = true;

        final List<AssetRetirementGlobalDetail> assetRetirementGlobalDetails =
                assetRetirementGlobal.getAssetRetirementGlobalDetails();

        if (assetRetirementGlobalDetails.size() == 0) {
            success = false;
            GlobalVariables.getMessageMap().putErrorForSectionId(
                    CamsConstants.AssetRetirementGlobal.SECTION_ID_ASSET_DETAIL_INFORMATION,
                    CamsKeyConstants.Retirement.ERROR_ASSET_RETIREMENT_GLOBAL_NO_ASSET);
        } else {
            // validate each asset retirement detail
            int index = 0;
            for (final AssetRetirementGlobalDetail detail : assetRetirementGlobalDetails) {
                final String errorPath = MAINTAINABLE_ERROR_PREFIX +
                                         CamsPropertyConstants.AssetRetirementGlobal.ASSET_RETIREMENT_GLOBAL_DETAILS + "[" + index++ +
                                         "]";
                GlobalVariables.getMessageMap().addToErrorPath(errorPath);

                success &= checkRetirementDetailOneLine(detail, assetRetirementGlobal, maintenanceDocument);

                GlobalVariables.getMessageMap().removeFromErrorPath(errorPath);
            }
        }
        return success;
    }

    /**
     * This method validates one asset is a valid asset and no duplicate with target asset when merge.
     *
     * @param assetRetirementGlobalDetail
     * @param assetRetirementGlobal
     * @param maintenanceDocument
     * @return
     */
    protected boolean checkRetirementDetailOneLine(
            final AssetRetirementGlobalDetail assetRetirementGlobalDetail,
            final AssetRetirementGlobal assetRetirementGlobal, final MaintenanceDocument maintenanceDocument) {
        boolean success;

        assetRetirementGlobalDetail.refreshReferenceObject(CamsPropertyConstants.AssetRetirementGlobalDetail.ASSET);

        final Asset asset = assetRetirementGlobalDetail.getAsset();

        if (ObjectUtils.isNull(asset)) {
            success = false;
            GlobalVariables.getMessageMap().putErrorForSectionId(
                    CamsConstants.AssetRetirementGlobal.SECTION_ID_ASSET_DETAIL_INFORMATION,
                    CamsKeyConstants.Retirement.ERROR_INVALID_CAPITAL_ASSET_NUMBER,
                    assetRetirementGlobalDetail.getCapitalAssetNumber().toString());
        } else {
            success = validateActiveCapitalAsset(asset);
            success &= validateNonMoveableAsset(asset, maintenanceDocument);

            if (getAssetRetirementService().isAssetRetiredByMerged(assetRetirementGlobal)) {
                success &= validateDuplicateAssetNumber(assetRetirementGlobal.getMergedTargetCapitalAssetNumber(),
                        assetRetirementGlobalDetail.getCapitalAssetNumber());
            }

            // Set asset non persistent fields
            getAssetService().setAssetSummaryFields(asset);
        }

        return success;
    }

    /**
     * Check for Merge Asset, no duplicate capitalAssetNumber between "from" and "to".
     *
     * @param targetAssetNumber
     * @param sourceAssetNumber
     * @return
     */
    protected boolean validateDuplicateAssetNumber(final Long targetAssetNumber, final Long sourceAssetNumber) {
        boolean success = true;

        if (getAssetService().isCapitalAssetNumberDuplicate(targetAssetNumber, sourceAssetNumber)) {
            GlobalVariables.getMessageMap().putError(
                    CamsPropertyConstants.AssetRetirementGlobalDetail.CAPITAL_ASSET_NUMBER,
                    CamsKeyConstants.Retirement.ERROR_DUPLICATE_CAPITAL_ASSET_NUMBER_WITH_TARGET,
                    sourceAssetNumber.toString());
            success = false;
        }

        return success;
    }

    /**
     * User must be in work group CM_SUPER_USERS to retire a non-moveable asset.
     *
     * @param asset
     * @param maintenanceDocument
     * @return
     */
    protected boolean validateNonMoveableAsset(final Asset asset, final MaintenanceDocument maintenanceDocument) {
        boolean success = true;

        final MaintenanceDocumentAuthorizer documentAuthorizer = (MaintenanceDocumentAuthorizer) SpringContext.getBean(
                DocumentDictionaryService.class).getDocumentAuthorizer(maintenanceDocument);
        final boolean isAuthorized = documentAuthorizer.isAuthorized(maintenanceDocument, CamsConstants.CAM_MODULE_CODE,
                CamsConstants.PermissionNames.RETIRE_NON_MOVABLE_ASSETS,
                GlobalVariables.getUserSession().getPerson().getPrincipalId());

        if (!getAssetService().isAssetMovableCheckByAsset(asset) && !isAuthorized) {
            GlobalVariables.getMessageMap().putError(
                    CamsPropertyConstants.AssetRetirementGlobalDetail.CAPITAL_ASSET_NUMBER,
                    CamsKeyConstants.Retirement.ERROR_INVALID_USER_GROUP_FOR_NON_MOVABLE_ASSET,
                    asset.getCapitalAssetNumber().toString());
            success = false;
        }

        return success;
    }

    /**
     * Validate Asset Retirement Global and Details.
     *
     * @param assetRetirementGlobal
     * @param maintenanceDocument
     * @return
     */
    protected boolean assetRetirementValidation(
            final AssetRetirementGlobal assetRetirementGlobal,
            final MaintenanceDocument maintenanceDocument) {
        boolean valid = validateRequiredGlobalFields(assetRetirementGlobal);
        if (getAssetRetirementService().isAssetRetiredByMerged(assetRetirementGlobal)) {
            valid &= validateMergeTargetAsset(assetRetirementGlobal);
        }

        if (!getAssetService().isDocumentEnrouting(maintenanceDocument)) {
            valid &= validateRetirementDetails(assetRetirementGlobal, maintenanceDocument);
            valid &= checkRetireMultipleAssets(assetRetirementGlobal.getRetirementReasonCode(),
                    assetRetirementGlobal.getAssetRetirementGlobalDetails(), 1, maintenanceDocument);
        }
        return valid;
    }

    /**
     * Validate mergedTargetCapitalAsset. Only valid and active capital asset is allowed.
     *
     * @param assetRetirementGlobal
     * @return
     */
    protected boolean validateMergeTargetAsset(final AssetRetirementGlobal assetRetirementGlobal) {
        boolean valid = true;
        final Asset targetAsset = assetRetirementGlobal.getMergedTargetCapitalAsset();
        final Long targetAssetNumber = assetRetirementGlobal.getMergedTargetCapitalAssetNumber();

        if (!checkEmptyValue(targetAssetNumber)) {
            putFieldError(CamsPropertyConstants.AssetRetirementGlobal.MERGED_TARGET_CAPITAL_ASSET_NUMBER,
                    CamsKeyConstants.Retirement.ERROR_BLANK_CAPITAL_ASSET_NUMBER);
            valid = false;
        } else if (ObjectUtils.isNull(targetAsset)) {
            putFieldError(CamsPropertyConstants.AssetRetirementGlobal.MERGED_TARGET_CAPITAL_ASSET_NUMBER,
                    CamsKeyConstants.Retirement.ERROR_INVALID_MERGED_TARGET_ASSET_NUMBER, targetAssetNumber.toString());
            valid = false;
        } else {
            // Check asset of capital and active
            if (!getAssetService().isCapitalAsset(targetAsset)) {
                putFieldError(CamsPropertyConstants.AssetRetirementGlobal.MERGED_TARGET_CAPITAL_ASSET_NUMBER,
                        CamsKeyConstants.Retirement.ERROR_NON_CAPITAL_ASSET_RETIREMENT, targetAssetNumber.toString());
                valid = false;
            }
            if (getAssetService().isAssetRetired(targetAsset)) {
                putFieldError(CamsPropertyConstants.AssetRetirementGlobal.MERGED_TARGET_CAPITAL_ASSET_NUMBER,
                        CamsKeyConstants.Retirement.ERROR_NON_ACTIVE_ASSET_RETIREMENT, targetAssetNumber.toString());
                valid = false;
            }
        }

        // Validating the mergeAssetDescription is not blank
        if (!checkEmptyValue(assetRetirementGlobal.getMergedTargetCapitalAssetDescription())) {
            final String label = SpringContext.getBean(BusinessObjectDictionaryService.class)
                    .getBusinessObjectEntry(AssetRetirementGlobal.class.getName()).getAttributeDefinition(
                            CamsPropertyConstants.AssetRetirementGlobal.MERGED_TARGET_CAPITAL_ASSET_DESC).getLabel();
            putFieldError(CamsPropertyConstants.AssetRetirementGlobal.MERGED_TARGET_CAPITAL_ASSET_DESC,
                    KFSKeyConstants.ERROR_REQUIRED, label);
            valid = false;
        }

        return valid;
    }

    /**
     * Only active capital equipment can be retired using the asset retirement document.
     *
     * @param asset
     * @return
     */
    protected boolean validateActiveCapitalAsset(final Asset asset) {
        boolean valid = true;

        if (!getAssetService().isCapitalAsset(asset)) {
            GlobalVariables.getMessageMap().putError(
                    CamsPropertyConstants.AssetRetirementGlobalDetail.CAPITAL_ASSET_NUMBER,
                    CamsKeyConstants.Retirement.ERROR_NON_CAPITAL_ASSET_RETIREMENT,
                    asset.getCapitalAssetNumber().toString());
            valid = false;
        } else if (getAssetService().isAssetRetired(asset)) {
            GlobalVariables.getMessageMap().putError(
                    CamsPropertyConstants.AssetRetirementGlobalDetail.CAPITAL_ASSET_NUMBER,
                    CamsKeyConstants.Retirement.ERROR_NON_ACTIVE_ASSET_RETIREMENT,
                    asset.getCapitalAssetNumber().toString());
            valid = false;
        } else if (!validateAssetOnLoan(asset)) {
            valid = false;
        }

        return valid;
    }

    /**
     * Validate required fields for given retirement reason code
     *
     * @param assetRetirementGlobal
     * @return
     */
    protected boolean validateRequiredGlobalFields(final AssetRetirementGlobal assetRetirementGlobal) {
        boolean valid = true;

        if (getAssetRetirementService().isAssetRetiredBySold(assetRetirementGlobal)) {
            if (StringUtils.isBlank(assetRetirementGlobal.getBuyerDescription())) {
                putFieldError(CamsPropertyConstants.AssetRetirementGlobalDetail.BUYER_DESCRIPTION,
                        CamsKeyConstants.Retirement.ERROR_RETIREMENT_DETAIL_INFO_NULL, new String[]{
                            CamsConstants.RetirementLabel.BUYER_DESCRIPTION,
                            getAssetRetirementService().getAssetRetirementReasonName(assetRetirementGlobal)});
                valid = false;
            }
            if (assetRetirementGlobal.getSalePrice() == null) {
                putFieldError(CamsPropertyConstants.AssetRetirementGlobalDetail.SALE_PRICE,
                        CamsKeyConstants.Retirement.ERROR_RETIREMENT_DETAIL_INFO_NULL, new String[]{
                            CamsConstants.RetirementLabel.SALE_PRICE,
                            getAssetRetirementService().getAssetRetirementReasonName(assetRetirementGlobal)});
                valid = false;
            }
            valid &= validateCashReceiptFinancialDocumentNumber(
                    assetRetirementGlobal.getCashReceiptFinancialDocumentNumber());
        } else if (getAssetRetirementService().isAssetRetiredByAuction(assetRetirementGlobal)) {
            valid = validateCashReceiptFinancialDocumentNumber(
                    assetRetirementGlobal.getCashReceiptFinancialDocumentNumber());
        } else if (getAssetRetirementService().isAssetRetiredByExternalTransferOrGift(assetRetirementGlobal)
                && StringUtils.isBlank(assetRetirementGlobal.getRetirementInstitutionName())) {
            putFieldError(CamsPropertyConstants.AssetRetirementGlobalDetail.RETIREMENT_INSTITUTION_NAME,
                    CamsKeyConstants.Retirement.ERROR_RETIREMENT_DETAIL_INFO_NULL, new String[]{
                        CamsConstants.RetirementLabel.RETIREMENT_INSTITUTION_NAME,
                        getAssetRetirementService().getAssetRetirementReasonName(assetRetirementGlobal)});
            valid = false;
        } else if (getAssetRetirementService().isAssetRetiredByTheft(assetRetirementGlobal)
                && StringUtils.isBlank(assetRetirementGlobal.getPaidCaseNumber())) {
            putFieldError(CamsPropertyConstants.AssetRetirementGlobalDetail.PAID_CASE_NUMBER,
                    CamsKeyConstants.Retirement.ERROR_RETIREMENT_DETAIL_INFO_NULL, new String[]{
                        CamsConstants.RetirementLabel.PAID_CASE_NUMBER,
                        getAssetRetirementService().getAssetRetirementReasonName(assetRetirementGlobal)});
            valid = false;
        }
        return valid;
    }

    /**
     * validates Cash Receipt Financial Document Number
     *
     * @param documentNumber
     * @return boolean
     */
    protected boolean validateCashReceiptFinancialDocumentNumber(final String documentNumber) {
        boolean valid = true;
        if (StringUtils.isNotBlank(documentNumber)) {
            final Map<String, String> retirementInfoMap = new HashMap<>();
            retirementInfoMap.put(CamsPropertyConstants.AssetGlobal.DOCUMENT_NUMBER, documentNumber);
            bo = SpringContext.getBean(BusinessObjectService.class).findByPrimaryKey(
                    DocumentHeader.class, retirementInfoMap);
            if (ObjectUtils.isNull(bo)) {
                putFieldError(CamsPropertyConstants.AssetRetirementGlobalDetail.CASH_RECEIPT_FINANCIAL_DOCUMENT_NUMBER,
                        CamsKeyConstants.Retirement.ERROR_INVALID_RETIREMENT_DETAIL_INFO,
                        new String[]{CamsConstants.RetirementLabel.CASH_RECEIPT_FINANCIAL_DOCUMENT_NUMBER,
                            documentNumber});
                valid = false;
            }
        }
        return valid;
    }

    /**
     * Validates whether or not asset is on loan status
     *
     * @param asset
     * @return boolean
     */
    protected boolean validateAssetOnLoan(final Asset asset) {
        boolean success = true;
        if (getAssetService().isAssetLoaned(asset)) {
            GlobalVariables.getMessageMap().putErrorForSectionId(
                    CamsConstants.AssetRetirementGlobal.SECTION_ID_ASSET_DETAIL_INFORMATION,
                    CamsKeyConstants.Retirement.ERROR_LOANED_ASSET_CANNOT_RETIRED);
            success = false;
        }
        return success;
    }

    protected AssetService getAssetService() {
        return SpringContext.getBean(AssetService.class);
    }

    protected AssetRetirementService getAssetRetirementService() {
        return SpringContext.getBean(AssetRetirementService.class);
    }

    protected AssetPaymentService getAssetPaymentService() {
        return SpringContext.getBean(AssetPaymentService.class);
    }

    protected AssetObjectCodeService getAssetObjectCodeService() {
        return SpringContext.getBean(AssetObjectCodeService.class);
    }
}
