/*
 * The Kuali Financial System, a comprehensive financial management system for higher education.
 *
 * Copyright 2005-2023 Kuali, Inc.
 *
 * This program is free software: you can redistribute it and/or modify
 * it under the terms of the GNU Affero General Public License as published by
 * the Free Software Foundation, either version 3 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU Affero General Public License for more details.
 *
 * You should have received a copy of the GNU Affero General Public License
 * along with this program.  If not, see <http://www.gnu.org/licenses/>.
 */
package org.kuali.kfs.module.cam.document.validation.impl;

import org.apache.commons.lang3.StringUtils;
import org.apache.logging.log4j.LogManager;
import org.apache.logging.log4j.Logger;
import org.kuali.kfs.coreservice.framework.parameter.ParameterService;
import org.kuali.kfs.kns.rules.TransactionalDocumentRuleBase;
import org.kuali.kfs.datadictionary.legacy.BusinessObjectDictionaryService;
import org.kuali.kfs.krad.document.Document;
import org.kuali.kfs.krad.service.BusinessObjectService;
import org.kuali.kfs.krad.util.ErrorMessage;
import org.kuali.kfs.krad.util.GlobalVariables;
import org.kuali.kfs.krad.util.ObjectUtils;
import org.kuali.kfs.module.cam.CamsConstants;
import org.kuali.kfs.module.cam.CamsKeyConstants;
import org.kuali.kfs.module.cam.CamsParameterConstants;
import org.kuali.kfs.module.cam.CamsPropertyConstants;
import org.kuali.kfs.module.cam.businessobject.Asset;
import org.kuali.kfs.module.cam.businessobject.AssetCondition;
import org.kuali.kfs.module.cam.businessobject.AssetType;
import org.kuali.kfs.module.cam.businessobject.BarcodeInventoryErrorDetail;
import org.kuali.kfs.module.cam.document.BarcodeInventoryErrorDocument;
import org.kuali.kfs.module.cam.document.service.AssetService;
import org.kuali.kfs.module.cam.service.AssetLockService;
import org.kuali.kfs.sys.KFSConstants;
import org.kuali.kfs.sys.KFSKeyConstants;
import org.kuali.kfs.sys.KFSPropertyConstants;
import org.kuali.kfs.sys.businessobject.Building;
import org.kuali.kfs.sys.businessobject.Campus;
import org.kuali.kfs.sys.businessobject.Room;
import org.kuali.kfs.sys.context.SpringContext;
import org.kuali.kfs.core.api.datetime.DateTimeService;

import java.sql.Timestamp;
import java.text.MessageFormat;
import java.util.ArrayList;
import java.util.Collection;
import java.util.HashMap;
import java.util.List;

/**
 * Business rule(s) applicable to Asset Barcode Inventory upload and Barcode inventory error document.
 */
public class BarcodeInventoryErrorDocumentRule extends TransactionalDocumentRuleBase {

    private static final Logger LOG = LogManager.getLogger();

    private AssetService assetService;

    @Override
    protected boolean processCustomSaveDocumentBusinessRules(final Document document) {
        return true;
    }

    /**
     * Invokes several methods that validates each barcode error record
     *
     * @param document
     * @param updateStatus
     * @return
     */
    public boolean validateBarcodeInventoryErrorDetail(final BarcodeInventoryErrorDocument document, final boolean updateStatus) {
        String errorPath;
        boolean valid;
        final List<BarcodeInventoryErrorDetail> barcodeInventoryErrorDetails = document.getBarcodeInventoryErrorDetail();

        long lineNumber = 0L;
        for (final BarcodeInventoryErrorDetail barcodeInventoryErrorDetail : barcodeInventoryErrorDetails) {
            barcodeInventoryErrorDetail.setErrorDescription("");
            if (barcodeInventoryErrorDetail.getErrorCorrectionStatusCode().equals(
                    CamsConstants.BarCodeInventoryError.STATUS_CODE_ERROR)) {
                errorPath = CamsConstants.DOCUMENT_PATH + "." +
                        CamsPropertyConstants.BarcodeInventory.BARCODE_INVENTORY_DETAIL + "[" + lineNumber + "]";
                GlobalVariables.getMessageMap().addToErrorPath(errorPath);

                final Asset asset = validateTagNumberAndRetrieveActiveAsset(barcodeInventoryErrorDetail.getAssetTagNumber());
                valid = ObjectUtils.isNotNull(asset);
                valid &= validateCampusCode(barcodeInventoryErrorDetail.getCampusCode(),
                        barcodeInventoryErrorDetail);
                if (ObjectUtils.isNotNull(asset)) {
                    valid &= validateBuildingCodeAndRoomNumber(barcodeInventoryErrorDetail, asset);
                }
                valid &= validateConditionCode(barcodeInventoryErrorDetail.getAssetConditionCode(),
                        barcodeInventoryErrorDetail);
                valid &= validateInventoryDate(barcodeInventoryErrorDetail.getUploadScanTimestamp());
                valid &= validateTaggingLock(barcodeInventoryErrorDetail.getAssetTagNumber(),
                        document.getDocumentNumber());

                if (!valid) {
                    barcodeInventoryErrorDetail.setErrorCorrectionStatusCode(
                            CamsConstants.BarCodeInventoryError.STATUS_CODE_ERROR);

                    // Getting the errors from GlobalVariables.
                    barcodeInventoryErrorDetail.setErrorDescription(getErrorMessages(errorPath));
                } else {
                    if (updateStatus) {
                        barcodeInventoryErrorDetail.setErrorCorrectionStatusCode(
                                CamsConstants.BarCodeInventoryError.STATUS_CODE_CORRECTED);
                    }

                    barcodeInventoryErrorDetail.setErrorDescription("NONE");
                }
                GlobalVariables.getMessageMap().removeFromErrorPath(errorPath);
            }
            lineNumber++;
        }

        // Since this document displays the asset lock error messages on the error description field, we don't want to
        // display the same message at the top of the document. Therefore, deleteLockErrorMessages method deletes such
        // errors from the GlobalVariables object.
        deleteLockErrorMessages();

        return true;
    }

    /**
     * validates the asset tag number exists in only one active asset.
     *
     * @param tagNumber
     * @return boolean
     * @deprecated this method is replaced by validateTagNumberAndRetrieveActiveAsset(String)
     */
    @Deprecated
    protected boolean validateTagNumber(final String tagNumber) {
        boolean result = true;
        // Getting a list of active assets.
        final Collection<Asset> assets = getAssetService().findAssetsMatchingTagNumber(tagNumber);

        if (ObjectUtils.isNull(assets) || assets.isEmpty()) {
            GlobalVariables.getMessageMap().putError(CamsPropertyConstants.BarcodeInventory.ASSET_TAG_NUMBER,
                    CamsKeyConstants.BarcodeInventory.ERROR_CAPITAL_ASSET_DOESNT_EXIST);
            result = false;
        } else {
            int activeAssets = assets.size();
            for (final Asset asset : assets) {
                if (getAssetService().isAssetRetired(asset)) {
                    activeAssets--;
                }
            }
            if (activeAssets == 0) {
                GlobalVariables.getMessageMap().putError(CamsPropertyConstants.BarcodeInventory.ASSET_TAG_NUMBER,
                        CamsKeyConstants.BarcodeInventory.ERROR_CAPITAL_ASSET_IS_RETIRED);
                result = false;
            } else if (activeAssets > 1) {
                GlobalVariables.getMessageMap().putError(CamsPropertyConstants.BarcodeInventory.ASSET_TAG_NUMBER,
                        CamsKeyConstants.BarcodeInventory.ERROR_DUPLICATED_TAG_NUMBER);
                result = false;
            }
        }
        return result;
    }

    /**
     * validates that the specified asset tag number exists in only one active asset;
     * and if so returns that asset; otherwise returns null.
     *
     * @param tagNumber the specified asset tag number
     * @return the only active asset with the specified tag number, or null if not existing.
     */
    protected Asset validateTagNumberAndRetrieveActiveAsset(final String tagNumber) {
        final Collection<Asset> assets = getAssetService().findAssetsMatchingTagNumber(tagNumber);

        if (ObjectUtils.isNull(assets) || assets.isEmpty()) {
            GlobalVariables.getMessageMap().putError(CamsPropertyConstants.BarcodeInventory.ASSET_TAG_NUMBER,
                    CamsKeyConstants.BarcodeInventory.ERROR_CAPITAL_ASSET_DOESNT_EXIST);
            return null;
        } else {
            assets.removeIf(asset -> getAssetService().isAssetRetired(asset));
            if (assets.size() == 0) {
                GlobalVariables.getMessageMap().putError(CamsPropertyConstants.BarcodeInventory.ASSET_TAG_NUMBER,
                        CamsKeyConstants.BarcodeInventory.ERROR_CAPITAL_ASSET_IS_RETIRED);
                return null;
            } else if (assets.size() > 1) {
                GlobalVariables.getMessageMap().putError(CamsPropertyConstants.BarcodeInventory.ASSET_TAG_NUMBER,
                        CamsKeyConstants.BarcodeInventory.ERROR_DUPLICATED_TAG_NUMBER);
                return null;
            }
        }

        return assets.iterator().next();
    }

    /**
     * Validates the inventory date does not equals null
     *
     * @param inventoryDate
     * @return boolean
     */
    protected boolean validateInventoryDate(final Timestamp inventoryDate) {
        boolean result = true;
        final String label = getBusinessObjectDictionaryService().getBusinessObjectEntry(
                BarcodeInventoryErrorDetail.class.getName()).getAttributeDefinition(
                        CamsPropertyConstants.BarcodeInventory.INVENTORY_DATE).getLabel();

        if (inventoryDate == null) {
            GlobalVariables.getMessageMap().putError(CamsPropertyConstants.BarcodeInventory.INVENTORY_DATE,
                    CamsKeyConstants.BarcodeInventory.ERROR_INVALID_FIELD, label);
            result = false;
        }
        return result;
    }

    /**
     * Validates the campus code exists in the campus code table
     *
     * @param campusCode
     * @param detail
     * @return boolean
     */
    protected boolean validateCampusCode(final String campusCode, final BarcodeInventoryErrorDetail detail) {
        boolean result = true;
        final String label = getBusinessObjectDictionaryService().getBusinessObjectEntry(
                BarcodeInventoryErrorDetail.class.getName()).getAttributeDefinition(
                        CamsPropertyConstants.BarcodeInventory.CAMPUS_CODE).getLabel();

        final Campus campus = getCampus(campusCode);

        if (ObjectUtils.isNull(campus)) {
            GlobalVariables.getMessageMap().putError(CamsPropertyConstants.BarcodeInventory.CAMPUS_CODE,
                    CamsKeyConstants.BarcodeInventory.ERROR_INVALID_FIELD, label);
            result = false;
        } else if (!campus.isActive()) {
            GlobalVariables.getMessageMap().putError(CamsPropertyConstants.BarcodeInventory.CAMPUS_CODE,
                    CamsKeyConstants.BarcodeInventory.ERROR_INACTIVE_FIELD, label);
            result = false;
        }
        return result;
    }

    private Campus getCampus(final String campusCode) {
        final BusinessObjectService businessObjectService = SpringContext.getBean(BusinessObjectService.class);
        return businessObjectService.findBySinglePrimaryKey(Campus.class, campusCode);
    }

    /**
     * Validates that the existence of the building code is consistent with the asset type requirements.
     *
     * @param buildingCode
     * @param detail
     * @param asset
     * @return boolean
     * @deprecated this method is replaced by validateBuildingCodeAndRoomNumber(BarcodeInventoryErrorDetail, Asset)
     */
    @Deprecated
    protected boolean validateBuildingCode(final String buildingCode, final BarcodeInventoryErrorDetail detail, final Asset asset) {
        boolean result = true;
        final String label = getBusinessObjectDictionaryService().getBusinessObjectEntry(
                BarcodeInventoryErrorDetail.class.getName()).getAttributeDefinition(
                        CamsPropertyConstants.BarcodeInventory.BUILDING_CODE).getLabel();
        final String description = asset.getCapitalAssetTypeCode();

        // if the asset has empty building code, then the BCIE should too
        if (StringUtils.isBlank(asset.getBuildingCode())) {
            if (StringUtils.isNotBlank(buildingCode)) {
                GlobalVariables.getMessageMap().putError(CamsPropertyConstants.BarcodeInventory.BUILDING_CODE,
                        CamsKeyConstants.BarcodeInventory.ERROR_NOT_ALLOWED_FIELD, label, description);
                result = false;
            }
        } else {
            // otherwise the BCIE should have a non-empty and existing active building code
            final HashMap<String, Object> fields = new HashMap<>();
            fields.put(KFSPropertyConstants.CAMPUS_CODE, detail.getCampusCode());
            fields.put(KFSPropertyConstants.BUILDING_CODE, detail.getBuildingCode());
            final Building building = getBusinessObjectService().findByPrimaryKey(Building.class, fields);

            if (StringUtils.isBlank(buildingCode)) {
                GlobalVariables.getMessageMap().putError(CamsPropertyConstants.BarcodeInventory.BUILDING_CODE,
                        CamsKeyConstants.BarcodeInventory.ERROR_REQUIRED_FIELD, label, description);
                result = false;
            } else if (ObjectUtils.isNull(building)) {
                GlobalVariables.getMessageMap().putError(CamsPropertyConstants.BarcodeInventory.BUILDING_CODE,
                        CamsKeyConstants.BarcodeInventory.ERROR_INVALID_FIELD, label);
                result = false;
            } else if (!building.isActive()) {
                GlobalVariables.getMessageMap().putError(CamsPropertyConstants.BarcodeInventory.BUILDING_CODE,
                        CamsKeyConstants.BarcodeInventory.ERROR_INACTIVE_FIELD, label);
                result = false;
            }
        }

        return result;
    }

    /**
     * Validates that the existence of the building room number is consistent with the asset type requirements.
     *
     * @param roomNumber
     * @param detail
     * @param asset
     * @return boolean
     * @deprecated this method is replaced by validateBuildingCodeAndRoomNumber(BarcodeInventoryErrorDetail, Asset)
     */
    @Deprecated
    protected boolean validateBuildingRoomNumber(final String roomNumber, final BarcodeInventoryErrorDetail detail, final Asset asset) {
        boolean result = true;
        final String label = getBusinessObjectDictionaryService().getBusinessObjectEntry(
                BarcodeInventoryErrorDetail.class.getName()).getAttributeDefinition(
                CamsPropertyConstants.BarcodeInventory.BUILDING_ROOM_NUMBER).getLabel();
        final String description = asset.getCapitalAssetTypeCode();

        // if the asset has empty building room number, then the BCIE should too
        if (StringUtils.isBlank(asset.getBuildingRoomNumber())) {
            if (StringUtils.isNotBlank(roomNumber)) {
                GlobalVariables.getMessageMap().putError(CamsPropertyConstants.BarcodeInventory.BUILDING_ROOM_NUMBER,
                        CamsKeyConstants.BarcodeInventory.ERROR_NOT_ALLOWED_FIELD, label, description);
                result = false;
            }
        } else {
            // otherwise the BCIE should have a non-empty and existing active building room number
            final HashMap<String, Object> fields = new HashMap<>();
            fields.put(KFSPropertyConstants.CAMPUS_CODE, detail.getCampusCode());
            fields.put(KFSPropertyConstants.BUILDING_CODE, detail.getBuildingCode());
            fields.put(KFSPropertyConstants.BUILDING_ROOM_NUMBER, detail.getBuildingRoomNumber());
            final Room room = getBusinessObjectService().findByPrimaryKey(Room.class, fields);

            if (StringUtils.isBlank(roomNumber)) {
                GlobalVariables.getMessageMap().putError(CamsPropertyConstants.BarcodeInventory.BUILDING_ROOM_NUMBER,
                        CamsKeyConstants.BarcodeInventory.ERROR_REQUIRED_FIELD, label, description);
                result = false;
            } else if (ObjectUtils.isNull(room)) {
                GlobalVariables.getMessageMap().putError(CamsPropertyConstants.BarcodeInventory.BUILDING_ROOM_NUMBER,
                        CamsKeyConstants.BarcodeInventory.ERROR_INVALID_FIELD, label);
                result = false;
            } else if (!room.isActive()) {
                GlobalVariables.getMessageMap().putError(CamsPropertyConstants.BarcodeInventory.BUILDING_ROOM_NUMBER,
                        CamsKeyConstants.BarcodeInventory.ERROR_INACTIVE_FIELD, label);
                result = false;
            }
        }

        return result;
    }

    /**
     * Validates that the existence of the building code and room number is consistent with the asset type
     * requirements.
     *
     * @param detail
     * @param asset
     * @return boolean
     */
    protected boolean validateBuildingCodeAndRoomNumber(final BarcodeInventoryErrorDetail detail, final Asset asset) {
        boolean result = true;

        final String campusCode = detail.getCampusCode();
        final String buildingCode = detail.getBuildingCode();
        final String roomNumber = detail.getBuildingRoomNumber();
        final String labelBuilding = getBusinessObjectDictionaryService().getBusinessObjectEntry(
                BarcodeInventoryErrorDetail.class.getName()).getAttributeDefinition(
                        CamsPropertyConstants.BarcodeInventory.BUILDING_CODE).getLabel();
        final String labelRoom = getBusinessObjectDictionaryService().getBusinessObjectEntry(
                BarcodeInventoryErrorDetail.class.getName()).getAttributeDefinition(
                        CamsPropertyConstants.BarcodeInventory.BUILDING_ROOM_NUMBER).getLabel();

        final String assetTypeCode = asset.getCapitalAssetTypeCode();
        final AssetType assetType = asset.getCapitalAssetType();

        // retrieve building
        final HashMap<String, Object> fields = new HashMap<>();
        fields.put(KFSPropertyConstants.CAMPUS_CODE, campusCode);
        fields.put(KFSPropertyConstants.BUILDING_CODE, buildingCode);
        final Building building = getBusinessObjectService().findByPrimaryKey(Building.class, fields);

        // retrieve room
        fields.put(KFSPropertyConstants.BUILDING_ROOM_NUMBER, roomNumber);
        final Room room = getBusinessObjectService().findByPrimaryKey(Room.class, fields);

        // if movingIndicator is true and requiredBuildingIndicator is false, then both building and room are required
        if (assetType.isMovingIndicator() && !assetType.isRequiredBuildingIndicator()) {
            // check building
            if (StringUtils.isBlank(buildingCode)) {
                GlobalVariables.getMessageMap().putError(CamsPropertyConstants.BarcodeInventory.BUILDING_CODE,
                        CamsKeyConstants.BarcodeInventory.ERROR_REQUIRED_FIELD, labelBuilding, assetTypeCode);
                result = false;
            } else if (ObjectUtils.isNull(building)) {
                GlobalVariables.getMessageMap().putError(CamsPropertyConstants.BarcodeInventory.BUILDING_CODE,
                        CamsKeyConstants.BarcodeInventory.ERROR_INVALID_FIELD, labelBuilding);
                result = false;
            } else if (!building.isActive()) {
                GlobalVariables.getMessageMap().putError(CamsPropertyConstants.BarcodeInventory.BUILDING_CODE,
                        CamsKeyConstants.BarcodeInventory.ERROR_INACTIVE_FIELD, labelBuilding);
                result = false;
            }

            // check room
            if (StringUtils.isBlank(roomNumber)) {
                GlobalVariables.getMessageMap().putError(CamsPropertyConstants.BarcodeInventory.BUILDING_ROOM_NUMBER,
                        CamsKeyConstants.BarcodeInventory.ERROR_REQUIRED_FIELD, labelRoom, assetTypeCode);
                result = false;
            } else if (ObjectUtils.isNull(room)) {
                GlobalVariables.getMessageMap().putError(CamsPropertyConstants.BarcodeInventory.BUILDING_ROOM_NUMBER,
                        CamsKeyConstants.BarcodeInventory.ERROR_INVALID_FIELD, labelRoom);
                result = false;
            } else if (!room.isActive()) {
                GlobalVariables.getMessageMap().putError(CamsPropertyConstants.BarcodeInventory.BUILDING_ROOM_NUMBER,
                        CamsKeyConstants.BarcodeInventory.ERROR_INACTIVE_FIELD, labelRoom);
                result = false;
            }
        } else if (!assetType.isMovingIndicator() && assetType.isRequiredBuildingIndicator()) {
            // if movingIndicator is false and requiredBuildingIndicator is true, then building is required while room
            // is not allowed check building
            if (StringUtils.isBlank(buildingCode)) {
                GlobalVariables.getMessageMap().putError(CamsPropertyConstants.BarcodeInventory.BUILDING_CODE,
                        CamsKeyConstants.BarcodeInventory.ERROR_REQUIRED_FIELD, labelBuilding, assetTypeCode);
                result = false;
            } else if (ObjectUtils.isNull(building)) {
                GlobalVariables.getMessageMap().putError(CamsPropertyConstants.BarcodeInventory.BUILDING_CODE,
                        CamsKeyConstants.BarcodeInventory.ERROR_INVALID_FIELD, labelBuilding);
                result = false;
            } else if (!building.isActive()) {
                GlobalVariables.getMessageMap().putError(CamsPropertyConstants.BarcodeInventory.BUILDING_CODE,
                        CamsKeyConstants.BarcodeInventory.ERROR_INACTIVE_FIELD, labelBuilding);
                result = false;
            }

            // check room
            if (StringUtils.isNotBlank(roomNumber)) {
                GlobalVariables.getMessageMap().putError(CamsPropertyConstants.BarcodeInventory.BUILDING_ROOM_NUMBER,
                        CamsKeyConstants.BarcodeInventory.ERROR_NOT_ALLOWED_FIELD, labelRoom, assetTypeCode);
                result = false;
            }
        } else if (!assetType.isMovingIndicator() && !assetType.isRequiredBuildingIndicator()) {
            // if both movingIndicator and requiredBuildingIndicator are false, then neither building nor room is
            // allowed check building
            if (StringUtils.isNotBlank(buildingCode)) {
                GlobalVariables.getMessageMap().putError(CamsPropertyConstants.BarcodeInventory.BUILDING_CODE,
                        CamsKeyConstants.BarcodeInventory.ERROR_NOT_ALLOWED_FIELD, labelBuilding, assetTypeCode);
                result = false;
            }

            // check room
            if (StringUtils.isNotBlank(roomNumber)) {
                GlobalVariables.getMessageMap().putError(CamsPropertyConstants.BarcodeInventory.BUILDING_ROOM_NUMBER,
                        CamsKeyConstants.BarcodeInventory.ERROR_NOT_ALLOWED_FIELD, labelRoom, assetTypeCode);
                result = false;
            }
        }

        return result;
    }

    /**
     * Validates the condition code exists in the condition table
     *
     * @param conditionCode
     * @param detail
     * @return boolean
     */
    protected boolean validateConditionCode(final String conditionCode, final BarcodeInventoryErrorDetail detail) {
        boolean result = true;
        final String label = getBusinessObjectDictionaryService().getBusinessObjectEntry(
                BarcodeInventoryErrorDetail.class.getName()).getAttributeDefinition(
                CamsPropertyConstants.BarcodeInventory.ASSET_CONDITION_CODE).getLabel();

        final AssetCondition condition;
        final HashMap<String, Object> fields = new HashMap<>();
        fields.put(CamsPropertyConstants.BarcodeInventory.ASSET_CONDITION_CODE, detail.getAssetConditionCode());
        condition = getBusinessObjectService().findByPrimaryKey(AssetCondition.class, fields);

        if (ObjectUtils.isNull(condition)) {
            GlobalVariables.getMessageMap().putError(CamsPropertyConstants.BarcodeInventory.ASSET_CONDITION_CODE,
                    CamsKeyConstants.BarcodeInventory.ERROR_INVALID_FIELD, label);
            result = false;
        } else if (!condition.isActive()) {
            GlobalVariables.getMessageMap().putError(CamsPropertyConstants.BarcodeInventory.ASSET_CONDITION_CODE,
                    CamsKeyConstants.BarcodeInventory.ERROR_INACTIVE_FIELD, label);
            result = false;
        }

        return result;
    }

    /**
     * Validates the Asset doesn't have any tagging locks as result of an existing document in process where the same
     * asset being affected
     *
     * @param tagNumber
     * @return boolean
     */
    protected boolean validateTaggingLock(final String tagNumber, final String documentNumber) {
        boolean result = true;
        final String skipAssetLockValidation;

        // Getting system parameter in order to determine whether or not the asset locks will be ignored.
        if (getParameterService().parameterExists(BarcodeInventoryErrorDocument.class,
                CamsParameterConstants.IGNORE_ASSET_LOCKS_IND
        )) {
            skipAssetLockValidation = getParameterService().getParameterValueAsString(
                    BarcodeInventoryErrorDocument.class,
                    CamsParameterConstants.IGNORE_ASSET_LOCKS_IND
            );
        } else {
            LOG.warn(
                    "CAMS Parameter '{}' not found! - Setting default value to 'N' ",
                    CamsParameterConstants.IGNORE_ASSET_LOCKS_IND
            );
            skipAssetLockValidation = CamsConstants.BarCodeInventoryError.BAR_CODE_ERROR_DOCUMENT_IGNORES_LOCKS_NO;
        }

        if (skipAssetLockValidation == null || StringUtils.isEmpty(skipAssetLockValidation)
                || StringUtils.equals(skipAssetLockValidation,
                CamsConstants.BarCodeInventoryError.BAR_CODE_ERROR_DOCUMENT_IGNORES_LOCKS_NO)) {
            // Getting a list of active assets.
            final List<Asset> assets = getAssetService().findActiveAssetsMatchingTagNumber(tagNumber);
            if (assets.size() > 1) {
                GlobalVariables.getMessageMap().putError(CamsPropertyConstants.BarcodeInventory.ASSET_TAG_NUMBER,
                        CamsKeyConstants.BarcodeInventory.ERROR_DUPLICATED_TAG_NUMBER);
                result = false;
            } else if (assets.size() > 0) {
                final List<Long> assetNumbers = new ArrayList<>();
                assetNumbers.add(assets.get(0).getCapitalAssetNumber());
                final List<String> lockingDocNumbers = SpringContext.getBean(AssetLockService.class)
                        .getAssetLockingDocuments(assetNumbers,
                                CamsConstants.DocumentTypeName.ASSET_BARCODE_INVENTORY_ERROR, documentNumber);
                if (lockingDocNumbers != null && !lockingDocNumbers.isEmpty()) {
                    for (final String lockingDocNumber : lockingDocNumbers) {
                        GlobalVariables.getMessageMap().putError(
                                CamsPropertyConstants.BarcodeInventory.ASSET_TAG_NUMBER,
                                CamsKeyConstants.BarcodeInventory.ERROR_ASSET_LOCKED, lockingDocNumber);
                    }
                    result = false;
                }
            }
        }
        return result;
    }

    /**
     * Iterates over the list of errors each records might have and returns a single string with all the errors for
     * each asset
     *
     * @param errorPath list of errors to be combined
     * @return String
     */
    protected String getErrorMessages(final String errorPath) {
        String message = "";
        final String[] fields = {CamsPropertyConstants.BarcodeInventory.ASSET_TAG_NUMBER,
            CamsPropertyConstants.BarcodeInventory.INVENTORY_DATE,
            CamsPropertyConstants.BarcodeInventory.CAMPUS_CODE,
            CamsPropertyConstants.BarcodeInventory.BUILDING_CODE,
            CamsPropertyConstants.BarcodeInventory.BUILDING_ROOM_NUMBER,
            CamsPropertyConstants.BarcodeInventory.ASSET_CONDITION_CODE};

        for (final String field : fields) {
            final String propertyName = errorPath + "." + field;
            if (GlobalVariables.getMessageMap().doesPropertyHaveError(propertyName)) {
                for (final Object errorMessage : GlobalVariables.getMessageMap().getMessages(propertyName)) {
                    final String errorMsg = getKualiConfigurationService().getPropertyValueAsString(
                            ((ErrorMessage) errorMessage).getErrorKey());
                    message += ", " + MessageFormat.format(errorMsg,
                            (Object[]) ((ErrorMessage) errorMessage).getMessageParameters());
                }
            }
        }
        return StringUtils.isEmpty(message) ? message : message.substring(2);
    }

    /**
     * Deletes the asset locking error messages from the GlobalVariables.
     */
    protected void deleteLockErrorMessages() {
        // Finding locking error messages
        final List<ErrorMessage> el = new ArrayList<>();

        if (GlobalVariables.getMessageMap().getMessages(KFSConstants.GLOBAL_ERRORS) == null) {
            return;
        }

        for (final ErrorMessage errorMessage : GlobalVariables.getMessageMap().getMessages(KFSConstants.GLOBAL_ERRORS)) {
            if (errorMessage.getErrorKey().equals(KFSKeyConstants.ERROR_MAINTENANCE_LOCKED)) {
                el.add(errorMessage);
            }
        }

        // Deleting asset locked error messages from global variable.
        for (final ErrorMessage em : el) {
            GlobalVariables.getMessageMap().getMessages(KFSConstants.GLOBAL_ERRORS).remove(em);
        }
    }

    protected ParameterService getParameterService() {
        return SpringContext.getBean(ParameterService.class);
    }

    protected AssetService getAssetService() {
        if (assetService == null) {
            assetService = SpringContext.getBean(AssetService.class);
        }

        return assetService;
    }

    protected void setAssetService(final AssetService assetService) {
        this.assetService = assetService;
    }

    protected DateTimeService getDateTimeService() {
        return SpringContext.getBean(DateTimeService.class);
    }

    protected BusinessObjectService getBusinessObjectService() {
        return SpringContext.getBean(BusinessObjectService.class);
    }

    private BusinessObjectDictionaryService getBusinessObjectDictionaryService() {
        return SpringContext.getBean(BusinessObjectDictionaryService.class);
    }
}
