/*
 * The Kuali Financial System, a comprehensive financial management system for higher education.
 *
 * Copyright 2005-2023 Kuali, Inc.
 *
 * This program is free software: you can redistribute it and/or modify
 * it under the terms of the GNU Affero General Public License as published by
 * the Free Software Foundation, either version 3 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU Affero General Public License for more details.
 *
 * You should have received a copy of the GNU Affero General Public License
 * along with this program.  If not, see <http://www.gnu.org/licenses/>.
 */
package org.kuali.kfs.module.cam.web.struts;

import org.apache.commons.lang3.StringUtils;
import org.apache.struts.action.ActionForm;
import org.apache.struts.action.ActionForward;
import org.apache.struts.action.ActionMapping;
import org.kuali.kfs.core.api.config.property.ConfigurationService;
import org.kuali.kfs.core.api.util.type.KualiDecimal;
import org.kuali.kfs.coreservice.framework.parameter.ParameterService;
import org.kuali.kfs.kns.question.ConfirmationQuestion;
import org.kuali.kfs.kns.util.KNSGlobalVariables;
import org.kuali.kfs.krad.service.BusinessObjectService;
import org.kuali.kfs.krad.util.GlobalVariables;
import org.kuali.kfs.krad.util.KRADConstants;
import org.kuali.kfs.krad.util.KRADUtils;
import org.kuali.kfs.krad.util.ObjectUtils;
import org.kuali.kfs.module.cam.CamsConstants;
import org.kuali.kfs.module.cam.CamsKeyConstants;
import org.kuali.kfs.module.cam.CamsParameterConstants;
import org.kuali.kfs.module.cam.CamsPropertyConstants;
import org.kuali.kfs.module.cam.businessobject.AssetGlobal;
import org.kuali.kfs.module.cam.businessobject.PurchasingAccountsPayableDocument;
import org.kuali.kfs.module.cam.businessobject.PurchasingAccountsPayableItemAsset;
import org.kuali.kfs.module.cam.businessobject.PurchasingAccountsPayableLineAssetAccount;
import org.kuali.kfs.module.cam.document.service.PurApInfoService;
import org.kuali.kfs.module.cam.document.service.PurApLineDocumentService;
import org.kuali.kfs.module.cam.document.service.PurApLineService;
import org.kuali.kfs.module.cam.document.web.PurApLineSession;
import org.kuali.kfs.module.cam.document.web.struts.CabActionBase;
import org.kuali.kfs.sys.KFSConstants;
import org.kuali.kfs.sys.KFSKeyConstants;
import org.kuali.kfs.sys.context.SpringContext;

import javax.servlet.http.HttpServletRequest;
import javax.servlet.http.HttpServletResponse;
import java.util.Collection;
import java.util.HashMap;
import java.util.List;
import java.util.Map;

public class PurApLineAction extends CabActionBase {

    PurApLineService purApLineService = SpringContext.getBean(PurApLineService.class);
    PurApInfoService purApInfoService = SpringContext.getBean(PurApInfoService.class);
    PurApLineDocumentService purApLineDocumentService = SpringContext.getBean(PurApLineDocumentService.class);

    public ActionForward start(
            final ActionMapping mapping, final ActionForm form, final HttpServletRequest request,
            final HttpServletResponse response) throws Exception {
        final PurApLineForm purApLineForm = (PurApLineForm) form;
        if (purApLineForm.getPurchaseOrderIdentifier() == null) {
            GlobalVariables.getMessageMap().putError(KFSConstants.GLOBAL_ERRORS, CamsKeyConstants.ERROR_PO_ID_EMPTY);
        } else {
            // set Contact Email Address and Phone Number from PurAp Purchase Order document
            purApInfoService.setPurchaseOrderFromPurAp(purApLineForm);

            // save PurAp document list into form
            buildPurApDocList(purApLineForm);

            if (!purApLineForm.getPurApDocs().isEmpty()) {
                // set item pre-populated fields
                purApLineService.buildPurApItemAssetList(purApLineForm.getPurApDocs());
                // create session object for current processing
                createPurApLineSession(purApLineForm.getPurchaseOrderIdentifier());
            }
        }
        return mapping.findForward(KFSConstants.MAPPING_BASIC);
    }

    protected void createPurApLineSession(final Integer purchaseOrderIdentifier) {
        GlobalVariables.getUserSession().addObject(CamsConstants.CAB_PURAP_SESSION.concat(
                purchaseOrderIdentifier.toString()), new PurApLineSession());
    }

    protected void clearPurApLineSession(final Integer purchaseOrderIdentifier) {
        if (purchaseOrderIdentifier != null) {
            GlobalVariables.getUserSession().removeObject(CamsConstants.CAB_PURAP_SESSION.concat(
                    purchaseOrderIdentifier.toString()));
        }
    }

    public ActionForward reload(
            final ActionMapping mapping, final ActionForm form, final HttpServletRequest request,
            final HttpServletResponse response) throws Exception {
        final PurApLineForm purApLineForm = (PurApLineForm) form;
        if (purApLineForm.getPurchaseOrderIdentifier() == null) {
            GlobalVariables.getMessageMap().putError(KFSConstants.GLOBAL_ERRORS, CamsKeyConstants.ERROR_PO_ID_EMPTY);
        }
        purApLineForm.getPurApDocs().clear();
        // clear the session and reload the page
        clearPurApLineSession(purApLineForm.getPurchaseOrderIdentifier());
        return start(mapping, form, request, response);
    }

    /**
     * Build PurchasingAccountsPayableDocument list in which all documents have the same PO_ID.
     *
     * @param purApLineForm
     */
    protected void buildPurApDocList(final PurApLineForm purApLineForm) {
        final Map<String, Object> cols = new HashMap<>();
        cols.put(CamsPropertyConstants.PurchasingAccountsPayableDocument.PURCHASE_ORDER_IDENTIFIER,
                purApLineForm.getPurchaseOrderIdentifier());
        final Collection<PurchasingAccountsPayableDocument> purApDocs = SpringContext.getBean(BusinessObjectService.class)
                .findMatchingOrderBy(PurchasingAccountsPayableDocument.class, cols,
                        CamsPropertyConstants.PurchasingAccountsPayableDocument.DOCUMENT_NUMBER, true);

        if (purApDocs == null || purApDocs.isEmpty()) {
            GlobalVariables.getMessageMap().putError(KFSConstants.GLOBAL_ERRORS, CamsKeyConstants.ERROR_PO_ID_INVALID,
                    purApLineForm.getPurchaseOrderIdentifier().toString());
        } else {
            boolean existActiveDoc = false;
            for (final PurchasingAccountsPayableDocument purApDoc : purApDocs) {
                if (ObjectUtils.isNotNull(purApDoc) && purApDoc.isActive()) {
                    // If there exists active document, set the existActiveDoc indicator.
                    existActiveDoc = true;
                    break;
                }
            }
            purApLineForm.getPurApDocs().clear();
            purApLineForm.getPurApDocs().addAll(purApDocs);
            setupObjectRelationship(purApLineForm.getPurApDocs());
            // If no active item exists or no exist document, display a message.
            if (!existActiveDoc) {
                KNSGlobalVariables.getMessageList().add(CamsKeyConstants.MESSAGE_NO_ACTIVE_PURAP_DOC);
            }
        }
    }

    /**
     * Setup relationship from account to item and item to doc. In this way, we keep all working objects in the same
     * view as form.
     *
     * @param purApDocs
     */
    protected void setupObjectRelationship(final List<PurchasingAccountsPayableDocument> purApDocs) {
        for (final PurchasingAccountsPayableDocument purApDoc : purApDocs) {
            for (final PurchasingAccountsPayableItemAsset item : purApDoc.getPurchasingAccountsPayableItemAssets()) {
                item.setPurchasingAccountsPayableDocument(purApDoc);
                for (final PurchasingAccountsPayableLineAssetAccount account :
                        item.getPurchasingAccountsPayableLineAssetAccounts()) {
                    account.setPurchasingAccountsPayableItemAsset(item);
                }
            }
        }
    }

    public ActionForward cancel(
            final ActionMapping mapping, final ActionForm form, final HttpServletRequest request,
            final HttpServletResponse response) throws Exception {
        return mapping.findForward(KRADConstants.MAPPING_PORTAL);
    }

    /**
     * save the information in the current form into underlying data store
     */
    public ActionForward save(
            final ActionMapping mapping, final ActionForm form, final HttpServletRequest request,
            final HttpServletResponse response) throws Exception {
        final PurApLineForm purApLineForm = (PurApLineForm) form;
        // get the current processing object from session
        final PurApLineSession purApLineSession = retrievePurApLineSession(purApLineForm);
        // persistent changes to CAB tables
        purApLineService.processSaveBusinessObjects(purApLineForm.getPurApDocs(), purApLineSession);
        KNSGlobalVariables.getMessageList().add(CamsKeyConstants.MESSAGE_CAB_CHANGES_SAVED_SUCCESS);
        return mapping.findForward(KFSConstants.MAPPING_BASIC);
    }

    /**
     * Handling for screen close. Default action is return to caller.
     */
    public ActionForward close(
            final ActionMapping mapping, final ActionForm form, final HttpServletRequest request,
            final HttpServletResponse response) throws Exception {
        final PurApLineForm purApLineForm = (PurApLineForm) form;

        // Create question page for save before close.
        final Object question = request.getParameter(KRADConstants.QUESTION_INST_ATTRIBUTE_NAME);
        final ConfigurationService kualiConfiguration = SpringContext.getBean(ConfigurationService.class);

        // logic for close question
        if (question == null) {
            // ask question if not already asked
            return performQuestionWithoutInput(mapping, form, request, response,
                    KRADConstants.DOCUMENT_SAVE_BEFORE_CLOSE_QUESTION,
                    kualiConfiguration.getPropertyValueAsString(KFSKeyConstants.QUESTION_SAVE_BEFORE_CLOSE),
                    KRADConstants.CONFIRMATION_QUESTION, KRADConstants.MAPPING_CLOSE, "");
        } else {
            final Object buttonClicked = request.getParameter(KRADConstants.QUESTION_CLICKED_BUTTON);
            final PurApLineSession purApLineSession = retrievePurApLineSession(purApLineForm);
            if (KRADConstants.DOCUMENT_SAVE_BEFORE_CLOSE_QUESTION.equals(question)
                && ConfirmationQuestion.YES.equals(buttonClicked)) {
                purApLineService.processSaveBusinessObjects(purApLineForm.getPurApDocs(), purApLineSession);
            }
            // remove current processing object from session
            removePurApLineSession(purApLineForm.getPurchaseOrderIdentifier());
        }

        return mapping.findForward(KRADConstants.MAPPING_PORTAL);
    }

    /**
     * Remove PurApLineSession object from user session.
     *
     * @param purchaseOrderIdentifier
     */
    private void removePurApLineSession(final Integer purchaseOrderIdentifier) {
        GlobalVariables.getUserSession().removeObject(
                CamsConstants.CAB_PURAP_SESSION.concat(purchaseOrderIdentifier.toString()));
    }

    /**
     * This method handles split action. Create one item with split quantity
     *
     * @param mapping
     * @param form
     * @param request
     * @param response
     * @return
     * @throws Exception
     */
    public ActionForward split(
            final ActionMapping mapping, final ActionForm form, final HttpServletRequest request,
            final HttpServletResponse response) throws Exception {
        final PurApLineForm purApLineForm = (PurApLineForm) form;
        // Get the line item for applying split action.
        final PurchasingAccountsPayableItemAsset selectedLineItem = getSelectedLineItem((PurApLineForm) form);

        if (selectedLineItem == null) {
            return mapping.findForward(KFSConstants.MAPPING_BASIC);
        }

        final String errorPath = CamsPropertyConstants.PurApLineForm.PURAP_DOCS + KFSConstants.SQUARE_BRACKET_LEFT +
                                 purApLineForm.getActionPurApDocIndex() + KFSConstants.SQUARE_BRACKET_RIGHT + "." +
                                 CamsPropertyConstants.PurchasingAccountsPayableDocument.PURCHASING_ACCOUNTS_PAYABLE_ITEM_ASSETS +
                                 KFSConstants.SQUARE_BRACKET_LEFT + purApLineForm.getActionItemAssetIndex() +
                                 KFSConstants.SQUARE_BRACKET_RIGHT;
        GlobalVariables.getMessageMap().addToErrorPath(errorPath);
        // check user input split quantity.
        checkSplitQty(selectedLineItem, errorPath);
        GlobalVariables.getMessageMap().removeFromErrorPath(errorPath);

        // apply split when error free
        if (GlobalVariables.getMessageMap().hasNoErrors() && selectedLineItem != null) {
            final PurApLineSession purApLineSession = retrievePurApLineSession(purApLineForm);
            // create a new item with split quantity from selected item
            purApLineService.processSplit(selectedLineItem, purApLineSession.getActionsTakenHistory());
        }

        return mapping.findForward(KFSConstants.MAPPING_BASIC);
    }

    /**
     * Get PurApLineSession object from user session.
     *
     * @param purApForm
     * @return
     */
    private PurApLineSession retrievePurApLineSession(final PurApLineForm purApForm) {
        PurApLineSession purApLineSession = (PurApLineSession) GlobalVariables.getUserSession().retrieveObject(
                CamsConstants.CAB_PURAP_SESSION.concat(purApForm.getPurchaseOrderIdentifier().toString()));
        if (purApLineSession == null) {
            purApLineSession = new PurApLineSession();
            GlobalVariables.getUserSession().addObject(
                    CamsConstants.CAB_PURAP_SESSION.concat(purApForm.getPurchaseOrderIdentifier().toString()),
                    purApLineSession);
        }
        return purApLineSession;
    }

    /**
     * Check user input splitQty. It must be required and can't be zero or greater than current quantity.
     *
     * @param itemAsset
     * @param errorPath
     */
    protected void checkSplitQty(final PurchasingAccountsPayableItemAsset itemAsset, final String errorPath) {
        if (itemAsset.getSplitQty() == null) {
            itemAsset.setSplitQty(KualiDecimal.ZERO);
        }

        if (itemAsset.getAccountsPayableItemQuantity() == null) {
            itemAsset.setAccountsPayableItemQuantity(KualiDecimal.ZERO);
        }

        final KualiDecimal splitQty = itemAsset.getSplitQty();
        final KualiDecimal oldQty = itemAsset.getAccountsPayableItemQuantity();
        final KualiDecimal maxAllowQty = oldQty.subtract(new KualiDecimal(0.1));

        if (splitQty == null) {
            GlobalVariables.getMessageMap().putError(
                    CamsPropertyConstants.PurchasingAccountsPayableItemAsset.SPLIT_QTY,
                    CamsKeyConstants.ERROR_SPLIT_QTY_REQUIRED);
        } else if (splitQty.isLessEqual(KualiDecimal.ZERO) || splitQty.isGreaterEqual(oldQty)) {
            GlobalVariables.getMessageMap().putError(
                    CamsPropertyConstants.PurchasingAccountsPayableItemAsset.SPLIT_QTY,
                    CamsKeyConstants.ERROR_SPLIT_QTY_INVALID, maxAllowQty.toString());
        }
    }

    /**
     * Merge Action includes merge all functionality.
     *
     * @param mapping
     * @param form
     * @param request
     * @param response
     * @return
     * @throws Exception
     */
    public ActionForward merge(
            final ActionMapping mapping, final ActionForm form, final HttpServletRequest request,
            final HttpServletResponse response) throws Exception {
        final PurApLineForm purApForm = (PurApLineForm) form;
        final boolean isMergeAll = purApLineService.isMergeAllAction(purApForm.getPurApDocs());
        final List<PurchasingAccountsPayableItemAsset> mergeLines = purApLineService.getSelectedMergeLines(isMergeAll,
                purApForm.getPurApDocs());

        final Object question = request.getParameter(KRADConstants.QUESTION_INST_ATTRIBUTE_NAME);
        // logic for trade-in allowance question
        if (question != null) {
            final Object buttonClicked = request.getParameter(KRADConstants.QUESTION_CLICKED_BUTTON);
            if (CamsConstants.TRADE_IN_INDICATOR_QUESTION.equals(question)
                    && ConfirmationQuestion.YES.equals(buttonClicked)) {
                if (purApLineService.mergeLinesHasDifferentObjectSubTypes(mergeLines)) {
                    // check if objectSubTypes are different and bring the obj sub types warning message
                    final String warningMessage = generateObjectSubTypeQuestion();
                    return performQuestionWithoutInput(mapping, form, request, response,
                            CamsConstants.PAYMENT_DIFFERENT_OBJECT_SUB_TYPE_CONFIRMATION_QUESTION, warningMessage,
                            KRADConstants.CONFIRMATION_QUESTION, CamsConstants.Actions.MERGE, "");
                } else {
                    performMerge(purApForm, mergeLines, isMergeAll);
                }
            } else if (CamsConstants.PAYMENT_DIFFERENT_OBJECT_SUB_TYPE_CONFIRMATION_QUESTION.equals(question)
                    && ConfirmationQuestion.YES.equals(buttonClicked)) {
                performMerge(purApForm, mergeLines, isMergeAll);
            }

            return mapping.findForward(KFSConstants.MAPPING_BASIC);
        }

        final boolean tradeInAllowanceInAllLines = purApLineService.isTradeInAllowanceExist(purApForm.getPurApDocs());
        final boolean tradeInIndicatorInSelectedLines = purApLineService.isTradeInIndExistInSelectedLines(mergeLines);
        // validating...
        validateMergeAction(purApForm, mergeLines, isMergeAll, tradeInAllowanceInAllLines,
                tradeInIndicatorInSelectedLines);

        if (GlobalVariables.getMessageMap().hasNoErrors()) {
            // Display a warning message without blocking the action if TI indicator exists but TI allowance not exist.
            if (tradeInIndicatorInSelectedLines && !tradeInAllowanceInAllLines) {
                return performQuestionWithoutInput(mapping, form, request, response,
                        CamsConstants.TRADE_IN_INDICATOR_QUESTION,
                        SpringContext.getBean(ConfigurationService.class).getPropertyValueAsString(
                                CamsKeyConstants.QUESTION_TRADE_IN_INDICATOR_EXISTING),
                        KRADConstants.CONFIRMATION_QUESTION, CamsConstants.Actions.MERGE, "");
            } else if (purApLineService.mergeLinesHasDifferentObjectSubTypes(mergeLines)) {
                // check if objectSubTypes are different and bring the obj sub types warning message
                final String warningMessage = generateObjectSubTypeQuestion();
                return performQuestionWithoutInput(mapping, form, request, response,
                        CamsConstants.PAYMENT_DIFFERENT_OBJECT_SUB_TYPE_CONFIRMATION_QUESTION, warningMessage,
                        KRADConstants.CONFIRMATION_QUESTION, CamsConstants.Actions.MERGE, "");
            } else {
                performMerge(purApForm, mergeLines, isMergeAll);
            }
        }

        return mapping.findForward(KFSConstants.MAPPING_BASIC);
    }

    /**
     * Generate the question string for different object sub type codes.
     *
     * @return
     */
    protected String generateObjectSubTypeQuestion() {
        final String parameterDetail = "(module:" + KRADUtils.getNamespaceAndComponentSimpleName(AssetGlobal.class) + ")";
        final ConfigurationService kualiConfiguration = getConfigurationService();

        final String continueQuestion = kualiConfiguration.getPropertyValueAsString(CamsKeyConstants.CONTINUE_QUESTION);
        return kualiConfiguration.getPropertyValueAsString(CamsKeyConstants.QUESTION_DIFFERENT_OBJECT_SUB_TYPES) +
                " " + CamsParameterConstants.OBJECT_SUB_TYPE_GROUPS + " " + parameterDetail + ". " +
                continueQuestion;
    }

    /**
     * Merge with service help.
     *
     * @param purApForm
     * @param mergeLines
     */
    protected void performMerge(
            final PurApLineForm purApForm, final List<PurchasingAccountsPayableItemAsset> mergeLines,
            final boolean isMergeAll) {
        final PurApLineSession purApLineSession = retrievePurApLineSession(purApForm);
        // handle merging lines including merge all situation.
        retrieveUserInputForMerge(mergeLines.get(0), purApForm);
        purApLineService.processMerge(mergeLines, purApLineSession.getActionsTakenHistory(), isMergeAll);
        // add all other mergeLines except the first one into processedItem list.
        mergeLines.remove(0);
        purApLineSession.getProcessedItems().addAll(mergeLines);
        clearForMerge(purApForm);
    }

    /**
     * Retrieve user input merge quantity and merge description.
     *
     * @param firstItem
     * @param purApForm
     */
    protected void retrieveUserInputForMerge(final PurchasingAccountsPayableItemAsset firstItem, final PurApLineForm purApForm) {
        if (ObjectUtils.isNotNull(firstItem)) {
            // Set new value for quantity and description.
            firstItem.setAccountsPayableItemQuantity(purApForm.getMergeQty());
            firstItem.setAccountsPayableLineItemDescription(purApForm.getMergeDesc());
        }
    }

    /**
     * Clear user input after merge.
     *
     * @param purApForm
     */
    protected void clearForMerge(final PurApLineForm purApForm) {
        // reset user input values.
        purApLineService.resetSelectedValue(purApForm.getPurApDocs());
        purApForm.setMergeQty(null);
        purApForm.setMergeDesc(null);
        purApForm.setSelectAll(false);
    }

    /**
     * Check if the merge action is valid or not.
     *
     * @param purApForm
     * @param mergeLines
     */
    protected void validateMergeAction(
            final PurApLineForm purApForm, final List<PurchasingAccountsPayableItemAsset> mergeLines,
            final boolean isMergeAll, final boolean tradeInAllowanceInAllLines, final boolean tradeInIndicatorInSelectedLines) {
        // check if the user entered merge quantity and merge description
        checkMergeRequiredFields(purApForm);

        // Check if the selected merge lines violate the business constraints.
        if (isMergeAll) {
            checkMergeAllValid(tradeInAllowanceInAllLines, tradeInIndicatorInSelectedLines);
        } else {
            checkMergeLinesValid(mergeLines, tradeInAllowanceInAllLines, tradeInIndicatorInSelectedLines);
        }

        // Check the associated pre-tagging data entries.
        checkPreTagValidForMerge(mergeLines, purApForm.getPurchaseOrderIdentifier());
    }

    /**
     * If to be merged items have: (1) No Pretag data: No problem; (2) 1 Pretag data entry: Associate this one with
     * the new item created after merge;(3) 1+ Pretag data entries: Display error, user has to manually fix data
     *
     * @param mergeLines
     */
    protected void checkPreTagValidForMerge(
            final List<PurchasingAccountsPayableItemAsset> mergeLines,
            final Integer purchaseOrderIdentifier) {
        final Map validNumberMap = getItemLineNumberMap(mergeLines);

        if (!validNumberMap.isEmpty() && validNumberMap.size() > 1
                && purApLineService.isMultipleTagExisting(purchaseOrderIdentifier, validNumberMap.keySet())) {
            GlobalVariables.getMessageMap().putError(CamsPropertyConstants.PurApLineForm.PURAP_DOCS,
                    CamsKeyConstants.ERROR_MERGE_WITH_PRETAGGING);
        }
    }

    /**
     * Build a Hashmap for itemLineNumber since itemLines could exist duplicate itemLineNumber
     *
     * @param itemLines
     * @return
     */
    protected Map getItemLineNumberMap(final List<PurchasingAccountsPayableItemAsset> itemLines) {
        final Map<Integer, Integer> validNumberMap = new HashMap<>();
        for (final PurchasingAccountsPayableItemAsset item : itemLines) {
            if (item.getItemLineNumber() != null) {
                validNumberMap.put(item.getItemLineNumber(), item.getItemLineNumber());
            }
        }
        return validNumberMap;
    }

    /**
     * For merge all, check if exists Trade-in allowance pending for allocate.
     *
     * @param tradeInAllowanceInAllLines
     * @param tradeInIndicatorInSelectedLines
     */
    protected void checkMergeAllValid(final boolean tradeInAllowanceInAllLines, final boolean tradeInIndicatorInSelectedLines) {
        if (tradeInAllowanceInAllLines && tradeInIndicatorInSelectedLines) {
            GlobalVariables.getMessageMap().putError(CamsPropertyConstants.PurApLineForm.PURAP_DOCS,
                    CamsKeyConstants.ERROR_TRADE_IN_PENDING);
        }
    }

    /**
     * Check if merge lines selected are allowed to continue this action. Constraints include: merge lines must more
     * than 1; no additional charges pending for allocate.
     *
     * @param mergeLines
     * @param tradeInAllowanceInAllLines
     * @param tradeInIndicatorInSelectedLines
     */
    protected void checkMergeLinesValid(
            final List<PurchasingAccountsPayableItemAsset> mergeLines,
            final boolean tradeInAllowanceInAllLines, final boolean tradeInIndicatorInSelectedLines) {
        if (mergeLines.size() <= 1) {
            GlobalVariables.getMessageMap().putError(CamsPropertyConstants.PurApLineForm.PURAP_DOCS,
                    CamsKeyConstants.ERROR_MERGE_LINE_SELECTED);
        } else {
            // if merge for different document lines and that document has additional charge allocation pending,
            // additional charges should be allocated first.
            if (purApLineService.isAdditionalChargePending(mergeLines)) {
                GlobalVariables.getMessageMap().putError(CamsPropertyConstants.PurApLineForm.PURAP_DOCS,
                        CamsKeyConstants.ERROR_ADDL_CHARGE_PENDING);
            }

            // if merge lines has indicator exists and trade-in allowance is pending for allocation, we will block
            // this action.
            if (tradeInIndicatorInSelectedLines && tradeInAllowanceInAllLines) {
                GlobalVariables.getMessageMap().putError(CamsPropertyConstants.PurApLineForm.PURAP_DOCS,
                        CamsKeyConstants.ERROR_TRADE_IN_PENDING);
            }
        }
    }

    /**
     * Check the required fields entered for merge.
     */
    protected void checkMergeRequiredFields(final PurApLineForm purApForm) {
        if (purApForm.getMergeQty() == null) {
            GlobalVariables.getMessageMap().putError(CamsPropertyConstants.PurApLineForm.MERGE_QTY,
                    CamsKeyConstants.ERROR_MERGE_QTY_EMPTY);
        }

        if (StringUtils.isBlank(purApForm.getMergeDesc())) {
            GlobalVariables.getMessageMap().putError(CamsPropertyConstants.PurApLineForm.MERGE_DESC,
                    CamsKeyConstants.ERROR_MERGE_DESCRIPTION_EMPTY);
        }
    }

    /**
     * Update the item quantity value from a decimal(less than 1) to 1.
     *
     * @param mapping
     * @param form
     * @param request
     * @param response
     * @return
     * @throws Exception
     */
    public ActionForward percentPayment(
            final ActionMapping mapping, final ActionForm form, final HttpServletRequest request,
            final HttpServletResponse response) throws Exception {
        final PurApLineForm purApform = (PurApLineForm) form;
        final PurchasingAccountsPayableItemAsset itemAsset = getSelectedLineItem(purApform);

        if (itemAsset != null) {
            final PurApLineSession purApLineSession = retrievePurApLineSession(purApform);
            purApLineService.processPercentPayment(itemAsset, purApLineSession.getActionsTakenHistory());
        }

        return mapping.findForward(KFSConstants.MAPPING_BASIC);
    }

    /**
     * Allocate line items including allocate additional charges functionality.
     *
     * @param mapping
     * @param form
     * @param request
     * @param response
     * @return
     * @throws Exception
     */
    public ActionForward allocate(
            final ActionMapping mapping, final ActionForm form, final HttpServletRequest request,
            final HttpServletResponse response) throws Exception {
        final PurApLineForm purApForm = (PurApLineForm) form;
        final PurchasingAccountsPayableItemAsset allocateSourceLine = getSelectedLineItem(purApForm);
        if (allocateSourceLine == null) {
            return mapping.findForward(KFSConstants.MAPPING_BASIC);
        }
        final List<PurchasingAccountsPayableItemAsset> allocateTargetLines =
                purApLineService.getAllocateTargetLines(allocateSourceLine, purApForm.getPurApDocs());

        final Object question = request.getParameter(KRADConstants.QUESTION_INST_ATTRIBUTE_NAME);
        // logic for trade-in allowance question
        if (question != null) {
            final Object buttonClicked = request.getParameter(KRADConstants.QUESTION_CLICKED_BUTTON);
            if (CamsConstants.TRADE_IN_INDICATOR_QUESTION.equals(question)
                    && ConfirmationQuestion.YES.equals(buttonClicked)) {
                if (purApLineService.allocateLinesHasDifferentObjectSubTypes(allocateTargetLines,
                        allocateSourceLine)) {
                    // check if objectSubTypes are different and bring the obj sub types warning message
                    final String warningMessage = generateObjectSubTypeQuestion();
                    return performQuestionWithoutInput(mapping, form, request, response,
                            CamsConstants.PAYMENT_DIFFERENT_OBJECT_SUB_TYPE_CONFIRMATION_QUESTION, warningMessage,
                            KRADConstants.CONFIRMATION_QUESTION, CamsConstants.Actions.ALLOCATE, "");
                } else {
                    performAllocate(purApForm, allocateSourceLine, allocateTargetLines);
                }
            } else if (CamsConstants.PAYMENT_DIFFERENT_OBJECT_SUB_TYPE_CONFIRMATION_QUESTION.equals(question)
                    && ConfirmationQuestion.YES.equals(buttonClicked)) {
                performAllocate(purApForm, allocateSourceLine, allocateTargetLines);
            }

            return mapping.findForward(KFSConstants.MAPPING_BASIC);
        }

        final boolean targetLineHasTradeIn = purApLineService.isTradeInIndExistInSelectedLines(allocateTargetLines);
        final boolean hasTradeInAllowance = purApLineService.isTradeInAllowanceExist(purApForm.getPurApDocs());
        // Check if this allocate is valid.
        validateAllocateAction(allocateSourceLine, allocateTargetLines, targetLineHasTradeIn, hasTradeInAllowance,
                purApForm.getPurApDocs());
        if (GlobalVariables.getMessageMap().hasNoErrors()) {
            // TI indicator exists in either source or target lines, but TI allowance not found, bring up a warning
            // message to confirm this action.
            if (!allocateSourceLine.isAdditionalChargeNonTradeInIndicator()
                    && !allocateSourceLine.isTradeInAllowance()
                    && (allocateSourceLine.isItemAssignedToTradeInIndicator()
                    || targetLineHasTradeIn)
                    && hasTradeInAllowance) {
                return performQuestionWithoutInput(mapping, form, request, response,
                        CamsConstants.TRADE_IN_INDICATOR_QUESTION, SpringContext.getBean(ConfigurationService.class)
                                .getPropertyValueAsString(CamsKeyConstants.QUESTION_TRADE_IN_INDICATOR_EXISTING),
                        KRADConstants.CONFIRMATION_QUESTION, CamsConstants.Actions.ALLOCATE, "");
            } else if (purApLineService.allocateLinesHasDifferentObjectSubTypes(allocateTargetLines,
                    allocateSourceLine)) {
                // check if objectSubTypes are different and bring the obj sub types warning message
                final String warningMessage = generateObjectSubTypeQuestion();
                return performQuestionWithoutInput(mapping, form, request, response,
                        CamsConstants.PAYMENT_DIFFERENT_OBJECT_SUB_TYPE_CONFIRMATION_QUESTION, warningMessage,
                        KRADConstants.CONFIRMATION_QUESTION, CamsConstants.Actions.ALLOCATE, "");
            } else {
                performAllocate(purApForm, allocateSourceLine, allocateTargetLines);
            }
        }

        return mapping.findForward(KFSConstants.MAPPING_BASIC);
    }

    /**
     * Allocate with service help.
     *
     * @param purApForm
     * @param allocateSourceLine
     * @param allocateTargetLines
     */
    protected void performAllocate(
            final PurApLineForm purApForm, final PurchasingAccountsPayableItemAsset allocateSourceLine,
            final List<PurchasingAccountsPayableItemAsset> allocateTargetLines) {
        final PurApLineSession purApLineSession = retrievePurApLineSession(purApForm);
        if (!purApLineService.processAllocate(allocateSourceLine, allocateTargetLines,
                purApLineSession.getActionsTakenHistory(), purApForm.getPurApDocs(), false)) {
            GlobalVariables.getMessageMap().putError(CamsPropertyConstants.PurApLineForm.PURAP_DOCS,
                    CamsKeyConstants.ERROR_ALLOCATE_NO_TARGET_ACCOUNT);
        } else {
            purApLineSession.getProcessedItems().add(allocateSourceLine);
            // clear select check box
            purApLineService.resetSelectedValue(purApForm.getPurApDocs());
            purApForm.setSelectAll(false);
        }
    }

    /**
     * Check if the line items are allowed to allocate.
     *
     * @param allocateSourceLine
     * @param allocateTargetLines
     * @param targetLineHasTradeIn
     * @param hasTradeInAllowance
     * @param purApDocs
     */
    protected void validateAllocateAction(
            final PurchasingAccountsPayableItemAsset allocateSourceLine,
            final List<PurchasingAccountsPayableItemAsset> allocateTargetLines, final boolean targetLineHasTradeIn,
            final boolean hasTradeInAllowance, final List<PurchasingAccountsPayableDocument> purApDocs) {
        // if no target selected...
        if (allocateTargetLines.isEmpty()) {
            GlobalVariables.getMessageMap().putError(CamsPropertyConstants.PurApLineForm.PURAP_DOCS,
                    CamsKeyConstants.ERROR_ALLOCATE_NO_LINE_SELECTED);
        }

        // For allocate trade-in allowance, additional charges(non trade-in) must be allocated before allocate trade-in.
        if (allocateSourceLine.isTradeInAllowance() && purApLineService.isAdditionalChargeExistInAllLines(purApDocs)) {
            GlobalVariables.getMessageMap().putError(CamsPropertyConstants.PurApLineForm.PURAP_DOCS,
                    CamsKeyConstants.ERROR_ADDL_CHARGE_PENDING);
        }
        // For line item, we need to check...
        if (!allocateSourceLine.isAdditionalChargeNonTradeInIndicator() && !allocateSourceLine.isTradeInAllowance()) {
            allocateTargetLines.add(allocateSourceLine);
            // Pending additional charges(non trade-in) can't associate with either source line or target lines.
            if (purApLineService.isAdditionalChargePending(allocateTargetLines)) {
                GlobalVariables.getMessageMap().putError(CamsPropertyConstants.PurApLineForm.PURAP_DOCS,
                        CamsKeyConstants.ERROR_ADDL_CHARGE_PENDING);
            }

            // For line item, check if trade-in allowance allocation pending.
            if (targetLineHasTradeIn && hasTradeInAllowance) {
                GlobalVariables.getMessageMap().putError(CamsPropertyConstants.PurApLineForm.PURAP_DOCS,
                        CamsKeyConstants.ERROR_TRADE_IN_PENDING);
            }
            allocateTargetLines.remove(allocateSourceLine);
        }
    }

    /**
     * Get the user selected line item and set the link reference to document
     *
     * @param purApLineForm
     * @return
     */
    private PurchasingAccountsPayableItemAsset getSelectedLineItem(final PurApLineForm purApLineForm) {
        final PurchasingAccountsPayableDocument purApDoc = purApLineForm.getPurApDocs().get(
                purApLineForm.getActionPurApDocIndex());
        PurchasingAccountsPayableItemAsset selectedItem = purApDoc.getPurchasingAccountsPayableItemAssets().get(
                purApLineForm.getActionItemAssetIndex());
        if (!selectedItem.isActive()) {
            selectedItem = null;
        }
        return selectedItem;
    }

    /**
     * @param purApLineForm
     * @return the user selected document.
     */
    private PurchasingAccountsPayableDocument getSelectedPurApDoc(final PurApLineForm purApLineForm) {
        return purApLineForm.getPurApDocs().get(purApLineForm.getActionPurApDocIndex());
    }

    /**
     * Handle apply payment action.
     *
     * @param mapping
     * @param form
     * @param request
     * @param response
     * @return
     * @throws Exception
     */
    public ActionForward applyPayment(
            final ActionMapping mapping, final ActionForm form, final HttpServletRequest request,
            final HttpServletResponse response) throws Exception {
        final PurApLineForm purApForm = (PurApLineForm) form;
        final PurchasingAccountsPayableItemAsset selectedLine = getSelectedLineItem(purApForm);

        if (selectedLine == null) {
            return mapping.findForward(KFSConstants.MAPPING_BASIC);
        }

        final Object question = request.getParameter(KRADConstants.QUESTION_INST_ATTRIBUTE_NAME);
        if (question != null) {
            final Object buttonClicked = request.getParameter(KRADConstants.QUESTION_CLICKED_BUTTON);
            if (CamsConstants.TRADE_IN_INDICATOR_QUESTION.equals(question)
                    && ConfirmationQuestion.YES.equals(buttonClicked)) {
                // create CAMS asset payment global document.
                return createApplyPaymentDocument(mapping, purApForm, selectedLine);
            } else {
                return mapping.findForward(KFSConstants.MAPPING_BASIC);
            }
        }
        if (selectedLine.isItemAssignedToTradeInIndicator()) {
            // TI indicator exists, bring up a warning message to confirm this action.
            return performQuestionWithoutInput(mapping, form, request, response,
                    CamsConstants.TRADE_IN_INDICATOR_QUESTION, SpringContext.getBean(ConfigurationService.class)
                            .getPropertyValueAsString(CamsKeyConstants.QUESTION_TRADE_IN_INDICATOR_EXISTING),
                    KRADConstants.CONFIRMATION_QUESTION, CamsConstants.Actions.APPLY_PAYMENT, "");
        }

        // create CAMS asset payment global document.
        return createApplyPaymentDocument(mapping, purApForm, selectedLine);
    }

    /**
     * Create CAMS asset payment document.
     *
     * @param mapping
     * @param purApForm
     * @param selectedLine
     * @return
     */
    private ActionForward createApplyPaymentDocument(
            final ActionMapping mapping, final PurApLineForm purApForm,
            final PurchasingAccountsPayableItemAsset selectedLine) {
        final PurApLineSession purApLineSession = retrievePurApLineSession(purApForm);
        final String documentNumber = purApLineDocumentService.processApplyPayment(selectedLine, purApForm.getPurApDocs(),
                purApLineSession, purApForm.getRequisitionIdentifier());
        // create CAMS asset payment global document.
        if (documentNumber != null) {
            purApForm.setDocumentNumber(documentNumber);
        }

        return mapping.findForward(KFSConstants.MAPPING_BASIC);
    }

    /**
     * Handle create asset action.
     *
     * @param mapping
     * @param form
     * @param request
     * @param response
     * @return
     * @throws Exception
     */
    public ActionForward createAsset(
            final ActionMapping mapping, final ActionForm form, final HttpServletRequest request,
            final HttpServletResponse response) throws Exception {
        final PurApLineForm purApForm = (PurApLineForm) form;
        final PurchasingAccountsPayableItemAsset selectedLine = getSelectedLineItem(purApForm);

        if (selectedLine == null) {
            return mapping.findForward(KFSConstants.MAPPING_BASIC);
        }

        final Object question = request.getParameter(KRADConstants.QUESTION_INST_ATTRIBUTE_NAME);
        if (question != null) {
            final Object buttonClicked = request.getParameter(KRADConstants.QUESTION_CLICKED_BUTTON);
            if (CamsConstants.TRADE_IN_INDICATOR_QUESTION.equals(question)
                && ConfirmationQuestion.YES.equals(buttonClicked)) {
                // If PurAp user set capitalAssetNumbers for apply Asset Payment document, bring up a warning message
                // for confirmation.
                if (isSettingAssetsInPurAp(selectedLine)) {
                    return performQuestionWithoutInput(mapping, form, request, response,
                            CamsConstants.SKIP_ASSET_NUMBERS_TO_ASSET_GLOBAL_QUESTION,
                            SpringContext.getBean(ConfigurationService.class).getPropertyValueAsString(
                                    CamsKeyConstants.QUESTION_SKIP_ASSET_NUMBERS_TO_ASSET_GLOBAL),
                            KRADConstants.CONFIRMATION_QUESTION, CamsConstants.Actions.CREATE_ASSET, "");
                } else {
                    return createAssetGlobalDocument(mapping, purApForm, selectedLine);
                }
            } else if (CamsConstants.SKIP_ASSET_NUMBERS_TO_ASSET_GLOBAL_QUESTION.equals(question)
                    && ConfirmationQuestion.YES.equals(buttonClicked)) {
                return createAssetGlobalDocument(mapping, purApForm, selectedLine);
            }
            return mapping.findForward(KFSConstants.MAPPING_BASIC);
        }

        // validate selected line item
        validateCreateAssetAction(selectedLine);

        if (GlobalVariables.getMessageMap().hasNoErrors()) {
            // TI indicator exists, bring up a warning message to confirm this action.
            if (selectedLine.isItemAssignedToTradeInIndicator()) {
                return performQuestionWithoutInput(mapping, form, request, response,
                        CamsConstants.TRADE_IN_INDICATOR_QUESTION, SpringContext.getBean(ConfigurationService.class)
                                .getPropertyValueAsString(CamsKeyConstants.QUESTION_TRADE_IN_INDICATOR_EXISTING),
                        KRADConstants.CONFIRMATION_QUESTION, CamsConstants.Actions.CREATE_ASSET, "");
            } else if (isSettingAssetsInPurAp(selectedLine)) {
                // If PurAp user set capitalAssetNumbers for apply Asset Payment document, bring up a warning message
                // to confirm using Asset Global document.
                return performQuestionWithoutInput(mapping, form, request, response,
                        CamsConstants.SKIP_ASSET_NUMBERS_TO_ASSET_GLOBAL_QUESTION,
                        SpringContext.getBean(ConfigurationService.class).getPropertyValueAsString(
                                CamsKeyConstants.QUESTION_SKIP_ASSET_NUMBERS_TO_ASSET_GLOBAL),
                        KRADConstants.CONFIRMATION_QUESTION, CamsConstants.Actions.CREATE_ASSET, "");
            } else {
                return createAssetGlobalDocument(mapping, purApForm, selectedLine);
            }
        }

        return mapping.findForward(KFSConstants.MAPPING_BASIC);
    }

    /**
     * check if PurAp set CAMS Assets information
     *
     * @param selectedLine
     * @return
     */
    private boolean isSettingAssetsInPurAp(final PurchasingAccountsPayableItemAsset selectedLine) {
        return selectedLine.getPurApItemAssets() != null && !selectedLine.getPurApItemAssets().isEmpty();
    }

    /**
     * Create asset global document
     *
     * @param mapping
     * @param purApForm
     * @param selectedLine
     * @return
     */
    private ActionForward createAssetGlobalDocument(
            final ActionMapping mapping, final PurApLineForm purApForm,
            final PurchasingAccountsPayableItemAsset selectedLine) {
        final PurApLineSession purApLineSession = retrievePurApLineSession(purApForm);
        final String documentNumber = purApLineDocumentService.processCreateAsset(selectedLine, purApForm.getPurApDocs(),
                purApLineSession, purApForm.getRequisitionIdentifier());
        // create CAMS asset global document.
        if (documentNumber != null) {
            // forward link to asset global
            purApForm.setDocumentNumber(documentNumber);
        }

        return mapping.findForward(KFSConstants.MAPPING_BASIC);
    }

    /**
     * Validate selected line item for asset global creation.
     *
     * @param selectedLine
     */
    protected void validateCreateAssetAction(final PurchasingAccountsPayableItemAsset selectedLine) {
        final KualiDecimal integerOne = new KualiDecimal(1);
        final KualiDecimal quantity = selectedLine.getAccountsPayableItemQuantity();
        // check if item quantity is a fractional value greater than 1.
        if (quantity.isGreaterThan(integerOne) && quantity.mod(integerOne).isNonZero()) {
            GlobalVariables.getMessageMap().putError(CamsPropertyConstants.PurApLineForm.PURAP_DOCS,
                    CamsKeyConstants.ERROR_FRACTIONAL_QUANTITY);
        } else if (quantity.isGreaterThan(KualiDecimal.ZERO) && quantity.isLessThan(integerOne)) {
            // if quantity is between (0,1) , set it to 1.
            selectedLine.setAccountsPayableItemQuantity(integerOne);
        }

    }

    protected ParameterService getParameterService() {
        return SpringContext.getBean(ParameterService.class);
    }

    protected ConfigurationService getConfigurationService() {
        return SpringContext.getBean(ConfigurationService.class);
    }
}
