/*
 * The Kuali Financial System, a comprehensive financial management system for higher education.
 *
 * Copyright 2005-2022 Kuali, Inc.
 *
 * This program is free software: you can redistribute it and/or modify
 * it under the terms of the GNU Affero General Public License as published by
 * the Free Software Foundation, either version 3 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU Affero General Public License for more details.
 *
 * You should have received a copy of the GNU Affero General Public License
 * along with this program.  If not, see <http://www.gnu.org/licenses/>.
 */
package org.kuali.kfs.module.cam.batch.service.impl;

import org.apache.logging.log4j.LogManager;
import org.apache.logging.log4j.Logger;
import org.kuali.kfs.kim.impl.identity.Person;
import org.kuali.kfs.krad.exception.AuthorizationException;
import org.kuali.kfs.krad.exception.ValidationException;
import org.kuali.kfs.module.cam.batch.AssetBarcodeInventoryInputFileType;
import org.kuali.kfs.module.cam.batch.service.AssetBarcodeInventoryInputFileService;
import org.kuali.kfs.module.cam.document.web.struts.AssetBarCodeInventoryInputFileForm;
import org.kuali.kfs.sys.KFSConstants;
import org.kuali.kfs.sys.batch.service.impl.BatchInputFileSetServiceImpl;
import org.kuali.kfs.sys.exception.FileStorageException;
import org.kuali.kfs.core.api.datetime.DateTimeService;

import java.io.File;
import java.io.FileInputStream;
import java.io.IOException;
import java.io.InputStream;
import java.util.ArrayList;
import java.util.Date;
import java.util.LinkedHashMap;
import java.util.List;
import java.util.Map;

public class AssetBarcodeInventoryInputFileServiceImpl extends BatchInputFileSetServiceImpl implements
        AssetBarcodeInventoryInputFileService {

    private static final Logger LOG = LogManager.getLogger();
    private DateTimeService dateTimeService;

    @Override
    public Map<String, String> save(
            final Person user, final AssetBarcodeInventoryInputFileType inputType,
            final String fileUserIdentifier, final Map<String, InputStream> typeToStreamMap,
            final AssetBarCodeInventoryInputFileForm form) throws AuthorizationException, FileStorageException {

        //add a step to check for file directory - add the AssetBarcodeInventoryInputFile to check for file location
        final List<String> directoryPathList = new ArrayList<>(super.getRequiredDirectoryNames());
        //fileType was not supposed to be used in the getDirectoryPath
        directoryPathList.add(inputType.getDirectoryPath(AssetBarcodeInventoryInputFileType.class.getSimpleName()));
        prepareDirectories(directoryPathList);

        final Date creationDate = dateTimeService.getCurrentDate();
        final Map<String, File> typeToTempFiles = copyStreamsToTemporaryDirectory(user, inputType, fileUserIdentifier,
                typeToStreamMap, creationDate);

        if (!inputType.validate(typeToTempFiles)) {
            deleteTempFiles(typeToTempFiles);
            LOG.error(
                    "Upload file validation failed for user {} identifier {}",
                    user::getName,
                    () -> fileUserIdentifier
            );
            throw new ValidationException("File validation failed");
        }

        final byte[] buf = new byte[1024];

        final Map<String, String> typeToFileNames = new LinkedHashMap<>();
        final Map<String, File> typeToFiles = new LinkedHashMap<>();
        try {
            for (final String fileType : inputType.getFileTypes()) {
                final File tempFile = typeToTempFiles.get(fileType);
                final String saveFileName = inputType.getDirectoryPath(fileType) + File.separator + tempFile.getName();
                try {
                    final InputStream fileContents = new FileInputStream(tempFile);
                    final File fileToSave = new File(saveFileName);

                    copyInputStreamToFile(fileContents, fileToSave, buf);
                    fileContents.close();
                    typeToFileNames.put(fileType, saveFileName);
                    typeToFiles.put(fileType, fileToSave);
                } catch (final IOException e) {
                    LOG.error("unable to save contents to file {}", saveFileName, e);
                    throw new RuntimeException("errors encountered while writing file " + saveFileName, e);
                }
            }
        } finally {
            deleteTempFiles(typeToTempFiles);
        }

        final String doneFileName = inputType.getDoneFileDirectoryPath() + File.separator +
                                    inputType.getDoneFileName(user, fileUserIdentifier, creationDate);
        final File doneFile = new File(doneFileName);
        try {
            doneFile.createNewFile();

            typeToFiles.put(KFSConstants.DONE_FILE_TYPE, doneFile);
        } catch (final IOException e) {
            LOG.error("unable to create done file", e);
            throw new RuntimeException("unable to create done file", e);
        }

        inputType.process(typeToFiles, form);

        return typeToFileNames;
    }

    @Override
    public void setDateTimeService(final DateTimeService dateTimeService) {
        this.dateTimeService = dateTimeService;
    }
}
