/*
 * The Kuali Financial System, a comprehensive financial management system for higher education.
 *
 * Copyright 2005-2022 Kuali, Inc.
 *
 * This program is free software: you can redistribute it and/or modify
 * it under the terms of the GNU Affero General Public License as published by
 * the Free Software Foundation, either version 3 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU Affero General Public License for more details.
 *
 * You should have received a copy of the GNU Affero General Public License
 * along with this program.  If not, see <http://www.gnu.org/licenses/>.
 */
package org.kuali.kfs.module.cam.businessobject;

import org.apache.commons.lang3.StringUtils;
import org.kuali.kfs.coa.businessobject.Account;
import org.kuali.kfs.coa.businessobject.AccountingPeriod;
import org.kuali.kfs.coa.businessobject.Chart;
import org.kuali.kfs.coa.service.AccountingPeriodService;
import org.kuali.kfs.krad.bo.GlobalBusinessObject;
import org.kuali.kfs.krad.bo.GlobalBusinessObjectDetail;
import org.kuali.kfs.krad.bo.PersistableBusinessObject;
import org.kuali.kfs.krad.bo.PersistableBusinessObjectBase;
import org.kuali.kfs.krad.service.BusinessObjectService;
import org.kuali.kfs.krad.util.ObjectUtils;
import org.kuali.kfs.module.cam.CamsConstants;
import org.kuali.kfs.module.cam.CamsPropertyConstants;
import org.kuali.kfs.module.cam.document.service.AssetPaymentService;
import org.kuali.kfs.module.cam.document.service.AssetRetirementService;
import org.kuali.kfs.module.cam.document.service.PaymentSummaryService;
import org.kuali.kfs.sys.businessobject.Country;
import org.kuali.kfs.sys.businessobject.DocumentHeader;
import org.kuali.kfs.sys.businessobject.GeneralLedgerPendingEntry;
import org.kuali.kfs.sys.businessobject.PostalCode;
import org.kuali.kfs.sys.businessobject.State;
import org.kuali.kfs.sys.businessobject.UniversityDate;
import org.kuali.kfs.sys.context.SpringContext;
import org.kuali.kfs.sys.service.UniversityDateService;
import org.kuali.kfs.core.api.datetime.DateTimeService;
import org.kuali.kfs.core.api.util.type.KualiDecimal;

import java.sql.Date;
import java.sql.Timestamp;
import java.util.ArrayList;
import java.util.Collection;
import java.util.List;

public class AssetRetirementGlobal extends PersistableBusinessObjectBase implements GlobalBusinessObject {

    protected String documentNumber;
    protected Long mergedTargetCapitalAssetNumber;
    protected String mergedTargetCapitalAssetDescription;
    protected String retirementReasonCode;
    protected String retirementChartOfAccountsCode;
    protected String retirementAccountNumber;
    protected String retirementContactName;
    protected String retirementInstitutionName;
    protected String retirementStreetAddress;
    protected String retirementCityName;
    protected String retirementStateCode;
    protected String retirementZipCode;
    protected String retirementCountryCode;
    protected String retirementPhoneNumber;
    protected KualiDecimal estimatedSellingPrice;
    protected KualiDecimal salePrice;
    protected String cashReceiptFinancialDocumentNumber;
    protected KualiDecimal handlingFeeAmount;
    protected KualiDecimal preventiveMaintenanceAmount;
    protected String buyerDescription;
    protected String paidCaseNumber;
    // persistent relationship
    protected Date retirementDate;
    protected Asset mergedTargetCapitalAsset;
    protected AssetRetirementReason retirementReason;
    protected DocumentHeader documentHeader;
    protected List<AssetRetirementGlobalDetail> assetRetirementGlobalDetails;
    protected Account retirementAccount;
    protected Chart retirementChartOfAccounts;
    protected DocumentHeader cashReceiptFinancialDocument;
    protected Country retirementCountry;
    protected State retirementState;
    protected PostalCode postalZipCode;

    protected List<GeneralLedgerPendingEntry> generalLedgerPendingEntries;

    protected Integer postingYear;
    protected String postingPeriodCode;
    protected AccountingPeriod accountingPeriod;
    protected static transient AccountingPeriodService accountingPeriodService;

    // Non-persistent
    protected KualiDecimal calculatedTotal;

    public AssetRetirementGlobal() {
        assetRetirementGlobalDetails = new ArrayList<>();
        generalLedgerPendingEntries = new ArrayList<>();
    }

    @Override
    public List<PersistableBusinessObject> generateDeactivationsToPersist() {
        return null;
    }

    @Override
    public List<PersistableBusinessObject> generateGlobalChangesToPersist() {
        final AssetRetirementService retirementService = SpringContext.getBean(AssetRetirementService.class);

        final List<PersistableBusinessObject> persistables = new ArrayList<>();

        if (retirementService.isAssetRetiredByMerged(this) && mergedTargetCapitalAsset != null) {
            setMergeObjectsForPersist(persistables, retirementService);
        }

        for (final AssetRetirementGlobalDetail detail : assetRetirementGlobalDetails) {
            // do a deep copy so the source asset isn't changed and subsequent calls to this method return the same list
            // of persistable BOs; otherwise an OptimisticLockException can occur when blanket approving an
            // asset retirement global
            setAssetForPersist((Asset) ObjectUtils.deepCopy(detail.getAsset()), persistables, retirementService);
        }

        return persistables;
    }

    @Override
    public List<Collection<PersistableBusinessObject>> buildListOfDeletionAwareLists() {
        final List<Collection<PersistableBusinessObject>> managedList = super.buildListOfDeletionAwareLists();
        managedList.add(new ArrayList<>(getAssetRetirementGlobalDetails()));
        return managedList;
    }

    /**
     * This method set asset fields for update
     *
     * @param persistables
     */
    protected void setAssetForPersist(
            final Asset asset, final List<PersistableBusinessObject> persistables,
            final AssetRetirementService retirementService) {
        final UniversityDateService universityDateService = SpringContext.getBean(UniversityDateService.class);

        // load the object by key
        asset.setInventoryStatusCode(CamsConstants.InventoryStatusCode.CAPITAL_ASSET_RETIRED);
        asset.setRetirementReasonCode(retirementReasonCode);

        // set retirement fiscal year and period code into asset
        final UniversityDate currentUniversityDate = universityDateService.getCurrentUniversityDate();
        if (ObjectUtils.isNotNull(currentUniversityDate)) {
            asset.setRetirementFiscalYear(universityDateService.getCurrentUniversityDate().getUniversityFiscalYear());
            asset.setRetirementPeriodCode(universityDateService.getCurrentUniversityDate()
                    .getUniversityFiscalAccountingPeriod());
        }

        if (retirementService.isAssetRetiredByTheft(this)
                && StringUtils.isNotBlank(getPaidCaseNumber())) {
            asset.setCampusPoliceDepartmentCaseNumber(getPaidCaseNumber());
        } else if (retirementService.isAssetRetiredBySold(this)
                || retirementService.isAssetRetiredByAuction(this)) {
            asset.setRetirementChartOfAccountsCode(getRetirementChartOfAccountsCode());
            asset.setRetirementAccountNumber(getRetirementAccountNumber());
            asset.setCashReceiptFinancialDocumentNumber(getCashReceiptFinancialDocumentNumber());
            asset.setSalePrice(getSalePrice());
            asset.setEstimatedSellingPrice(getEstimatedSellingPrice());
        } else if (retirementService.isAssetRetiredByMerged(this)) {
            asset.setTotalCostAmount(KualiDecimal.ZERO);
            asset.setSalvageAmount(KualiDecimal.ZERO);
        } else if (retirementService.isAssetRetiredByExternalTransferOrGift(this)) {
            persistables.add(setOffCampusLocationObjectsForPersist(asset));
        }
        asset.setLastInventoryDate(new Timestamp(SpringContext.getBean(DateTimeService.class).getCurrentSqlDate()
                .getTime()));
        persistables.add(asset);
    }

    /**
     * This method set off campus location for persist
     *
     * @param asset Asset to populate AssetLocation
     * @return the AssetLocation.
     */
    protected AssetLocation setOffCampusLocationObjectsForPersist(final Asset asset) {
        AssetLocation offCampusLocation = new AssetLocation();
        offCampusLocation.setCapitalAssetNumber(asset.getCapitalAssetNumber());
        offCampusLocation.setAssetLocationTypeCode(CamsConstants.AssetLocationTypeCode.RETIREMENT);
        offCampusLocation = (AssetLocation) SpringContext.getBean(BusinessObjectService.class)
                .retrieve(offCampusLocation);
        if (offCampusLocation == null) {
            offCampusLocation = new AssetLocation();
            offCampusLocation.setCapitalAssetNumber(asset.getCapitalAssetNumber());
            offCampusLocation.setAssetLocationTypeCode(CamsConstants.AssetLocationTypeCode.RETIREMENT);
            asset.getAssetLocations().add(offCampusLocation);
        }

        offCampusLocation.setAssetLocationContactName(getRetirementContactName());
        offCampusLocation.setAssetLocationInstitutionName(getRetirementInstitutionName());
        offCampusLocation.setAssetLocationPhoneNumber(getRetirementPhoneNumber());
        offCampusLocation.setAssetLocationStreetAddress(getRetirementStreetAddress());
        offCampusLocation.setAssetLocationCityName(getRetirementCityName());
        offCampusLocation.setAssetLocationStateCode(getRetirementStateCode());
        offCampusLocation.setAssetLocationCountryCode(getRetirementCountryCode());
        offCampusLocation.setAssetLocationZipCode(getRetirementZipCode());

        return offCampusLocation;
    }

    /**
     * This method set target payment and source payment; set target/source asset salvageAmount/totalCostAmount
     *
     * @param persistables
     */
    protected void setMergeObjectsForPersist(
            final List<PersistableBusinessObject> persistables,
            final AssetRetirementService retirementService) {
        final PaymentSummaryService paymentSummaryService = SpringContext.getBean(PaymentSummaryService.class);
        final AssetPaymentService assetPaymentService = SpringContext.getBean(AssetPaymentService.class);

        Integer maxTargetSequenceNo = assetPaymentService.getMaxSequenceNumber(mergedTargetCapitalAssetNumber);

        KualiDecimal salvageAmount = KualiDecimal.ZERO;
        KualiDecimal totalCostAmount = KualiDecimal.ZERO;
        Asset sourceAsset;

        // update for each merge source asset
        for (final AssetRetirementGlobalDetail detail : getAssetRetirementGlobalDetails()) {
            detail.refreshReferenceObject(CamsPropertyConstants.AssetRetirementGlobalDetail.ASSET);
            sourceAsset = detail.getAsset();

            totalCostAmount = totalCostAmount.add(paymentSummaryService.calculatePaymentTotalCost(sourceAsset));
            salvageAmount = salvageAmount.add(sourceAsset.getSalvageAmount());

            retirementService.generateOffsetPaymentsForEachSource(sourceAsset, persistables, detail.getDocumentNumber());
            maxTargetSequenceNo = retirementService.generateNewPaymentForTarget(mergedTargetCapitalAsset, sourceAsset,
                    persistables, maxTargetSequenceNo, detail.getDocumentNumber());

        }
        final KualiDecimal mergedTargetSalvageAmount = mergedTargetCapitalAsset.getSalvageAmount() != null ?
                mergedTargetCapitalAsset.getSalvageAmount() : KualiDecimal.ZERO;

        // update merged target asset
        mergedTargetCapitalAsset.setTotalCostAmount(totalCostAmount.add(paymentSummaryService
                .calculatePaymentTotalCost(mergedTargetCapitalAsset)));
        mergedTargetCapitalAsset.setSalvageAmount(salvageAmount.add(mergedTargetSalvageAmount));
        mergedTargetCapitalAsset.setLastInventoryDate(new Timestamp(SpringContext.getBean(DateTimeService.class)
                .getCurrentSqlDate().getTime()));
        mergedTargetCapitalAsset.setCapitalAssetDescription(getMergedTargetCapitalAssetDescription());
        persistables.add(mergedTargetCapitalAsset);
    }

    @Override
    public List<? extends GlobalBusinessObjectDetail> getAllDetailObjects() {
        return getAssetRetirementGlobalDetails();
    }

    @Override
    public boolean isPersistable() {
        return true;
    }

    @Override
    public String getDocumentNumber() {
        return documentNumber;
    }

    @Override
    public void setDocumentNumber(final String documentNumber) {
        this.documentNumber = documentNumber;
    }

    public Long getMergedTargetCapitalAssetNumber() {
        return mergedTargetCapitalAssetNumber;
    }

    public void setMergedTargetCapitalAssetNumber(final Long mergedTargetCapitalAssetNumber) {
        this.mergedTargetCapitalAssetNumber = mergedTargetCapitalAssetNumber;
    }

    public String getRetirementReasonCode() {
        return retirementReasonCode;
    }

    public void setRetirementReasonCode(final String retirementReasonCode) {
        this.retirementReasonCode = retirementReasonCode;
    }

    public Date getRetirementDate() {
        return retirementDate;
    }

    public void setRetirementDate(final Date retirementDate) {
        this.retirementDate = retirementDate;
    }

    public Asset getMergedTargetCapitalAsset() {
        return mergedTargetCapitalAsset;
    }

    /**
     * @deprecated
     */
    public void setMergedTargetCapitalAsset(final Asset mergedTargetCapitalAsset) {
        this.mergedTargetCapitalAsset = mergedTargetCapitalAsset;
    }

    public AssetRetirementReason getRetirementReason() {
        return retirementReason;
    }

    /**
     * @deprecated
     */
    public void setRetirementReason(final AssetRetirementReason retirementReason) {
        this.retirementReason = retirementReason;
    }

    public DocumentHeader getDocumentHeader() {
        return documentHeader;
    }

    public void setDocumentHeader(final DocumentHeader documentHeader) {
        this.documentHeader = documentHeader;
    }

    public List<AssetRetirementGlobalDetail> getAssetRetirementGlobalDetails() {
        return assetRetirementGlobalDetails;
    }

    public void setAssetRetirementGlobalDetails(final List<AssetRetirementGlobalDetail> assetRetirementGlobalDetails) {
        this.assetRetirementGlobalDetails = assetRetirementGlobalDetails;
    }

    public List<GeneralLedgerPendingEntry> getGeneralLedgerPendingEntries() {
        return generalLedgerPendingEntries;
    }

    public void setGeneralLedgerPendingEntries(final List<GeneralLedgerPendingEntry> glPendingEntries) {
        generalLedgerPendingEntries = glPendingEntries;
    }

    public String getMergedTargetCapitalAssetDescription() {
        return mergedTargetCapitalAssetDescription;
    }

    public void setMergedTargetCapitalAssetDescription(final String mergedTargetCapitalAssetDescription) {
        this.mergedTargetCapitalAssetDescription = mergedTargetCapitalAssetDescription;
    }

    public String getRetirementChartOfAccountsCode() {
        return retirementChartOfAccountsCode;
    }

    public void setRetirementChartOfAccountsCode(final String retirementChartOfAccountsCode) {
        this.retirementChartOfAccountsCode = retirementChartOfAccountsCode;
    }

    public String getRetirementAccountNumber() {
        return retirementAccountNumber;
    }

    public void setRetirementAccountNumber(final String retirementAccountNumber) {
        this.retirementAccountNumber = retirementAccountNumber;
    }

    public String getRetirementContactName() {
        return retirementContactName;
    }

    public void setRetirementContactName(final String retirementContactName) {
        this.retirementContactName = retirementContactName;
    }

    public String getRetirementInstitutionName() {
        return retirementInstitutionName;
    }

    public void setRetirementInstitutionName(final String retirementInstitutionName) {
        this.retirementInstitutionName = retirementInstitutionName;
    }

    public String getRetirementStreetAddress() {
        return retirementStreetAddress;
    }

    public void setRetirementStreetAddress(final String retirementStreetAddress) {
        this.retirementStreetAddress = retirementStreetAddress;
    }

    public String getRetirementCityName() {
        return retirementCityName;
    }

    public void setRetirementCityName(final String retirementCityName) {
        this.retirementCityName = retirementCityName;
    }

    public String getRetirementStateCode() {
        return retirementStateCode;
    }

    public void setRetirementStateCode(final String retirementStateCode) {
        this.retirementStateCode = retirementStateCode;
    }

    public String getRetirementZipCode() {
        return retirementZipCode;
    }

    public void setRetirementZipCode(final String retirementZipCode) {
        this.retirementZipCode = retirementZipCode;
    }

    public PostalCode getPostalZipCode() {
        return postalZipCode;
    }

    public void setPostalZipCode(final PostalCode postalZipCode) {
        this.postalZipCode = postalZipCode;
    }

    public String getRetirementCountryCode() {
        return retirementCountryCode;
    }

    public void setRetirementCountryCode(final String retirementCountryCode) {
        this.retirementCountryCode = retirementCountryCode;
    }

    public String getRetirementPhoneNumber() {
        return retirementPhoneNumber;
    }

    public void setRetirementPhoneNumber(final String retirementPhoneNumber) {
        this.retirementPhoneNumber = retirementPhoneNumber;
    }

    public KualiDecimal getEstimatedSellingPrice() {
        return estimatedSellingPrice;
    }

    public void setEstimatedSellingPrice(final KualiDecimal estimatedSellingPrice) {
        this.estimatedSellingPrice = estimatedSellingPrice;
    }

    public KualiDecimal getSalePrice() {
        return salePrice;
    }

    public void setSalePrice(final KualiDecimal salePrice) {
        this.salePrice = salePrice;
    }

    public String getCashReceiptFinancialDocumentNumber() {
        return cashReceiptFinancialDocumentNumber;
    }

    public void setCashReceiptFinancialDocumentNumber(final String cashReceiptFinancialDocumentNumber) {
        this.cashReceiptFinancialDocumentNumber = cashReceiptFinancialDocumentNumber;
    }

    public KualiDecimal getHandlingFeeAmount() {
        return handlingFeeAmount;
    }

    public void setHandlingFeeAmount(final KualiDecimal handlingFeeAmount) {
        this.handlingFeeAmount = handlingFeeAmount;
    }

    public KualiDecimal getPreventiveMaintenanceAmount() {
        return preventiveMaintenanceAmount;
    }

    public void setPreventiveMaintenanceAmount(final KualiDecimal preventiveMaintenanceAmount) {
        this.preventiveMaintenanceAmount = preventiveMaintenanceAmount;
    }

    public String getBuyerDescription() {
        return buyerDescription;
    }

    public void setBuyerDescription(final String buyerDescription) {
        this.buyerDescription = buyerDescription;
    }

    public String getPaidCaseNumber() {
        return paidCaseNumber;
    }

    public void setPaidCaseNumber(final String paidCaseNumber) {
        this.paidCaseNumber = paidCaseNumber;
    }

    public Chart getRetirementChartOfAccounts() {
        return retirementChartOfAccounts;
    }

    /**
     * @deprecated
     */
    public void setRetirementChartOfAccounts(final Chart retirementChartOfAccounts) {
        this.retirementChartOfAccounts = retirementChartOfAccounts;
    }

    public Account getRetirementAccount() {
        return retirementAccount;
    }

    /**
     * @deprecated
     */
    public void setRetirementAccount(final Account retirementAccount) {
        this.retirementAccount = retirementAccount;
    }

    public DocumentHeader getCashReceiptFinancialDocument() {
        return cashReceiptFinancialDocument;
    }

    /**
     * @deprecated
     */
    public void setCashReceiptFinancialDocument(final DocumentHeader cashReceiptFinancialDocument) {
        this.cashReceiptFinancialDocument = cashReceiptFinancialDocument;
    }

    public Country getRetirementCountry() {
        return retirementCountry;
    }

    /**
     * @deprecated
     */
    public void setRetirementCountry(final Country retirementCountry) {
        this.retirementCountry = retirementCountry;
    }

    public State getRetirementState() {
        return retirementState;
    }

    /**
     * @deprecated
     */
    public void setRetirementState(final State retirementState) {
        this.retirementState = retirementState;
    }

    public KualiDecimal getCalculatedTotal() {
        calculatedTotal = KualiDecimal.ZERO;
        if (handlingFeeAmount != null) {
            calculatedTotal = calculatedTotal.add(handlingFeeAmount);
        }
        if (preventiveMaintenanceAmount != null) {
            calculatedTotal = calculatedTotal.add(preventiveMaintenanceAmount);
        }
        if (salePrice != null) {
            calculatedTotal = calculatedTotal.add(salePrice);
        }
        return calculatedTotal;
    }

    public Integer getPostingYear() {
        return postingYear;
    }

    public void setPostingYear(final Integer postingYear) {
        this.postingYear = postingYear;
    }

    public static AccountingPeriodService getAccountingPeriodService() {
        if (accountingPeriodService == null) {
            accountingPeriodService = SpringContext.getBean(AccountingPeriodService.class);
        }
        return accountingPeriodService;
    }

    /**
     * Creates a composite of postingPeriodCode and postingyear.
     *
     * @return composite or an empty string if either postingPeriodCode or postingYear is null
     */
    public String getAccountingPeriodCompositeString() {
        if (postingPeriodCode == null || postingYear == null) {
            return "";
        }
        return postingPeriodCode + postingYear;
    }

    public void setAccountingPeriodCompositeString(final String accountingPeriodString) {
        if (StringUtils.isNotBlank(accountingPeriodString)) {
            final String period = StringUtils.left(accountingPeriodString, 2);
            final Integer year = Integer.valueOf(StringUtils.right(accountingPeriodString, 4));
            final AccountingPeriod accountingPeriod = getAccountingPeriodService().getByPeriod(period, year);
            setAccountingPeriod(accountingPeriod);
        }
    }

    public void setAccountingPeriodCompositeString(final AccountingPeriod accountingPeriod) {
        setAccountingPeriod(accountingPeriod);
    }

    public String getPostingPeriodCode() {
        return postingPeriodCode;
    }

    public void setPostingPeriodCode(final String postingPeriodCode) {
        this.postingPeriodCode = postingPeriodCode;
    }

    /**
     * Set postingYear and postingPeriodCode
     *
     * @param accountingPeriod
     */
    public void setAccountingPeriod(final AccountingPeriod accountingPeriod) {
        this.accountingPeriod = accountingPeriod;
        if (ObjectUtils.isNotNull(accountingPeriod)) {
            setPostingYear(accountingPeriod.getUniversityFiscalYear());
            setPostingPeriodCode(accountingPeriod.getUniversityFiscalPeriodCode());
        }
    }

    public AccountingPeriod getAccountingPeriod() {
        return accountingPeriod;
    }

}
