/*
 * The Kuali Financial System, a comprehensive financial management system for higher education.
 *
 * Copyright 2005-2022 Kuali, Inc.
 *
 * This program is free software: you can redistribute it and/or modify
 * it under the terms of the GNU Affero General Public License as published by
 * the Free Software Foundation, either version 3 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU Affero General Public License for more details.
 *
 * You should have received a copy of the GNU Affero General Public License
 * along with this program.  If not, see <http://www.gnu.org/licenses/>.
 */
package org.kuali.kfs.module.cam.businessobject;

import org.apache.logging.log4j.LogManager;
import org.apache.logging.log4j.Logger;
import org.kuali.kfs.krad.util.ObjectUtils;
import org.kuali.kfs.module.cam.CamsPropertyConstants;
import org.kuali.kfs.module.purap.businessobject.AccountsPayableItem;
import org.kuali.kfs.module.purap.businessobject.CreditMemoAccountRevision;
import org.kuali.kfs.module.purap.businessobject.PurApAccountingLineBase;
import org.kuali.kfs.module.purap.document.AccountsPayableDocumentBase;

import java.util.ArrayList;
import java.util.List;

/**
 * Accounting line grouped data for GL Line
 */
public class PurApAccountLineGroup extends AccountLineGroup {

    private static final Logger LOG = LogManager.getLogger();

    private List<PurApAccountingLineBase> sourceEntries = new ArrayList<>();

    protected PurApAccountLineGroup() {
    }

    /**
     * Constructs a PurApAccountLineGroup from a PurApAccountingLineBase Line Entry
     *
     * @param entry PurApAccountingLineBase Line
     */
    public PurApAccountLineGroup(final PurApAccountingLineBase entry) {
        entry.refreshReferenceObject(CamsPropertyConstants.PURAP_ITEM);
        final AccountsPayableItem purapItem = entry.getPurapItem();
        if (ObjectUtils.isNotNull(purapItem)) {
            purapItem.refreshReferenceObject(CamsPropertyConstants.PURAP_DOCUMENT);
            final AccountsPayableDocumentBase purapDocument = ((AccountsPayableItem) entry.getPurapItem()).getPurapDocument();
            if (ObjectUtils.isNotNull(purapDocument)) {
                setReferenceFinancialDocumentNumber(purapDocument.getPurchaseOrderIdentifier() != null ?
                        purapDocument.getPurchaseOrderIdentifier().toString() : "");
                setDocumentNumber(purapDocument.getDocumentNumber());
            }
        } else {
            LOG.error("Could not load PurAP document details for {}", entry);
        }
        setUniversityFiscalYear(entry.getPostingYear());
        setUniversityFiscalPeriodCode(entry.getPostingPeriodCode());
        setChartOfAccountsCode(entry.getChartOfAccountsCode());
        setAccountNumber(entry.getAccountNumber());
        setSubAccountNumber(entry.getSubAccountNumber());
        setFinancialObjectCode(entry.getFinancialObjectCode());
        setFinancialSubObjectCode(entry.getFinancialSubObjectCode());
        setOrganizationReferenceId(entry.getOrganizationReferenceId());
        setProjectCode(entry.getProjectCode());
        sourceEntries.add(entry);
        if (CreditMemoAccountRevision.class.isAssignableFrom(entry.getClass())) {
            setAmount(entry.getAmount().negated());
        } else {
            setAmount(entry.getAmount());
        }
    }

    /**
     * @param entry PurApAccountingLineBase
     * @return true if PurApAccountingLineBase belongs to same account line group
     */
    public boolean isAccounted(final PurApAccountingLineBase entry) {
        final PurApAccountLineGroup test = new PurApAccountLineGroup(entry);
        return equals(test);
    }

    /**
     * This method will combine multiple Purap account entries for the same account line group.
     *
     * @param newEntry PurApAccountingLineBase
     */
    public void combineEntry(final PurApAccountingLineBase newEntry) {
        sourceEntries.add(newEntry);
        if (CreditMemoAccountRevision.class.isAssignableFrom(newEntry.getClass())) {
            amount = amount.add(newEntry.getAmount().negated());
        } else {
            amount = amount.add(newEntry.getAmount());
        }
    }

    public List<PurApAccountingLineBase> getSourceEntries() {
        return sourceEntries;
    }

    public void setSourceEntries(final List<PurApAccountingLineBase> sourceGlEntries) {
        sourceEntries = sourceGlEntries;
    }
}
