/*
 * The Kuali Financial System, a comprehensive financial management system for higher education.
 *
 * Copyright 2005-2022 Kuali, Inc.
 *
 * This program is free software: you can redistribute it and/or modify
 * it under the terms of the GNU Affero General Public License as published by
 * the Free Software Foundation, either version 3 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU Affero General Public License for more details.
 *
 * You should have received a copy of the GNU Affero General Public License
 * along with this program.  If not, see <http://www.gnu.org/licenses/>.
 */
package org.kuali.kfs.module.cam.document;

import org.apache.commons.lang3.StringUtils;
import org.apache.logging.log4j.LogManager;
import org.apache.logging.log4j.Logger;
import org.kuali.kfs.core.api.config.property.ConfigurationService;
import org.kuali.kfs.core.api.datetime.DateTimeService;
import org.kuali.kfs.coreservice.framework.parameter.ParameterService;
import org.kuali.kfs.integration.cam.CapitalAssetManagementModuleService;
import org.kuali.kfs.kew.api.WorkflowDocument;
import org.kuali.kfs.kim.api.identity.IdentityService;
import org.kuali.kfs.kns.document.MaintenanceDocument;
import org.kuali.kfs.kns.maintenance.Maintainable;
import org.kuali.kfs.kns.web.ui.Section;
import org.kuali.kfs.krad.bo.Note;
import org.kuali.kfs.krad.maintenance.MaintenanceLock;
import org.kuali.kfs.krad.service.DocumentService;
import org.kuali.kfs.krad.util.ErrorMessage;
import org.kuali.kfs.krad.util.GlobalVariables;
import org.kuali.kfs.krad.util.KRADConstants;
import org.kuali.kfs.krad.util.ObjectUtils;
import org.kuali.kfs.module.cam.CamsConstants;
import org.kuali.kfs.module.cam.CamsKeyConstants;
import org.kuali.kfs.module.cam.CamsParameterConstants;
import org.kuali.kfs.module.cam.CamsPropertyConstants;
import org.kuali.kfs.module.cam.businessobject.Asset;
import org.kuali.kfs.module.cam.businessobject.AssetFabrication;
import org.kuali.kfs.module.cam.businessobject.defaultvalue.NextAssetNumberFinder;
import org.kuali.kfs.module.cam.document.service.AssetLocationService;
import org.kuali.kfs.module.cam.document.service.AssetService;
import org.kuali.kfs.module.cam.document.service.EquipmentLoanOrReturnService;
import org.kuali.kfs.module.cam.document.service.PaymentSummaryService;
import org.kuali.kfs.module.cam.document.service.RetirementInfoService;
import org.kuali.kfs.module.cam.service.AssetLockService;
import org.kuali.kfs.module.cam.util.MaintainableWorkflowUtils;
import org.kuali.kfs.sys.KFSConstants;
import org.kuali.kfs.sys.KFSPropertyConstants;
import org.kuali.kfs.sys.businessobject.DocumentHeader;
import org.kuali.kfs.sys.context.SpringContext;
import org.kuali.kfs.sys.document.FinancialSystemMaintainable;
import org.springframework.util.AutoPopulatingList;

import java.sql.Timestamp;
import java.text.MessageFormat;
import java.util.ArrayList;
import java.util.HashMap;
import java.util.Iterator;
import java.util.List;
import java.util.Map;

/**
 * This class implements custom data preparation for displaying asset edit screen.
 */
public class AssetMaintainableImpl extends FinancialSystemMaintainable {

    private static final Logger LOG = LogManager.getLogger();

    private Asset asset;
    private Asset copyAsset;
    private boolean fabricationOn;
    protected static volatile IdentityService identityService;
    protected static volatile DocumentService documentService;
    protected static volatile AssetService assetService;

    private static final Map<String, String> FINANCIAL_DOC_NAME_MAP = new HashMap<>();

    static {
        FINANCIAL_DOC_NAME_MAP.put(KFSConstants.FinancialDocumentTypeCodes.CASH_RECEIPT,
                KFSConstants.FinancialDocumentTypeNames.CASH_RECEIPT);
        FINANCIAL_DOC_NAME_MAP.put(KFSConstants.FinancialDocumentTypeCodes.DISTRIBUTION_OF_INCOME_AND_EXPENSE,
                KFSConstants.FinancialDocumentTypeNames.DISTRIBUTION_OF_INCOME_AND_EXPENSE);
        FINANCIAL_DOC_NAME_MAP.put(KFSConstants.FinancialDocumentTypeCodes.GENERAL_ERROR_CORRECTION,
                KFSConstants.FinancialDocumentTypeNames.GENERAL_ERROR_CORRECTION);
        FINANCIAL_DOC_NAME_MAP.put(KFSConstants.FinancialDocumentTypeCodes.INTERNAL_BILLING,
                KFSConstants.FinancialDocumentTypeNames.INTERNAL_BILLING);
        FINANCIAL_DOC_NAME_MAP.put(KFSConstants.FinancialDocumentTypeCodes.SERVICE_BILLING,
                KFSConstants.FinancialDocumentTypeNames.SERVICE_BILLING);
        FINANCIAL_DOC_NAME_MAP.put(
                KFSConstants.FinancialDocumentTypeCodes.YEAR_END_DISTRIBUTION_OF_INCOME_AND_EXPENSE,
                KFSConstants.FinancialDocumentTypeNames.YEAR_END_DISTRIBUTION_OF_INCOME_AND_EXPENSE);
        FINANCIAL_DOC_NAME_MAP.put(KFSConstants.FinancialDocumentTypeCodes.YEAR_END_GENERAL_ERROR_CORRECTION,
                KFSConstants.FinancialDocumentTypeNames.YEAR_END_GENERAL_ERROR_CORRECTION);
        FINANCIAL_DOC_NAME_MAP.put(KFSConstants.FinancialDocumentTypeCodes.PROCUREMENT_CARD,
                KFSConstants.FinancialDocumentTypeNames.PROCUREMENT_CARD);
    }

    /**
     * We are using a substitute mechanism for asset locking which can lock on assets when rule check passed.
     *
     * @return empty list from this method.
     */
    @Override
    public List<MaintenanceLock> generateMaintenanceLocks() {
        return new ArrayList<>();
    }

    @Override
    public void doRouteStatusChange(final DocumentHeader documentHeader) {
        super.doRouteStatusChange(documentHeader);
        final WorkflowDocument workflowDoc = documentHeader.getWorkflowDocument();
        // release lock for asset edit...
        if (getBusinessObject() instanceof Asset && !(getBusinessObject() instanceof AssetFabrication)
            && (workflowDoc.isCanceled() || workflowDoc.isDisapproved() || workflowDoc.isProcessed()
                    || workflowDoc.isFinal())) {
            getCapitalAssetManagementModuleService().deleteAssetLocks(getDocumentNumber(), null);
        }
    }

    protected CapitalAssetManagementModuleService getCapitalAssetManagementModuleService() {
        return SpringContext.getBean(CapitalAssetManagementModuleService.class);
    }

    /**
     * @param document   Maintenance Document used for editing
     * @param parameters Parameters available
     */
    @Override
    public void processAfterEdit(final MaintenanceDocument document, final Map parameters) {
        initializeAttributes(document);
        // Identifies the latest location information
        getAssetLocationService().setOffCampusLocation(copyAsset);
        getAssetLocationService().setOffCampusLocation(asset);

        // Calculates payment summary and depreciation summary based on available payment records
        final PaymentSummaryService paymentSummaryService = SpringContext.getBean(PaymentSummaryService.class);
        paymentSummaryService.calculateAndSetPaymentSummary(copyAsset);
        paymentSummaryService.calculateAndSetPaymentSummary(asset);

        // Identifies the merge history and separation history based on asset disposition records
        getAssetService().setSeparateHistory(copyAsset);
        getAssetService().setSeparateHistory(asset);

        // Finds out the latest retirement info, is asset is currently retired.
        final RetirementInfoService retirementInfoService = SpringContext.getBean(RetirementInfoService.class);
        retirementInfoService.setRetirementInfo(copyAsset);
        retirementInfoService.setRetirementInfo(asset);

        retirementInfoService.setMergeHistory(copyAsset);
        retirementInfoService.setMergeHistory(asset);

        // Finds out the latest equipment loan or return information if available
        final EquipmentLoanOrReturnService equipmentLoanOrReturnService = SpringContext.getBean(
                EquipmentLoanOrReturnService.class);
        equipmentLoanOrReturnService.setEquipmentLoanInfo(copyAsset);
        equipmentLoanOrReturnService.setEquipmentLoanInfo(asset);

        super.processAfterEdit(document, parameters);
    }

    /**
     * Uses data dictionary for bo Asset to get the core section ids to set section titles.
     */
    @Override
    public List<Section> getCoreSections(final MaintenanceDocument document, final Maintainable oldMaintainable) {
        final List<Section> sections = super.getCoreSections(document, oldMaintainable);

        final Asset asset = (Asset) getBusinessObject();
        if (asset.getAssetPayments().size() == 0) {
            for (final Section section : sections) {
                if (CamsConstants.Asset.SECTION_ID_PAYMENT_INFORMATION.equals(section.getSectionId())) {
                    section.setSectionTitle(section.getSectionTitle() + CamsConstants.Asset.SECTION_TITLE_NO_PAYMENT +
                            asset.getCapitalAssetNumber());
                }
            }
        }

        return sections;
    }

    /**
     * This method gets old and new maintainable objects and creates convenience handles to them
     *
     * @param document Asset Edit Document
     */
    private void initializeAttributes(final MaintenanceDocument document) {
        if (asset == null) {
            asset = (Asset) document.getNewMaintainableObject().getBusinessObject();
            asset.setTagged();
        }
        if (copyAsset == null) {
            copyAsset = (Asset) document.getOldMaintainableObject().getBusinessObject();
        }

        setFabricationOn(document.getNewMaintainableObject().getBusinessObject() instanceof AssetFabrication);
    }

    /**
     * KFSMI-5964: added refresh to Asset object after retrieve to prevent updated depreciation data from
     * wiped on existing saved/enrouted maint. doc
     */
    @Override
    public void processAfterRetrieve() {
        if (getBusinessObject() instanceof Asset
            && MaintainableWorkflowUtils.isDocumentSavedOrEnroute(getDocumentNumber())) {
            final Asset asset = (Asset) getBusinessObject();
            asset.refreshReferenceObject(CamsPropertyConstants.Asset.ASSET_PAYMENTS);

            final PaymentSummaryService paymentSummaryService = SpringContext.getBean(PaymentSummaryService.class);
            paymentSummaryService.calculateAndSetPaymentSummary(asset);
        }
    }

    @Override
    public void processAfterPost(final MaintenanceDocument document, final Map<String, String[]> parameters) {
        final String customAction = findCustomAction(parameters);

        if (StringUtils.equals(CamsPropertyConstants.Asset.LAST_INVENTORY_DATE_UPDATE_BUTTON, customAction)) {
            processLastInventoryDateUpdate(document);
        }

        deleteObsoleteNotes(document);

        super.processAfterPost(document, parameters);
    }

    private String findCustomAction(final Map<String, String[]> parameters) {
        String customAction = null;
        final String[] customActions = parameters.get(KRADConstants.CUSTOM_ACTION);
        if (customActions != null) {
            customAction = customActions[0];
        }
        return customAction;
    }

    private void processLastInventoryDateUpdate(final MaintenanceDocument document) {
        final WorkflowDocument workflowDoc = document.getDocumentHeader().getWorkflowDocument();
        if (workflowDoc != null && workflowDoc.isInitiated()) {
            final Asset oldAsset = (Asset) document.getOldMaintainableObject().getBusinessObject();
            final Asset newAsset = (Asset) document.getNewMaintainableObject().getBusinessObject();

            if (getAssetService().hasAssetLocationChanged(oldAsset, newAsset)) {
                GlobalVariables.getMessageMap().addToErrorPath(KFSConstants.MAINTENANCE_NEW_MAINTAINABLE);
                GlobalVariables.getMessageMap().putWarning(CamsPropertyConstants.Asset.CAMPUS_CODE,
                        CamsKeyConstants.AssetLocationGlobal.WARNING_LOCATION_UPDATE_UNNECESSARY,
                        newAsset.getCapitalAssetNumber().toString());
                GlobalVariables.getMessageMap().removeFromErrorPath(KFSConstants.MAINTENANCE_NEW_MAINTAINABLE);
            } else {
                final String noteText = buildNoteText();
                final Note lastInventoryDateUpdatedNote = getDocumentService().createNoteFromDocument(document, noteText);

                if (shouldAddNote(document, noteText)) {
                    clearObsoleteWarningMessage();
                    asset.setLastInventoryDate(
                        new Timestamp(SpringContext.getBean(DateTimeService.class).getCurrentSqlDate().getTime()));

                    lastInventoryDateUpdatedNote.setAuthorUniversalIdentifier(
                        getIdentityService().getPrincipalByPrincipalName(KFSConstants.SYSTEM_USER).getPrincipalId());
                    document.addNote(lastInventoryDateUpdatedNote);
                    getDocumentService().saveDocumentNotes(document);
                }
            }
        }
    }

    private String buildNoteText() {
        final String userPrincipalName = GlobalVariables.getUserSession().getUserToLog();
        final String noteTextPattern =
            SpringContext.getBean(ConfigurationService.class).getPropertyValueAsString(
                CamsKeyConstants.Asset.LAST_INVENTORY_DATE_UPDATE_NOTE_TEXT);
        return MessageFormat.format(
            noteTextPattern,
            userPrincipalName,
            ((Asset) businessObject).getCapitalAssetNumber().toString());
    }

    private boolean shouldAddNote(final MaintenanceDocument document, final String noteText) {
        boolean shouldAddNote = true;

        for (final Note note : document.getNotes()) {
            if (note.getNoteText().equals(noteText)) {
                shouldAddNote = false;
                break;
            }
        }

        return shouldAddNote;
    }

    private void clearObsoleteWarningMessage() {
        final Map<String, AutoPopulatingList<ErrorMessage>> warningMessages = GlobalVariables.getMessageMap()
                .getWarningMessages();
        if (ObjectUtils.isNotNull(warningMessages) && !warningMessages.isEmpty()) {
            warningMessages.entrySet().removeIf(nextWarningMessage -> nextWarningMessage.getValue().get(0)
                    .getErrorKey().equals(CamsKeyConstants.AssetLocationGlobal.WARNING_LOCATION_UPDATE_UNNECESSARY));
        }
    }

    /**
     * If location details have been updated, the last inventory date will be updated from that. Notes from when the
     * user had previously clicked the update button are no longer needed and should be deleted.
     *
     * @param document
     */
    private void deleteObsoleteNotes(final MaintenanceDocument document) {
        final Asset oldAsset = (Asset) document.getOldMaintainableObject().getBusinessObject();
        final Asset newAsset = (Asset) document.getNewMaintainableObject().getBusinessObject();

        if (getAssetService().hasAssetLocationChanged(oldAsset, newAsset)) {
            if (!newAsset.getLastInventoryDate().equals(oldAsset.getLastInventoryDate())) {
                newAsset.setLastInventoryDate(oldAsset.getLastInventoryDate());
            }
            final String noteText = buildNoteText();
            for (final Iterator<Note> noteIterator = document.getNotes().iterator(); noteIterator.hasNext(); ) {
                final Note note = noteIterator.next();
                if (note.getNoteText().equals(noteText)) {
                    noteIterator.remove();
                    getDocumentService().saveDocumentNotes(document);
                }
            }
        }
    }

    @Override
    public void saveBusinessObject() {
        final Asset asset = (Asset) businessObject;
        if (asset.getCapitalAssetNumber() == null) {
            asset.setCapitalAssetNumber(NextAssetNumberFinder.getLongValue());
        }
        asset.refreshReferenceObject(CamsPropertyConstants.Asset.ASSET_LOCATIONS);
        // update and save asset location
        if (getAssetLocationService().isOffCampusLocationExists(asset.getOffCampusLocation())) {
            getAssetLocationService().updateOffCampusLocation(asset);
        }
        super.saveBusinessObject();
    }

    @Override
    public void processAfterNew(final MaintenanceDocument document, final Map<String, String[]> parameters) {
        super.processAfterNew(document, parameters);
        initializeAttributes(document);
        if (asset.getCreateDate() == null) {
            asset.setCreateDate(SpringContext.getBean(DateTimeService.class).getCurrentSqlDate());
            asset.setAcquisitionTypeCode(CamsConstants.Asset.ACQUISITION_TYPE_CODE_C);
            asset.setVendorName(CamsConstants.Asset.VENDOR_NAME_CONSTRUCTED);
            asset.setInventoryStatusCode(CamsConstants.InventoryStatusCode.CAPITAL_ASSET_UNDER_CONSTRUCTION);
            asset.setPrimaryDepreciationMethodCode(CamsConstants.Asset.DEPRECIATION_METHOD_STRAIGHT_LINE_CODE);
            asset.setCapitalAssetTypeCode(SpringContext.getBean(ParameterService.class).getParameterValueAsString(
                    AssetFabrication.class, CamsParameterConstants.ASSET_TYPE));
            asset.setManufacturerName(SpringContext.getBean(ParameterService.class).getParameterValueAsString(
                    AssetFabrication.class, CamsParameterConstants.MANUFACTURER));
            getAssetService().setFiscalPeriod(asset);
        }
        // setup offCampusLocation
        getAssetLocationService().setOffCampusLocation(asset);
    }

    @Override
    protected boolean answerSplitNodeQuestion(final String nodeName) throws UnsupportedOperationException {
        if (CamsConstants.RouteLevelNames.ORGANIZATION_INACTIVE.equals(nodeName)) {
            return isRequiresOrganizationInactiveRouteNode();
        }

        throw new UnsupportedOperationException("Cannot answer split question for this node you call \"" + nodeName +
                "\"");
    }

    /**
     * @return if the organization inactive route node needs to be stopped at
     */
    protected boolean isRequiresOrganizationInactiveRouteNode() {
        final Asset asset = (Asset) getBusinessObject();
        asset.getOrganizationOwnerAccount().refreshReferenceObject(KFSPropertyConstants.ORGANIZATION);

        return !asset.getOrganizationOwnerAccount().getOrganization().isActive();
    }

    @Override
    public void setGenerateDefaultValues(final String docTypeName) {

    }

    public List<String> getFpLinks() {
        final Asset asset = (Asset) getBusinessObject();
        final List<Long> assetNumbers = new ArrayList<>();
        assetNumbers.add(asset.getCapitalAssetNumber());
        return SpringContext.getBean(AssetLockService.class).getAssetLockingDocuments(assetNumbers,
                CamsConstants.DocumentTypeName.ASSET_FP_INQUIRY, "");
    }

    public List<String> getPreqLinks() {
        final Asset asset = (Asset) getBusinessObject();
        final List<Long> assetNumbers = new ArrayList<>();
        assetNumbers.add(asset.getCapitalAssetNumber());
        return SpringContext.getBean(AssetLockService.class).getAssetLockingDocuments(assetNumbers,
                CamsConstants.DocumentTypeName.ASSET_PREQ_INQUIRY, "");
    }

    public List<String> getFpLinkedDocumentInfo() {
        final List<String> documentInfo = new ArrayList<>();
        for (final String aDocumentNumber : getFpLinks()) {
            final String docTypeName = getDocumentService().getByDocumentHeaderId(aDocumentNumber).getDocumentHeader()
                    .getWorkflowDocument().getDocumentTypeName();
            documentInfo.add(FINANCIAL_DOC_NAME_MAP.get(docTypeName) + "-" + aDocumentNumber);
        }
        return documentInfo;
    }

    public boolean isFabricationOn() {
        return fabricationOn;
    }

    public void setFabricationOn(final boolean fabricationOn) {
        this.fabricationOn = fabricationOn;
    }

    protected static AssetService getAssetService() {
        if (assetService == null) {
            assetService = SpringContext.getBean(AssetService.class);
        }
        return assetService;
    }

    private AssetLocationService getAssetLocationService() {
        return SpringContext.getBean(AssetLocationService.class);
    }

    protected static DocumentService getDocumentService() {
        if (documentService == null) {
            documentService = SpringContext.getBean(DocumentService.class);
        }
        return documentService;
    }

    public IdentityService getIdentityService() {
        if (identityService == null) {
            identityService = SpringContext.getBean(IdentityService.class);
        }
        return identityService;
    }

    public void setIdentityService(final IdentityService identityService) {
        this.identityService = identityService;
    }
}
