/*
 * The Kuali Financial System, a comprehensive financial management system for higher education.
 *
 * Copyright 2005-2022 Kuali, Inc.
 *
 * This program is free software: you can redistribute it and/or modify
 * it under the terms of the GNU Affero General Public License as published by
 * the Free Software Foundation, either version 3 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU Affero General Public License for more details.
 *
 * You should have received a copy of the GNU Affero General Public License
 * along with this program.  If not, see <http://www.gnu.org/licenses/>.
 */
package org.kuali.kfs.module.cam.document.authorization;

import org.kuali.kfs.coa.service.AccountService;
import org.kuali.kfs.kns.datadictionary.MaintainableCollectionDefinition;
import org.kuali.kfs.kns.document.MaintenanceDocument;
import org.kuali.kfs.datadictionary.legacy.MaintenanceDocumentDictionaryService;
import org.kuali.kfs.krad.util.KRADConstants;
import org.kuali.kfs.module.cam.CamsConstants;
import org.kuali.kfs.module.cam.CamsPropertyConstants;
import org.kuali.kfs.module.cam.businessobject.AssetGlobal;
import org.kuali.kfs.module.cam.businessobject.AssetGlobalDetail;
import org.kuali.kfs.module.cam.businessobject.AssetPaymentDetail;
import org.kuali.kfs.module.cam.document.service.AssetGlobalService;
import org.kuali.kfs.sys.KFSConstants;
import org.kuali.kfs.sys.KFSPropertyConstants;
import org.kuali.kfs.sys.context.SpringContext;
import org.kuali.kfs.sys.document.authorization.FinancialSystemMaintenanceDocumentPresentationControllerBase;
import org.kuali.kfs.sys.service.impl.KfsParameterConstants;
import org.kuali.kfs.core.api.parameter.ParameterEvaluator;
import org.kuali.kfs.krad.bo.BusinessObject;

import java.util.HashSet;
import java.util.Set;

/**
 * Presentation Controller for Asset Global Maintenance Documents
 */
public class AssetGlobalPresentationController extends FinancialSystemMaintenanceDocumentPresentationControllerBase {

    @Override
    public Set<String> getConditionallyHiddenPropertyNames(final BusinessObject businessObject) {
        final Set<String> fields = super.getConditionallyHiddenPropertyNames(businessObject);
        final MaintenanceDocument document = (MaintenanceDocument) businessObject;
        final AssetGlobal assetGlobal = (AssetGlobal) document.getNewMaintainableObject().getBusinessObject();
        final MaintainableCollectionDefinition maintCollDef = SpringContext.getBean(MaintenanceDocumentDictionaryService.class)
                .getMaintainableCollection("AA", "assetPaymentDetails");
        final boolean isAssetSeparate = SpringContext.getBean(AssetGlobalService.class).isAssetSeparate(assetGlobal);

        if (assetGlobal.isCapitalAssetBuilderOriginIndicator() || isAssetSeparate) {
            // do not include payment add section within the payment details collection
            maintCollDef.setIncludeAddLine(false);
            // No update could be made to payment if it's created from CAB. Here, disable delete button if payment already added into
            // collection.
            for (final AssetPaymentDetail payment : assetGlobal.getAssetPaymentDetails()) {
                payment.setNewCollectionRecord(false);
            }
        } else {
            // conversely allow add during any other case. This is important because the attribute is set on the DD and the DD is
            // only loaded on project startup. Hence setting is important to avoid state related bugs
            maintCollDef.setIncludeAddLine(true);
        }

        final AssetGlobalService assetGlobalService = SpringContext.getBean(AssetGlobalService.class);
        if (!assetGlobalService.isAssetSeparateByPayment(assetGlobal)) {
            // Show payment sequence number field only if a separate by payment was selected
            fields.add(CamsPropertyConstants.AssetGlobal.SEPARATE_SOURCE_PAYMENT_SEQUENCE_NUMBER);
        }

        if (!isAssetSeparate) {
            fields.addAll(getAssetGlobalLocationHiddenFields(assetGlobal));

            if (assetGlobalService.existsInGroup(assetGlobalService.getNonNewAcquisitionCodeGroup(), assetGlobal.getAcquisitionTypeCode())) {
                // Fields in the add section
                fields.add(KFSConstants.ADD_PREFIX + "." + CamsPropertyConstants.AssetGlobal.ASSET_PAYMENT_DETAILS + "." + CamsPropertyConstants.AssetPaymentDetail.DOCUMENT_POSTING_DATE);
                fields.add(KFSConstants.ADD_PREFIX + "." + CamsPropertyConstants.AssetGlobal.ASSET_PAYMENT_DETAILS + "." + CamsPropertyConstants.AssetPaymentDetail.DOCUMENT_NUMBER);
                fields.add(KFSConstants.ADD_PREFIX + "." + CamsPropertyConstants.AssetGlobal.ASSET_PAYMENT_DETAILS + "." + CamsPropertyConstants.AssetPaymentDetail.DOCUMENT_TYPE_CODE);
                fields.add(KFSConstants.ADD_PREFIX + "." + CamsPropertyConstants.AssetGlobal.ASSET_PAYMENT_DETAILS + "." + CamsPropertyConstants.AssetPaymentDetail.ORIGINATION_CODE);

                // Hiding some fields when the status of the document is not final.
                if (!document.getDocumentHeader().getWorkflowDocument().isFinal()) {
                    fields.addAll(getAssetGlobalPaymentsHiddenFields(assetGlobal));
                }
            }
        }
        // if no separate asset is added, hide the button
        if (isAssetSeparate && (assetGlobal.getAssetSharedDetails().isEmpty() || assetGlobal.getAssetSharedDetails().get(0).getAssetGlobalUniqueDetails().isEmpty())) {
            fields.add("calculateEqualSourceAmountsButton");
        }

        return fields;
    }

    @Override
    public Set<String> getConditionallyReadOnlyPropertyNames(final MaintenanceDocument document) {
        final Set<String> fields = super.getConditionallyReadOnlyPropertyNames(document);

        final AssetGlobal assetGlobal = (AssetGlobal) document.getNewMaintainableObject().getBusinessObject();

        // "Asset Separate" document functionality
        if (SpringContext.getBean(AssetGlobalService.class).isAssetSeparate(assetGlobal)) {
            fields.addAll(getAssetGlobalDetailsReadOnlyFields());
            fields.addAll(getAssetGlobalPaymentsReadOnlyFields(assetGlobal));
        } else if (assetGlobal.isCapitalAssetBuilderOriginIndicator()) {
            // If asset global document is created from CAB, disallow add payment to collection.
            fields.addAll(getAssetGlobalPaymentsReadOnlyFields(assetGlobal));
        }

        // if accounts can't cross charts, then add the extra chartOfAccountsCode field to be displayed readOnly
        if (!SpringContext.getBean(AccountService.class).accountsCanCrossCharts()) {
            final String COA_CODE_NAME = KRADConstants.ADD_PREFIX + "." + CamsPropertyConstants.AssetGlobal.ASSET_PAYMENT_DETAILS + "." + KFSPropertyConstants.CHART_OF_ACCOUNTS_CODE;
            fields.add(COA_CODE_NAME);
        }

        return fields;
    }

    @Override
    public Set<String> getConditionallyHiddenSectionIds(final BusinessObject businessObject) {
        final Set<String> fields = super.getConditionallyHiddenSectionIds(businessObject);

        final MaintenanceDocument document = (MaintenanceDocument) businessObject;
        final AssetGlobal assetGlobal = (AssetGlobal) document.getNewMaintainableObject().getBusinessObject();

        if (!isAccountPeriodEnabledForThisDoc(document)) {
            fields.add(KFSConstants.ACCOUNTING_PERIOD_TAB_ID);
        }

        // hide "Asset Information", "Recalculate Total Amount" tabs if not "Asset Separate" doc
        if (!SpringContext.getBean(AssetGlobalService.class).isAssetSeparate(assetGlobal)) {
            fields.add(CamsConstants.AssetGlobal.SECTION_ID_ASSET_INFORMATION);
            fields.add(CamsConstants.AssetGlobal.SECTION_ID_RECALCULATE_SEPARATE_SOURCE_AMOUNT);
        }

        return fields;
    }

    private boolean isAccountPeriodEnabledForThisDoc(final MaintenanceDocument document) {
        final String docType = document.getDocumentHeader().getWorkflowDocument().getDocumentTypeName();
        // check accounting period is enabled for doc type in system parameter
        final ParameterEvaluator evaluator = getParameterEvaluatorService().getParameterEvaluator(
                KFSConstants.CoreModuleNamespaces.KFS,
                KfsParameterConstants.YEAR_END_ACCOUNTING_PERIOD_PARAMETER_NAMES.DETAIL_PARAMETER_TYPE,
                KfsParameterConstants.YEAR_END_ACCOUNTING_PERIOD_PARAMETER_NAMES.FISCAL_PERIOD_SELECTION_DOCUMENT_TYPES,
                docType);
        return evaluator.evaluationSucceeds();
    }

    /**
     * @param assetGlobal
     * @return Asset Location fields with index that are present on AssetGlobal. Includes add line. Useful for hiding them.
     */
    protected Set<String> getAssetGlobalLocationHiddenFields(final AssetGlobal assetGlobal) {
        final Set<String> fields = new HashSet<>();

        // hide it for the add line
        int i = 0;
        for (final AssetGlobalDetail assetSharedDetail : assetGlobal.getAssetSharedDetails()) {
            fields.add(KFSConstants.ADD_PREFIX + "." + CamsPropertyConstants.AssetGlobal.ASSET_SHARED_DETAILS + "["
                    + i + "]." + CamsPropertyConstants.AssetGlobalDetail.ASSET_UNIQUE_DETAILS + "."
                    + CamsPropertyConstants.AssetGlobalDetail.REPRESENTATIVE_UNIVERSAL_IDENTIFIER);
            fields.add(KFSConstants.ADD_PREFIX + "." + CamsPropertyConstants.AssetGlobal.ASSET_SHARED_DETAILS + "["
                    + i + "]." + CamsPropertyConstants.AssetGlobalDetail.ASSET_UNIQUE_DETAILS + "."
                    + CamsPropertyConstants.AssetGlobalDetail.CAPITAL_ASSET_TYPE_CODE);
            fields.add(KFSConstants.ADD_PREFIX + "." + CamsPropertyConstants.AssetGlobal.ASSET_SHARED_DETAILS + "["
                    + i + "]." + CamsPropertyConstants.AssetGlobalDetail.ASSET_UNIQUE_DETAILS + "."
                    + CamsPropertyConstants.AssetGlobalDetail.CAPITAL_ASSET_DESCRIPTION);
            fields.add(KFSConstants.ADD_PREFIX + "." + CamsPropertyConstants.AssetGlobal.ASSET_SHARED_DETAILS + "["
                    + i + "]." + CamsPropertyConstants.AssetGlobalDetail.ASSET_UNIQUE_DETAILS + "."
                    + CamsPropertyConstants.AssetGlobalDetail.MANUFACTURER_NAME);
            fields.add(KFSConstants.ADD_PREFIX + "." + CamsPropertyConstants.AssetGlobal.ASSET_SHARED_DETAILS + "[" + i
                    + "]." + CamsPropertyConstants.AssetGlobalDetail.ASSET_UNIQUE_DETAILS + "."
                    + CamsPropertyConstants.AssetGlobalDetail.ORGANIZATION_TEXT);
            fields.add(KFSConstants.ADD_PREFIX + "." + CamsPropertyConstants.AssetGlobal.ASSET_SHARED_DETAILS + "[" + i
                    + "]." + CamsPropertyConstants.AssetGlobalDetail.ASSET_UNIQUE_DETAILS + "."
                    + CamsPropertyConstants.AssetGlobalDetail.MANUFACTURER_MODEL_NUMBER);
            fields.add(KFSConstants.ADD_PREFIX + "." + CamsPropertyConstants.AssetGlobal.ASSET_SHARED_DETAILS + "[" + i
                    + "]." + CamsPropertyConstants.AssetGlobalDetail.ASSET_UNIQUE_DETAILS + "."
                    + CamsPropertyConstants.AssetGlobalDetail.SEPARATE_SOURCE_AMOUNT);

            // hide it for the existing lines
            int j = 0;
            for (final AssetGlobalDetail assetGlobalUniqueDetail : assetSharedDetail.getAssetGlobalUniqueDetails()) {
                fields.add(CamsPropertyConstants.AssetGlobal.ASSET_SHARED_DETAILS + "[" + i + "]."
                        + CamsPropertyConstants.AssetGlobalDetail.ASSET_UNIQUE_DETAILS + "[" + j + "]."
                        + CamsPropertyConstants.AssetGlobalDetail.REPRESENTATIVE_UNIVERSAL_IDENTIFIER);
                fields.add(CamsPropertyConstants.AssetGlobal.ASSET_SHARED_DETAILS + "[" + i + "]."
                        + CamsPropertyConstants.AssetGlobalDetail.ASSET_UNIQUE_DETAILS + "[" + j + "]."
                        + CamsPropertyConstants.AssetGlobalDetail.CAPITAL_ASSET_TYPE_CODE);
                fields.add(CamsPropertyConstants.AssetGlobal.ASSET_SHARED_DETAILS + "[" + i + "]."
                        + CamsPropertyConstants.AssetGlobalDetail.ASSET_UNIQUE_DETAILS + "[" + j + "]."
                        + CamsPropertyConstants.AssetGlobalDetail.CAPITAL_ASSET_DESCRIPTION);
                fields.add(CamsPropertyConstants.AssetGlobal.ASSET_SHARED_DETAILS + "[" + i + "]."
                        + CamsPropertyConstants.AssetGlobalDetail.ASSET_UNIQUE_DETAILS + "[" + j + "]."
                        + CamsPropertyConstants.AssetGlobalDetail.MANUFACTURER_NAME);
                fields.add(CamsPropertyConstants.AssetGlobal.ASSET_SHARED_DETAILS + "[" + i + "]."
                        + CamsPropertyConstants.AssetGlobalDetail.ASSET_UNIQUE_DETAILS + "[" + j + "]."
                        + CamsPropertyConstants.AssetGlobalDetail.ORGANIZATION_TEXT);
                fields.add(CamsPropertyConstants.AssetGlobal.ASSET_SHARED_DETAILS + "[" + i + "]."
                        + CamsPropertyConstants.AssetGlobalDetail.ASSET_UNIQUE_DETAILS + "[" + j + "]."
                        + CamsPropertyConstants.AssetGlobalDetail.MANUFACTURER_MODEL_NUMBER);
                fields.add(CamsPropertyConstants.AssetGlobal.ASSET_SHARED_DETAILS + "[" + i + "]."
                        + CamsPropertyConstants.AssetGlobalDetail.ASSET_UNIQUE_DETAILS + "[" + j + "]."
                        + CamsPropertyConstants.AssetGlobalDetail.SEPARATE_SOURCE_AMOUNT);
                j++;
            }

            i++;
        }

        return fields;
    }

    /**
     * @param assetGlobal
     * @return posting year and fiscal month on every payment. Useful for hiding them.
     */
    protected Set<String> getAssetGlobalPaymentsHiddenFields(final AssetGlobal assetGlobal) {
        final Set<String> fields = new HashSet<>();

        int i = 0;
        for (final AssetPaymentDetail assetPaymentDetail : assetGlobal.getAssetPaymentDetails()) {
            fields.add(CamsPropertyConstants.AssetGlobal.ASSET_PAYMENT_DETAILS + "[" + i + "]."
                    + CamsPropertyConstants.AssetPaymentDetail.DOCUMENT_POSTING_DATE);
            fields.add(CamsPropertyConstants.AssetGlobal.ASSET_PAYMENT_DETAILS + "[" + i + "]."
                    + CamsPropertyConstants.AssetPaymentDetail.DOCUMENT_NUMBER);
            fields.add(CamsPropertyConstants.AssetGlobal.ASSET_PAYMENT_DETAILS + "[" + i + "]."
                    + CamsPropertyConstants.AssetPaymentDetail.DOCUMENT_TYPE_CODE);
            fields.add(CamsPropertyConstants.AssetGlobal.ASSET_PAYMENT_DETAILS + "[" + i + "]."
                    + CamsPropertyConstants.AssetPaymentDetail.ORIGINATION_CODE);
            i++;
        }

        return fields;
    }

    /**
     * @return Asset Global Detail fields that should be read only
     */
    protected Set<String> getAssetGlobalDetailsReadOnlyFields() {
        final Set<String> fields = new HashSet<>();

        fields.add(CamsPropertyConstants.Asset.ORGANIZATION_OWNER_CHART_OF_ACCOUNTS_CODE);
        fields.add(CamsPropertyConstants.Asset.ORGANIZATION_OWNER_ACCOUNT_NUMBER);
        fields.add(CamsPropertyConstants.Asset.ORGANIZATION_CODE);
        fields.add(CamsPropertyConstants.Asset.AGENCY_NUMBER);
        fields.add(CamsPropertyConstants.Asset.ACQUISITION_TYPE_CODE);
        fields.add(CamsPropertyConstants.Asset.INVENTORY_STATUS_CODE);
        fields.add(CamsPropertyConstants.Asset.CONDITION_CODE);
        fields.add(CamsPropertyConstants.AssetGlobal.CAPITAL_ASSET_DESCRIPTION);
        fields.add(CamsPropertyConstants.Asset.CAPITAL_ASSET_TYPE_CODE);
        fields.add(CamsPropertyConstants.Asset.VENDOR_NAME);
        fields.add(CamsPropertyConstants.Asset.MANUFACTURER_NAME);
        fields.add(CamsPropertyConstants.Asset.MANUFACTURER_MODEL_NUMBER);
        fields.add(CamsPropertyConstants.AssetGlobal.ORGANIZATION_TEXT);
        fields.add(CamsPropertyConstants.Asset.REP_USER_AUTH_ID);
        fields.add(CamsPropertyConstants.Asset.LAST_INVENTORY_DATE);
        fields.add(CamsPropertyConstants.Asset.CREATE_DATE);
        fields.add(CamsPropertyConstants.Asset.ASSET_DATE_OF_SERVICE);
        fields.add(CamsPropertyConstants.AssetGlobal.CAPITAL_ASSET_DEPRECIATION_DATE);
        fields.add(CamsPropertyConstants.Asset.LAND_COUNTRY_NAME);
        fields.add(CamsPropertyConstants.Asset.LAND_ACREAGE_SIZE);
        fields.add(CamsPropertyConstants.Asset.LAND_PARCEL_NUMBER);

        return fields;
    }

    /**
     * @param assetGlobal
     * @return Asset Global Payment lines with index that should be set to read only.
     */
    protected Set<String> getAssetGlobalPaymentsReadOnlyFields(final AssetGlobal assetGlobal) {
        final Set<String> fields = new HashSet<>();

        // set all payment detail fields to read only.
        int i = 0;
        for (final AssetPaymentDetail assetPaymentDetail : assetGlobal.getAssetPaymentDetails()) {
            fields.add(CamsPropertyConstants.AssetGlobal.ASSET_PAYMENT_DETAILS + "[" + i + "]." + CamsPropertyConstants.AssetPaymentDetail.SEQUENCE_NUMBER);
            fields.add(CamsPropertyConstants.AssetGlobal.ASSET_PAYMENT_DETAILS + "[" + i + "]." + CamsPropertyConstants.AssetPaymentDetail.CHART_OF_ACCOUNTS_CODE);
            fields.add(CamsPropertyConstants.AssetGlobal.ASSET_PAYMENT_DETAILS + "[" + i + "]." + CamsPropertyConstants.AssetPaymentDetail.ACCOUNT_NUMBER);
            fields.add(CamsPropertyConstants.AssetGlobal.ASSET_PAYMENT_DETAILS + "[" + i + "]." + CamsPropertyConstants.AssetPaymentDetail.SUB_ACCOUNT_NUMBER);
            fields.add(CamsPropertyConstants.AssetGlobal.ASSET_PAYMENT_DETAILS + "[" + i + "]." + CamsPropertyConstants.AssetPaymentDetail.FINANCIAL_OBJECT_CODE);
            fields.add(CamsPropertyConstants.AssetGlobal.ASSET_PAYMENT_DETAILS + "[" + i + "]." + CamsPropertyConstants.AssetPaymentDetail.SUB_OBJECT_CODE);
            fields.add(CamsPropertyConstants.AssetGlobal.ASSET_PAYMENT_DETAILS + "[" + i + "]." + CamsPropertyConstants.AssetPaymentDetail.PROJECT_CODE);
            fields.add(CamsPropertyConstants.AssetGlobal.ASSET_PAYMENT_DETAILS + "[" + i + "]." + CamsPropertyConstants.AssetPaymentDetail.ORGANIZATION_REFERENCE_ID);
            fields.add(CamsPropertyConstants.AssetGlobal.ASSET_PAYMENT_DETAILS + "[" + i + "]." + CamsPropertyConstants.AssetPaymentDetail.DOCUMENT_NUMBER);
            fields.add(CamsPropertyConstants.AssetGlobal.ASSET_PAYMENT_DETAILS + "[" + i + "]." + CamsPropertyConstants.AssetPaymentDetail.DOCUMENT_TYPE_CODE);
            fields.add(CamsPropertyConstants.AssetGlobal.ASSET_PAYMENT_DETAILS + "[" + i + "]." + CamsPropertyConstants.AssetPaymentDetail.PURCHASE_ORDER);
            fields.add(CamsPropertyConstants.AssetGlobal.ASSET_PAYMENT_DETAILS + "[" + i + "]." + CamsPropertyConstants.AssetPaymentDetail.REQUISITION_NUMBER);
            fields.add(CamsPropertyConstants.AssetGlobal.ASSET_PAYMENT_DETAILS + "[" + i + "]." + CamsPropertyConstants.AssetPaymentDetail.DOCUMENT_POSTING_DATE);
            fields.add(CamsPropertyConstants.AssetGlobal.ASSET_PAYMENT_DETAILS + "[" + i + "]." + CamsPropertyConstants.AssetPaymentDetail.DOCUMENT_POSTING_FISCAL_YEAR);
            fields.add(CamsPropertyConstants.AssetGlobal.ASSET_PAYMENT_DETAILS + "[" + i + "]." + CamsPropertyConstants.AssetPaymentDetail.DOCUMENT_POSTING_FISCAL_MONTH);
            fields.add(CamsPropertyConstants.AssetGlobal.ASSET_PAYMENT_DETAILS + "[" + i + "]." + CamsPropertyConstants.AssetPaymentDetail.AMOUNT);
            fields.add(CamsPropertyConstants.AssetGlobal.ASSET_PAYMENT_DETAILS + "[" + i + "]." + CamsPropertyConstants.AssetPaymentDetail.ORIGINATION_CODE);

            i++;
        }

        return fields;
    }
}
