/*
 * The Kuali Financial System, a comprehensive financial management system for higher education.
 *
 * Copyright 2005-2022 Kuali, Inc.
 *
 * This program is free software: you can redistribute it and/or modify
 * it under the terms of the GNU Affero General Public License as published by
 * the Free Software Foundation, either version 3 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU Affero General Public License for more details.
 *
 * You should have received a copy of the GNU Affero General Public License
 * along with this program.  If not, see <http://www.gnu.org/licenses/>.
 */
package org.kuali.kfs.module.cam.document.service.impl;

import org.apache.commons.beanutils.PropertyUtils;
import org.apache.commons.lang3.StringUtils;
import org.apache.logging.log4j.LogManager;
import org.apache.logging.log4j.Logger;
import org.kuali.kfs.coa.businessobject.ObjectCode;
import org.kuali.kfs.coa.service.ObjectCodeService;
import org.kuali.kfs.coreservice.framework.parameter.ParameterService;
import org.kuali.kfs.krad.bo.PersistableBusinessObject;
import org.kuali.kfs.krad.service.BusinessObjectService;
import org.kuali.kfs.krad.util.GlobalVariables;
import org.kuali.kfs.krad.util.ObjectUtils;
import org.kuali.kfs.module.cam.CamsConstants;
import org.kuali.kfs.module.cam.CamsKeyConstants;
import org.kuali.kfs.module.cam.CamsParameterConstants;
import org.kuali.kfs.module.cam.CamsPropertyConstants;
import org.kuali.kfs.module.cam.batch.AssetDepreciationStep;
import org.kuali.kfs.module.cam.businessobject.Asset;
import org.kuali.kfs.module.cam.businessobject.AssetGlobal;
import org.kuali.kfs.module.cam.businessobject.AssetPayment;
import org.kuali.kfs.module.cam.businessobject.AssetPaymentAllocationType;
import org.kuali.kfs.module.cam.businessobject.AssetPaymentAssetDetail;
import org.kuali.kfs.module.cam.businessobject.AssetPaymentDetail;
import org.kuali.kfs.module.cam.businessobject.AssetPaymentInProcessPayment;
import org.kuali.kfs.module.cam.document.AssetPaymentDocument;
import org.kuali.kfs.module.cam.document.dataaccess.AssetPaymentDao;
import org.kuali.kfs.module.cam.document.service.AssetPaymentService;
import org.kuali.kfs.module.cam.document.service.AssetRetirementService;
import org.kuali.kfs.module.cam.document.service.AssetService;
import org.kuali.kfs.module.cam.util.distribution.AssetDistribution;
import org.kuali.kfs.sys.KFSConstants;
import org.kuali.kfs.sys.KFSPropertyConstants;
import org.kuali.kfs.sys.businessobject.UniversityDate;
import org.kuali.kfs.sys.service.UniversityDateService;
import org.kuali.kfs.core.api.util.type.KualiDecimal;
import org.springframework.transaction.annotation.Transactional;

import java.beans.PropertyDescriptor;
import java.lang.reflect.InvocationTargetException;
import java.lang.reflect.Method;
import java.util.ArrayList;
import java.util.Arrays;
import java.util.HashMap;
import java.util.List;
import java.util.Locale;
import java.util.Map;
import java.util.regex.Pattern;

@Transactional
public class AssetPaymentServiceImpl implements AssetPaymentService {

    private static final Logger LOG = LogManager.getLogger();

    private BusinessObjectService businessObjectService;
    private AssetPaymentDao assetPaymentDao;
    private ParameterService parameterService;
    private UniversityDateService universityDateService;
    private ObjectCodeService objectCodeService;
    private AssetRetirementService assetRetirementService;
    private AssetService assetService;

    @Override
    public Integer getMaxSequenceNumber(final Long capitalAssetNumber) {
        return getAssetPaymentDao().getMaxSquenceNumber(capitalAssetNumber);
    }

    @Override
    public boolean isPaymentFederalOwned(final AssetPayment assetPayment) {
        assetPayment.refreshReferenceObject(CamsPropertyConstants.AssetPayment.OBJECT_CODE_CURRENT);
        if (ObjectUtils.isNotNull(assetPayment.getObjectCodeCurrent())) {
            final String subTypeCode = assetPayment.getObjectCodeCurrent().getFinancialObjectSubTypeCode();
            return getParameterService().getParameterValuesAsString(CamsParameterConstants.NAMESPACE,
                    CamsParameterConstants.DETAIL_TYPE_DOCUMENT,
                    CamsParameterConstants.FEDERAL_OWNED_SUB_TYPES).contains(subTypeCode);
        }
        return false;
    }

    @Override
    public boolean isPaymentEligibleForAccumDeprGLPosting(final AssetPayment assetPayment) {
        final KualiDecimal accumulatedDepreciationAmount = assetPayment.getAccumulatedPrimaryDepreciationAmount();
        return accumulatedDepreciationAmount != null && !accumulatedDepreciationAmount.isZero();
    }

    @Override
    public boolean isPaymentEligibleForCapitalizationGLPosting(final AssetPayment assetPayment) {
        final KualiDecimal accountChargeAmount = assetPayment.getAccountChargeAmount();
        return accountChargeAmount != null && !accountChargeAmount.isZero();
    }

    @Override
    public boolean isPaymentEligibleForOffsetGLPosting(final AssetPayment assetPayment) {
        final KualiDecimal accountChargeAmount = assetPayment.getAccountChargeAmount();
        if (assetPayment.getAccumulatedPrimaryDepreciationAmount() == null) {
            assetPayment.setAccumulatedPrimaryDepreciationAmount(KualiDecimal.ZERO);
        }
        final KualiDecimal accumulatedDepreciationAmount = assetPayment.getAccumulatedPrimaryDepreciationAmount();
        return accountChargeAmount != null && !accountChargeAmount.subtract(accumulatedDepreciationAmount).isZero();
    }

    @Override
    public boolean isPaymentFinancialObjectActive(final AssetPayment assetPayment) {
        ObjectCode financialObjectCode = new ObjectCode();
        financialObjectCode.setUniversityFiscalYear(getUniversityDateService().getCurrentFiscalYear());
        financialObjectCode.setChartOfAccountsCode(assetPayment.getChartOfAccountsCode());
        financialObjectCode.setFinancialObjectCode(assetPayment.getFinancialObjectCode());
        financialObjectCode = (ObjectCode) getBusinessObjectService().retrieve(financialObjectCode);
        if (ObjectUtils.isNotNull(financialObjectCode)) {
            return financialObjectCode.isActive();
        }
        return false;
    }

    @Override
    public void processApprovedAssetPayment(final AssetPaymentDocument document) {
        // Creating new asset payment records
        processPayments(document);
    }

    /**
     * Creates a new asset payment record for each new asset payment detail record and then save them
     *
     * @param document
     */
    protected void processPayments(final AssetPaymentDocument document) {
        final List<AssetPaymentDetail> assetPaymentDetailLines = document.getSourceAccountingLines();
        final List<AssetPaymentAssetDetail> assetPaymentAssetDetails = document.getAssetPaymentAssetDetail();
        final List<PersistableBusinessObject> assetPayments = new ArrayList<>();
        Integer maxSequenceNo;

        //instantiate asset payment distributor
        final AssetDistribution paymentDistributor = document.getAssetPaymentDistributor();

        // Calculating the asset payments distributions for each individual asset on the list
        final Map<String, Map<AssetPaymentAssetDetail, KualiDecimal>> assetPaymentDistributionMap =
                paymentDistributor.getAssetPaymentDistributions();

        try {
            // Creating a new payment record for each asset that has payments.
            for (final AssetPaymentAssetDetail assetPaymentAssetDetail : assetPaymentAssetDetails) {
                maxSequenceNo = getMaxSequenceNumber(assetPaymentAssetDetail.getCapitalAssetNumber());

                KualiDecimal totalAmount = KualiDecimal.ZERO;
                for (final AssetPaymentDetail assetPaymentDetail : assetPaymentDetailLines) {

                    // Retrieve the asset payment from the distribution map. It's possible not to find an entry because,
                    // for example, AssetDistributionManual.findExistingMap will merge some line items. Skip in that
                    // scenario
                    final Map<AssetPaymentAssetDetail, KualiDecimal> assetPaymentDistributionMapElement =
                            assetPaymentDistributionMap.get(assetPaymentDetail.getAssetPaymentDetailKey());
                    if (ObjectUtils.isNotNull(assetPaymentDistributionMapElement)) {
                        final KualiDecimal amount = assetPaymentDistributionMapElement.get(assetPaymentAssetDetail);

                        final AssetPayment assetPayment = createAssetPayment(document.getDocumentNumber(), ++maxSequenceNo,
                                assetPaymentAssetDetail.getCapitalAssetNumber(),
                                document.getAssetPaymentInProcessPayments(), assetPaymentDetail, amount);
                        totalAmount = totalAmount.add(amount);

                        // add new payment
                        assetPayments.add(assetPayment);
                    }
                }
                // *********************BEGIN - Updating Asset ***********************************************************
                // Retrieving the asset that will have its cost updated
                final HashMap<String, Long> keys = new HashMap<>();
                keys.put(CamsPropertyConstants.Asset.CAPITAL_ASSET_NUMBER,
                        assetPaymentAssetDetail.getCapitalAssetNumber());
                final Asset asset = getBusinessObjectService().findByPrimaryKey(Asset.class, keys);

                // Set previous total cost
                if (assetPaymentAssetDetail.getPreviousTotalCostAmount() == null) {
                    assetPaymentAssetDetail.setPreviousTotalCostAmount(new KualiDecimal(0));
                }

                // Setting the asset's new cost.
                asset.setTotalCostAmount(assetPaymentAssetDetail.getPreviousTotalCostAmount().add(totalAmount));

                // Setting the asset's financial object sub-type Code. Only when the asset doesn't have one.
                if (asset.getFinancialObjectSubTypeCode() == null
                        || asset.getFinancialObjectSubTypeCode().trim().equals("")) {
                    asset.setFinancialObjectSubTypeCode(assetPaymentDetailLines.get(0).getObjectCode()
                            .getFinancialObjectSubTypeCode());
                }
                // Saving changes
                getBusinessObjectService().save(asset);
                // *********************END - Updating Asset ***********************************************************
            }
        } catch (final Exception e) {
            throw new RuntimeException(e);
        }
        // Finally, saving all the asset payment records.
        getBusinessObjectService().save(assetPayments);
    }

    protected AssetPayment createAssetPayment(
            final String documentNumber, final Integer maxSequenceNo, final Long capitalAssetNumber,
            final List<AssetPaymentInProcessPayment> assetPaymentInProcessPayments, final AssetPaymentDetail assetPaymentDetail,
            final KualiDecimal amount) throws IllegalAccessException, InvocationTargetException {
        final AssetPayment assetPayment = new AssetPayment(assetPaymentDetail);
        assetPayment.setCapitalAssetNumber(capitalAssetNumber);
        assetPayment.setPaymentSequenceNumber(maxSequenceNo);

        // Resetting each period field its value with nulls
        adjustPaymentAmounts(assetPayment, false, true);

        if (assetPaymentDetail.isTransferPaymentIndicator()) {
            final boolean depreciationGenerated = generateDepreciationAmountsOnAssetPayment(
                    assetPaymentDetail.getSequenceNumber(), assetPayment, assetPaymentInProcessPayments);
            if (depreciationGenerated) {
                assetPayment.setTransferPaymentCode(CamsConstants.AssetPayment.TRANSFER_PAYMENT_CODE_Y);
            } else {
                assetPayment.setTransferPaymentCode(CamsConstants.AssetPayment.TRANSFER_PAYMENT_CODE_N);
            }
        } else {
            assetPayment.setTransferPaymentCode(CamsConstants.AssetPayment.TRANSFER_PAYMENT_CODE_N);
        }

        if (StringUtils.isBlank(assetPayment.getDocumentNumber())) {
            assetPayment.setDocumentNumber(documentNumber);
        }
        assetPayment.setAccountChargeAmount(amount);

        KualiDecimal baseAmount = KualiDecimal.ZERO;

        // If the object sub type is not in the list of federally owned object sub types, then...
        final ObjectCode objectCode = getObjectCodeService().getByPrimaryId(assetPaymentDetail.getPostingYear(),
                assetPaymentDetail.getChartOfAccountsCode(), assetPaymentDetail.getFinancialObjectCode());

        // Depreciation Base Amount will be assigned to each payment only when the object code's sub type code is not a
        // federally owned one
        if (!isNonDepreciableFederallyOwnedObjSubType(objectCode.getFinancialObjectSubTypeCode())) {
            baseAmount = baseAmount.add(amount);
        }
        assetPayment.setPrimaryDepreciationBaseAmount(baseAmount);

        return assetPayment;
    }

    /**
     * Generates and set the depreciation amounts on payments that are not to be depreciated anymore after they had
     * zero'ed out their amounts on an Asset Manual Payment document.
     * @param paymentSequenceNumber the source payment sequence number
     * @param assetPayment on which to set accumulatedPrimaryDepreciationAmount and period1Depreciation1Amount
     * @param assetPaymentInProcessPayments which have the selected payments for the in process payments
     * @return if a matching selection has been found. Can be used for additional processing on the passed in
     *         assetPayment
     */
    protected boolean generateDepreciationAmountsOnAssetPayment(
            final Integer paymentSequenceNumber,
            final AssetPayment assetPayment, final List<AssetPaymentInProcessPayment> assetPaymentInProcessPayments) {
        boolean matchFound = false;
        KualiDecimal totalAccumulatedPrimaryDepreciationAmount = new KualiDecimal(0);
        for (final AssetPaymentInProcessPayment assetPaymentInProcessPayment : assetPaymentInProcessPayments) {
            if (assetPayment.getCapitalAssetNumber().equals(assetPaymentInProcessPayment.getCapitalAssetNumber())
                    && paymentSequenceNumber.equals(assetPaymentInProcessPayment.getSequenceNumber())) {
                final AssetPayment apipAssetPayment = assetPaymentInProcessPayment.getAssetPayment();
                matchFound = true;

                apipAssetPayment.setTransferPaymentCode(CamsConstants.AssetPayment.TRANSFER_PAYMENT_CODE_Y);
                if (ObjectUtils.isNotNull(apipAssetPayment.getAccumulatedPrimaryDepreciationAmount())) {
                    totalAccumulatedPrimaryDepreciationAmount = totalAccumulatedPrimaryDepreciationAmount
                            .add(apipAssetPayment.getAccumulatedPrimaryDepreciationAmount());
                }
            }
        }

        // Set accumulated depreciation to the sum of the accumulated depreciation of the payments for the acct line
        // on this asset
        assetPayment.setAccumulatedPrimaryDepreciationAmount(totalAccumulatedPrimaryDepreciationAmount.negated());

        final String fiscalMonth = universityDateService.getCurrentUniversityDate().getUniversityFiscalAccountingPeriod();
        setCurrentDepreciation(fiscalMonth, assetPayment.getAccumulatedPrimaryDepreciationAmount(), assetPayment);

        return matchFound;
    }

    @Override
    public void adjustPaymentAmounts(final AssetPayment assetPayment, final boolean reverseAmount, final boolean nullPeriodDepreciation)
            throws IllegalAccessException, InvocationTargetException {
        LOG.debug("Starting - adjustAmounts() ");
        final PropertyDescriptor[] propertyDescriptors = PropertyUtils.getPropertyDescriptors(AssetPayment.class);
        for (final PropertyDescriptor propertyDescriptor : propertyDescriptors) {
            final Method readMethod = propertyDescriptor.getReadMethod();
            if (readMethod != null && propertyDescriptor.getPropertyType() != null
                    && KualiDecimal.class.isAssignableFrom(propertyDescriptor.getPropertyType())) {
                final KualiDecimal amount = (KualiDecimal) readMethod.invoke(assetPayment);
                final Method writeMethod = propertyDescriptor.getWriteMethod();
                if (writeMethod != null && amount != null) {
                    // Reset periodic depreciation expenses
                    if (nullPeriodDepreciation && Pattern.matches(CamsConstants.SET_PERIOD_DEPRECIATION_AMOUNT_REGEX,
                            writeMethod.getName().toLowerCase(Locale.US))) {
                        final Object[] nullVal = new Object[]{null};
                        writeMethod.invoke(assetPayment, nullVal);
                    } else if (reverseAmount) {
                        // reverse the amounts
                        writeMethod.invoke(assetPayment, amount.negated());
                    }
                }

            }
        }
        LOG.debug("Finished - adjustAmounts()");
    }

    @Override
    public boolean isPaymentEligibleForGLPosting(final AssetPayment assetPayment) {
        // Transfer payment code flag is not Y
        boolean isEligible = !CamsConstants.AssetPayment.TRANSFER_PAYMENT_CODE_Y.equals(
                assetPayment.getTransferPaymentCode());
        // Financial object code is currently active
        isEligible &= isPaymentFinancialObjectActive(assetPayment);
        // Payment is not federally funded
        isEligible &= !isPaymentFederalOwned(assetPayment);
        return isEligible;
    }

    /**
     * Checks if object sub type is a non-depreciable federally owned object sub type
     *
     * @param objectSubType
     * @return true if is FEDERAL_OWNED_SUB_TYPES
     */
    @Override
    public boolean isNonDepreciableFederallyOwnedObjSubType(final String objectSubType) {
        List<String> federallyOwnedObjectSubTypes = new ArrayList<>();
        if (getParameterService().parameterExists(AssetDepreciationStep.class,
                CamsParameterConstants.FEDERAL_OWNED_SUB_TYPES)) {
            federallyOwnedObjectSubTypes = new ArrayList<>(getParameterService().getParameterValuesAsString(
                    AssetDepreciationStep.class,
                    CamsParameterConstants.FEDERAL_OWNED_SUB_TYPES));
        }
        return federallyOwnedObjectSubTypes.contains(objectSubType);
    }

    @Override
    public boolean extractPostedDatePeriod(final AssetPaymentDetail assetPaymentDetail) {
        final Map<String, Object> primaryKeys = new HashMap<>();
        primaryKeys.put(KFSPropertyConstants.UNIVERSITY_DATE,
                assetPaymentDetail.getExpenditureFinancialDocumentPostedDate());
        final UniversityDate universityDate = businessObjectService.findByPrimaryKey(UniversityDate.class, primaryKeys);
        if (universityDate != null) {
            assetPaymentDetail.setPostingYear(universityDate.getUniversityFiscalYear());
            assetPaymentDetail.setPostingPeriodCode(universityDate.getUniversityFiscalAccountingPeriod());
            return true;
        } else {
            return false;
        }
    }

    @Override
    public Integer getAssetPaymentDetailQuantity(final AssetGlobal assetGlobal) {
        Integer assetPaymentDetailQuantity = 0;
        for (final AssetPaymentDetail assetPaymentDetail : assetGlobal.getAssetPaymentDetails()) {
            assetPaymentDetailQuantity++;
        }
        return assetPaymentDetailQuantity;
    }

    @Override
    public boolean validateAssets(final String errorPath, final Asset asset) {
        boolean valid = true;

        // Validating the asset is a capital asset
        if (!getAssetService().isCapitalAsset(asset)) {
            GlobalVariables.getMessageMap().putError(errorPath, CamsKeyConstants.Payment.ERROR_NON_CAPITAL_ASSET,
                    asset.getCapitalAssetNumber().toString());
            valid = false;
        }

        // Validating the asset hasn't been retired
        if (getAssetService().isAssetRetired(asset)) {
            GlobalVariables.getMessageMap().putError(errorPath,
                    CamsKeyConstants.Retirement.ERROR_NON_ACTIVE_ASSET_RETIREMENT,
                    asset.getCapitalAssetNumber().toString());
            valid = false;
        }
        return valid;
    }

    /**
     * This method determines whether or not an asset has different object sub type codes in its documents.
     *
     * @return true when the asset has payments with object codes that point to different object sub type codes
     */
    @Override
    public boolean hasDifferentObjectSubTypes(final AssetPaymentDocument document) {
        final List<String> subTypes;
        subTypes = new ArrayList<>(parameterService.getParameterValuesAsString(Asset.class,
                CamsParameterConstants.OBJECT_SUB_TYPE_GROUPS));

        final List<AssetPaymentDetail> assetPaymentDetails = document.getSourceAccountingLines();
        List<String> validObjectSubTypes;

        /*
         * Expected system parameter elements (object sub types). [BD,BF] [CM,CF,CO] [UC,UF,UO] [LI,LF]
         */

        // Getting the list of object sub type codes from the asset payments on the jsp.
        final List<String> objectSubTypeList = new ArrayList<>();
        for (final AssetPaymentDetail assetPaymentDetail : assetPaymentDetails) {
            assetPaymentDetail.refreshReferenceObject(CamsPropertyConstants.AssetPaymentDetail.OBJECT_CODE);
            if (ObjectUtils.isNull(assetPaymentDetail.getObjectCode())) {
                return false;
            }
            objectSubTypeList.add(assetPaymentDetail.getObjectCode().getFinancialObjectSubTypeCode());
        }

        if (!assetService.isObjectSubTypeCompatible(objectSubTypeList)) {
            return true;
        }

        final List<AssetPaymentAssetDetail> assetPaymentAssetDetails = document.getAssetPaymentAssetDetail();
        for (final AssetPaymentAssetDetail assetPaymentAssetDetail : assetPaymentAssetDetails) {
            assetPaymentAssetDetail.getAsset().refreshReferenceObject(CamsPropertyConstants.Asset.ASSET_PAYMENTS);
            final List<AssetPayment> assetPayments = assetPaymentAssetDetail.getAsset().getAssetPayments();

            // Comparing against the already approved asset payments
            if (!assetPayments.isEmpty()) {
                for (final AssetPayment assetPayment : assetPayments) {
                    final String paymentSubObjectType = assetPayment.getFinancialObject().getFinancialObjectSubTypeCode();

                    validObjectSubTypes = new ArrayList<>();
                    for (final String subType : subTypes) {
                        validObjectSubTypes = Arrays.asList(StringUtils.split(subType, ","));
                        if (validObjectSubTypes.contains(paymentSubObjectType)) {
                            break;
                        }
                        validObjectSubTypes = new ArrayList<>();
                    }
                    if (validObjectSubTypes.isEmpty()) {
                        validObjectSubTypes.add(paymentSubObjectType);
                    }

                    // Comparing the same asset payment document
                    for (final AssetPaymentDetail assetPaymentDetail : assetPaymentDetails) {
                        if (!validObjectSubTypes.contains(assetPaymentDetail.getObjectCode()
                                .getFinancialObjectSubTypeCode())) {
                            // Differences where found.
                            return true;
                        }
                    }
                }
            }
        }
        // If none object sub types are different...
        return false;
    }

    public void setBusinessObjectService(final BusinessObjectService businessObjectService) {
        this.businessObjectService = businessObjectService;
    }

    public BusinessObjectService getBusinessObjectService() {
        return businessObjectService;
    }

    public ParameterService getParameterService() {
        return parameterService;
    }

    public void setParameterService(final ParameterService parameterService) {
        this.parameterService = parameterService;
    }

    public AssetPaymentDao getAssetPaymentDao() {
        return assetPaymentDao;
    }

    public void setAssetPaymentDao(final AssetPaymentDao assetPaymentDao) {
        this.assetPaymentDao = assetPaymentDao;
    }

    public ObjectCodeService getObjectCodeService() {
        return objectCodeService;
    }

    public void setObjectCodeService(final ObjectCodeService objectCodeService) {
        this.objectCodeService = objectCodeService;
    }

    public UniversityDateService getUniversityDateService() {
        return universityDateService;
    }

    public void setUniversityDateService(final UniversityDateService universityDateService) {
        this.universityDateService = universityDateService;
    }

    public AssetRetirementService getAssetRetirementService() {
        return assetRetirementService;
    }

    public void setAssetRetirementService(final AssetRetirementService assetRetirementService) {
        this.assetRetirementService = assetRetirementService;
    }

    public AssetService getAssetService() {
        return assetService;
    }

    public void setAssetService(final AssetService assetService) {
        this.assetService = assetService;
    }

    @Override
    public AssetPaymentAllocationType getAssetDistributionType(final String distributionCode) {
        final HashMap<String, String> keys = new HashMap<>();
        keys.put("distributionCode", distributionCode);
        return getBusinessObjectService().findByPrimaryKey(AssetPaymentAllocationType.class, keys);
    }

    @Override
    public void setCurrentDepreciation(final String fiscalMonth, final KualiDecimal depreciationAmount, final AssetPayment assetPayment) {
        switch (fiscalMonth) {
            case KFSConstants.MONTH1:
                assetPayment.setPeriod1Depreciation1Amount(depreciationAmount);
                break;
            case KFSConstants.MONTH2:
                assetPayment.setPeriod2Depreciation1Amount(depreciationAmount);
                break;
            case KFSConstants.MONTH3:
                assetPayment.setPeriod3Depreciation1Amount(depreciationAmount);
                break;
            case KFSConstants.MONTH4:
                assetPayment.setPeriod4Depreciation1Amount(depreciationAmount);
                break;
            case KFSConstants.MONTH5:
                assetPayment.setPeriod5Depreciation1Amount(depreciationAmount);
                break;
            case KFSConstants.MONTH6:
                assetPayment.setPeriod6Depreciation1Amount(depreciationAmount);
                break;
            case KFSConstants.MONTH7:
                assetPayment.setPeriod7Depreciation1Amount(depreciationAmount);
                break;
            case KFSConstants.MONTH8:
                assetPayment.setPeriod8Depreciation1Amount(depreciationAmount);
                break;
            case KFSConstants.MONTH9:
                assetPayment.setPeriod9Depreciation1Amount(depreciationAmount);
                break;
            case KFSConstants.MONTH10:
                assetPayment.setPeriod10Depreciation1Amount(depreciationAmount);
                break;
            case KFSConstants.MONTH11:
                assetPayment.setPeriod11Depreciation1Amount(depreciationAmount);
                break;
            case KFSConstants.MONTH12:
                assetPayment.setPeriod12Depreciation1Amount(depreciationAmount);
                break;
            default:
                break;
        }
    }
}
