/*
 * The Kuali Financial System, a comprehensive financial management system for higher education.
 *
 * Copyright 2005-2022 Kuali, Inc.
 *
 * This program is free software: you can redistribute it and/or modify
 * it under the terms of the GNU Affero General Public License as published by
 * the Free Software Foundation, either version 3 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU Affero General Public License for more details.
 *
 * You should have received a copy of the GNU Affero General Public License
 * along with this program.  If not, see <http://www.gnu.org/licenses/>.
 */
package org.kuali.kfs.module.cam.document.service.impl;

import org.apache.commons.lang3.StringUtils;
import org.apache.logging.log4j.LogManager;
import org.apache.logging.log4j.Logger;
import org.kuali.kfs.coa.businessobject.ObjectCode;
import org.kuali.kfs.coa.service.ObjectCodeService;
import org.kuali.kfs.coreservice.framework.parameter.ParameterService;
import org.kuali.kfs.kns.document.MaintenanceDocument;
import org.kuali.kfs.krad.document.Document;
import org.kuali.kfs.krad.exception.ValidationException;
import org.kuali.kfs.krad.service.BusinessObjectService;
import org.kuali.kfs.krad.util.ObjectUtils;
import org.kuali.kfs.module.cam.CamsConstants;
import org.kuali.kfs.module.cam.CamsParameterConstants;
import org.kuali.kfs.module.cam.CamsPropertyConstants;
import org.kuali.kfs.module.cam.businessobject.Asset;
import org.kuali.kfs.module.cam.businessobject.AssetGlobal;
import org.kuali.kfs.module.cam.businessobject.AssetGlobalDetail;
import org.kuali.kfs.module.cam.businessobject.AssetLocation;
import org.kuali.kfs.module.cam.businessobject.AssetLocationGlobalDetail;
import org.kuali.kfs.module.cam.businessobject.AssetPayment;
import org.kuali.kfs.module.cam.businessobject.AssetType;
import org.kuali.kfs.module.cam.document.service.AssetService;
import org.kuali.kfs.module.cam.document.service.PaymentSummaryService;
import org.kuali.kfs.sys.KFSConstants.DocumentStatusCodes;
import org.kuali.kfs.sys.KFSPropertyConstants;
import org.kuali.kfs.sys.businessobject.UniversityDate;
import org.kuali.kfs.sys.service.UniversityDateService;
import org.kuali.kfs.kew.api.WorkflowDocument;

import java.util.ArrayList;
import java.util.Arrays;
import java.util.Collection;
import java.util.HashMap;
import java.util.List;
import java.util.Map;
import java.util.Set;

public class AssetServiceImpl implements AssetService {

    private static final Logger LOG = LogManager.getLogger();

    protected BusinessObjectService businessObjectService;
    protected ObjectCodeService objectCodeService;
    protected ParameterService parameterService;
    protected PaymentSummaryService paymentSummaryService;
    private UniversityDateService universityDateService;

    @Override
    public boolean isAssetMovableCheckByAsset(final Asset asset) {
        asset.refreshReferenceObject(CamsPropertyConstants.Asset.CAPITAL_ASSET_TYPE);
        return asset.getCapitalAssetType().isMovingIndicator();
    }

    @Override
    public boolean isAssetDepreciationStarted(final Asset asset) {
        // check non-persistent field accumulatedDepreciation first since it's the sum of
        // assetPayment.accumulatedPrimaryDepreciationAmount. If it's not set yet, we'll check assetPayment one by one.
        if (ObjectUtils.isNotNull(asset.getAccumulatedDepreciation()) && asset.getAccumulatedDepreciation().isPositive()) {
            return true;
        } else {
            for (final AssetPayment assetPayment : asset.getAssetPayments()) {
                if (ObjectUtils.isNotNull(assetPayment.getAccumulatedPrimaryDepreciationAmount())
                        && assetPayment.getAccumulatedPrimaryDepreciationAmount().isPositive()) {
                    return true;
                }
            }
        }
        return false;
    }

    @Override
    public boolean isCapitalAsset(final Asset asset) {
        return parameterService.getParameterValuesAsString(Asset.class,
                CamsParameterConstants.CAPITAL_ASSET_STATUS_CODES).contains(asset.getInventoryStatusCode());
    }

    @Override
    public boolean isAssetRetired(final Asset asset) {
        return parameterService.getParameterValuesAsString(Asset.class,
                CamsParameterConstants.RETIRED_STATUS_CODES).contains(asset.getInventoryStatusCode());
    }

    @Override
    public boolean isInServiceDateChanged(final Asset oldAsset, final Asset newAsset) {
        return !(ObjectUtils.isNull(oldAsset.getCapitalAssetInServiceDate()) ? ObjectUtils.isNull(newAsset.getCapitalAssetInServiceDate()) : oldAsset.getCapitalAssetInServiceDate().equals(newAsset.getCapitalAssetInServiceDate()));
    }

    @Override
    public boolean isAssetFabrication(final MaintenanceDocument maintenanceDocument) {
        return maintenanceDocument.getNewMaintainableObject().getBusinessObject() instanceof Asset && maintenanceDocument.isNew();
    }

    @Override
    public boolean isAssetLoaned(final Asset asset) {
        return ObjectUtils.isNotNull(asset.getExpectedReturnDate()) && ObjectUtils.isNull(asset.getLoanReturnDate());
    }

    @Override
    public boolean isAssetTaggedInPriorFiscalYear(final Asset asset) {
        return StringUtils.isNotBlank(asset.getCampusTagNumber())
                && ObjectUtils.isNotNull(asset.getFinancialDocumentPostingYear())
                && !universityDateService.getCurrentFiscalYear().equals(asset.getFinancialDocumentPostingYear());
    }

    @Override
    public boolean isTagNumberCheckExclude(final Asset asset) {
        final String status = asset.getInventoryStatusCode();

        return StringUtils.equalsIgnoreCase(status, CamsConstants.InventoryStatusCode.CAPITAL_ASSET_RETIRED)
                || StringUtils.equalsIgnoreCase(status, CamsConstants.InventoryStatusCode.NON_CAPITAL_ASSET_RETIRED)
                || StringUtils.equalsIgnoreCase(asset.getCampusTagNumber(), CamsConstants.Asset.NON_TAGGABLE_ASSET);
    }

    @Override
    public boolean isOffCampusLocationEntered(final Asset asset) {
        final AssetLocation offCampus = asset.getOffCampusLocation();
        return StringUtils.isNotBlank(offCampus.getAssetLocationContactName())
                || StringUtils.isNotBlank(offCampus.getAssetLocationStreetAddress())
                || StringUtils.isNotBlank(offCampus.getAssetLocationCityName())
                || StringUtils.isNotBlank(offCampus.getAssetLocationStateCode())
                || StringUtils.isNotBlank(offCampus.getAssetLocationZipCode())
                || StringUtils.isNotBlank(offCampus.getAssetLocationCountryCode());
    }

    @Override
    public boolean isFinancialObjectSubTypeCodeChanged(final Asset oldAsset, final Asset newAsset) {
        return !StringUtils.equalsIgnoreCase(oldAsset.getFinancialObjectSubTypeCode(), newAsset.getFinancialObjectSubTypeCode());
    }

    @Override
    public boolean isAssetTypeCodeChanged(final Asset oldAsset, final Asset newAsset) {
        return !StringUtils.equalsIgnoreCase(oldAsset.getCapitalAssetTypeCode(), newAsset.getCapitalAssetTypeCode());
    }

    @Override
    public boolean isAssetDepreciableLifeLimitZero(final Asset asset) {
        asset.refreshReferenceObject(CamsPropertyConstants.Asset.CAPITAL_ASSET_TYPE);
        final AssetType capitalAssetType = asset.getCapitalAssetType();
        if (ObjectUtils.isNotNull(capitalAssetType)) {
            return Integer.valueOf(0).equals(capitalAssetType.getDepreciableLifeLimit());
        }
        return false;
    }

    @Override
    public boolean isCapitalAssetNumberDuplicate(final Long capitalAssetNumber1, final Long capitalAssetNumber2) {
        return capitalAssetNumber1 != null && capitalAssetNumber2 != null && capitalAssetNumber1.compareTo(
                capitalAssetNumber2) == 0;
    }

    /**
     * This method calls the service codes to calculate the summary fields for each asset
     *
     * @param asset
     */
    @Override
    public void setAssetSummaryFields(final Asset asset) {
        if (ObjectUtils.isNotNull(asset)) {
            asset.setFederalContribution(paymentSummaryService.calculateFederalContribution(asset));
            asset.setAccumulatedDepreciation(paymentSummaryService.calculatePrimaryAccumulatedDepreciation(asset));
            asset.setBookValue(paymentSummaryService.calculatePrimaryBookValue(asset));
        }
    }

    @Override
    public boolean isAssetMovableCheckByPayment(final String financialObjectSubTypeCode) {
        if (ObjectUtils.isNull(financialObjectSubTypeCode)) {
            return true;
        }

        if (parameterService.getParameterValuesAsString(Asset.class, CamsParameterConstants.MOVABLE_EQUIPMENT_OBJECT_SUB_TYPES).contains(financialObjectSubTypeCode)) {
            return true;
        } else if (parameterService.getParameterValuesAsString(Asset.class, CamsParameterConstants.NON_MOVABLE_EQUIPMENT_OBJECT_SUB_TYPES).contains(financialObjectSubTypeCode)) {
            return false;
        } else {
            throw new ValidationException("Could not determine movable or non-movable for this object sub-type code " + financialObjectSubTypeCode);
        }
    }

    @Override
    public boolean isAssetMovableCheckByPayment(final Asset asset) {
        final String financialObjectSubTypeCode = determineFinancialObjectSubTypeCode(asset);

        return isAssetMovableCheckByPayment(financialObjectSubTypeCode);
    }

    @Override
    public String determineFinancialObjectSubTypeCode(final Asset asset) {
        String financialObjectSubTypeCode = asset.getFinancialObjectSubTypeCode();

        if (ObjectUtils.isNotNull(asset.getAssetPayments()) && !asset.getAssetPayments().isEmpty()) {
            // use the last payment, so it's more recent and less likely to hit an out-dated object code
            final int size = asset.getAssetPayments().size();
            final AssetPayment lastAssetPayment = asset.getAssetPayments().get(size - 1);
            lastAssetPayment.refreshReferenceObject(CamsPropertyConstants.AssetPayment.FINANCIAL_OBJECT);
            final String chartCode = lastAssetPayment.getChartOfAccountsCode();
            final String finObjectCode = lastAssetPayment.getFinancialObjectCode();
            final ObjectCode objectCode = objectCodeService.getByPrimaryIdForCurrentYear(chartCode, finObjectCode);

            // if objectCode is null, we likely have hit an out-dated object code from some old payment
            if (ObjectUtils.isNotNull(objectCode)) {
                financialObjectSubTypeCode = objectCode.getFinancialObjectSubTypeCode();
            } else {
                LOG.warn(
                        "Possibly out-dated object code {} for chart {} for current fiscal year in payments for asset"
                        + " {}",
                        () -> finObjectCode,
                        () -> chartCode,
                        asset::getCapitalAssetNumber
                );
            }
        }
        return financialObjectSubTypeCode;
    }

    @Override
    public List<Asset> findActiveAssetsMatchingTagNumber(final String campusTagNumber) {
        final List<Asset> activeMatches = new ArrayList<>();
        // find all assets matching this tag number
        final Map<String, String> params = new HashMap<>();
        params.put(CamsPropertyConstants.Asset.CAMPUS_TAG_NUMBER, campusTagNumber);
        final Collection<Asset> tagMatches = businessObjectService.findMatching(Asset.class, params);
        if (tagMatches != null && !tagMatches.isEmpty()) {
            for (final Asset asset : tagMatches) {
                // if found matching, check if status is not retired
                if (!isAssetRetired(asset)) {
                    activeMatches.add(asset);
                }
            }
        }
        return activeMatches;
    }

    @Override
    public Collection<Asset> findAssetsMatchingTagNumber(final String campusTagNumber) {
        // find all assets matching this tag number
        final Map<String, String> params = new HashMap<>();
        params.put(CamsPropertyConstants.Asset.CAMPUS_TAG_NUMBER, campusTagNumber);
        return businessObjectService.findMatching(Asset.class, params);
    }

    @Override
    public boolean isObjectSubTypeCompatible(final List<String> financialObjectSubTypeCode) {
        if (financialObjectSubTypeCode == null || financialObjectSubTypeCode.size() <= 1) {
            return true;
        }

        final List<String> subTypes = new ArrayList<>(parameterService.getParameterValuesAsString(Asset.class,
                CamsParameterConstants.OBJECT_SUB_TYPE_GROUPS));
        final String firstObjectSubType = financialObjectSubTypeCode.get(0);
        List<String> validObjectSubTypes = new ArrayList<>();

        // Get the set for compatible object sub type code by the first financial object sub type code
        for (final String subType : subTypes) {
            if (subType.contains(firstObjectSubType)) {
                validObjectSubTypes = Arrays.asList(StringUtils.split(subType, ","));
                break;
            }
        }

        if (validObjectSubTypes.isEmpty()) {
            validObjectSubTypes.add(firstObjectSubType);
        }

        // Check in the whole list if all object sub type code are compatible.
        for (final String subTypeCode : financialObjectSubTypeCode) {
            if (!validObjectSubTypes.contains(subTypeCode)) {
                return false;
            }
        }

        return true;
    }

    @Override
    public void setSeparateHistory(final Asset asset) {
        // Find the asset this was separated from. It should only be one, error if more
        final Map<String, String> paramsAssetGlobalDetail = new HashMap<>();
        paramsAssetGlobalDetail.put(CamsPropertyConstants.AssetGlobalDetail.CAPITAL_ASSET_NUMBER,
                asset.getCapitalAssetNumber().toString());
        final Collection<AssetGlobalDetail> assetGlobalDetails = businessObjectService.findMatching(AssetGlobalDetail.class,
                paramsAssetGlobalDetail);

        if (assetGlobalDetails.size() > 1) {
            throw new IllegalStateException("Asset #" + asset.getCapitalAssetNumber().toString() +
                    " was created from more then one asset document.");
        } else if (assetGlobalDetails.size() == 1) {
            // Find the document associated to that
            final Map<String, String> paramsAssetGlobal = new HashMap<>();
            paramsAssetGlobal.put(CamsPropertyConstants.AssetGlobal.DOCUMENT_NUMBER,
                    assetGlobalDetails.iterator().next().getDocumentNumber());
            final AssetGlobal assetGlobal = businessObjectService.findByPrimaryKey(AssetGlobal.class, paramsAssetGlobal);

            // Only set it if it is in status approved
            if (DocumentStatusCodes.APPROVED.equals(assetGlobal.getDocumentHeader().getFinancialDocumentStatusCode())) {
                asset.setSeparateHistory(assetGlobal);
            }
        }

        // Else no history, just return
    }

    @Override
    public List<String> getDocumentNumbersThatSeparatedThisAsset(final Long capitalAssetNumber) {
        final Map<String, String> paramsAssetGlobal = new HashMap<>();
        paramsAssetGlobal.put(CamsPropertyConstants.AssetGlobal.SEPARATE_SOURCE_CAPITAL_ASSET_NUMBER,
                capitalAssetNumber.toString());
        final Collection<AssetGlobal> assetGlobals = businessObjectService.findMatching(AssetGlobal.class, paramsAssetGlobal);

        final List<String> separateDocumentNumbers = new ArrayList<>();
        for (final AssetGlobal assetGlobal : assetGlobals) {
            if (DocumentStatusCodes.APPROVED.equals(
                    assetGlobal.getDocumentHeader().getFinancialDocumentStatusCode())) {
                separateDocumentNumbers.add(assetGlobal.getDocumentNumber());
            }
        }

        return separateDocumentNumbers;
    }

    /**
     * sets the posting year and posting month based on the asset creation date
     *
     * @param asset
     * @return none
     */
    @Override
    public void setFiscalPeriod(final Asset asset) {
        if (asset.getCreateDate() == null) {
            return;
        }

        final Map<String, Object> primaryKeys = new HashMap<>();
        primaryKeys.put(KFSPropertyConstants.UNIVERSITY_DATE, asset.getCreateDate());
        final UniversityDate universityDate = businessObjectService.findByPrimaryKey(UniversityDate.class, primaryKeys);
        if (universityDate != null) {
            asset.setFinancialDocumentPostingYear(universityDate.getUniversityFiscalYear());
            asset.setFinancialDocumentPostingPeriodCode(universityDate.getUniversityFiscalAccountingPeriod());
        }
    }

    @Override
    public Set<String> getCurrentRouteLevels(final WorkflowDocument workflowDocument) {
        return workflowDocument.getCurrentNodeNames();
    }

    @Override
    public boolean isDocumentEnrouting(final Document document) {
        return document.getDocumentHeader().getWorkflowDocument().isEnroute();
    }

    @Override
    public boolean hasCapitalAssetLocationDetailsChanged(final AssetLocationGlobalDetail assetLocationGlobalDetail) {
        boolean success = false;

        if (ObjectUtils.isNotNull(assetLocationGlobalDetail.getCapitalAssetNumber())) {
            assetLocationGlobalDetail.refreshReferenceObject(CamsPropertyConstants.AssetLocationGlobalDetail.ASSET);
            final Asset asset = assetLocationGlobalDetail.getAsset();

            if (ObjectUtils.isNotNull(assetLocationGlobalDetail.getAsset())) {
                if (!StringUtils.equalsIgnoreCase(asset.getCampusCode(), assetLocationGlobalDetail.getCampusCode())) {
                    success = true;
                }
                if (!StringUtils.equalsIgnoreCase(asset.getBuildingCode(), assetLocationGlobalDetail.getBuildingCode())) {
                    success = true;
                }
                if (!StringUtils.equalsIgnoreCase(asset.getBuildingRoomNumber(), assetLocationGlobalDetail.getBuildingRoomNumber())) {
                    success = true;
                }
                if (!StringUtils.equalsIgnoreCase(asset.getBuildingSubRoomNumber(), assetLocationGlobalDetail.getBuildingSubRoomNumber())) {
                    success = true;
                }
                if (!StringUtils.equalsIgnoreCase(asset.getCampusTagNumber(), assetLocationGlobalDetail.getCampusTagNumber())) {
                    success = true;
                }
            }
        }

        return success;
    }

    @Override
    public boolean hasAssetLocationChanged(final Asset oldAsset, final Asset newAsset) {
        final AssetLocation oldOffCampusLocation = oldAsset.getOffCampusLocation();
        final AssetLocation newOffCampusLocation = newAsset.getOffCampusLocation();
        if (ObjectUtils.isNull(oldOffCampusLocation) || ObjectUtils.isNull(newOffCampusLocation)) {
            return false;
        }

        return !StringUtils.equalsIgnoreCase(oldAsset.getCampusCode(), newAsset.getCampusCode())
                || !StringUtils.equalsIgnoreCase(oldAsset.getBuildingCode(), newAsset.getBuildingCode())
                || !StringUtils.equalsIgnoreCase(oldAsset.getBuildingRoomNumber(), newAsset.getBuildingRoomNumber())
                || !StringUtils.equalsIgnoreCase(oldAsset.getBuildingSubRoomNumber(), newAsset.getBuildingSubRoomNumber())
                || !StringUtils.equalsIgnoreCase(oldAsset.getCampusTagNumber(), newAsset.getCampusTagNumber())
                || !StringUtils.equalsIgnoreCase(oldOffCampusLocation.getAssetLocationContactName(),
                    newOffCampusLocation.getAssetLocationContactName())
                || !StringUtils.equalsIgnoreCase(oldOffCampusLocation.getAssetLocationStreetAddress(),
                    newOffCampusLocation.getAssetLocationStreetAddress())
                || !StringUtils.equalsIgnoreCase(oldOffCampusLocation.getAssetLocationCityName(),
                    newOffCampusLocation.getAssetLocationCityName())
                || !StringUtils.equalsIgnoreCase(oldOffCampusLocation.getAssetLocationStateCode(),
                    newOffCampusLocation.getAssetLocationStateCode())
                || !StringUtils.equalsIgnoreCase(oldOffCampusLocation.getAssetLocationZipCode(),
                    newOffCampusLocation.getAssetLocationZipCode())
                || !StringUtils.equalsIgnoreCase(oldOffCampusLocation.getAssetLocationCountryCode(),
                    newOffCampusLocation.getAssetLocationCountryCode());
    }

    public BusinessObjectService getBusinessObjectService() {
        return businessObjectService;
    }

    public void setBusinessObjectService(final BusinessObjectService businessObjectService) {
        this.businessObjectService = businessObjectService;
    }

    public ObjectCodeService getObjectCodeService() {
        return objectCodeService;
    }

    public void setObjectCodeService(final ObjectCodeService objectCodeService) {
        this.objectCodeService = objectCodeService;
    }

    public ParameterService getParameterService() {
        return parameterService;
    }

    public void setParameterService(final ParameterService parameterService) {
        this.parameterService = parameterService;
    }

    public PaymentSummaryService getPaymentSummaryService() {
        return paymentSummaryService;
    }

    public void setPaymentSummaryService(final PaymentSummaryService paymentSummaryService) {
        this.paymentSummaryService = paymentSummaryService;
    }

    public void setUniversityDateService(final UniversityDateService universityDateService) {
        this.universityDateService = universityDateService;
    }
}
