/*
 * The Kuali Financial System, a comprehensive financial management system for higher education.
 *
 * Copyright 2005-2022 Kuali, Inc.
 *
 * This program is free software: you can redistribute it and/or modify
 * it under the terms of the GNU Affero General Public License as published by
 * the Free Software Foundation, either version 3 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU Affero General Public License for more details.
 *
 * You should have received a copy of the GNU Affero General Public License
 * along with this program.  If not, see <http://www.gnu.org/licenses/>.
 */
package org.kuali.kfs.module.cam.document.web.struts;

import org.apache.commons.lang3.StringUtils;
import org.apache.logging.log4j.LogManager;
import org.apache.logging.log4j.Logger;
import org.apache.struts.action.ActionForm;
import org.apache.struts.action.ActionForward;
import org.apache.struts.action.ActionMapping;
import org.apache.struts.upload.FormFile;
import org.kuali.kfs.kns.util.KNSGlobalVariables;
import org.kuali.kfs.krad.exception.ValidationException;
import org.kuali.kfs.krad.util.GlobalVariables;
import org.kuali.kfs.module.cam.CamsKeyConstants;
import org.kuali.kfs.module.cam.batch.AssetBarcodeInventoryInputFileType;
import org.kuali.kfs.module.cam.batch.service.AssetBarcodeInventoryInputFileService;
import org.kuali.kfs.sys.KFSConstants;
import org.kuali.kfs.sys.KFSKeyConstants;
import org.kuali.kfs.sys.batch.BatchInputFileSetType;
import org.kuali.kfs.sys.batch.service.BatchInputFileSetService;
import org.kuali.kfs.sys.businessobject.BatchUpload;
import org.kuali.kfs.sys.context.SpringContext;
import org.kuali.kfs.sys.exception.FileStorageException;
import org.kuali.kfs.sys.web.struts.KualiBatchInputFileSetAction;
import org.kuali.kfs.sys.web.struts.KualiBatchInputFileSetForm;
import org.kuali.kfs.core.api.util.ConcreteKeyValue;
import org.kuali.kfs.core.api.util.KeyValue;

import javax.servlet.http.HttpServletRequest;
import javax.servlet.http.HttpServletResponse;
import java.io.InputStream;
import java.util.ArrayList;
import java.util.HashMap;
import java.util.List;
import java.util.Map;

/**
 * Action class for the CAMS Barcode Inventory upload.
 */
public class AssetBarCodeInventoryInputFileAction extends KualiBatchInputFileSetAction {

    private static final Logger LOG = LogManager.getLogger();

    /**
     * Overridden because I needed to validate the file type is correct.
     */
    @Override
    public ActionForward save(
            final ActionMapping mapping, final ActionForm form, final HttpServletRequest request,
            final HttpServletResponse response) throws Exception {
        final BatchUpload batchUpload = ((AssetBarCodeInventoryInputFileForm) form).getBatchUpload();
        final AssetBarcodeInventoryInputFileType batchType =
                (AssetBarcodeInventoryInputFileType) retrieveBatchInputFileSetTypeImpl(batchUpload.getBatchInputTypeName());

        final Map<String, FormFile> uploadedFiles = ((KualiBatchInputFileSetForm) form).getUploadedFiles();

        final String fileTypeExtension = batchType.getFileExtension();
        final String fileName = uploadedFiles.get(fileTypeExtension.substring(1)).getFileName();

        //Validating the file type is the correct one before saving.
        if (StringUtils.isNotBlank(fileName) && !fileName.endsWith(fileTypeExtension)) {
            GlobalVariables.getMessageMap().putError(KFSConstants.GLOBAL_ERRORS,
                    CamsKeyConstants.BarcodeInventory.ERROR_INVALID_FILE_TYPE);
            return mapping.findForward(KFSConstants.MAPPING_BASIC);
        }

        final String uploadDescription = ((AssetBarCodeInventoryInputFileForm) form).getUploadDescription();

        boolean requiredValuesForFilesMissing = false;
        if (StringUtils.isBlank(batchUpload.getFileUserIdentifer())) {
            GlobalVariables.getMessageMap().putError(KFSConstants.GLOBAL_ERRORS,
                    KFSKeyConstants.ERROR_BATCH_UPLOAD_NO_FILE_SET_IDENTIFIER_SELECTED);
            requiredValuesForFilesMissing = true;
        }

        if (StringUtils.isBlank(uploadDescription)) {
            GlobalVariables.getMessageMap().putError(KFSConstants.GLOBAL_ERRORS,
                    KFSKeyConstants.ERROR_DOCUMENT_NO_DESCRIPTION);
            requiredValuesForFilesMissing = true;
        }

        final AssetBarcodeInventoryInputFileService batchInputFileSetService = SpringContext.getBean(AssetBarcodeInventoryInputFileService.class);

        if (!batchInputFileSetService.isFileUserIdentifierProperlyFormatted(batchUpload.getFileUserIdentifer())) {
            GlobalVariables.getMessageMap().putError(KFSConstants.GLOBAL_ERRORS,
                    KFSKeyConstants.ERROR_BATCH_UPLOAD_FILE_SET_IDENTIFIER_BAD_FORMAT);
            requiredValuesForFilesMissing = true;
        }

        final Map<String, InputStream> typeToStreamMap = new HashMap<>();

        for (final String fileType : uploadedFiles.keySet()) {
            final FormFile uploadedFile = uploadedFiles.get(fileType);
            if (uploadedFile == null || uploadedFile.getInputStream() == null
                    || uploadedFile.getInputStream().available() == 0) {
                GlobalVariables.getMessageMap().putError(KFSConstants.GLOBAL_ERRORS,
                        KFSKeyConstants.ERROR_BATCH_UPLOAD_NO_FILE_SELECTED_SAVE_FOR_FILE_TYPE,
                        batchType.getFileTypeDescription().get(fileType));
                requiredValuesForFilesMissing = true;
            } else {
                typeToStreamMap.put(fileType, uploadedFile.getInputStream());
            }
        }

        if (requiredValuesForFilesMissing) {
            return mapping.findForward(KFSConstants.MAPPING_BASIC);
        }

        try {
            batchInputFileSetService.save(GlobalVariables.getUserSession().getPerson(), batchType,
                    batchUpload.getFileUserIdentifer(), typeToStreamMap, (AssetBarCodeInventoryInputFileForm) form);
        } catch (final FileStorageException e) {
            LOG.error("Error occurred while trying to save file set (probably tried to save a file that already exists).", e);
            GlobalVariables.getMessageMap().putError(KFSConstants.GLOBAL_ERRORS,
                    KFSKeyConstants.ERROR_BATCH_UPLOAD_FILE_SAVE_ERROR, e.getMessage());
            return mapping.findForward(KFSConstants.MAPPING_BASIC);
        } catch (final ValidationException e) {
            LOG.error("Error occurred while trying to validate file set.", e);
            GlobalVariables.getMessageMap().putError(KFSConstants.GLOBAL_ERRORS,
                    KFSKeyConstants.ERROR_BATCH_UPLOAD_FILE_VALIDATION_ERROR);
            return mapping.findForward(KFSConstants.MAPPING_BASIC);
        }
        KNSGlobalVariables.getMessageList().add(KFSKeyConstants.MESSAGE_BATCH_UPLOAD_SAVE_SUCCESSFUL);

        return mapping.findForward(KFSConstants.MAPPING_BASIC);

    }

    /**
     * Builds list of filenames that the user has permission to manage, and populates the form member. Throws an
     * exception if the batch file set type is not active. Sets the title key from the batch input type. This method
     * must be called before the action handler to ensure proper authorization.
     */
    @Override
    public void setupForm(final KualiBatchInputFileSetForm form) {
        final BatchInputFileSetType batchInputFileSetType = retrieveBatchInputFileSetTypeImpl(form.getBatchUpload().getBatchInputTypeName());

        if (batchInputFileSetType == null) {
            LOG.error(
                    "Batch input type implementation not found for id {}",
                    () -> form.getBatchUpload().getBatchInputTypeName()
            );
            throw new RuntimeException("Batch input type implementation not found for id " +
                    form.getBatchUpload().getBatchInputTypeName());
        }

        if (!SpringContext.getBean(BatchInputFileSetService.class).isBatchInputTypeActive(batchInputFileSetType)) {
            throw new RuntimeException("Batch input file set type is not active.");
        }
        form.setBatchInputFileSetType(batchInputFileSetType);

        final List<KeyValue> fileTypes = new ArrayList<>();
        fileTypes.add(new ConcreteKeyValue("", "Select a file type to download"));

        for (final String fileAlias : batchInputFileSetType.getFileTypes()) {
            fileTypes.add(new ConcreteKeyValue(fileAlias, batchInputFileSetType.getFileTypeDescription().get(fileAlias)));
        }
        form.setFileTypes(fileTypes);

        form.setTitleKey(batchInputFileSetType.getTitleKey());
    }
}
