/*
 * The Kuali Financial System, a comprehensive financial management system for higher education.
 *
 * Copyright 2005-2022 Kuali, Inc.
 *
 * This program is free software: you can redistribute it and/or modify
 * it under the terms of the GNU Affero General Public License as published by
 * the Free Software Foundation, either version 3 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU Affero General Public License for more details.
 *
 * You should have received a copy of the GNU Affero General Public License
 * along with this program.  If not, see <http://www.gnu.org/licenses/>.
 */
package org.kuali.kfs.module.cam.util;

import org.apache.commons.collections4.Closure;
import org.apache.commons.collections4.Predicate;
import org.apache.commons.lang3.StringUtils;
import org.kuali.kfs.module.cam.CamsConstants;
import org.kuali.kfs.module.cam.businessobject.BarcodeInventoryErrorDetail;
import org.kuali.kfs.module.cam.document.BarcodeInventoryErrorDocument;

/**
 * Helps filter out records from a collection of BCIE and replace its elements with the inputted data
 */
public class BarcodeInventoryErrorDetailPredicate implements Predicate, Closure {

    private final BarcodeInventoryErrorDocument doc;

    public BarcodeInventoryErrorDetailPredicate(final BarcodeInventoryErrorDocument doc) {
        this.doc = doc;
    }

    @Override
    public boolean evaluate(final Object object) {
        boolean satisfies = true;

        if (object instanceof BarcodeInventoryErrorDetail) {

            final BarcodeInventoryErrorDetail detail = (BarcodeInventoryErrorDetail) object;

            // It will only replace when the status is equals to error
            if (detail.getErrorCorrectionStatusCode().equals(CamsConstants.BarCodeInventoryError.STATUS_CODE_ERROR)) {
                if (doc.getCurrentTagNumber() != null && StringUtils.isNotBlank(doc.getCurrentTagNumber())) {
                    if (!StringUtils.equals(doc.getCurrentTagNumber(), detail.getAssetTagNumber())) {
                        satisfies = false;
                    }
                }

                if (doc.getCurrentScanCode() != null && StringUtils.isNotBlank(doc.getCurrentScanCode())) {
                    satisfies = doc.getCurrentScanCode().equals("Y") && detail.isUploadScanIndicator();
                }

                if (doc.getCurrentCampusCode() != null && StringUtils.isNotBlank(doc.getCurrentCampusCode())) {
                    if (!StringUtils.equals(doc.getCurrentCampusCode(), detail.getCampusCode())) {
                        satisfies = false;
                    }
                }

                if (doc.getCurrentBuildingNumber() != null && StringUtils.isNotBlank(doc.getCurrentBuildingNumber())) {
                    if (!StringUtils.equals(doc.getCurrentBuildingNumber(), detail.getBuildingCode())) {
                        satisfies = false;
                    }
                }

                if (doc.getCurrentRoom() != null && StringUtils.isNotBlank(doc.getCurrentRoom())) {
                    if (!StringUtils.equals(doc.getCurrentRoom(), detail.getBuildingRoomNumber())) {
                        satisfies = false;
                    }
                }

                if (doc.getCurrentSubroom() != null && StringUtils.isNotBlank(doc.getCurrentSubroom())) {
                    if (!StringUtils.equals(doc.getCurrentSubroom(), detail.getBuildingSubRoomNumber())) {
                        satisfies = false;
                    }
                }

                if (doc.getCurrentConditionCode() != null && StringUtils.isNotBlank(doc.getCurrentConditionCode())) {
                    if (!StringUtils.equals(doc.getCurrentConditionCode(), detail.getAssetConditionCode())) {
                        satisfies = false;
                    }
                }
            } else {
                satisfies = false;
            }
        }
        return satisfies;
    }

    /**
     * @see org.apache.commons.collections4.Closure#execute(java.lang.Object)
     */
    @Override
    public void execute(final Object object) {
        if (evaluate(object)) {
            final BarcodeInventoryErrorDetail detail = (BarcodeInventoryErrorDetail) object;
            if (doc.getNewCampusCode() != null && StringUtils.isNotBlank(doc.getNewCampusCode())) {
                detail.setCampusCode(doc.getNewCampusCode());
            }

            if (doc.getNewBuildingNumber() != null && StringUtils.isNotBlank(doc.getNewBuildingNumber())) {
                detail.setBuildingCode(doc.getNewBuildingNumber());
            }

            if (doc.getNewRoom() != null && StringUtils.isNotBlank(doc.getNewRoom())) {
                detail.setBuildingRoomNumber(doc.getNewRoom());
            }

            if (doc.getNewSubroom() != null && StringUtils.isNotBlank(doc.getNewSubroom())) {
                detail.setBuildingSubRoomNumber(doc.getNewSubroom());
            }

            if (doc.getNewConditionCode() != null && StringUtils.isNotBlank(doc.getNewConditionCode())) {
                detail.setAssetConditionCode(doc.getNewConditionCode());
            }
        }
    }
}
