/*
 * The Kuali Financial System, a comprehensive financial management system for higher education.
 *
 * Copyright 2005-2021 Kuali, Inc.
 *
 * This program is free software: you can redistribute it and/or modify
 * it under the terms of the GNU Affero General Public License as published by
 * the Free Software Foundation, either version 3 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU Affero General Public License for more details.
 *
 * You should have received a copy of the GNU Affero General Public License
 * along with this program.  If not, see <http://www.gnu.org/licenses/>.
 */
package org.kuali.kfs.module.cam.util.distribution;

import org.kuali.kfs.module.cam.businessobject.AssetPaymentAssetDetail;
import org.kuali.kfs.module.cam.businessobject.AssetPaymentDetail;
import org.kuali.kfs.module.cam.document.AssetPaymentDocument;
import org.kuali.kfs.core.api.util.type.KualiDecimal;

import java.util.Collection;
import java.util.List;
import java.util.Map;

/**
 * Provides distribution of asset payments to assets.
 */
public abstract class AssetDistribution {

    protected AssetPaymentDocument doc;

    /**
     * @param doc
     */
    public AssetDistribution(AssetPaymentDocument doc) {
        super();
        this.doc = doc;
    }

    /**
     * @return list of payment details
     */
    protected List<AssetPaymentDetail> getAssetPaymentDetailLines() {
        return doc.getSourceAccountingLines();
    }

    /**
     * Retrieve the asset payment distribution hierarchy
     * <p>
     * {@code
     * Source Line Key
     * |
     * +-- Asset 1
     * |     |
     * |     +-- Allocated Amount
     * |
     * +-- Asset 2
     * |
     * +-- Allocated Amount
     * }
     *
     * @return A map of each asset detail with its associated allocation
     */
    public abstract Map<String, Map<AssetPaymentAssetDetail, KualiDecimal>> getAssetPaymentDistributions();

    /**
     * Apply the distributions/allocations to the payment docs
     */
    public void applyDistributionsToDocument() {

        // Reset the allocated amounts so we are populating fresh
        for (AssetPaymentAssetDetail apad : doc.getAssetPaymentAssetDetail()) {
            apad.setAllocatedAmount(KualiDecimal.ZERO);
        }

        // iterate all the distributions
        Collection<Map<AssetPaymentAssetDetail, KualiDecimal>> values = getAssetPaymentDistributions().values();
        for (Map<AssetPaymentAssetDetail, KualiDecimal> assetDistribution : values) {
            for (AssetPaymentAssetDetail apad : assetDistribution.keySet()) {
                KualiDecimal amount = apad.getAllocatedAmount().add(assetDistribution.get(apad));
                apad.setAllocatedAmount(amount);
            }
        }

    }
}
